"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PluginLoader = void 0;
const fs = require("fs");
const path = require("path");
const hook = require("require-in-the-middle");
const constants_1 = require("../constants");
var HookState;
(function (HookState) {
    HookState[HookState["UNINITIALIZED"] = 0] = "UNINITIALIZED";
    HookState[HookState["ENABLED"] = 1] = "ENABLED";
    HookState[HookState["DISABLED"] = 2] = "DISABLED";
})(HookState || (HookState = {}));
class PluginLoader {
    constructor(logger, tracer) {
        this.plugins = [];
        this.hookState = HookState.UNINITIALIZED;
        this.tracer = tracer;
        this.logger = logger;
    }
    static defaultPackageName(moduleName) {
        return `${constants_1.Constants.OPENCENSUS_SCOPE}/${constants_1.Constants.DEFAULT_PLUGIN_PACKAGE_NAME_PREFIX}-${moduleName}`;
    }
    static defaultPluginsFromArray(modulesToPatch) {
        const plugins = modulesToPatch.reduce((plugins, moduleName) => {
            plugins[moduleName] = PluginLoader.defaultPackageName(moduleName);
            return plugins;
        }, {});
        return plugins;
    }
    getPackageVersion(name, basedir) {
        let version = '';
        if (basedir) {
            const pkgJson = path.join(basedir, 'package.json');
            try {
                version = JSON.parse(fs.readFileSync(pkgJson).toString()).version;
            }
            catch (e) {
                this.logger.error('could not get version of %s module: %s', name, e.message);
            }
        }
        else {
            version = process.versions.node;
        }
        return version;
    }
    loadPlugins(pluginList) {
        if (this.hookState === HookState.UNINITIALIZED) {
            hook(Object.keys(pluginList), (exports, name, basedir) => {
                if (this.hookState !== HookState.ENABLED) {
                    return exports;
                }
                const plugin = pluginList[name];
                const version = this.getPackageVersion(name, basedir);
                this.logger.info('trying loading %s.%s', name, version);
                if (!version) {
                    return exports;
                }
                this.logger.debug('applying patch to %s@%s module', name, version);
                let moduleName;
                let moduleConfig = {};
                if (typeof plugin === 'string') {
                    moduleName = plugin;
                }
                else {
                    moduleConfig = plugin.config;
                    moduleName = plugin.module;
                }
                this.logger.debug('using package %s to patch %s', moduleName, name);
                try {
                    const plugin = require(moduleName).plugin;
                    this.plugins.push(plugin);
                    return plugin.enable(exports, this.tracer, version, moduleConfig, basedir);
                }
                catch (e) {
                    this.logger.error('could not load plugin %s of module %s. Error: %s', moduleName, name, e.message);
                    return exports;
                }
            });
        }
        this.hookState = HookState.ENABLED;
    }
    unloadPlugins() {
        for (const plugin of this.plugins) {
            plugin.disable();
        }
        this.plugins = [];
        this.hookState = HookState.DISABLED;
    }
    static set searchPathForTest(searchPath) {
        module.paths.push(searchPath);
    }
}
exports.PluginLoader = PluginLoader;
//# sourceMappingURL=data:application/json;base64,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