"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateDataFrameMeta = exports.isGeoPoint = exports.getUniqueValuesForRawAggs = exports.getTimeField = exports.getRawQueryString = exports.getRawDataFrame = exports.getRawAggs = exports.getFieldType = exports.getAggConfigForRawAggs = exports.getAggConfig = exports.formatTimePickerDate = exports.formatFieldValue = exports.dataFrameToSpec = exports.createDataFrame = exports.convertResult = void 0;
var _datemath = _interopRequireDefault(require("@opensearch/datemath"));
var _types = require("./types");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Returns the raw data frame from the search request.
 *
 * @param searchRequest - search request object.
 * @returns dataframe
 */
const getRawDataFrame = searchRequest => {
  var _searchRequest$params;
  return (_searchRequest$params = searchRequest.params) === null || _searchRequest$params === void 0 || (_searchRequest$params = _searchRequest$params.body) === null || _searchRequest$params === void 0 ? void 0 : _searchRequest$params.df;
};

/**
 * Returns the raw query string from the search request.
 * Gets current state query if exists, otherwise gets the initial query.
 *
 * @param searchRequest - search request object
 * @returns query string
 */
exports.getRawDataFrame = getRawDataFrame;
const getRawQueryString = searchRequest => {
  var _searchRequest$params2, _searchRequest$params3, _searchRequest$params4;
  return (_searchRequest$params2 = (_searchRequest$params3 = searchRequest.params) === null || _searchRequest$params3 === void 0 || (_searchRequest$params3 = _searchRequest$params3.body) === null || _searchRequest$params3 === void 0 || (_searchRequest$params3 = _searchRequest$params3.query) === null || _searchRequest$params3 === void 0 || (_searchRequest$params3 = _searchRequest$params3.queries[1]) === null || _searchRequest$params3 === void 0 ? void 0 : _searchRequest$params3.query) !== null && _searchRequest$params2 !== void 0 ? _searchRequest$params2 : (_searchRequest$params4 = searchRequest.params) === null || _searchRequest$params4 === void 0 || (_searchRequest$params4 = _searchRequest$params4.body) === null || _searchRequest$params4 === void 0 || (_searchRequest$params4 = _searchRequest$params4.query) === null || _searchRequest$params4 === void 0 || (_searchRequest$params4 = _searchRequest$params4.queries[0]) === null || _searchRequest$params4 === void 0 ? void 0 : _searchRequest$params4.query;
};

/**
 * Returns the raw aggregations from the search request.
 *
 * @param searchRequest - search request object
 * @returns aggregations
 */
exports.getRawQueryString = getRawQueryString;
const getRawAggs = searchRequest => {
  var _searchRequest$params5;
  return (_searchRequest$params5 = searchRequest.params) === null || _searchRequest$params5 === void 0 || (_searchRequest$params5 = _searchRequest$params5.body) === null || _searchRequest$params5 === void 0 ? void 0 : _searchRequest$params5.aggs;
};

/**
 * Returns the unique values for raw aggregations. This is used
 * with `other-filter` aggregation. To get the values that were not
 * included in the aggregation response prior to this request.
 *
 * @param rawAggs - raw aggregations object
 * @returns object containing the field and its unique values
 */
exports.getRawAggs = getRawAggs;
const getUniqueValuesForRawAggs = rawAggs => {
  var _rawAggs$filters;
  const filters = (_rawAggs$filters = rawAggs.filters) === null || _rawAggs$filters === void 0 || (_rawAggs$filters = _rawAggs$filters.filters) === null || _rawAggs$filters === void 0 || (_rawAggs$filters = _rawAggs$filters['']) === null || _rawAggs$filters === void 0 || (_rawAggs$filters = _rawAggs$filters.bool) === null || _rawAggs$filters === void 0 ? void 0 : _rawAggs$filters.must_not;
  if (!filters || !Array.isArray(filters)) {
    return null;
  }
  const values = [];
  let field;
  filters.forEach(agg => {
    Object.values(agg).forEach(aggValue => {
      Object.entries(aggValue).forEach(([key, value]) => {
        field = key;
        values.push(value);
      });
    });
  });
  return {
    field,
    values
  };
};

/**
 * Returns the aggregation configuration for raw aggregations.
 * Aggregations are nested objects, so this function recursively
 * builds an object that is easier to work with.
 *
 * @param rawAggs - raw aggregations object
 * @returns aggregation configuration
 */
exports.getUniqueValuesForRawAggs = getUniqueValuesForRawAggs;
const getAggConfigForRawAggs = rawAggs => {
  const aggConfig = {
    id: '',
    type: ''
  };
  Object.entries(rawAggs).forEach(([aggKey, agg]) => {
    aggConfig.id = aggKey;
    Object.entries(agg).forEach(([name, value]) => {
      if (name === 'aggs') {
        aggConfig.aggs = {};
        Object.entries(value).forEach(([subAggKey, subRawAgg]) => {
          const subAgg = getAggConfigForRawAggs(subRawAgg);
          if (subAgg) {
            aggConfig.aggs[subAgg.id] = {
              ...subAgg,
              id: subAggKey
            };
          }
        });
      } else {
        aggConfig.type = name;
        Object.assign(aggConfig, {
          [name]: value
        });
      }
    });
  });
  return aggConfig;
};

/**
 * Returns the aggregation configuration.
 *
 * @param searchRequest - search request object
 * @param aggConfig - aggregation configuration object
 * @param getAggTypeFn - function to get the aggregation type from the aggsService
 * @returns aggregation configuration
 */
exports.getAggConfigForRawAggs = getAggConfigForRawAggs;
const getAggConfig = (searchRequest, aggConfig = {}, getAggTypeFn) => {
  const rawAggs = getRawAggs(searchRequest);
  Object.entries(rawAggs).forEach(([aggKey, agg]) => {
    aggConfig.id = aggKey;
    Object.entries(agg).forEach(([name, value]) => {
      if (name === 'aggs' && value) {
        aggConfig.aggs = {};
        Object.entries(value).forEach(([subAggKey, subRawAgg]) => {
          const subAgg = getAggConfigForRawAggs(subRawAgg);
          if (subAgg) {
            aggConfig.aggs[subAgg.id] = {
              ...subAgg,
              id: subAggKey
            };
          }
        });
      } else {
        var _getAggTypeFn$type, _getAggTypeFn;
        aggConfig.type = (_getAggTypeFn$type = (_getAggTypeFn = getAggTypeFn(name)) === null || _getAggTypeFn === void 0 ? void 0 : _getAggTypeFn.type) !== null && _getAggTypeFn$type !== void 0 ? _getAggTypeFn$type : name;
        Object.assign(aggConfig, {
          [name]: value
        });
      }
    });
  });
  return aggConfig;
};

/**
 * Converts the data frame response to a search response.
 * This function is used to convert the data frame response to a search response
 * to be used by the rest of the application.
 *
 * @param response - data frame response object
 * @returns converted search response
 */
exports.getAggConfig = getAggConfig;
const convertResult = response => {
  const body = response.body;
  if (body.hasOwnProperty('error')) {
    return response;
  }
  const data = body;
  const hits = [];
  for (let index = 0; index < data.size; index++) {
    const hit = {};
    data.fields.forEach(field => {
      hit[field.name] = field.values[index];
    });
    hits.push({
      _index: data.name,
      _source: hit
    });
  }
  const searchResponse = {
    took: response.took,
    timed_out: false,
    _shards: {
      total: 1,
      successful: 1,
      skipped: 0,
      failed: 0
    },
    hits: {
      total: 0,
      max_score: 0,
      hits
    }
  };
  if (data.hasOwnProperty('aggs')) {
    const dataWithAggs = data;
    if (!dataWithAggs.aggs) {
      // TODO: MQL best guess, get timestamp field and caculate it here
      return searchResponse;
    }
    searchResponse.aggregations = Object.entries(dataWithAggs.aggs).reduce((acc, [id, value]) => {
      var _dataWithAggs$meta;
      const aggConfig = (_dataWithAggs$meta = dataWithAggs.meta) === null || _dataWithAggs$meta === void 0 ? void 0 : _dataWithAggs$meta.aggs;
      if (id === 'other-filter') {
        const buckets = value;
        buckets.forEach(bucket => {
          const bucketValue = bucket.value;
          searchResponse.hits.total += bucketValue;
        });
        acc[id] = {
          buckets: [{
            '': {
              doc_count: 0
            }
          }]
        };
        return acc;
      }
      if (aggConfig && aggConfig.type === 'buckets') {
        const buckets = value;
        acc[id] = {
          buckets: buckets.map(bucket => {
            const bucketValue = bucket.value;
            searchResponse.hits.total += bucketValue;
            return {
              key_as_string: bucket.key,
              key: aggConfig.date_histogram ? new Date(bucket.key).getTime() : bucket.key,
              doc_count: bucketValue
            };
          })
        };
        return acc;
      }
      acc[id] = Array.isArray(value) ? value[0] : value;
      return acc;
    }, {});
  }
  return searchResponse;
};

/**
 * Formats the field value.
 *
 * @param field - field object
 * @param value - value to format
 * @returns formatted value
 */
exports.convertResult = convertResult;
const formatFieldValue = (field, value) => {
  return field.format && field.format.convert ? field.format.convert(value) : value;
};

/**
 * Returns the field type. This function is used to determine the field type so that can
 * be used by the rest of the application. The field type must map to a OsdFieldType
 * to be used by the rest of the application.
 *
 * @param field - field object
 * @returns field type
 */
exports.formatFieldValue = formatFieldValue;
const getFieldType = field => {
  var _field$name, _field$values;
  const fieldName = (_field$name = field.name) === null || _field$name === void 0 ? void 0 : _field$name.toLowerCase();
  if (fieldName !== null && fieldName !== void 0 && fieldName.includes('date') || fieldName !== null && fieldName !== void 0 && fieldName.includes('timestamp')) {
    return 'date';
  }
  if ((_field$values = field.values) !== null && _field$values !== void 0 && _field$values.some(value => value instanceof Date || _datemath.default.isDateTime(value))) {
    return 'date';
  }
  if (field.type === 'struct') {
    return 'object';
  }
  return field.type;
};

/**
 * Returns the time field. If there is an aggConfig then we do not have to guess.
 * If there is no aggConfig then we will try to guess the time field.
 *
 * @param data - data frame object.
 * @param aggConfig - aggregation configuration object.
 * @returns time field.
 */
exports.getFieldType = getFieldType;
const getTimeField = (data, aggConfig) => {
  const fields = data.schema || data.fields;
  return aggConfig && aggConfig.date_histogram && aggConfig.date_histogram.field ? fields.find(field => {
    var _aggConfig$date_histo;
    return field.name === (aggConfig === null || aggConfig === void 0 || (_aggConfig$date_histo = aggConfig.date_histogram) === null || _aggConfig$date_histo === void 0 ? void 0 : _aggConfig$date_histo.field);
  }) : fields.find(field => field.type === 'date');
};

/**
 * Parses timepicker datetimes using datemath package. Will attempt to parse strings such as
 * "now - 15m"
 *
 * @param dateRange - of type TimeRange
 * @param dateFormat - formatting string (should work with Moment)
 * @returns object with `fromDate` and `toDate` strings, both of which will be in utc time and formatted to
 * the `dateFormat` parameter
 */
exports.getTimeField = getTimeField;
const formatTimePickerDate = (dateRange, dateFormat) => {
  const dateMathParse = date => {
    const parsedDate = _datemath.default.parse(date);
    return parsedDate ? parsedDate.utc().format(dateFormat) : '';
  };
  const fromDate = dateMathParse(dateRange.from);
  const toDate = dateMathParse(dateRange.to);
  return {
    fromDate,
    toDate
  };
};

/**
 * Checks if the value is a GeoPoint. Expects an object with lat and lon properties.
 *
 * @param value - value to check
 * @returns True if the value is a GeoPoint, false otherwise
 */
exports.formatTimePickerDate = formatTimePickerDate;
const isGeoPoint = value => {
  return typeof value === 'object' && value !== null && 'lat' in value && 'lon' in value && typeof value.lat === 'number' && typeof value.lon === 'number';
};

/**
 * Creates a data frame.
 *
 * @param partial - partial data frame object
 * @returns data frame.
 */
exports.isGeoPoint = isGeoPoint;
const createDataFrame = partial => {
  var _partial$schema, _partial$fields;
  let size = 0;
  const processField = field => {
    field.type = getFieldType(field);
    if (!field.values) {
      field.values = new Array(size);
    } else if (field.values.length > size) {
      size = field.values.length;
    }
    return field;
  };
  const schema = (_partial$schema = partial.schema) === null || _partial$schema === void 0 ? void 0 : _partial$schema.map(processField);
  const fields = (_partial$fields = partial.fields) === null || _partial$fields === void 0 ? void 0 : _partial$fields.map(processField);
  return {
    ...partial,
    schema,
    fields,
    size
  };
};

/**
 * Updates the data frame metadata. Metadata is used to store the aggregation configuration.
 * It also stores the query string used to fetch the data frame aggregations.
 *
 * @param params - { dataFrame, qs, aggConfig, timeField, timeFilter, getAggQsFn }
 */
exports.createDataFrame = createDataFrame;
const updateDataFrameMeta = ({
  dataFrame,
  qs,
  aggConfig,
  timeField,
  timeFilter,
  getAggQsFn
}) => {
  dataFrame.meta = {
    aggs: aggConfig,
    aggsQs: {
      [aggConfig.id]: getAggQsFn({
        qs,
        aggConfig,
        timeField,
        timeFilter
      })
    }
  };
  if (aggConfig.aggs) {
    const subAggs = aggConfig.aggs;
    for (const [key, subAgg] of Object.entries(subAggs)) {
      const subAggConfig = {
        [key]: subAgg
      };
      dataFrame.meta.aggsQs[subAgg.id] = getAggQsFn({
        qs,
        aggConfig: subAggConfig,
        timeField,
        timeFilter
      });
    }
  }
};

/**
 * Converts a data frame to index pattern spec which can be used to create an index pattern.
 *
 * @param dataFrame - data frame object
 * @param id - index pattern id if it exists
 * @returns index pattern spec
 */
exports.updateDataFrameMeta = updateDataFrameMeta;
const dataFrameToSpec = (dataFrame, id) => {
  var _getTimeField;
  const fields = dataFrame.schema || dataFrame.fields;
  const toFieldSpec = (field, overrides) => {
    var _field$aggregatable, _field$searchable;
    return {
      ...field,
      ...overrides,
      aggregatable: (_field$aggregatable = field.aggregatable) !== null && _field$aggregatable !== void 0 ? _field$aggregatable : true,
      searchable: (_field$searchable = field.searchable) !== null && _field$searchable !== void 0 ? _field$searchable : true
    };
  };
  const flattenFields = (acc, field) => {
    switch (field.type) {
      case 'object':
        const dataField = dataFrame.fields.find(f => f.name === field.name) || field;
        if (dataField) {
          const subField = dataField.values[0];
          if (!subField) {
            acc[field.name] = toFieldSpec(field, {});
            break;
          }
          Object.entries(subField).forEach(([key, value]) => {
            const subFieldName = `${dataField.name}.${key}`;
            const subFieldType = typeof value;
            if (subFieldType === 'object' && isGeoPoint(value)) {
              acc[subFieldName] = toFieldSpec(subField, {
                name: subFieldName,
                type: 'geo_point'
              });
            } else {
              var _Object$entries;
              acc = flattenFields(acc, {
                name: subFieldName,
                type: subFieldType,
                values: subFieldType === 'object' ? (_Object$entries = Object.entries(value)) === null || _Object$entries === void 0 ? void 0 : _Object$entries.map(([k, v]) => ({
                  name: `${subFieldName}.${k}`,
                  type: typeof v
                })) : []
              });
            }
          });
        }
        break;
      default:
        acc[field.name] = toFieldSpec(field, {});
        break;
    }
    return acc;
  };
  return {
    id: id !== null && id !== void 0 ? id : _types.DATA_FRAME_TYPES.DEFAULT,
    title: dataFrame.name,
    timeFieldName: (_getTimeField = getTimeField(dataFrame)) === null || _getTimeField === void 0 ? void 0 : _getTimeField.name,
    type: !id ? _types.DATA_FRAME_TYPES.DEFAULT : undefined,
    fields: fields.reduce(flattenFields, {})
  };
};
exports.dataFrameToSpec = dataFrameToSpec;