"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.plugin = exports.Http2Plugin = void 0;
const core_1 = require("@opencensus/core");
const http_1 = require("./http");
const shimmer = require("shimmer");
const url = require("url");
const uuid = require("uuid");
class Http2Plugin extends http_1.HttpPlugin {
    constructor() {
        super('http2');
    }
    applyPatch() {
        shimmer.wrap(this.moduleExports, 'createServer', this.getPatchCreateServerFunction());
        shimmer.wrap(this.moduleExports, 'createSecureServer', this.getPatchCreateServerFunction());
        shimmer.wrap(this.moduleExports, 'connect', this.getPatchConnectFunction());
        return this.moduleExports;
    }
    applyUnpatch() {
        shimmer.unwrap(this.moduleExports, 'createServer');
        shimmer.unwrap(this.moduleExports, 'createSecureServer');
        shimmer.unwrap(this.moduleExports, 'connect');
    }
    getPatchConnectFunction() {
        const plugin = this;
        return (original) => {
            return function patchedConnect(authority) {
                const client = original.apply(this, arguments);
                shimmer.wrap(client, 'request', (original) => (plugin.getPatchRequestFunction())(original, authority));
                shimmer.unwrap(plugin.moduleExports, 'connect');
                return client;
            };
        };
    }
    getPatchRequestFunction() {
        const plugin = this;
        return (original, authority) => {
            return function patchedRequest(headers) {
                if (headers['x-opencensus-outgoing-request']) {
                    return original.apply(this, arguments);
                }
                const request = original.apply(this, arguments);
                plugin.tracer.wrapEmitter(request);
                const traceOptions = {
                    name: `http2-${(headers[':method'] || 'GET').toLowerCase()}`,
                    kind: core_1.SpanKind.CLIENT
                };
                if (!plugin.tracer.currentRootSpan) {
                    return plugin.tracer.startRootSpan(traceOptions, plugin.getMakeHttp2RequestTraceFunction(request, headers, authority, plugin));
                }
                else {
                    const span = plugin.tracer.startChildSpan(traceOptions.name, traceOptions.kind);
                    return (plugin.getMakeHttp2RequestTraceFunction(request, headers, authority, plugin))(span);
                }
            };
        };
    }
    getMakeHttp2RequestTraceFunction(request, headers, authority, plugin) {
        return (span) => {
            if (!span)
                return request;
            const setter = {
                setHeader(name, value) {
                    headers[name] = value;
                }
            };
            const propagation = plugin.tracer.propagation;
            if (propagation) {
                propagation.inject(setter, span.spanContext);
            }
            request.on('response', (responseHeaders) => {
                const status = `${responseHeaders[':status']}`;
                span.addAttribute(Http2Plugin.ATTRIBUTE_HTTP_STATUS_CODE, status);
                span.setStatus(Http2Plugin.convertTraceStatus(parseInt(status, 10)));
            });
            request.on('end', () => {
                const userAgent = headers['user-agent'] || headers['User-Agent'] || null;
                span.addAttribute(Http2Plugin.ATTRIBUTE_HTTP_HOST, `${url.parse(authority).host}`);
                span.addAttribute(Http2Plugin.ATTRIBUTE_HTTP_METHOD, `${headers[':method']}`);
                span.addAttribute(Http2Plugin.ATTRIBUTE_HTTP_PATH, `${headers[':path']}`);
                span.addAttribute(Http2Plugin.ATTRIBUTE_HTTP_ROUTE, `${headers[':path']}`);
                if (userAgent) {
                    span.addAttribute(Http2Plugin.ATTRIBUTE_HTTP_USER_AGENT, `${userAgent}`);
                }
                span.addMessageEvent(core_1.MessageEventType.SENT, uuid.v4().split('-').join(''));
                span.end();
            });
            request.on('error', (err) => {
                span.addAttribute(http_1.HttpPlugin.ATTRIBUTE_HTTP_ERROR_NAME, err.name);
                span.addAttribute(http_1.HttpPlugin.ATTRIBUTE_HTTP_ERROR_MESSAGE, err.message);
                span.setStatus(core_1.CanonicalCode.UNKNOWN, err.message);
                span.end();
            });
            return request;
        };
    }
    getPatchCreateServerFunction() {
        const plugin = this;
        return (original) => {
            return function patchedCreateServer() {
                const server = original.apply(this, arguments);
                shimmer.wrap(server.constructor.prototype, 'emit', plugin.getPatchEmitFunction());
                shimmer.unwrap(plugin.moduleExports, 'createServer');
                shimmer.unwrap(plugin.moduleExports, 'createSecureServer');
                return server;
            };
        };
    }
    getPatchEmitFunction() {
        const plugin = this;
        return (original) => {
            return function patchedEmit(event, stream, headers) {
                if (event !== 'stream') {
                    return original.apply(this, arguments);
                }
                const propagation = plugin.tracer.propagation;
                const getter = {
                    getHeader(name) {
                        return headers[name];
                    }
                };
                const traceOptions = {
                    name: headers[':path'],
                    kind: core_1.SpanKind.SERVER,
                    spanContext: propagation ? propagation.extract(getter) : null
                };
                let statusCode = 0;
                const originalRespond = stream.respond;
                stream.respond = function () {
                    stream.respond = originalRespond;
                    statusCode = arguments[0][':status'];
                    return stream.respond.apply(this, arguments);
                };
                return plugin.tracer.startRootSpan(traceOptions, rootSpan => {
                    if (!rootSpan)
                        return original.apply(this, arguments);
                    plugin.tracer.wrapEmitter(stream);
                    const originalEnd = stream.end;
                    stream.end = function () {
                        stream.end = originalEnd;
                        const returned = stream.end.apply(this, arguments);
                        const userAgent = (headers['user-agent'] || headers['User-Agent'] ||
                            null);
                        rootSpan.addAttribute(Http2Plugin.ATTRIBUTE_HTTP_HOST, `${headers[':authority']}`);
                        rootSpan.addAttribute(Http2Plugin.ATTRIBUTE_HTTP_METHOD, `${headers[':method']}`);
                        rootSpan.addAttribute(Http2Plugin.ATTRIBUTE_HTTP_PATH, `${headers[':path']}`);
                        rootSpan.addAttribute(Http2Plugin.ATTRIBUTE_HTTP_ROUTE, `${headers[':path']}`);
                        rootSpan.addAttribute(Http2Plugin.ATTRIBUTE_HTTP_USER_AGENT, userAgent);
                        rootSpan.addAttribute(Http2Plugin.ATTRIBUTE_HTTP_STATUS_CODE, `${statusCode}`);
                        rootSpan.setStatus(Http2Plugin.convertTraceStatus(statusCode));
                        rootSpan.addMessageEvent(core_1.MessageEventType.RECEIVED, uuid.v4().split('-').join(''));
                        rootSpan.end();
                        return returned;
                    };
                    return original.apply(this, arguments);
                });
            };
        };
    }
}
exports.Http2Plugin = Http2Plugin;
const plugin = new Http2Plugin();
exports.plugin = plugin;
//# sourceMappingURL=data:application/json;base64,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