"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const logger_1 = require("../utils/logger");
const NS = 'zh:controller:database';
class Database {
    entries;
    path;
    maxId;
    constructor(entries, path) {
        this.entries = entries;
        this.maxId = Math.max(...Object.keys(entries).map((t) => Number(t)), 0);
        this.path = path;
    }
    static open(path) {
        const entries = {};
        if (fs_1.default.existsSync(path)) {
            const file = fs_1.default.readFileSync(path, 'utf-8');
            for (const row of file.split('\n')) {
                if (!row) {
                    continue;
                }
                try {
                    const json = JSON.parse(row);
                    if (json.id != undefined) {
                        entries[json.id] = json;
                    }
                }
                catch (error) {
                    logger_1.logger.error(`Corrupted database line, ignoring. ${error}`, NS);
                }
            }
        }
        return new Database(entries, path);
    }
    *getEntriesIterator(type) {
        for (const id in this.entries) {
            const entry = this.entries[id];
            if (type.includes(entry.type)) {
                yield entry;
            }
        }
    }
    insert(databaseEntry) {
        if (this.entries[databaseEntry.id]) {
            throw new Error(`DatabaseEntry with ID '${databaseEntry.id}' already exists`);
        }
        this.entries[databaseEntry.id] = databaseEntry;
        this.write();
    }
    update(databaseEntry, write) {
        if (!this.entries[databaseEntry.id]) {
            throw new Error(`DatabaseEntry with ID '${databaseEntry.id}' does not exist`);
        }
        this.entries[databaseEntry.id] = databaseEntry;
        if (write) {
            this.write();
        }
    }
    remove(id) {
        if (!this.entries[id]) {
            throw new Error(`DatabaseEntry with ID '${id}' does not exist`);
        }
        delete this.entries[id];
        this.write();
    }
    has(id) {
        return Boolean(this.entries[id]);
    }
    newID() {
        this.maxId += 1;
        return this.maxId;
    }
    write() {
        logger_1.logger.debug(`Writing database to '${this.path}'`, NS);
        let lines = '';
        for (const id in this.entries) {
            lines += JSON.stringify(this.entries[id]) + `\n`;
        }
        const tmpPath = this.path + '.tmp';
        fs_1.default.writeFileSync(tmpPath, lines.slice(0, -1)); // remove last newline, no effect if empty string
        // Ensure file is on disk https://github.com/Koenkk/zigbee2mqtt/issues/11759
        const fd = fs_1.default.openSync(tmpPath, 'r+');
        fs_1.default.fsyncSync(fd);
        fs_1.default.closeSync(fd);
        fs_1.default.renameSync(tmpPath, this.path);
    }
}
exports.default = Database;
//# sourceMappingURL=database.js.map