/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.timeseries.rest.handler;

import java.io.IOException;
import java.time.Clock;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.action.search.SearchRequest;
import org.opensearch.action.search.SearchResponse;
import org.opensearch.client.Client;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.commons.authuser.User;
import org.opensearch.core.action.ActionListener;
import org.opensearch.index.query.BoolQueryBuilder;
import org.opensearch.index.query.QueryBuilder;
import org.opensearch.index.query.QueryBuilders;
import org.opensearch.search.aggregations.AggregationBuilder;
import org.opensearch.search.aggregations.bucket.histogram.Histogram;
import org.opensearch.search.aggregations.bucket.histogram.LongBounds;
import org.opensearch.search.builder.SearchSourceBuilder;
import org.opensearch.timeseries.AnalysisType;
import org.opensearch.timeseries.common.exception.ValidationException;
import org.opensearch.timeseries.constant.CommonMessages;
import org.opensearch.timeseries.model.Config;
import org.opensearch.timeseries.model.IntervalTimeConfiguration;
import org.opensearch.timeseries.model.ValidationAspect;
import org.opensearch.timeseries.model.ValidationIssueType;
import org.opensearch.timeseries.rest.handler.HistogramAggregationHelper;
import org.opensearch.timeseries.util.SecurityClientUtil;

public class IntervalCalculation {
    private final Logger logger = LogManager.getLogger(IntervalCalculation.class);
    private final Config config;
    private final TimeValue requestTimeout;
    private final HistogramAggregationHelper histogramAggHelper;
    private final Client client;
    private final SecurityClientUtil clientUtil;
    private final User user;
    private final AnalysisType context;
    private final Clock clock;
    private final FullBucketRatePredicate acceptanceCriteria;

    public IntervalCalculation(Config config, TimeValue requestTimeout, Client client, SecurityClientUtil clientUtil, User user, AnalysisType context, Clock clock) {
        this.config = config;
        this.requestTimeout = requestTimeout;
        this.histogramAggHelper = new HistogramAggregationHelper(config, requestTimeout);
        this.client = client;
        this.clientUtil = clientUtil;
        this.user = user;
        this.context = context;
        this.clock = clock;
        this.acceptanceCriteria = new FullBucketRatePredicate();
    }

    public void findInterval(long latestTime, Map<String, Object> topEntity, ActionListener<IntervalTimeConfiguration> listener) {
        ActionListener minimumIntervalListener = ActionListener.wrap(minIntervalAndValidity -> {
            if (((Boolean)minIntervalAndValidity.getRight()).booleanValue()) {
                listener.onResponse((Object)((IntervalTimeConfiguration)minIntervalAndValidity.getLeft()));
            } else if (minIntervalAndValidity.getLeft() == null) {
                listener.onResponse(null);
            } else {
                this.getBucketAggregates(latestTime, topEntity, (IntervalTimeConfiguration)minIntervalAndValidity.getLeft(), listener);
            }
        }, arg_0 -> listener.onFailure(arg_0));
        LongBounds longBounds = this.histogramAggHelper.getTimeRangeBounds(latestTime, 60000L);
        this.findMinimumInterval(topEntity, longBounds, (ActionListener<Pair<IntervalTimeConfiguration, Boolean>>)minimumIntervalListener);
    }

    private void getBucketAggregates(long latestTime, Map<String, Object> topEntity, IntervalTimeConfiguration minimumInterval, ActionListener<IntervalTimeConfiguration> listener) throws IOException {
        try {
            int newIntervalInMinutes = this.increaseAndGetNewInterval(minimumInterval);
            LongBounds timeStampBounds = this.histogramAggHelper.getTimeRangeBounds(latestTime, newIntervalInMinutes);
            SearchRequest searchRequest = this.composeIntervalQuery(topEntity, newIntervalInMinutes, timeStampBounds);
            ActionListener intervalListener = ActionListener.wrap(interval -> listener.onResponse(interval), exception -> {
                listener.onFailure(exception);
                this.logger.error("Failed to get interval recommendation", (Throwable)exception);
            });
            IntervalRecommendationListener searchResponseListener = new IntervalRecommendationListener((ActionListener<IntervalTimeConfiguration>)intervalListener, searchRequest.source(), (IntervalTimeConfiguration)this.config.getInterval(), this.clock.millis() + 10000L, latestTime, timeStampBounds);
            this.clientUtil.asyncRequestWithInjectedSecurity(searchRequest, (arg_0, arg_1) -> ((Client)this.client).search(arg_0, arg_1), this.user, this.client, this.context, searchResponseListener);
        }
        catch (ValidationException ex) {
            listener.onFailure((Exception)ex);
        }
    }

    private int increaseAndGetNewInterval(IntervalTimeConfiguration oldInterval) {
        return (int)Math.ceil((double)IntervalTimeConfiguration.getIntervalInMinute(oldInterval) * 1.2);
    }

    private void findMinimumInterval(Map<String, Object> topEntity, LongBounds timeStampBounds, ActionListener<Pair<IntervalTimeConfiguration, Boolean>> listener) {
        try {
            SearchRequest searchRequest = this.composeIntervalQuery(topEntity, 1, timeStampBounds);
            ActionListener searchResponseListener = ActionListener.wrap(response -> {
                Histogram aggregate = null;
                try {
                    aggregate = this.histogramAggHelper.checkBucketResultErrors((SearchResponse)response);
                }
                catch (ValidationException e) {
                    listener.onFailure((Exception)e);
                }
                if (aggregate == null) {
                    this.logger.warn("Fail to get aggregated result");
                    listener.onResponse((Object)Pair.of((Object)new IntervalTimeConfiguration(1L, ChronoUnit.MINUTES), (Object)Boolean.FALSE));
                    return;
                }
                List<Long> timestamps = aggregate.getBuckets().stream().map(entry -> HistogramAggregationHelper.convertKeyToEpochMillis(entry.getKey())).collect(Collectors.toList());
                if (timestamps.isEmpty()) {
                    this.logger.warn("empty data, return one minute by default");
                    listener.onResponse((Object)Pair.of((Object)new IntervalTimeConfiguration(1L, ChronoUnit.MINUTES), (Object)Boolean.FALSE));
                    return;
                }
                double medianDifference = IntervalCalculation.calculateMedianDifference(timestamps);
                long minimumMinutes = IntervalCalculation.millisecondsToCeilMinutes(Double.valueOf(medianDifference).longValue());
                if (minimumMinutes > 60L) {
                    this.logger.warn("The minimum interval is too large: {}", (Object)minimumMinutes);
                    listener.onResponse((Object)Pair.of(null, (Object)false));
                    return;
                }
                listener.onResponse((Object)Pair.of((Object)new IntervalTimeConfiguration(minimumMinutes, ChronoUnit.MINUTES), (Object)this.acceptanceCriteria.test(aggregate, minimumMinutes)));
            }, arg_0 -> listener.onFailure(arg_0));
            this.clientUtil.asyncRequestWithInjectedSecurity(searchRequest, (arg_0, arg_1) -> ((Client)this.client).search(arg_0, arg_1), this.user, this.client, this.context, searchResponseListener);
        }
        catch (Exception e) {
            listener.onFailure(e);
        }
    }

    private static double calculateMedianDifference(List<Long> timestamps) {
        ArrayList<Long> differences = new ArrayList<Long>();
        for (int i = 1; i < timestamps.size(); ++i) {
            differences.add(timestamps.get(i) - timestamps.get(i - 1));
        }
        Collections.sort(differences);
        int middle = differences.size() / 2;
        if (differences.size() % 2 == 0) {
            return (double)((Long)differences.get(middle - 1) + (Long)differences.get(middle)) / 2.0;
        }
        return ((Long)differences.get(middle)).longValue();
    }

    private static long millisecondsToCeilMinutes(long milliseconds) {
        return (milliseconds + 59999L) / 60000L;
    }

    private SearchRequest composeIntervalQuery(Map<String, Object> topEntity, int intervalInMinutes, LongBounds timeStampBounds) {
        AggregationBuilder aggregation = this.histogramAggHelper.getBucketAggregation(intervalInMinutes, timeStampBounds);
        BoolQueryBuilder query = QueryBuilders.boolQuery().filter(this.config.getFilterQuery());
        if (this.config.isHighCardinality()) {
            if (topEntity.isEmpty()) {
                throw new ValidationException(CommonMessages.CATEGORY_FIELD_TOO_SPARSE, ValidationIssueType.CATEGORY, ValidationAspect.MODEL);
            }
            for (Map.Entry<String, Object> entry : topEntity.entrySet()) {
                query.filter((QueryBuilder)QueryBuilders.termQuery((String)entry.getKey(), (Object)entry.getValue()));
            }
        }
        SearchSourceBuilder searchSourceBuilder = new SearchSourceBuilder().query((QueryBuilder)query).aggregation(aggregation).size(0).timeout(this.requestTimeout);
        return new SearchRequest(this.config.getIndices().toArray(new String[0])).source(searchSourceBuilder);
    }

    class FullBucketRatePredicate
    implements HistogramPredicate {
        FullBucketRatePredicate() {
        }

        @Override
        public boolean test(Histogram histogram, long minimumMinutes) {
            double fullBucketRate = IntervalCalculation.this.histogramAggHelper.processBucketAggregationResults(histogram, minimumMinutes * 60000L, IntervalCalculation.this.config);
            return fullBucketRate > 0.75;
        }
    }

    static interface HistogramPredicate {
        public boolean test(Histogram var1, long var2);
    }

    class IntervalRecommendationListener
    implements ActionListener<SearchResponse> {
        private final ActionListener<IntervalTimeConfiguration> intervalListener;
        SearchSourceBuilder searchSourceBuilder;
        IntervalTimeConfiguration currentIntervalToTry;
        private final long expirationEpochMs;
        private final long latestTime;
        private LongBounds currentTimeStampBounds;

        IntervalRecommendationListener(ActionListener<IntervalTimeConfiguration> intervalListener, SearchSourceBuilder searchSourceBuilder, IntervalTimeConfiguration currentIntervalToTry, long expirationEpochMs, long latestTime, LongBounds timeStampBounds) {
            this.intervalListener = intervalListener;
            this.searchSourceBuilder = searchSourceBuilder;
            this.currentIntervalToTry = currentIntervalToTry;
            this.expirationEpochMs = expirationEpochMs;
            this.latestTime = latestTime;
            this.currentTimeStampBounds = timeStampBounds;
        }

        public void onResponse(SearchResponse response) {
            try {
                Histogram aggregate = null;
                try {
                    aggregate = IntervalCalculation.this.histogramAggHelper.checkBucketResultErrors(response);
                }
                catch (ValidationException e) {
                    this.intervalListener.onFailure((Exception)e);
                }
                if (aggregate == null) {
                    this.intervalListener.onResponse(null);
                    return;
                }
                int newIntervalMinute = IntervalCalculation.this.increaseAndGetNewInterval(this.currentIntervalToTry);
                double fullBucketRate = IntervalCalculation.this.histogramAggHelper.processBucketAggregationResults(aggregate, newIntervalMinute * 60000, IntervalCalculation.this.config);
                if (fullBucketRate > 0.75) {
                    this.intervalListener.onResponse((Object)this.currentIntervalToTry);
                } else if (this.expirationEpochMs < IntervalCalculation.this.clock.millis()) {
                    this.intervalListener.onFailure((Exception)new ValidationException(CommonMessages.TIMEOUT_ON_INTERVAL_REC, ValidationIssueType.TIMEOUT, ValidationAspect.MODEL));
                    IntervalCalculation.this.logger.info(CommonMessages.TIMEOUT_ON_INTERVAL_REC);
                } else if ((long)newIntervalMinute < 60L) {
                    this.searchWithDifferentInterval(newIntervalMinute);
                } else {
                    this.intervalListener.onResponse(null);
                }
            }
            catch (Exception e) {
                this.onFailure(e);
            }
        }

        private void searchWithDifferentInterval(int newIntervalMinuteValue) {
            this.currentIntervalToTry = new IntervalTimeConfiguration(newIntervalMinuteValue, ChronoUnit.MINUTES);
            this.currentTimeStampBounds = IntervalCalculation.this.histogramAggHelper.getTimeRangeBounds(this.latestTime, newIntervalMinuteValue);
            SearchSourceBuilder updatedSearchSourceBuilder = IntervalCalculation.this.histogramAggHelper.getSearchSourceBuilder(this.searchSourceBuilder.query(), IntervalCalculation.this.histogramAggHelper.getBucketAggregation(newIntervalMinuteValue, this.currentTimeStampBounds));
            IntervalCalculation.this.clientUtil.asyncRequestWithInjectedSecurity(new SearchRequest().indices(IntervalCalculation.this.config.getIndices().toArray(new String[0])).source(updatedSearchSourceBuilder), (arg_0, arg_1) -> ((Client)IntervalCalculation.this.client).search(arg_0, arg_1), IntervalCalculation.this.user, IntervalCalculation.this.client, IntervalCalculation.this.context, this);
        }

        public void onFailure(Exception e) {
            IntervalCalculation.this.logger.error("Failed to recommend new interval", (Throwable)e);
            this.intervalListener.onFailure((Exception)new ValidationException(CommonMessages.MODEL_VALIDATION_FAILED_UNEXPECTEDLY, ValidationIssueType.AGGREGATION, ValidationAspect.MODEL));
        }
    }
}

