"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultBackend = void 0;
var _performanceNow = _interopRequireDefault(require("performance-now"));
var _uuid = require("uuid");
var _query_helpers = require("../../common/helpers/notebooks/query_helpers");
var _sample_notebooks = require("../../common/helpers/notebooks/sample_notebooks");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class DefaultBackend {
  constructor() {
    _defineProperty(this, "backend", 'Default Backend');
    // Creates a new notebooks with sample markdown text
    _defineProperty(this, "createNewNotebook", newNotebookName => {
      const noteObject = {
        dateCreated: new Date().toISOString(),
        name: newNotebookName,
        dateModified: new Date().toISOString(),
        backend: 'Default',
        paragraphs: []
      };
      return {
        object: noteObject
      };
    });
    // indexes a notebook with body provided
    _defineProperty(this, "indexNote", async function (client, body) {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            notebook: body
          }
        });
        return response;
      } catch (error) {
        throw new Error('Index Doc Error:' + error);
      }
    });
    // updates a notebook with updateBody provided as parameter
    _defineProperty(this, "updateNote", async function (client, noteId, updateBody) {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: noteId,
          body: {
            notebook: updateBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Update Doc Error:' + error);
      }
    });
    // fetched a notebook by Id
    _defineProperty(this, "getNote", async function (client, noteId) {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: noteId
        });
        if (response.observabilityObjectList.length === 0) {
          throw new Error('notebook id not found');
        }
        return response.observabilityObjectList[0];
      } catch (error) {
        throw new Error('Get Doc Error:' + error);
      }
    });
    // gets first `FETCH_SIZE` notebooks available
    _defineProperty(this, "viewNotes", async function (client, _wreckOptions) {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'notebook'
        });
        return response.observabilityObjectList.map(notebook => ({
          path: notebook.notebook.name,
          id: notebook.objectId,
          dateCreated: notebook.notebook.dateCreated,
          dateModified: notebook.notebook.dateModified
        }));
      } catch (error) {
        if (error.body.error.type === 'index_not_found_exception') {
          return [];
        } else throw new Error('View Notebooks Error:' + error);
      }
    });
    /* Fetches a notebook by id
     * Param: noteId -> Id of notebook to be fetched
     */
    _defineProperty(this, "fetchNote", async function (client, noteId, _wreckOptions) {
      try {
        const noteObject = await this.getNote(client, noteId);
        return {
          path: noteObject.notebook.name,
          dateCreated: noteObject.notebook.dateCreated,
          dateModified: noteObject.notebook.dateModified,
          paragraphs: noteObject.notebook.paragraphs
        };
      } catch (error) {
        throw new Error('Fetching Notebook Error:' + error);
      }
    });
    /* Adds a notebook to storage
     * Param: name -> name of new notebook
     */
    _defineProperty(this, "addNote", async function (client, params, _wreckOptions) {
      try {
        const newNotebook = this.createNewNotebook(params.name);
        const opensearchClientResponse = await this.indexNote(client, newNotebook.object);
        return {
          status: 'OK',
          message: opensearchClientResponse,
          body: opensearchClientResponse.objectId
        };
      } catch (error) {
        throw new Error('Creating New Notebook Error:' + error);
      }
    });
    /* Adds sample notebooks to storage
     * Param: name -> name of new notebook
     */
    _defineProperty(this, "addSampleNotes", async function (client, visIds, _wreckOptions) {
      try {
        const notebooks = (0, _sample_notebooks.getSampleNotebooks)(visIds);
        const newNotebooks = [];
        for (let i = 0; i < notebooks.length; i++) {
          const notebook = notebooks[i];
          await this.indexNote(client, notebook.notebook).then(response => {
            newNotebooks.push({
              id: response.objectId,
              name: notebook.notebook.name,
              dateModified: notebook.dateModified,
              dateCreated: notebook.dateCreated
            });
          });
        }
        return {
          status: 'OK',
          message: '',
          body: newNotebooks
        };
      } catch (error) {
        throw new Error('Creating New Notebook Error:' + error);
      }
    });
    /* Renames a notebook
     * Params: name -> new name for the notebook to be renamed
     *         noteId -> Id of notebook to be fetched
     */
    _defineProperty(this, "renameNote", async function (client, params, _wreckOptions) {
      try {
        const updateNotebook = {
          name: params.name,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return {
          status: 'OK',
          message: opensearchClientResponse
        };
      } catch (error) {
        throw new Error('Renaming Notebook Error:' + error);
      }
    });
    /* Clone a notebook
     * Params: name -> new name for the cloned notebook
     *         noteId -> Id for the notebook to be cloned
     */
    _defineProperty(this, "cloneNote", async function (client, params, _wreckOptions) {
      try {
        const noteObject = await this.getNote(client, params.noteId);
        const newNotebook = this.createNewNotebook(params.name);
        const cloneNotebook = {
          ...newNotebook.object
        };
        cloneNotebook.paragraphs = noteObject.notebook.paragraphs;
        const opensearchClientIndexResponse = await this.indexNote(client, cloneNotebook);
        return {
          status: 'OK',
          body: {
            ...cloneNotebook,
            id: opensearchClientIndexResponse.objectId
          }
        };
      } catch (error) {
        throw new Error('Cloning Notebook Error:' + error);
      }
    });
    /* Delete a notebook
     * Param: noteId -> Id for the notebook to be deleted
     */
    _defineProperty(this, "deleteNote", async function (client, noteList, _wreckOptions) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: noteList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Deleting Notebook Error:' + error);
      }
    });
    /* Export a notebook
     * Param: noteId -> Id for the notebook to be exported
     */
    _defineProperty(this, "exportNote", async function (client, noteId, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, noteId);
        return {
          status: 'OK',
          body: opensearchClientGetResponse
        };
      } catch (error) {
        throw new Error('Export Notebook Error:' + error);
      }
    });
    /* Import a notebook
     * Params: noteObj -> note Object to be imported
     */
    _defineProperty(this, "importNote", async function (client, noteObj, _wreckOptions) {
      try {
        const newNoteObject = {
          ...noteObj
        };
        newNoteObject.id = 'note_' + (0, _uuid.v4)();
        newNoteObject.dateCreated = new Date().toISOString();
        newNoteObject.dateModified = new Date().toISOString();
        const opensearchClientIndexResponse = await this.indexNote(client, newNoteObject);
        return {
          status: 'OK',
          message: opensearchClientIndexResponse,
          body: opensearchClientIndexResponse.objectId
        };
      } catch (error) {
        throw new Error('Import Notebook Error:' + error);
      }
    });
    /* Updates input for required paragraphs
     * Params: paragraphs -> list of paragraphs
     *         paragraphId -> Id of paragraph to be updated
     *         paragraphInput -> Input to be added
     */
    _defineProperty(this, "updateParagraph", function (paragraphs, paragraphId, paragraphInput, paragraphType) {
      try {
        const updatedParagraphs = [];
        paragraphs.map(paragraph => {
          const updatedParagraph = {
            ...paragraph
          };
          if (paragraph.id === paragraphId) {
            updatedParagraph.dateModified = new Date().toISOString();
            updatedParagraph.input.inputText = paragraphInput;
            if (paragraphType.length > 0) {
              updatedParagraph.input.inputType = paragraphType;
            }
          }
          updatedParagraphs.push(updatedParagraph);
        });
        return updatedParagraphs;
      } catch (error) {
        throw new Error('Update Paragraph Error:' + error);
      }
    });
    // Creates new paragraph with the given input and input type
    _defineProperty(this, "createParagraph", function (paragraphInput, inputType) {
      try {
        let paragraphType = 'MARKDOWN';
        if (inputType === 'VISUALIZATION') {
          paragraphType = 'VISUALIZATION';
        }
        if (inputType === 'OBSERVABILITY_VISUALIZATION') {
          paragraphType = 'OBSERVABILITY_VISUALIZATION';
        }
        if (paragraphInput.substring(0, 3) === '%sql' || paragraphInput.substring(0, 3) === '%ppl') {
          paragraphType = 'QUERY';
        }
        const inputObject = {
          inputType: paragraphType,
          inputText: paragraphInput
        };
        const outputObjects = [{
          outputType: paragraphType,
          result: '',
          execution_time: '0s'
        }];
        const newParagraph = {
          id: 'paragraph_' + (0, _uuid.v4)(),
          dateCreated: new Date().toISOString(),
          dateModified: new Date().toISOString(),
          input: inputObject,
          output: outputObjects
        };
        return newParagraph;
      } catch (error) {
        throw new Error('Create Paragraph Error:' + error);
      }
    });
    /* Runs a paragraph
     * Currently only runs markdown by copying input.inputText to result
     * UI renders Markdown
     */
    _defineProperty(this, "runParagraph", async function (paragraphs, paragraphId, client) {
      try {
        const updatedParagraphs = [];
        let index = 0;
        for (index = 0; index < paragraphs.length; ++index) {
          const startTime = (0, _performanceNow.default)();
          const updatedParagraph = {
            ...paragraphs[index]
          };
          if (paragraphs[index].id === paragraphId) {
            updatedParagraph.dateModified = new Date().toISOString();
            if ((0, _query_helpers.inputIsQuery)(paragraphs[index].input.inputText)) {
              updatedParagraph.output = [{
                outputType: 'QUERY',
                result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputText.substring(0, 3) === '%md') {
              updatedParagraph.output = [{
                outputType: 'MARKDOWN',
                result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputType === 'VISUALIZATION') {
              updatedParagraph.dateModified = new Date().toISOString();
              updatedParagraph.output = [{
                outputType: 'VISUALIZATION',
                result: '',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputType === 'OBSERVABILITY_VISUALIZATION') {
              updatedParagraph.dateModified = new Date().toISOString();
              updatedParagraph.output = [{
                outputType: 'OBSERVABILITY_VISUALIZATION',
                result: '',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if ((0, _query_helpers.formatNotRecognized)(paragraphs[index].input.inputText)) {
              updatedParagraph.output = [{
                outputType: 'MARKDOWN',
                result: 'Please select an input type (%sql, %ppl, or %md)',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            }
          }
          updatedParagraphs.push(updatedParagraph);
        }
        return updatedParagraphs;
      } catch (error) {
        throw new Error('Running Paragraph Error:' + error);
      }
    });
    /* --> Updates a Paragraph with input content
     * --> Runs it
     * --> Updates the notebook in index
     * --> Fetches the updated Paragraph (with new input content and output result)
     * Params: noteId -> Id of the notebook
     *         paragraphId -> Id of the paragraph to be updated
     *         paragraphInput -> paragraph input code
     */
    _defineProperty(this, "updateRunFetchParagraph", async function (client, request, _wreckOptions) {
      try {
        const scopedClient = client.asScoped(request);
        const params = request.body;
        const opensearchClientGetResponse = await this.getNote(scopedClient, params.noteId);
        const updatedInputParagraphs = this.updateParagraph(opensearchClientGetResponse.notebook.paragraphs, params.paragraphId, params.paragraphInput, params.paragraphType);
        const updatedOutputParagraphs = await this.runParagraph(updatedInputParagraphs, params.paragraphId, client);
        const updateNotebook = {
          paragraphs: updatedOutputParagraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(scopedClient, params.noteId, updateNotebook);
        let resultParagraph = {};
        let index = 0;
        for (index = 0; index < updatedOutputParagraphs.length; ++index) {
          if (params.paragraphId === updatedOutputParagraphs[index].id) {
            resultParagraph = updatedOutputParagraphs[index];
          }
        }
        return resultParagraph;
      } catch (error) {
        throw new Error('Update/Run Paragraph Error:' + error);
      }
    });
    /* --> Updates a Paragraph with input content
     * --> Updates the notebook in index
     * --> Fetches the updated Paragraph (with new input content)
     * Params: noteId -> Id of the notebook
     *         paragraphId -> Id of the paragraph to be updated
     *         paragraphInput -> paragraph input code
     */
    _defineProperty(this, "updateFetchParagraph", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedInputParagraphs = this.updateParagraph(opensearchClientGetResponse.notebook.paragraphs, params.paragraphId, params.paragraphInput);
        const updateNotebook = {
          paragraphs: updatedInputParagraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        let resultParagraph = {};
        updatedInputParagraphs.map(paragraph => {
          if (params.paragraphId === paragraph.id) {
            resultParagraph = paragraph;
          }
        });
        return resultParagraph;
      } catch (error) {
        throw new Error('Save Paragraph Error:' + error);
      }
    });
    /* --> Fetches the Paragraph
     * --> Adds a Paragraph with input content
     * --> Updates the notebook in index
     * Params: noteId -> Id of the notebook
     *         paragraphId -> Id of the paragraph to be fetched
     */
    _defineProperty(this, "addFetchNewParagraph", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const paragraphs = opensearchClientGetResponse.notebook.paragraphs;
        const newParagraph = this.createParagraph(params.paragraphInput, params.inputType);
        paragraphs.splice(params.paragraphIndex, 0, newParagraph);
        const updateNotebook = {
          paragraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return newParagraph;
      } catch (error) {
        throw new Error('add/Fetch Paragraph Error:' + error);
      }
    });
    /* --> Deletes a Paragraph with id
     * --> Fetches the all other Paragraphs as a list
     * --> Updates the notebook in index
     * Params: noteId -> Id of the notebook
     *         paragraphId -> Id of the paragraph to be deleted
     */
    _defineProperty(this, "deleteFetchParagraphs", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedparagraphs = [];
        if (params.paragraphId !== undefined) {
          opensearchClientGetResponse.notebook.paragraphs.map((paragraph, index) => {
            if (paragraph.id !== params.paragraphId) {
              updatedparagraphs.push(paragraph);
            }
          });
        }
        const updateNotebook = {
          paragraphs: updatedparagraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return {
          paragraphs: updatedparagraphs
        };
      } catch (error) {
        console.log('error', error);
        throw new Error('Delete Paragraph Error:' + error);
      }
    });
    /* --> Clears output for all the paragraphs
     * --> Fetches the all Paragraphs as a list (with cleared outputs)
     * --> Updates the notebook in index
     * Param: noteId -> Id of notebook to be cleared
     */
    _defineProperty(this, "clearAllFetchParagraphs", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedparagraphs = [];
        opensearchClientGetResponse.notebook.paragraphs.map((paragraph, index) => {
          const updatedParagraph = {
            ...paragraph
          };
          updatedParagraph.output = [];
          updatedparagraphs.push(updatedParagraph);
        });
        const updateNotebook = {
          paragraphs: updatedparagraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return {
          paragraphs: updatedparagraphs
        };
      } catch (error) {
        throw new Error('Clear Paragraph Error:' + error);
      }
    });
  }
}
exports.DefaultBackend = DefaultBackend;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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