"use strict";
/* istanbul ignore file */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ZBOSSAdapter = void 0;
const __1 = require("../..");
const utils_1 = require("../../../utils");
const logger_1 = require("../../../utils/logger");
const Zcl = __importStar(require("../../../zspec/zcl"));
const serialPortUtils_1 = __importDefault(require("../../serialPortUtils"));
const socketPortUtils_1 = __importDefault(require("../../socketPortUtils"));
const driver_1 = require("../driver");
const enums_1 = require("../enums");
const frame_1 = require("../frame");
const NS = 'zh:zboss';
const autoDetectDefinitions = [
    // Nordic Zigbee NCP
    { manufacturer: 'ZEPHYR', vendorId: '2fe3', productId: '0100' },
];
class ZBOSSAdapter extends __1.Adapter {
    queue;
    driver;
    waitress;
    coordinator;
    constructor(networkOptions, serialPortOptions, backupPath, adapterOptions) {
        super(networkOptions, serialPortOptions, backupPath, adapterOptions);
        const concurrent = adapterOptions && adapterOptions.concurrent ? adapterOptions.concurrent : 8;
        logger_1.logger.debug(`Adapter concurrent: ${concurrent}`, NS);
        this.queue = new utils_1.Queue(concurrent);
        this.waitress = new utils_1.Waitress(this.waitressValidator, this.waitressTimeoutFormatter);
        this.driver = new driver_1.ZBOSSDriver(serialPortOptions, networkOptions);
        this.driver.on('frame', this.processMessage.bind(this));
    }
    async processMessage(frame) {
        logger_1.logger.debug(`processMessage: ${JSON.stringify(frame)}`, NS);
        if (frame.type == frame_1.FrameType.INDICATION &&
            frame.commandId == enums_1.CommandId.ZDO_DEV_UPDATE_IND &&
            frame.payload.status == enums_1.DeviceUpdateStatus.LEFT) {
            logger_1.logger.debug(`Device left network request received: ${frame.payload.nwk} ${frame.payload.ieee}`, NS);
            const payload = {
                networkAddress: frame.payload.nwk,
                ieeeAddr: frame.payload.ieee,
            };
            this.emit('deviceLeave', payload);
        }
        if (frame.type == frame_1.FrameType.INDICATION && frame.commandId == enums_1.CommandId.NWK_LEAVE_IND) {
            logger_1.logger.debug(`Device left network request received from ${frame.payload.ieee}`, NS);
            const payload = {
                networkAddress: frame.payload.nwk,
                ieeeAddr: frame.payload.ieee,
            };
            this.emit('deviceLeave', payload);
        }
        if (frame.type == frame_1.FrameType.INDICATION && frame.commandId == enums_1.CommandId.ZDO_DEV_ANNCE_IND) {
            logger_1.logger.debug(`Device join request received: ${frame.payload.nwk} ${frame.payload.ieee}`, NS);
            const payload = {
                networkAddress: frame.payload.nwk,
                ieeeAddr: frame.payload.ieee,
            };
            this.emit('deviceJoined', payload);
        }
        if (frame.type == frame_1.FrameType.INDICATION && frame.commandId == enums_1.CommandId.APSDE_DATA_IND) {
            logger_1.logger.debug(`ZCL frame received from ${frame.payload.srcNwk} ${frame.payload.srcEndpoint}`, NS);
            const payload = {
                clusterID: frame.payload.clusterID,
                header: Zcl.Header.fromBuffer(frame.payload.data),
                data: frame.payload.data,
                address: frame.payload.srcNwk,
                endpoint: frame.payload.srcEndpoint,
                linkquality: frame.payload.lqi,
                groupID: frame.payload.grpNwk,
                wasBroadcast: false,
                destinationEndpoint: frame.payload.dstEndpoint,
            };
            this.waitress.resolve(payload);
            this.emit('zclPayload', payload);
        }
        //this.emit('event', frame);
    }
    static async isValidPath(path) {
        // For TCP paths we cannot get device information, therefore we cannot validate it.
        if (socketPortUtils_1.default.isTcpPath(path)) {
            return false;
        }
        try {
            return serialPortUtils_1.default.is((0, utils_1.RealpathSync)(path), autoDetectDefinitions);
        }
        catch (error) {
            logger_1.logger.debug(`Failed to determine if path is valid: '${error}'`, NS);
            return false;
        }
    }
    static async autoDetectPath() {
        const paths = await serialPortUtils_1.default.find(autoDetectDefinitions);
        paths.sort((a, b) => (a < b ? -1 : 1));
        return paths.length > 0 ? paths[0] : null;
    }
    async start() {
        logger_1.logger.info(`ZBOSS Adapter starting`, NS);
        await this.driver.connect();
        return this.driver.startup();
    }
    async stop() {
        await this.driver.stop();
        logger_1.logger.info(`ZBOSS Adapter stopped`, NS);
    }
    async getCoordinator() {
        return this.queue.execute(async () => {
            const info = await this.driver.getCoordinator();
            logger_1.logger.debug(`ZBOSS Adapter Coordinator description:\n${JSON.stringify(info)}`, NS);
            this.coordinator = {
                networkAddress: info.networkAddress,
                manufacturerID: 0,
                ieeeAddr: info.ieeeAddr,
                endpoints: info.endpoints,
            };
            return this.coordinator;
        });
    }
    async getCoordinatorVersion() {
        return this.driver.getCoordinatorVersion();
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async reset(type) {
        return Promise.reject(new Error('Not supported'));
    }
    async supportsBackup() {
        return false;
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async backup(ieeeAddressesInDatabase) {
        throw new Error('This adapter does not support backup');
    }
    async getNetworkParameters() {
        return this.queue.execute(async () => {
            const channel = this.driver.netInfo.network.channel;
            const panID = this.driver.netInfo.network.panID;
            const extendedPanID = this.driver.netInfo.network.extendedPanID;
            return {
                panID,
                extendedPanID: parseInt(Buffer.from(extendedPanID).toString('hex'), 16),
                channel,
            };
        });
    }
    async supportsChangeChannel() {
        return false;
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async changeChannel(newChannel) {
        throw new Error(`Channel change is not supported for 'zboss' yet`);
    }
    async setTransmitPower(value) {
        if (this.driver.isInitialized()) {
            return this.queue.execute(async () => {
                await this.driver.setTXPower(value);
            });
        }
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async addInstallCode(ieeeAddress, key) {
        throw new Error(`Install code is not supported for 'zboss' yet`);
    }
    async permitJoin(seconds, networkAddress) {
        if (this.driver.isInitialized()) {
            return this.queue.execute(async () => {
                await this.driver.permitJoin(networkAddress, seconds);
                if (!networkAddress) {
                    // send broadcast permit
                    await this.driver.permitJoin(0xfffc, seconds);
                }
            });
        }
    }
    async lqi(networkAddress) {
        return this.queue.execute(async () => {
            const neighbors = [];
            const request = async (startIndex) => {
                try {
                    const result = await this.driver.lqi(networkAddress, startIndex);
                    return result;
                }
                catch (error) {
                    throw new Error(`LQI for '${networkAddress}' failed: ${error}`);
                }
            };
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            const add = (list) => {
                for (const entry of list) {
                    neighbors.push({
                        linkquality: entry.lqi,
                        networkAddress: entry.nwk,
                        ieeeAddr: entry.ieee,
                        relationship: (entry.relationship >> 4) & 0x7,
                        depth: entry.depth,
                    });
                }
            };
            let response = (await request(0)).payload;
            add(response.neighbors);
            const size = response.entries;
            let nextStartIndex = response.neighbors.length;
            while (neighbors.length < size) {
                response = await request(nextStartIndex);
                add(response.neighbors);
                nextStartIndex += response.neighbors.length;
            }
            return { neighbors };
        }, networkAddress);
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async routingTable(networkAddress) {
        throw new Error(`Routing table is not supported for 'zboss' yet`);
    }
    async nodeDescriptor(networkAddress) {
        return this.queue.execute(async () => {
            try {
                logger_1.logger.debug(`Requesting 'Node Descriptor' for '${networkAddress}'`, NS);
                const descriptor = await this.driver.nodeDescriptor(networkAddress);
                const logicaltype = descriptor.payload.flags & 0x07;
                return {
                    manufacturerCode: descriptor.payload.manufacturerCode,
                    type: logicaltype == 0 ? 'Coordinator' : logicaltype == 1 ? 'Router' : 'EndDevice',
                };
            }
            catch (error) {
                logger_1.logger.debug(`Node descriptor request for '${networkAddress}' failed (${error}), retry`, NS);
                throw error;
            }
        });
    }
    async activeEndpoints(networkAddress) {
        logger_1.logger.debug(`Requesting 'Active endpoints' for '${networkAddress}'`, NS);
        return this.queue.execute(async () => {
            const endpoints = await this.driver.activeEndpoints(networkAddress);
            return { endpoints: [...endpoints.payload.endpoints] };
        }, networkAddress);
    }
    async simpleDescriptor(networkAddress, endpointID) {
        logger_1.logger.debug(`Requesting 'Simple Descriptor' for '${networkAddress}' endpoint ${endpointID}`, NS);
        return this.queue.execute(async () => {
            const sd = await this.driver.simpleDescriptor(networkAddress, endpointID);
            return {
                profileID: sd.payload.profileID,
                endpointID: sd.payload.endpoint,
                deviceID: sd.payload.deviceID,
                inputClusters: sd.payload.inputClusters,
                outputClusters: sd.payload.outputClusters,
            };
        }, networkAddress);
    }
    async bind(destinationNetworkAddress, sourceIeeeAddress, sourceEndpoint, clusterID, destinationAddressOrGroup, type, destinationEndpoint) {
        return this.queue.execute(async () => {
            await this.driver.bind(destinationNetworkAddress, sourceIeeeAddress, sourceEndpoint, clusterID, destinationAddressOrGroup, type, destinationEndpoint);
        }, destinationNetworkAddress);
    }
    async unbind(destinationNetworkAddress, sourceIeeeAddress, sourceEndpoint, clusterID, destinationAddressOrGroup, type, destinationEndpoint) {
        return this.queue.execute(async () => {
            await this.driver.unbind(destinationNetworkAddress, sourceIeeeAddress, sourceEndpoint, clusterID, destinationAddressOrGroup, type, destinationEndpoint);
        }, destinationNetworkAddress);
    }
    async removeDevice(networkAddress, ieeeAddr) {
        return this.queue.execute(async () => {
            await this.driver.removeDevice(networkAddress, ieeeAddr);
        }, networkAddress);
    }
    async sendZclFrameToEndpoint(ieeeAddr, networkAddress, endpoint, zclFrame, timeout, disableResponse, disableRecovery, sourceEndpoint) {
        return this.queue.execute(async () => {
            return this.sendZclFrameToEndpointInternal(ieeeAddr, networkAddress, endpoint, sourceEndpoint || 1, zclFrame, timeout, disableResponse, disableRecovery, 0, 0, false, false, false, null);
        }, networkAddress);
    }
    async sendZclFrameToEndpointInternal(ieeeAddr, networkAddress, endpoint, sourceEndpoint, zclFrame, timeout, disableResponse, disableRecovery, responseAttempt, dataRequestAttempt, checkedNetworkAddress, discoveredRoute, assocRemove, assocRestore) {
        if (ieeeAddr == null) {
            ieeeAddr = this.coordinator.ieeeAddr;
        }
        logger_1.logger.debug(`sendZclFrameToEndpointInternal ${ieeeAddr}:${networkAddress}/${endpoint} ` +
            `(${responseAttempt},${dataRequestAttempt},${this.queue.count()}), timeout=${timeout}`, NS);
        let response = null;
        const command = zclFrame.command;
        if (command.response && disableResponse === false) {
            response = this.waitFor(networkAddress, endpoint, zclFrame.header.transactionSequenceNumber, zclFrame.cluster.ID, command.response, timeout);
        }
        else if (!zclFrame.header.frameControl.disableDefaultResponse) {
            response = this.waitFor(networkAddress, endpoint, zclFrame.header.transactionSequenceNumber, zclFrame.cluster.ID, Zcl.Foundation.defaultRsp.ID, timeout);
        }
        try {
            const dataConfirmResult = await this.driver.request(ieeeAddr, 0x0104, zclFrame.cluster.ID, endpoint, sourceEndpoint || 0x01, zclFrame.toBuffer());
            if (!dataConfirmResult) {
                if (response != null) {
                    response.cancel();
                }
                throw Error('sendZclFrameToEndpointInternal error');
            }
            if (response !== null) {
                try {
                    const result = await response.start().promise;
                    return result;
                }
                catch (error) {
                    logger_1.logger.debug(`Response timeout (${ieeeAddr}:${networkAddress},${responseAttempt})`, NS);
                    if (responseAttempt < 1 && !disableRecovery) {
                        return this.sendZclFrameToEndpointInternal(ieeeAddr, networkAddress, endpoint, sourceEndpoint, zclFrame, timeout, disableResponse, disableRecovery, responseAttempt + 1, dataRequestAttempt, checkedNetworkAddress, discoveredRoute, assocRemove, assocRestore);
                    }
                    else {
                        throw error;
                    }
                }
            }
            else {
                return;
            }
        }
        catch (error) {
            if (response != null) {
                response.cancel();
            }
            throw error;
        }
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async sendZclFrameToGroup(groupID, zclFrame, sourceEndpoint) {
        return;
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async sendZclFrameToAll(endpoint, zclFrame, sourceEndpoint, destination) {
        return;
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async setChannelInterPAN(channel) {
        return;
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async sendZclFrameInterPANToIeeeAddr(zclFrame, ieeeAddress) {
        return;
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async sendZclFrameInterPANBroadcast(zclFrame, timeout) {
        throw new Error(`Is not supported for 'zboss' yet`);
    }
    async restoreChannelInterPAN() {
        return;
    }
    waitFor(networkAddress, endpoint, 
    // frameType: Zcl.FrameType,
    // direction: Zcl.Direction,
    transactionSequenceNumber, clusterID, commandIdentifier, timeout) {
        const payload = {
            address: networkAddress,
            endpoint,
            clusterID,
            commandIdentifier,
            transactionSequenceNumber,
        };
        const waiter = this.waitress.waitFor(payload, timeout);
        const cancel = () => this.waitress.remove(waiter.ID);
        return { cancel: cancel, promise: waiter.start().promise, start: waiter.start };
    }
    waitressTimeoutFormatter(matcher, timeout) {
        return (`Timeout - ${matcher.address} - ${matcher.endpoint}` +
            ` - ${matcher.transactionSequenceNumber} - ${matcher.clusterID}` +
            ` - ${matcher.commandIdentifier} after ${timeout}ms`);
    }
    waitressValidator(payload, matcher) {
        return ((payload.header &&
            (!matcher.address || payload.address === matcher.address) &&
            payload.endpoint === matcher.endpoint &&
            (!matcher.transactionSequenceNumber || payload.header.transactionSequenceNumber === matcher.transactionSequenceNumber) &&
            payload.clusterID === matcher.clusterID &&
            matcher.commandIdentifier === payload.header.commandIdentifier) ||
            false);
    }
}
exports.ZBOSSAdapter = ZBOSSAdapter;
//# sourceMappingURL=zbossAdapter.js.map