"use strict";
/* istanbul ignore file */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const buffalo_1 = require("../../../buffalo");
const constants_1 = require("./constants");
const parameterType_1 = __importDefault(require("./parameterType"));
class BuffaloZiGate extends buffalo_1.Buffalo {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any, @typescript-eslint/no-unused-vars
    write(type, value, options) {
        switch (type) {
            case parameterType_1.default.UINT8: {
                return this.writeUInt8(value);
            }
            case parameterType_1.default.UINT16: {
                return this.writeUInt16BE(value);
            }
            case parameterType_1.default.UINT32: {
                return this.writeUInt32BE(value);
            }
            case parameterType_1.default.IEEEADDR: {
                return this.writeIeeeAddrBE(value);
            }
            case parameterType_1.default.BUFFER: {
                return this.writeBuffer(value, value.length);
            }
            case parameterType_1.default.BUFFER8: {
                return this.writeBuffer(value, 8);
            }
            case parameterType_1.default.BUFFER16: {
                return this.writeBuffer(value, 16);
            }
            case parameterType_1.default.BUFFER18: {
                return this.writeBuffer(value, 18);
            }
            case parameterType_1.default.BUFFER32: {
                return this.writeBuffer(value, 32);
            }
            case parameterType_1.default.BUFFER42: {
                return this.writeBuffer(value, 42);
            }
            case parameterType_1.default.BUFFER100: {
                return this.writeBuffer(value, 100);
            }
            case parameterType_1.default.LIST_UINT8: {
                return this.writeListUInt8(value);
            }
            case parameterType_1.default.LIST_UINT16: {
                return this.writeListUInt16BE(value);
            }
            case parameterType_1.default.INT8: {
                return this.writeInt8(value);
            }
            case parameterType_1.default.ADDRESS_WITH_TYPE_DEPENDENCY: {
                const addressMode = this.buffer.readUInt8(this.position - 1);
                return addressMode == 3 ? this.writeIeeeAddrBE(value) : this.writeUInt16BE(value);
            }
            case parameterType_1.default.RAW: {
                return this.writeRaw(value);
            }
        }
        throw new Error(`Write for '${type}' not available`);
    }
    read(type, options) {
        switch (type) {
            case parameterType_1.default.UINT8: {
                return this.readUInt8();
            }
            case parameterType_1.default.UINT16: {
                return this.readUInt16BE();
            }
            case parameterType_1.default.UINT32: {
                return this.readUInt32BE();
            }
            case parameterType_1.default.IEEEADDR: {
                return this.readIeeeAddrBE();
            }
            case parameterType_1.default.BUFFER: {
                // if length option not specified, read the whole buffer
                return this.readBuffer(options.length ?? this.buffer.length);
            }
            case parameterType_1.default.BUFFER8: {
                return this.readBuffer(8);
            }
            case parameterType_1.default.BUFFER16: {
                return this.readBuffer(16);
            }
            case parameterType_1.default.BUFFER18: {
                return this.readBuffer(18);
            }
            case parameterType_1.default.BUFFER32: {
                return this.readBuffer(32);
            }
            case parameterType_1.default.BUFFER42: {
                return this.readBuffer(42);
            }
            case parameterType_1.default.BUFFER100: {
                return this.readBuffer(100);
            }
            case parameterType_1.default.LIST_UINT8: {
                return this.readListUInt8(options.length ?? 0); // XXX: should always be valid?
            }
            case parameterType_1.default.LIST_UINT16: {
                return this.readListUInt16BE(options.length ?? 0); // XXX: should always be valid?
            }
            case parameterType_1.default.INT8: {
                return this.readInt8();
            }
            case parameterType_1.default.MACCAPABILITY: {
                const result = {};
                const mac = this.readUInt8();
                //
                result.alternatePanCoordinator = !!(mac & 0b00000001);
                // bit 0: Alternative PAN Coordinator, always 0
                result.fullFunctionDevice = !!(mac & 0b00000010);
                // bit 1: Device Type, 1 = FFD , 0 = RFD ; cf. https://fr.wikipedia.org/wiki/IEEE_802.15.4
                result.mainsPowerSource = !!(mac & 0b00000100);
                // bit 2: Power Source, 1 = mains power, 0 = other
                result.receiverOnWhenIdle = !!(mac & 0b00001000);
                // bit 3: Receiver on when Idle, 1 = non-sleepy, 0 = sleepy
                result.reserved = (mac & 0b00110000) >> 4;
                // bit 4&5: Reserved
                result.securityCapability = !!(mac & 0b01000000);
                // bit 6: Security capacity, always 0 (standard security)
                result.allocateAddress = !!(mac & 0b10000000);
                // bit 7: 1 = joining device must be issued network address
                return result;
            }
            case parameterType_1.default.ADDRESS_WITH_TYPE_DEPENDENCY: {
                const addressMode = this.buffer.readUInt8(this.position - 1);
                return addressMode == 3 ? this.readIeeeAddrBE() : this.readUInt16BE();
            }
            case parameterType_1.default.BUFFER_RAW: {
                const buffer = this.buffer.subarray(this.position);
                this.position += buffer.length;
                return buffer;
            }
            case parameterType_1.default.STRING: {
                const buffer = this.buffer.subarray(this.position);
                this.position += buffer.length;
                return unescape(buffer.toString());
            }
            case parameterType_1.default.LOG_LEVEL: {
                return constants_1.LOG_LEVEL[this.readUInt8()];
            }
            case parameterType_1.default.MAYBE_UINT8: {
                return this.isMore() ? this.readUInt8() : null;
            }
        }
        throw new Error(`Read for '${type}' not available`);
    }
    writeRaw(value) {
        this.buffer.set(value, this.position);
        this.position += value.length;
    }
    readUInt16BE() {
        const value = this.buffer.readUInt16BE(this.position);
        this.position += 2;
        return value;
    }
    writeUInt16BE(value) {
        this.buffer.writeUInt16BE(value, this.position);
        this.position += 2;
    }
    readUInt32BE() {
        const value = this.buffer.readUInt32BE(this.position);
        this.position += 4;
        return value;
    }
    writeUInt32BE(value) {
        this.buffer.writeUInt32BE(value, this.position);
        this.position += 4;
    }
    readListUInt16BE(length) {
        const value = [];
        for (let i = 0; i < length; i++) {
            value.push(this.readUInt16BE());
        }
        return value;
    }
    writeListUInt16BE(values) {
        for (const value of values) {
            this.writeUInt16BE(value);
        }
    }
    readIeeeAddrBE() {
        const octets = Array.from(this.readBuffer(8));
        return `0x${octets.map((octet) => octet.toString(16).padStart(2, '0')).join('')}`;
    }
    writeIeeeAddrBE(value /*TODO: EUI64*/) {
        this.writeUInt32BE(parseInt(value.slice(2, 10), 16));
        this.writeUInt32BE(parseInt(value.slice(10), 16));
    }
}
exports.default = BuffaloZiGate;
//# sourceMappingURL=buffaloZiGate.js.map