"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MultipleAuthentication = void 0;
var _authentication_type = require("../authentication_type");
var _common = require("../../../../common");
var _next_url = require("../../../utils/next_url");
var _routes = require("./routes");
var _types = require("../../types");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class MultipleAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);
    _defineProperty(this, "authTypes", void 0);
    _defineProperty(this, "authHandlers", void 0);
    this.authTypes = this.config.auth.type;
    this.authHandlers = new Map();
  }
  async init() {
    const routes = new _routes.MultiAuthRoutes(this.router, this.sessionStorageFactory);
    routes.setupRoutes();
    for (let i = 0; i < this.authTypes.length; i++) {
      switch (this.authTypes[i].toLowerCase()) {
        case _common.AuthType.BASIC:
          {
            const BasicAuth = new _types.BasicAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await BasicAuth.init();
            this.authHandlers.set(_common.AuthType.BASIC, BasicAuth);
            break;
          }
        case _common.AuthType.OPEN_ID:
          {
            const OidcAuth = new _types.OpenIdAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await OidcAuth.init();
            this.authHandlers.set(_common.AuthType.OPEN_ID, OidcAuth);
            break;
          }
        case _common.AuthType.SAML:
          {
            const SamlAuth = new _types.SamlAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await SamlAuth.init();
            this.authHandlers.set(_common.AuthType.SAML, SamlAuth);
            break;
          }
        default:
          {
            throw new Error(`Unsupported authentication type: ${this.authTypes[i]}`);
          }
      }
    }
  }

  // override functions inherited from AuthenticationType
  requestIncludesAuthInfo(request) {
    for (const key of this.authHandlers.keys()) {
      if (this.authHandlers.get(key).requestIncludesAuthInfo(request)) {
        return true;
      }
    }
    return false;
  }
  async getAdditionalAuthHeader(request) {
    var _cookie$authType;
    // To Do: refactor this method to improve the effiency to get cookie, get cookie from input parameter
    const cookie = await this.sessionStorageFactory.asScoped(request).get();
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType = cookie.authType) === null || _cookie$authType === void 0 ? void 0 : _cookie$authType.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return await this.authHandlers.get(reqAuthType).getAdditionalAuthHeader(request);
    } else {
      return {};
    }
  }
  getCookie(request, authInfo) {
    return {};
  }
  getKeepAliveExpiry(cookie, request) {
    var _cookie$authType2;
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType2 = cookie.authType) === null || _cookie$authType2 === void 0 ? void 0 : _cookie$authType2.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).getKeepAliveExpiry(cookie, request);
    } else {
      // default to TTL setting
      return Date.now() + this.config.session.ttl;
    }
  }
  async isValidCookie(cookie, request) {
    var _cookie$authType3;
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType3 = cookie.authType) === null || _cookie$authType3 === void 0 ? void 0 : _cookie$authType3.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).isValidCookie(cookie, request);
    } else {
      return false;
    }
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      const nextUrlParam = (0, _next_url.composeNextUrlQueryParam)(request, this.coreSetup.http.basePath.serverBasePath);
      return response.redirected({
        headers: {
          location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}?${nextUrlParam}`
        }
      });
    } else {
      return response.unauthorized();
    }
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$authType4;
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType4 = cookie.authType) === null || _cookie$authType4 === void 0 ? void 0 : _cookie$authType4.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).buildAuthHeaderFromCookie(cookie, request);
    } else {
      return {};
    }
  }
}
exports.MultipleAuthentication = MultipleAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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