"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AuthenticationType = void 0;
var _opensearch_security_client = require("../../backend/opensearch_security_client");
var _tenant_resolver = require("../../multitenancy/tenant_resolver");
var _errors = require("../../errors");
var _common = require("../../../common");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.router = router;
    this.esClient = esClient;
    this.coreSetup = coreSetup;
    this.logger = logger;
    _defineProperty(this, "type", void 0);
    _defineProperty(this, "securityClient", void 0);
    _defineProperty(this, "authHandler", async (request, response, toolkit) => {
      var _this$config$multiten;
      // skip auth for APIs that do not require auth
      if (this.authNotRequired(request)) {
        return toolkit.authenticated();
      }
      const authState = {};

      // if browser request, auth logic is:
      //   1. check if request includes auth header or parameter(e.g. jwt in url params) is present, if so, authenticate with auth header.
      //   2. if auth header not present, check if auth cookie is present, if no cookie, send to authentication workflow
      //   3. verify whether auth cookie is valid, if not valid, send to authentication workflow
      //   4. if cookie is valid, pass to route handlers
      const authHeaders = {};
      let cookie;
      let authInfo;
      // if this is an REST API call, suppose the request includes necessary auth header
      if (this.requestIncludesAuthInfo(request)) {
        try {
          const additionalAuthHeader = await this.getAdditionalAuthHeader(request);
          Object.assign(authHeaders, additionalAuthHeader);
          authInfo = await this.securityClient.authinfo(request, additionalAuthHeader);
          cookie = this.getCookie(request, authInfo);

          // set tenant from cookie if exist
          const browserCookie = await this.sessionStorageFactory.asScoped(request).get();
          if (browserCookie && (0, _tenant_resolver.isValidTenant)(browserCookie.tenant)) {
            cookie.tenant = browserCookie.tenant;
          }
          this.sessionStorageFactory.asScoped(request).set(cookie);
        } catch (error) {
          return response.unauthorized({
            body: error.message
          });
        }
      } else {
        // no auth header in request, try cookie
        try {
          cookie = await this.sessionStorageFactory.asScoped(request).get();
        } catch (error) {
          this.logger.error(`Error parsing cookie: ${error.message}`);
          cookie = undefined;
        }
        if (!cookie || !(await this.isValidCookie(cookie, request))) {
          // clear cookie
          this.sessionStorageFactory.asScoped(request).clear();

          // for assets, we can still pass it to resource handler as notHandled.
          // marking it as authenticated may result in login pop up when auth challenge
          // is enabled.
          if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
            return toolkit.notHandled();
          }

          // allow optional authentication
          if (this.authOptional(request)) {
            return toolkit.authenticated();
          }

          // send to auth workflow
          return this.handleUnauthedRequest(request, response, toolkit);
        }

        // extend session expiration time
        if (this.config.session.keepalive) {
          cookie.expiryTime = this.getKeepAliveExpiry(cookie, request);
          this.sessionStorageFactory.asScoped(request).set(cookie);
        }
        // cookie is valid
        // build auth header
        const authHeadersFromCookie = this.buildAuthHeaderFromCookie(cookie, request);
        Object.assign(authHeaders, authHeadersFromCookie);
        const additionalAuthHeader = await this.getAdditionalAuthHeader(request);
        Object.assign(authHeaders, additionalAuthHeader);
      }

      // resolve tenant if necessary
      if ((_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled) {
        try {
          const tenant = await this.resolveTenant(request, cookie, authHeaders, authInfo);
          // return 401 if no tenant available
          if (!(0, _tenant_resolver.isValidTenant)(tenant)) {
            return response.badRequest({
              body: 'No available tenant for current user, please reach out to your system administrator'
            });
          }
          authState.selectedTenant = tenant;

          // set tenant in header
          if (this.config.multitenancy.enabled && this.config.multitenancy.enable_aggregation_view) {
            // Store all saved objects in a single kibana index.
            Object.assign(authHeaders, {
              securitytenant: _common.GLOBAL_TENANT_SYMBOL
            });
          } else {
            Object.assign(authHeaders, {
              securitytenant: tenant
            });
          }

          // set tenant to cookie
          if (tenant !== cookie.tenant) {
            cookie.tenant = tenant;
            this.sessionStorageFactory.asScoped(request).set(cookie);
          }
        } catch (error) {
          this.logger.error(`Failed to resolve user tenant: ${error}`);
          if (error instanceof _errors.UnauthenticatedError) {
            if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
              return toolkit.notHandled();
            }
            this.sessionStorageFactory.asScoped(request).clear();
            return this.handleUnauthedRequest(request, response, toolkit);
          }
          throw error;
        }
      }
      if (!authInfo) {
        authInfo = await this.securityClient.authinfo(request, authHeaders);
      }
      authState.authInfo = authInfo;
      return toolkit.authenticated({
        requestHeaders: authHeaders,
        state: authState
      });
    });
    this.securityClient = new _opensearch_security_client.SecurityClient(esClient);
    this.type = '';
    this.config = config;
  }
  authNotRequired(request) {
    const pathname = request.url.pathname;
    if (!pathname) {
      return false;
    }
    // allow requests to ignored routes
    if (AuthenticationType.ROUTES_TO_IGNORE.includes(pathname)) {
      return true;
    }
    // allow requests to routes that doesn't require authentication
    if (this.config.auth.unauthenticated_routes.indexOf(pathname) > -1) {
      // TODO: use opensearch-dashboards server user
      return true;
    }
    return false;
  }
  authOptional(request) {
    const pathname = request.url.pathname;
    if (!pathname) {
      return false;
    }
    return AuthenticationType.ROUTES_AUTH_OPTIONAL.includes(pathname);
  }
  async resolveTenant(request, cookie, authHeader, authInfo) {
    if (!authInfo) {
      try {
        authInfo = await this.securityClient.authinfo(request, authHeader);
      } catch (error) {
        throw new _errors.UnauthenticatedError(error);
      }
    }
    const dashboardsInfo = await this.securityClient.dashboardsinfo(request, authHeader);
    return (0, _tenant_resolver.resolveTenant)({
      request,
      username: authInfo.user_name,
      roles: authInfo.roles,
      availableTenants: authInfo.tenants,
      config: this.config,
      cookie,
      multitenancyEnabled: dashboardsInfo.multitenancy_enabled,
      privateTenantEnabled: dashboardsInfo.private_tenant_enabled,
      defaultTenant: dashboardsInfo.default_tenant
    });
  }
  getKeepAliveExpiry(cookie, request) {
    return Date.now() + this.config.session.ttl;
  }
  isPageRequest(request) {
    const path = request.url.pathname || '/';
    return path.startsWith('/app/') || path === '/' || path.startsWith('/goto/');
  }

  // abstract functions for concrete auth types to implement
}
exports.AuthenticationType = AuthenticationType;
_defineProperty(AuthenticationType, "ROUTES_TO_IGNORE", ['/app/login']);
_defineProperty(AuthenticationType, "ROUTES_AUTH_OPTIONAL", ['/api/core/capabilities']);
_defineProperty(AuthenticationType, "REST_API_CALL_HEADER", 'osd-xsrf');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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