"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerTraceAnalyticsDslRouter = registerTraceAnalyticsDslRouter;
var _configSchema = require("@osd/config-schema");
var _trace_analytics = require("../../common/constants/trace_analytics");
var _metrics_helper = require("../common/metrics/metrics_helper");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerTraceAnalyticsDslRouter(router, dataSourceEnabled) {
  router.post({
    path: _trace_analytics.TRACE_ANALYTICS_DATA_PREPPER_INDICES_ROUTE,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const params = {
      index: [_trace_analytics.DATA_PREPPER_INDEX_NAME, _trace_analytics.DATA_PREPPER_SERVICE_INDEX_NAME],
      allow_no_indices: false
    };
    const {
      dataSourceMDSId
    } = request.query;
    try {
      let resp;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        resp = await client.callAPI('indices.exists', params);
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('indices.exists', params);
      }
      return response.ok({
        body: resp
      });
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _trace_analytics.TRACE_ANALYTICS_JAEGER_INDICES_ROUTE,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const {
      dataSourceMDSId
    } = request.query;
    const params = {
      index: [_trace_analytics.JAEGER_INDEX_NAME, _trace_analytics.JAEGER_SERVICE_INDEX_NAME],
      allow_no_indices: false
    };
    try {
      let resp;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        resp = await client.callAPI('indices.exists', params);
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('indices.exists', params);
      }
      return response.ok({
        body: resp
      });
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _trace_analytics.TRACE_ANALYTICS_DSL_ROUTE,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.maybe(_configSchema.schema.string()),
        from: _configSchema.schema.maybe(_configSchema.schema.number()),
        size: _configSchema.schema.number(),
        query: _configSchema.schema.maybe(_configSchema.schema.object({
          bool: _configSchema.schema.object({
            filter: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
              unknowns: 'allow'
            }))),
            must: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
              unknowns: 'allow'
            }))),
            should: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
              unknowns: 'allow'
            }))),
            must_not: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
              unknowns: 'allow'
            }))),
            minimum_should_match: _configSchema.schema.maybe(_configSchema.schema.number()),
            adjust_pure_negative: _configSchema.schema.maybe(_configSchema.schema.boolean()),
            boost: _configSchema.schema.maybe(_configSchema.schema.any())
          })
        })),
        aggs: _configSchema.schema.maybe(_configSchema.schema.any()),
        aggregations: _configSchema.schema.maybe(_configSchema.schema.any()),
        sort: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.any())),
        _source: _configSchema.schema.maybe(_configSchema.schema.object({
          includes: _configSchema.schema.arrayOf(_configSchema.schema.string())
        })),
        script_fields: _configSchema.schema.maybe(_configSchema.schema.any()),
        track_total_hits: _configSchema.schema.maybe(_configSchema.schema.boolean())
      }),
      query: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('trace_analytics', 'get', 'count');
    const {
      index,
      size,
      ...rest
    } = request.body;
    const {
      dataSourceMDSId
    } = request.query;
    const params = {
      index: index || _trace_analytics.DATA_PREPPER_INDEX_NAME,
      size,
      body: rest
    };
    try {
      let resp;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        resp = await client.callAPI('search', params);
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
      }
      return response.ok({
        body: resp
      });
    } catch (error) {
      (0, _metrics_helper.addRequestToMetric)('trace_analytics', 'get', error);
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsIl90cmFjZV9hbmFseXRpY3MiLCJfbWV0cmljc19oZWxwZXIiLCJyZWdpc3RlclRyYWNlQW5hbHl0aWNzRHNsUm91dGVyIiwicm91dGVyIiwiZGF0YVNvdXJjZUVuYWJsZWQiLCJwb3N0IiwicGF0aCIsIlRSQUNFX0FOQUxZVElDU19EQVRBX1BSRVBQRVJfSU5ESUNFU19ST1VURSIsInZhbGlkYXRlIiwiYm9keSIsInNjaGVtYSIsImFueSIsInF1ZXJ5Iiwib2JqZWN0IiwiZGF0YVNvdXJjZU1EU0lkIiwibWF5YmUiLCJzdHJpbmciLCJkZWZhdWx0VmFsdWUiLCJjb250ZXh0IiwicmVxdWVzdCIsInJlc3BvbnNlIiwicGFyYW1zIiwiaW5kZXgiLCJEQVRBX1BSRVBQRVJfSU5ERVhfTkFNRSIsIkRBVEFfUFJFUFBFUl9TRVJWSUNFX0lOREVYX05BTUUiLCJhbGxvd19ub19pbmRpY2VzIiwicmVzcCIsImNsaWVudCIsImRhdGFTb3VyY2UiLCJvcGVuc2VhcmNoIiwibGVnYWN5IiwiZ2V0Q2xpZW50IiwiY2FsbEFQSSIsImNvcmUiLCJjYWxsQXNDdXJyZW50VXNlciIsIm9rIiwiZXJyb3IiLCJjb25zb2xlIiwiY3VzdG9tIiwic3RhdHVzQ29kZSIsIm1lc3NhZ2UiLCJUUkFDRV9BTkFMWVRJQ1NfSkFFR0VSX0lORElDRVNfUk9VVEUiLCJKQUVHRVJfSU5ERVhfTkFNRSIsIkpBRUdFUl9TRVJWSUNFX0lOREVYX05BTUUiLCJUUkFDRV9BTkFMWVRJQ1NfRFNMX1JPVVRFIiwiZnJvbSIsIm51bWJlciIsInNpemUiLCJib29sIiwiZmlsdGVyIiwiYXJyYXlPZiIsInVua25vd25zIiwibXVzdCIsInNob3VsZCIsIm11c3Rfbm90IiwibWluaW11bV9zaG91bGRfbWF0Y2giLCJhZGp1c3RfcHVyZV9uZWdhdGl2ZSIsImJvb2xlYW4iLCJib29zdCIsImFnZ3MiLCJhZ2dyZWdhdGlvbnMiLCJzb3J0IiwiX3NvdXJjZSIsImluY2x1ZGVzIiwic2NyaXB0X2ZpZWxkcyIsInRyYWNrX3RvdGFsX2hpdHMiLCJhZGRSZXF1ZXN0VG9NZXRyaWMiLCJyZXN0Il0sInNvdXJjZXMiOlsidHJhY2VfYW5hbHl0aWNzX2RzbF9yb3V0ZXIudHMiXSwic291cmNlc0NvbnRlbnQiOlsiLypcbiAqIENvcHlyaWdodCBPcGVuU2VhcmNoIENvbnRyaWJ1dG9yc1xuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG5pbXBvcnQgeyBSZXF1ZXN0UGFyYW1zIH0gZnJvbSAnQG9wZW5zZWFyY2gtcHJvamVjdC9vcGVuc2VhcmNoJztcbmltcG9ydCB7IHNjaGVtYSB9IGZyb20gJ0Bvc2QvY29uZmlnLXNjaGVtYSc7XG5pbXBvcnQgeyBJUm91dGVyIH0gZnJvbSAnLi4vLi4vLi4vLi4vc3JjL2NvcmUvc2VydmVyJztcbmltcG9ydCB7XG4gIERBVEFfUFJFUFBFUl9JTkRFWF9OQU1FLFxuICBEQVRBX1BSRVBQRVJfU0VSVklDRV9JTkRFWF9OQU1FLFxuICBKQUVHRVJfSU5ERVhfTkFNRSxcbiAgSkFFR0VSX1NFUlZJQ0VfSU5ERVhfTkFNRSxcbiAgVFJBQ0VfQU5BTFlUSUNTX0RBVEFfUFJFUFBFUl9JTkRJQ0VTX1JPVVRFLFxuICBUUkFDRV9BTkFMWVRJQ1NfRFNMX1JPVVRFLFxuICBUUkFDRV9BTkFMWVRJQ1NfSkFFR0VSX0lORElDRVNfUk9VVEUsXG59IGZyb20gJy4uLy4uL2NvbW1vbi9jb25zdGFudHMvdHJhY2VfYW5hbHl0aWNzJztcbmltcG9ydCB7IGFkZFJlcXVlc3RUb01ldHJpYyB9IGZyb20gJy4uL2NvbW1vbi9tZXRyaWNzL21ldHJpY3NfaGVscGVyJztcblxuZXhwb3J0IGZ1bmN0aW9uIHJlZ2lzdGVyVHJhY2VBbmFseXRpY3NEc2xSb3V0ZXIocm91dGVyOiBJUm91dGVyLCBkYXRhU291cmNlRW5hYmxlZDogYm9vbGVhbikge1xuICByb3V0ZXIucG9zdChcbiAgICB7XG4gICAgICBwYXRoOiBUUkFDRV9BTkFMWVRJQ1NfREFUQV9QUkVQUEVSX0lORElDRVNfUk9VVEUsXG4gICAgICB2YWxpZGF0ZToge1xuICAgICAgICBib2R5OiBzY2hlbWEuYW55KCksXG4gICAgICAgIHF1ZXJ5OiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICBkYXRhU291cmNlTURTSWQ6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKHsgZGVmYXVsdFZhbHVlOiAnJyB9KSksXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9LFxuICAgIGFzeW5jIChjb250ZXh0LCByZXF1ZXN0LCByZXNwb25zZSkgPT4ge1xuICAgICAgY29uc3QgcGFyYW1zOiBSZXF1ZXN0UGFyYW1zLkluZGljZXNFeGlzdHMgPSB7XG4gICAgICAgIGluZGV4OiBbREFUQV9QUkVQUEVSX0lOREVYX05BTUUsIERBVEFfUFJFUFBFUl9TRVJWSUNFX0lOREVYX05BTUVdLFxuICAgICAgICBhbGxvd19ub19pbmRpY2VzOiBmYWxzZSxcbiAgICAgIH07XG4gICAgICBjb25zdCB7IGRhdGFTb3VyY2VNRFNJZCB9ID0gcmVxdWVzdC5xdWVyeTtcbiAgICAgIHRyeSB7XG4gICAgICAgIGxldCByZXNwO1xuICAgICAgICBpZiAoZGF0YVNvdXJjZUVuYWJsZWQgJiYgZGF0YVNvdXJjZU1EU0lkKSB7XG4gICAgICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5kYXRhU291cmNlLm9wZW5zZWFyY2gubGVnYWN5LmdldENsaWVudChkYXRhU291cmNlTURTSWQpO1xuICAgICAgICAgIHJlc3AgPSBhd2FpdCBjbGllbnQuY2FsbEFQSSgnaW5kaWNlcy5leGlzdHMnLCBwYXJhbXMpO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIHJlc3AgPSBhd2FpdCBjb250ZXh0LmNvcmUub3BlbnNlYXJjaC5sZWdhY3kuY2xpZW50LmNhbGxBc0N1cnJlbnRVc2VyKFxuICAgICAgICAgICAgJ2luZGljZXMuZXhpc3RzJyxcbiAgICAgICAgICAgIHBhcmFtc1xuICAgICAgICAgICk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiByZXNwLFxuICAgICAgICB9KTtcbiAgICAgIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoZXJyb3IpO1xuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcbiAgICAgICAgICBzdGF0dXNDb2RlOiBlcnJvci5zdGF0dXNDb2RlIHx8IDUwMCxcbiAgICAgICAgICBib2R5OiBlcnJvci5tZXNzYWdlLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICk7XG4gIHJvdXRlci5wb3N0KFxuICAgIHtcbiAgICAgIHBhdGg6IFRSQUNFX0FOQUxZVElDU19KQUVHRVJfSU5ESUNFU19ST1VURSxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5hbnkoKSxcbiAgICAgICAgcXVlcnk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGRhdGFTb3VyY2VNRFNJZDogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoeyBkZWZhdWx0VmFsdWU6ICcnIH0pKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICBjb25zdCB7IGRhdGFTb3VyY2VNRFNJZCB9ID0gcmVxdWVzdC5xdWVyeTtcbiAgICAgIGNvbnN0IHBhcmFtczogUmVxdWVzdFBhcmFtcy5JbmRpY2VzRXhpc3RzID0ge1xuICAgICAgICBpbmRleDogW0pBRUdFUl9JTkRFWF9OQU1FLCBKQUVHRVJfU0VSVklDRV9JTkRFWF9OQU1FXSxcbiAgICAgICAgYWxsb3dfbm9faW5kaWNlczogZmFsc2UsXG4gICAgICB9O1xuICAgICAgdHJ5IHtcbiAgICAgICAgbGV0IHJlc3A7XG4gICAgICAgIGlmIChkYXRhU291cmNlRW5hYmxlZCAmJiBkYXRhU291cmNlTURTSWQpIHtcbiAgICAgICAgICBjb25zdCBjbGllbnQgPSBjb250ZXh0LmRhdGFTb3VyY2Uub3BlbnNlYXJjaC5sZWdhY3kuZ2V0Q2xpZW50KGRhdGFTb3VyY2VNRFNJZCk7XG4gICAgICAgICAgcmVzcCA9IGF3YWl0IGNsaWVudC5jYWxsQVBJKCdpbmRpY2VzLmV4aXN0cycsIHBhcmFtcyk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgcmVzcCA9IGF3YWl0IGNvbnRleHQuY29yZS5vcGVuc2VhcmNoLmxlZ2FjeS5jbGllbnQuY2FsbEFzQ3VycmVudFVzZXIoXG4gICAgICAgICAgICAnaW5kaWNlcy5leGlzdHMnLFxuICAgICAgICAgICAgcGFyYW1zXG4gICAgICAgICAgKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgICAgIGJvZHk6IHJlc3AsXG4gICAgICAgIH0pO1xuICAgICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcihlcnJvcik7XG4gICAgICAgIHJldHVybiByZXNwb25zZS5jdXN0b20oe1xuICAgICAgICAgIHN0YXR1c0NvZGU6IGVycm9yLnN0YXR1c0NvZGUgfHwgNTAwLFxuICAgICAgICAgIGJvZHk6IGVycm9yLm1lc3NhZ2UsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgKTtcbiAgcm91dGVyLnBvc3QoXG4gICAge1xuICAgICAgcGF0aDogVFJBQ0VfQU5BTFlUSUNTX0RTTF9ST1VURSxcbiAgICAgIHZhbGlkYXRlOiB7XG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGluZGV4OiBzY2hlbWEubWF5YmUoc2NoZW1hLnN0cmluZygpKSxcbiAgICAgICAgICBmcm9tOiBzY2hlbWEubWF5YmUoc2NoZW1hLm51bWJlcigpKSxcbiAgICAgICAgICBzaXplOiBzY2hlbWEubnVtYmVyKCksXG4gICAgICAgICAgcXVlcnk6IHNjaGVtYS5tYXliZShcbiAgICAgICAgICAgIHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgICAgICBib29sOiBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICAgICAgICBmaWx0ZXI6IHNjaGVtYS5tYXliZShzY2hlbWEuYXJyYXlPZihzY2hlbWEub2JqZWN0KHt9LCB7IHVua25vd25zOiAnYWxsb3cnIH0pKSksXG4gICAgICAgICAgICAgICAgbXVzdDogc2NoZW1hLm1heWJlKHNjaGVtYS5hcnJheU9mKHNjaGVtYS5vYmplY3Qoe30sIHsgdW5rbm93bnM6ICdhbGxvdycgfSkpKSxcbiAgICAgICAgICAgICAgICBzaG91bGQ6IHNjaGVtYS5tYXliZShzY2hlbWEuYXJyYXlPZihzY2hlbWEub2JqZWN0KHt9LCB7IHVua25vd25zOiAnYWxsb3cnIH0pKSksXG4gICAgICAgICAgICAgICAgbXVzdF9ub3Q6IHNjaGVtYS5tYXliZShzY2hlbWEuYXJyYXlPZihzY2hlbWEub2JqZWN0KHt9LCB7IHVua25vd25zOiAnYWxsb3cnIH0pKSksXG4gICAgICAgICAgICAgICAgbWluaW11bV9zaG91bGRfbWF0Y2g6IHNjaGVtYS5tYXliZShzY2hlbWEubnVtYmVyKCkpLFxuICAgICAgICAgICAgICAgIGFkanVzdF9wdXJlX25lZ2F0aXZlOiBzY2hlbWEubWF5YmUoc2NoZW1hLmJvb2xlYW4oKSksXG4gICAgICAgICAgICAgICAgYm9vc3Q6IHNjaGVtYS5tYXliZShzY2hlbWEuYW55KCkpLFxuICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgIH0pXG4gICAgICAgICAgKSxcbiAgICAgICAgICBhZ2dzOiBzY2hlbWEubWF5YmUoc2NoZW1hLmFueSgpKSxcbiAgICAgICAgICBhZ2dyZWdhdGlvbnM6IHNjaGVtYS5tYXliZShzY2hlbWEuYW55KCkpLFxuICAgICAgICAgIHNvcnQ6IHNjaGVtYS5tYXliZShzY2hlbWEuYXJyYXlPZihzY2hlbWEuYW55KCkpKSxcbiAgICAgICAgICBfc291cmNlOiBzY2hlbWEubWF5YmUoXG4gICAgICAgICAgICBzY2hlbWEub2JqZWN0KHtcbiAgICAgICAgICAgICAgaW5jbHVkZXM6IHNjaGVtYS5hcnJheU9mKHNjaGVtYS5zdHJpbmcoKSksXG4gICAgICAgICAgICB9KVxuICAgICAgICAgICksXG4gICAgICAgICAgc2NyaXB0X2ZpZWxkczogc2NoZW1hLm1heWJlKHNjaGVtYS5hbnkoKSksXG4gICAgICAgICAgdHJhY2tfdG90YWxfaGl0czogc2NoZW1hLm1heWJlKHNjaGVtYS5ib29sZWFuKCkpLFxuICAgICAgICB9KSxcbiAgICAgICAgcXVlcnk6IHNjaGVtYS5vYmplY3Qoe1xuICAgICAgICAgIGRhdGFTb3VyY2VNRFNJZDogc2NoZW1hLm1heWJlKHNjaGVtYS5zdHJpbmcoeyBkZWZhdWx0VmFsdWU6ICcnIH0pKSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0sXG4gICAgYXN5bmMgKGNvbnRleHQsIHJlcXVlc3QsIHJlc3BvbnNlKSA9PiB7XG4gICAgICBhZGRSZXF1ZXN0VG9NZXRyaWMoJ3RyYWNlX2FuYWx5dGljcycsICdnZXQnLCAnY291bnQnKTtcbiAgICAgIGNvbnN0IHsgaW5kZXgsIHNpemUsIC4uLnJlc3QgfSA9IHJlcXVlc3QuYm9keTtcbiAgICAgIGNvbnN0IHsgZGF0YVNvdXJjZU1EU0lkIH0gPSByZXF1ZXN0LnF1ZXJ5O1xuICAgICAgY29uc3QgcGFyYW1zOiBSZXF1ZXN0UGFyYW1zLlNlYXJjaCA9IHtcbiAgICAgICAgaW5kZXg6IGluZGV4IHx8IERBVEFfUFJFUFBFUl9JTkRFWF9OQU1FLFxuICAgICAgICBzaXplLFxuICAgICAgICBib2R5OiByZXN0LFxuICAgICAgfTtcbiAgICAgIHRyeSB7XG4gICAgICAgIGxldCByZXNwO1xuICAgICAgICBpZiAoZGF0YVNvdXJjZUVuYWJsZWQgJiYgZGF0YVNvdXJjZU1EU0lkKSB7XG4gICAgICAgICAgY29uc3QgY2xpZW50ID0gY29udGV4dC5kYXRhU291cmNlLm9wZW5zZWFyY2gubGVnYWN5LmdldENsaWVudChkYXRhU291cmNlTURTSWQpO1xuICAgICAgICAgIHJlc3AgPSBhd2FpdCBjbGllbnQuY2FsbEFQSSgnc2VhcmNoJywgcGFyYW1zKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXNwID0gYXdhaXQgY29udGV4dC5jb3JlLm9wZW5zZWFyY2gubGVnYWN5LmNsaWVudC5jYWxsQXNDdXJyZW50VXNlcignc2VhcmNoJywgcGFyYW1zKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmVzcG9uc2Uub2soe1xuICAgICAgICAgIGJvZHk6IHJlc3AsXG4gICAgICAgIH0pO1xuICAgICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgICAgYWRkUmVxdWVzdFRvTWV0cmljKCd0cmFjZV9hbmFseXRpY3MnLCAnZ2V0JywgZXJyb3IpO1xuICAgICAgICBpZiAoZXJyb3Iuc3RhdHVzQ29kZSAhPT0gNDA0KSBjb25zb2xlLmVycm9yKGVycm9yKTtcbiAgICAgICAgcmV0dXJuIHJlc3BvbnNlLmN1c3RvbSh7XG4gICAgICAgICAgc3RhdHVzQ29kZTogZXJyb3Iuc3RhdHVzQ29kZSB8fCA1MDAsXG4gICAgICAgICAgYm9keTogZXJyb3IubWVzc2FnZSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICApO1xufVxuIl0sIm1hcHBpbmdzIjoiOzs7Ozs7QUFNQSxJQUFBQSxhQUFBLEdBQUFDLE9BQUE7QUFFQSxJQUFBQyxnQkFBQSxHQUFBRCxPQUFBO0FBU0EsSUFBQUUsZUFBQSxHQUFBRixPQUFBO0FBakJBO0FBQ0E7QUFDQTtBQUNBOztBQWdCTyxTQUFTRywrQkFBK0JBLENBQUNDLE1BQWUsRUFBRUMsaUJBQTBCLEVBQUU7RUFDM0ZELE1BQU0sQ0FBQ0UsSUFBSSxDQUNUO0lBQ0VDLElBQUksRUFBRUMsMkRBQTBDO0lBQ2hEQyxRQUFRLEVBQUU7TUFDUkMsSUFBSSxFQUFFQyxvQkFBTSxDQUFDQyxHQUFHLENBQUMsQ0FBQztNQUNsQkMsS0FBSyxFQUFFRixvQkFBTSxDQUFDRyxNQUFNLENBQUM7UUFDbkJDLGVBQWUsRUFBRUosb0JBQU0sQ0FBQ0ssS0FBSyxDQUFDTCxvQkFBTSxDQUFDTSxNQUFNLENBQUM7VUFBRUMsWUFBWSxFQUFFO1FBQUcsQ0FBQyxDQUFDO01BQ25FLENBQUM7SUFDSDtFQUNGLENBQUMsRUFDRCxPQUFPQyxPQUFPLEVBQUVDLE9BQU8sRUFBRUMsUUFBUSxLQUFLO0lBQ3BDLE1BQU1DLE1BQW1DLEdBQUc7TUFDMUNDLEtBQUssRUFBRSxDQUFDQyx3Q0FBdUIsRUFBRUMsZ0RBQStCLENBQUM7TUFDakVDLGdCQUFnQixFQUFFO0lBQ3BCLENBQUM7SUFDRCxNQUFNO01BQUVYO0lBQWdCLENBQUMsR0FBR0ssT0FBTyxDQUFDUCxLQUFLO0lBQ3pDLElBQUk7TUFDRixJQUFJYyxJQUFJO01BQ1IsSUFBSXRCLGlCQUFpQixJQUFJVSxlQUFlLEVBQUU7UUFDeEMsTUFBTWEsTUFBTSxHQUFHVCxPQUFPLENBQUNVLFVBQVUsQ0FBQ0MsVUFBVSxDQUFDQyxNQUFNLENBQUNDLFNBQVMsQ0FBQ2pCLGVBQWUsQ0FBQztRQUM5RVksSUFBSSxHQUFHLE1BQU1DLE1BQU0sQ0FBQ0ssT0FBTyxDQUFDLGdCQUFnQixFQUFFWCxNQUFNLENBQUM7TUFDdkQsQ0FBQyxNQUFNO1FBQ0xLLElBQUksR0FBRyxNQUFNUixPQUFPLENBQUNlLElBQUksQ0FBQ0osVUFBVSxDQUFDQyxNQUFNLENBQUNILE1BQU0sQ0FBQ08saUJBQWlCLENBQ2xFLGdCQUFnQixFQUNoQmIsTUFDRixDQUFDO01BQ0g7TUFDQSxPQUFPRCxRQUFRLENBQUNlLEVBQUUsQ0FBQztRQUNqQjFCLElBQUksRUFBRWlCO01BQ1IsQ0FBQyxDQUFDO0lBQ0osQ0FBQyxDQUFDLE9BQU9VLEtBQUssRUFBRTtNQUNkQyxPQUFPLENBQUNELEtBQUssQ0FBQ0EsS0FBSyxDQUFDO01BQ3BCLE9BQU9oQixRQUFRLENBQUNrQixNQUFNLENBQUM7UUFDckJDLFVBQVUsRUFBRUgsS0FBSyxDQUFDRyxVQUFVLElBQUksR0FBRztRQUNuQzlCLElBQUksRUFBRTJCLEtBQUssQ0FBQ0k7TUFDZCxDQUFDLENBQUM7SUFDSjtFQUNGLENBQ0YsQ0FBQztFQUNEckMsTUFBTSxDQUFDRSxJQUFJLENBQ1Q7SUFDRUMsSUFBSSxFQUFFbUMscURBQW9DO0lBQzFDakMsUUFBUSxFQUFFO01BQ1JDLElBQUksRUFBRUMsb0JBQU0sQ0FBQ0MsR0FBRyxDQUFDLENBQUM7TUFDbEJDLEtBQUssRUFBRUYsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO1FBQ25CQyxlQUFlLEVBQUVKLG9CQUFNLENBQUNLLEtBQUssQ0FBQ0wsb0JBQU0sQ0FBQ00sTUFBTSxDQUFDO1VBQUVDLFlBQVksRUFBRTtRQUFHLENBQUMsQ0FBQztNQUNuRSxDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0QsT0FBT0MsT0FBTyxFQUFFQyxPQUFPLEVBQUVDLFFBQVEsS0FBSztJQUNwQyxNQUFNO01BQUVOO0lBQWdCLENBQUMsR0FBR0ssT0FBTyxDQUFDUCxLQUFLO0lBQ3pDLE1BQU1TLE1BQW1DLEdBQUc7TUFDMUNDLEtBQUssRUFBRSxDQUFDb0Isa0NBQWlCLEVBQUVDLDBDQUF5QixDQUFDO01BQ3JEbEIsZ0JBQWdCLEVBQUU7SUFDcEIsQ0FBQztJQUNELElBQUk7TUFDRixJQUFJQyxJQUFJO01BQ1IsSUFBSXRCLGlCQUFpQixJQUFJVSxlQUFlLEVBQUU7UUFDeEMsTUFBTWEsTUFBTSxHQUFHVCxPQUFPLENBQUNVLFVBQVUsQ0FBQ0MsVUFBVSxDQUFDQyxNQUFNLENBQUNDLFNBQVMsQ0FBQ2pCLGVBQWUsQ0FBQztRQUM5RVksSUFBSSxHQUFHLE1BQU1DLE1BQU0sQ0FBQ0ssT0FBTyxDQUFDLGdCQUFnQixFQUFFWCxNQUFNLENBQUM7TUFDdkQsQ0FBQyxNQUFNO1FBQ0xLLElBQUksR0FBRyxNQUFNUixPQUFPLENBQUNlLElBQUksQ0FBQ0osVUFBVSxDQUFDQyxNQUFNLENBQUNILE1BQU0sQ0FBQ08saUJBQWlCLENBQ2xFLGdCQUFnQixFQUNoQmIsTUFDRixDQUFDO01BQ0g7TUFDQSxPQUFPRCxRQUFRLENBQUNlLEVBQUUsQ0FBQztRQUNqQjFCLElBQUksRUFBRWlCO01BQ1IsQ0FBQyxDQUFDO0lBQ0osQ0FBQyxDQUFDLE9BQU9VLEtBQUssRUFBRTtNQUNkQyxPQUFPLENBQUNELEtBQUssQ0FBQ0EsS0FBSyxDQUFDO01BQ3BCLE9BQU9oQixRQUFRLENBQUNrQixNQUFNLENBQUM7UUFDckJDLFVBQVUsRUFBRUgsS0FBSyxDQUFDRyxVQUFVLElBQUksR0FBRztRQUNuQzlCLElBQUksRUFBRTJCLEtBQUssQ0FBQ0k7TUFDZCxDQUFDLENBQUM7SUFDSjtFQUNGLENBQ0YsQ0FBQztFQUNEckMsTUFBTSxDQUFDRSxJQUFJLENBQ1Q7SUFDRUMsSUFBSSxFQUFFc0MsMENBQXlCO0lBQy9CcEMsUUFBUSxFQUFFO01BQ1JDLElBQUksRUFBRUMsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO1FBQ2xCUyxLQUFLLEVBQUVaLG9CQUFNLENBQUNLLEtBQUssQ0FBQ0wsb0JBQU0sQ0FBQ00sTUFBTSxDQUFDLENBQUMsQ0FBQztRQUNwQzZCLElBQUksRUFBRW5DLG9CQUFNLENBQUNLLEtBQUssQ0FBQ0wsb0JBQU0sQ0FBQ29DLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDbkNDLElBQUksRUFBRXJDLG9CQUFNLENBQUNvQyxNQUFNLENBQUMsQ0FBQztRQUNyQmxDLEtBQUssRUFBRUYsb0JBQU0sQ0FBQ0ssS0FBSyxDQUNqQkwsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO1VBQ1ptQyxJQUFJLEVBQUV0QyxvQkFBTSxDQUFDRyxNQUFNLENBQUM7WUFDbEJvQyxNQUFNLEVBQUV2QyxvQkFBTSxDQUFDSyxLQUFLLENBQUNMLG9CQUFNLENBQUN3QyxPQUFPLENBQUN4QyxvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUU7Y0FBRXNDLFFBQVEsRUFBRTtZQUFRLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDOUVDLElBQUksRUFBRTFDLG9CQUFNLENBQUNLLEtBQUssQ0FBQ0wsb0JBQU0sQ0FBQ3dDLE9BQU8sQ0FBQ3hDLG9CQUFNLENBQUNHLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRTtjQUFFc0MsUUFBUSxFQUFFO1lBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM1RUUsTUFBTSxFQUFFM0Msb0JBQU0sQ0FBQ0ssS0FBSyxDQUFDTCxvQkFBTSxDQUFDd0MsT0FBTyxDQUFDeEMsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFO2NBQUVzQyxRQUFRLEVBQUU7WUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzlFRyxRQUFRLEVBQUU1QyxvQkFBTSxDQUFDSyxLQUFLLENBQUNMLG9CQUFNLENBQUN3QyxPQUFPLENBQUN4QyxvQkFBTSxDQUFDRyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUU7Y0FBRXNDLFFBQVEsRUFBRTtZQUFRLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDaEZJLG9CQUFvQixFQUFFN0Msb0JBQU0sQ0FBQ0ssS0FBSyxDQUFDTCxvQkFBTSxDQUFDb0MsTUFBTSxDQUFDLENBQUMsQ0FBQztZQUNuRFUsb0JBQW9CLEVBQUU5QyxvQkFBTSxDQUFDSyxLQUFLLENBQUNMLG9CQUFNLENBQUMrQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQ3BEQyxLQUFLLEVBQUVoRCxvQkFBTSxDQUFDSyxLQUFLLENBQUNMLG9CQUFNLENBQUNDLEdBQUcsQ0FBQyxDQUFDO1VBQ2xDLENBQUM7UUFDSCxDQUFDLENBQ0gsQ0FBQztRQUNEZ0QsSUFBSSxFQUFFakQsb0JBQU0sQ0FBQ0ssS0FBSyxDQUFDTCxvQkFBTSxDQUFDQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQ2hDaUQsWUFBWSxFQUFFbEQsb0JBQU0sQ0FBQ0ssS0FBSyxDQUFDTCxvQkFBTSxDQUFDQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQ3hDa0QsSUFBSSxFQUFFbkQsb0JBQU0sQ0FBQ0ssS0FBSyxDQUFDTCxvQkFBTSxDQUFDd0MsT0FBTyxDQUFDeEMsb0JBQU0sQ0FBQ0MsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2hEbUQsT0FBTyxFQUFFcEQsb0JBQU0sQ0FBQ0ssS0FBSyxDQUNuQkwsb0JBQU0sQ0FBQ0csTUFBTSxDQUFDO1VBQ1prRCxRQUFRLEVBQUVyRCxvQkFBTSxDQUFDd0MsT0FBTyxDQUFDeEMsb0JBQU0sQ0FBQ00sTUFBTSxDQUFDLENBQUM7UUFDMUMsQ0FBQyxDQUNILENBQUM7UUFDRGdELGFBQWEsRUFBRXRELG9CQUFNLENBQUNLLEtBQUssQ0FBQ0wsb0JBQU0sQ0FBQ0MsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUN6Q3NELGdCQUFnQixFQUFFdkQsb0JBQU0sQ0FBQ0ssS0FBSyxDQUFDTCxvQkFBTSxDQUFDK0MsT0FBTyxDQUFDLENBQUM7TUFDakQsQ0FBQyxDQUFDO01BQ0Y3QyxLQUFLLEVBQUVGLG9CQUFNLENBQUNHLE1BQU0sQ0FBQztRQUNuQkMsZUFBZSxFQUFFSixvQkFBTSxDQUFDSyxLQUFLLENBQUNMLG9CQUFNLENBQUNNLE1BQU0sQ0FBQztVQUFFQyxZQUFZLEVBQUU7UUFBRyxDQUFDLENBQUM7TUFDbkUsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNELE9BQU9DLE9BQU8sRUFBRUMsT0FBTyxFQUFFQyxRQUFRLEtBQUs7SUFDcEMsSUFBQThDLGtDQUFrQixFQUFDLGlCQUFpQixFQUFFLEtBQUssRUFBRSxPQUFPLENBQUM7SUFDckQsTUFBTTtNQUFFNUMsS0FBSztNQUFFeUIsSUFBSTtNQUFFLEdBQUdvQjtJQUFLLENBQUMsR0FBR2hELE9BQU8sQ0FBQ1YsSUFBSTtJQUM3QyxNQUFNO01BQUVLO0lBQWdCLENBQUMsR0FBR0ssT0FBTyxDQUFDUCxLQUFLO0lBQ3pDLE1BQU1TLE1BQTRCLEdBQUc7TUFDbkNDLEtBQUssRUFBRUEsS0FBSyxJQUFJQyx3Q0FBdUI7TUFDdkN3QixJQUFJO01BQ0p0QyxJQUFJLEVBQUUwRDtJQUNSLENBQUM7SUFDRCxJQUFJO01BQ0YsSUFBSXpDLElBQUk7TUFDUixJQUFJdEIsaUJBQWlCLElBQUlVLGVBQWUsRUFBRTtRQUN4QyxNQUFNYSxNQUFNLEdBQUdULE9BQU8sQ0FBQ1UsVUFBVSxDQUFDQyxVQUFVLENBQUNDLE1BQU0sQ0FBQ0MsU0FBUyxDQUFDakIsZUFBZSxDQUFDO1FBQzlFWSxJQUFJLEdBQUcsTUFBTUMsTUFBTSxDQUFDSyxPQUFPLENBQUMsUUFBUSxFQUFFWCxNQUFNLENBQUM7TUFDL0MsQ0FBQyxNQUFNO1FBQ0xLLElBQUksR0FBRyxNQUFNUixPQUFPLENBQUNlLElBQUksQ0FBQ0osVUFBVSxDQUFDQyxNQUFNLENBQUNILE1BQU0sQ0FBQ08saUJBQWlCLENBQUMsUUFBUSxFQUFFYixNQUFNLENBQUM7TUFDeEY7TUFDQSxPQUFPRCxRQUFRLENBQUNlLEVBQUUsQ0FBQztRQUNqQjFCLElBQUksRUFBRWlCO01BQ1IsQ0FBQyxDQUFDO0lBQ0osQ0FBQyxDQUFDLE9BQU9VLEtBQUssRUFBRTtNQUNkLElBQUE4QixrQ0FBa0IsRUFBQyxpQkFBaUIsRUFBRSxLQUFLLEVBQUU5QixLQUFLLENBQUM7TUFDbkQsSUFBSUEsS0FBSyxDQUFDRyxVQUFVLEtBQUssR0FBRyxFQUFFRixPQUFPLENBQUNELEtBQUssQ0FBQ0EsS0FBSyxDQUFDO01BQ2xELE9BQU9oQixRQUFRLENBQUNrQixNQUFNLENBQUM7UUFDckJDLFVBQVUsRUFBRUgsS0FBSyxDQUFDRyxVQUFVLElBQUksR0FBRztRQUNuQzlCLElBQUksRUFBRTJCLEtBQUssQ0FBQ0k7TUFDZCxDQUFDLENBQUM7SUFDSjtFQUNGLENBQ0YsQ0FBQztBQUNIIn0=