"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerGettingStartedRoutes = registerGettingStartedRoutes;
var _configSchema = require("@osd/config-schema");
var _stream = require("stream");
var _utils = require("../../../../../src/core/server/saved_objects/routes/utils");
var _helper = require("./helper");
var _utils2 = require("../../../../../src/core/server/utils");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerGettingStartedRoutes(router) {
  // Fetch the tutorial assets
  router.get({
    path: `/api/observability/gettingStarted/{tutorialId}`,
    validate: {
      params: _configSchema.schema.object({
        tutorialId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const fileData = await (0, _helper.loadAssetsFromFile)(request.params.tutorialId);
      return response.ok({
        body: {
          data: fileData
        }
      });
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: 'Issue in fetching NDJSON file for tutorialId: ' + request.params.tutorialId
      });
    }
  });

  // Fetch the tutorial dashboards
  router.get({
    path: `/api/observability/gettingStarted/dashboards/{tutorialId}`,
    validate: {
      params: _configSchema.schema.object({
        tutorialId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const fileData = await (0, _helper.loadAssetsFromFile)(request.params.tutorialId);
      const objects = await (0, _utils.createSavedObjectsStreamFromNdJson)(_stream.Readable.from(fileData));
      const loadedObjects = await objects.toArray();
      const loadDashboardIds = loadedObjects.filter(savedObject => savedObject.type === 'dashboard').map(dashboard => ({
        id: dashboard.id,
        title: dashboard.attributes.title
      }));
      return response.ok({
        body: {
          data: loadDashboardIds
        }
      });
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: 'Issue in fetching dashboards for tutorialId: ' + request.params.tutorialId
      });
    }
  });

  // Fetch the tutorial saved searches
  router.get({
    path: `/api/observability/gettingStarted/indexPatterns/{tutorialId}`,
    validate: {
      params: _configSchema.schema.object({
        tutorialId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const fileData = await (0, _helper.loadAssetsFromFile)(request.params.tutorialId);
      const objects = await (0, _utils.createSavedObjectsStreamFromNdJson)(_stream.Readable.from(fileData));
      const loadedObjects = await objects.toArray();
      const loadDashboardIds = loadedObjects.filter(savedObject => savedObject.type === 'index-pattern').map(indexPattern => ({
        id: indexPattern.id,
        title: indexPattern.attributes.title
      }));
      return response.ok({
        body: {
          data: loadDashboardIds
        }
      });
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: 'Issue in fetching index-patterns for tutorialId: ' + request.params.tutorialId
      });
    }
  });
  router.post({
    path: `/api/observability/gettingStarted/createAssets`,
    validate: {
      body: _configSchema.schema.object({
        mdsId: _configSchema.schema.string(),
        mdsLabel: _configSchema.schema.string(),
        tutorialId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const {
        mdsId,
        mdsLabel,
        tutorialId
      } = request.body;
      const {
        requestWorkspaceId
      } = (0, _utils2.getWorkspaceState)(request);
      const fileData = await (0, _helper.loadAssetsFromFile)(tutorialId);
      const objects = await (0, _utils.createSavedObjectsStreamFromNdJson)(_stream.Readable.from(fileData));
      const loadedObjects = await objects.toArray();
      const updatedObjects = loadedObjects.map(obj => {
        let newId = obj.id;
        let references = obj.references;
        if (requestWorkspaceId) {
          var _references;
          newId = `workspaceId-${requestWorkspaceId}-${newId}`;
          references = (_references = references) === null || _references === void 0 ? void 0 : _references.map(ref => {
            return {
              ...ref,
              id: `workspaceId-${requestWorkspaceId}-${ref.id}`
            };
          });
        }
        if (mdsId) {
          var _references2;
          newId = `mds-${mdsId}-objectId-${obj.id}`;
          references = (_references2 = references) === null || _references2 === void 0 ? void 0 : _references2.map(ref => {
            return {
              ...ref,
              id: `mds-${mdsId}-objectId-${ref.id}`
            };
          });
          if (obj.type === 'index-pattern' && references) {
            references.push({
              id: mdsId,
              type: 'data-source',
              name: mdsLabel
            });
          }
        }
        return {
          ...obj,
          id: newId,
          ...(references && {
            references
          })
        };
      });
      await context.core.savedObjects.client.bulkCreate(updatedObjects);
      return response.ok({
        body: {
          message: 'Objects loaded successfully'
        }
      });
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: 'Issue in loading objects for mdsId:'
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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