"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerOpenSearchRoutes = registerOpenSearchRoutes;
var _lodash = require("lodash");
var _adHelpers = require("./utils/adHelpers");
var _helpers = require("../utils/helpers");
var _opensearchHelpers = require("./utils/opensearchHelpers");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
function registerOpenSearchRoutes(apiRouter, opensearchService) {
  apiRouter.get('/_indices', opensearchService.getIndices);
  apiRouter.get('/_indices/{dataSourceId}', opensearchService.getIndices);
  apiRouter.get('/_aliases', opensearchService.getAliases);
  apiRouter.get('/_aliases/{dataSourceId}', opensearchService.getAliases);
  apiRouter.get('/_mappings', opensearchService.getMapping);
  apiRouter.get('/_mappings/{dataSourceId}', opensearchService.getMapping);
  apiRouter.post('/_search', opensearchService.executeSearch);
  apiRouter.put('/create_index', opensearchService.createIndex);
  apiRouter.put('/create_index/{dataSourceId}', opensearchService.createIndex);
  apiRouter.post('/bulk', opensearchService.bulk);
  apiRouter.post('/bulk/{dataSourceId}', opensearchService.bulk);
  apiRouter.post('/delete_index', opensearchService.deleteIndex);
  apiRouter.get('/_remote/info', opensearchService.getClustersInfo);
  apiRouter.get('/_remote/info/', opensearchService.getClustersInfo);
  apiRouter.get('/_remote/info/{dataSourceId}', opensearchService.getClustersInfo);
  apiRouter.get('/_indices_and_aliases', opensearchService.getIndicesAndAliases);
  apiRouter.get('/_indices_and_aliases/{dataSourceId}', opensearchService.getIndicesAndAliases);
}
class OpenSearchService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "executeSearch", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          index,
          query,
          size = 0,
          sort = undefined,
          collapse = undefined,
          aggs = undefined,
          rawQuery = undefined
        } = request.body;
        const requestBody = rawQuery ? rawQuery : {
          query: query,
          ...(sort !== undefined && {
            sort: sort
          }),
          ...(collapse !== undefined && {
            collapse: collapse
          }),
          ...(aggs !== undefined && {
            aggs: aggs
          })
        };
        const params = {
          index,
          size,
          body: requestBody
        };
        const results = await this.client.asScoped(request).callAsCurrentUser('search', params);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: results
          }
        });
      } catch (err) {
        console.error('Anomaly detector - Unable to execute search', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getIndices", async (context, request, opensearchDashboardsResponse) => {
      const {
        index,
        clusters
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let indices = [];
        let resolve_resp;
        let response = await callWithRequest('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        response = response.map(item => ({
          ...item,
          localCluster: true
        }));

        // only call cat indices
        if (clusters != '') {
          if (index == '') {
            resolve_resp = await callWithRequest('transport.request', {
              method: 'GET',
              path: '/_resolve/index/' + clusters + ':*'
            });
          } else {
            resolve_resp = await callWithRequest('transport.request', {
              method: 'GET',
              path: '/_resolve/index/' + clusters + ':' + index
            });
          }
          indices = resolve_resp.indices.map(item => ({
            index: item.name,
            format: 'json',
            health: 'undefined',
            localCluster: false
          }));
          response = response.concat(indices);
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        // In case no matching indices is found it throws an error.
        if (err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception') {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                indices: []
              }
            }
          });
        }
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getAliases", async (context, request, opensearchDashboardsResponse) => {
      const {
        alias
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('cat.aliases', {
          alias,
          format: 'json',
          h: 'alias,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              aliases: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get aliases', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "createIndex", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;

      //@ts-ignore
      const index = request.body.index;
      //@ts-ignore
      const body = request.body.body;
      const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
      try {
        await callWithRequest('indices.create', {
          index: index,
          body: body
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to create index', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
      try {
        const response = await callWithRequest('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "bulk", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;
      const body = request.body;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('bulk', {
          body: body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform bulk action', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "deleteIndex", async (context, request, opensearchDashboardsResponse) => {
      const index = request.query;
      try {
        await callWithRequest('indices.delete', {
          index: index
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform delete index action', err);
        // Ignore the error if it's an index_not_found_exception
        if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: false,
              error: (0, _adHelpers.getErrorMessage)(err)
            }
          });
        }
      }
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getMapping", async (context, request, opensearchDashboardsResponse) => {
      let {
        indices
      } = request.query;
      // If indices is not an array, convert it to an array, server framework auto converts single item in string array to a string
      if (!Array.isArray(indices)) {
        indices = [indices];
      }
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let mappings = {};
        let remoteMappings = {};
        let localIndices = indices.filter(index => !index.includes(':'));
        let remoteIndices = indices.filter(index => index.includes(':'));
        if (localIndices.length > 0) {
          mappings = await callWithRequest('indices.getMapping', {
            index: localIndices
          });
        }

        // make call to fields_caps
        if (remoteIndices.length) {
          const fieldCapsResponse = await callWithRequest('transport.request', {
            method: 'GET',
            path: remoteIndices.toString() + '/_field_caps?fields=*&include_unmapped'
          });
          remoteMappings = (0, _opensearchHelpers.convertFieldCapsToMappingStructure)(fieldCapsResponse);
        }
        Object.assign(mappings, remoteMappings);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              mappings: mappings
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get mappings', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    // we use this to retrieve indices and aliases from both the local cluster and remote clusters
    // 3 different OS APIs are called here, _cat/indices, _cat/aliases and _resolve/index
    _defineProperty(this, "getIndicesAndAliases", async (context, request, opensearchDashboardsResponse) => {
      const {
        indexOrAliasQuery,
        clusters,
        queryForLocalCluster
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let indicesResponse = [];
        let aliasesResponse = [];
        if (queryForLocalCluster == 'true') {
          indicesResponse = await callWithRequest('cat.indices', {
            index: indexOrAliasQuery,
            format: 'json',
            h: 'health,index'
          });
          indicesResponse = indicesResponse.map(item => ({
            ...item,
            localCluster: true
          }));
          aliasesResponse = await callWithRequest('cat.aliases', {
            alias: indexOrAliasQuery,
            format: 'json',
            h: 'alias,index'
          });
          aliasesResponse = aliasesResponse.map(item => ({
            ...item,
            localCluster: true
          }));
        }

        // only call cat indices and cat aliases
        if (clusters != '') {
          let remoteIndices = [];
          let remoteAliases = [];
          let resolveResponse;
          const resolveIndexQuery = indexOrAliasQuery == '' ? clusters.split(',').map(cluster => `${cluster}:*`).join(',') : clusters.split(',').map(cluster => `${cluster}:${indexOrAliasQuery}`).join(',');
          resolveResponse = await callWithRequest('transport.request', {
            method: 'GET',
            path: '/_resolve/index/' + resolveIndexQuery
          });
          remoteIndices = resolveResponse.indices.map(item => ({
            index: item.name,
            format: 'json',
            health: 'undefined',
            localCluster: false
          }));
          remoteAliases = resolveResponse.aliases.map(item => ({
            alias: item.name,
            index: item.indices,
            format: 'json',
            localCluster: false
          }));
          indicesResponse = indicesResponse.concat(remoteIndices);
          aliasesResponse = aliasesResponse.concat(remoteAliases);
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              aliases: aliasesResponse,
              indices: indicesResponse
            }
          }
        });
      } catch (err) {
        // In case no matching indices is found it throws an error.
        if (err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception') {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                indices: [],
                aliases: []
              }
            }
          });
        }
        console.log('Anomaly detector - Unable to get indices and aliases', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getClustersInfo", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let clustersResponse = [];
        const remoteInfo = await callWithRequest('transport.request', {
          method: 'GET',
          path: '/_remote/info'
        });
        clustersResponse = Object.keys(remoteInfo).map(key => ({
          name: key,
          localCluster: false
        }));
        const clusterHealth = await callWithRequest('cat.health', {
          format: 'json',
          h: 'cluster'
        });
        clustersResponse.push({
          name: clusterHealth[0].cluster,
          localCluster: true
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              clusters: clustersResponse
            }
          }
        });
      } catch (err) {
        console.error('Alerting - OpensearchService - getClusterHealth:', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.default = OpenSearchService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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