"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ObservabilityPlugin = void 0;
var _configSchema = require("@osd/config-schema");
var _migrations = require("./adaptors/integrations/migrations");
var _opensearch_observability_plugin = require("./adaptors/opensearch_observability_plugin");
var _ppl_plugin = require("./adaptors/ppl_plugin");
var _ppl_parser = require("./parsers/ppl_parser");
var _register_settings = require("./plugin_helper/register_settings");
var _index = require("./routes/index");
var _observability_saved_object = require("./saved_objects/observability_saved_object");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class ObservabilityPlugin {
  constructor(initializerContext) {
    this.initializerContext = initializerContext;
    _defineProperty(this, "logger", void 0);
    this.logger = initializerContext.logger.get();
  }
  async setup(core, deps) {
    const {
      assistantDashboards,
      dataSource
    } = deps;
    this.logger.debug('Observability: Setup');
    const router = core.http.createRouter();
    const dataSourceEnabled = !!dataSource;
    const openSearchObservabilityClient = core.opensearch.legacy.createClient('opensearch_observability', {
      plugins: [_ppl_plugin.PPLPlugin, _opensearch_observability_plugin.OpenSearchObservabilityPlugin]
    });
    if (dataSourceEnabled) {
      dataSource.registerCustomApiSchema(_ppl_plugin.PPLPlugin);
      dataSource.registerCustomApiSchema(_opensearch_observability_plugin.OpenSearchObservabilityPlugin);
    }
    // @ts-ignore
    core.http.registerRouteHandlerContext('observability_plugin', (_context, _request) => {
      return {
        logger: this.logger,
        observabilityClient: openSearchObservabilityClient
      };
    });
    const obsPanelType = {
      name: 'observability-panel',
      hidden: false,
      namespaceType: 'single',
      mappings: {
        dynamic: false,
        properties: {
          title: {
            type: 'text'
          },
          description: {
            type: 'text'
          }
        }
      },
      management: {
        importableAndExportable: true,
        getInAppUrl() {
          return {
            path: `/app/management/observability/settings`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return `Observability Settings [${obj.id}]`;
        }
      },
      migrations: {
        '3.0.0': doc => ({
          ...doc,
          description: ''
        }),
        '3.0.1': doc => ({
          ...doc,
          description: 'Some Description Text'
        }),
        '3.0.2': doc => ({
          ...doc,
          dateCreated: parseInt(doc.dateCreated || '0', 10)
        })
      }
    };
    const integrationInstanceType = {
      name: 'integration-instance',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/installed/${obj.id}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          templateName: {
            type: 'text'
          },
          dataSource: {
            type: 'text'
          },
          creationDate: {
            type: 'date'
          },
          assets: {
            type: 'nested'
          }
        }
      }
    };
    const integrationTemplateType = {
      name: 'integration-template',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/available/${obj.attributes.name}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          var _obj$attributes$displ;
          return (_obj$attributes$displ = obj.attributes.displayName) !== null && _obj$attributes$displ !== void 0 ? _obj$attributes$displ : obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          version: {
            type: 'text'
          },
          displayName: {
            type: 'text'
          },
          license: {
            type: 'text'
          },
          type: {
            type: 'text'
          },
          labels: {
            type: 'text'
          },
          author: {
            type: 'text'
          },
          description: {
            type: 'text'
          },
          sourceUrl: {
            type: 'text'
          },
          statics: {
            type: 'nested'
          },
          components: {
            type: 'nested'
          },
          assets: {
            type: 'nested'
          },
          sampleData: {
            type: 'nested'
          }
        }
      },
      migrations: {
        '3.0.0': _migrations.migrateV1IntegrationToV2Integration
      }
    };
    core.savedObjects.registerType(obsPanelType);
    core.savedObjects.registerType(integrationInstanceType);
    core.savedObjects.registerType(integrationTemplateType);

    // Register server side APIs
    (0, _index.setupRoutes)({
      router,
      client: openSearchObservabilityClient,
      dataSourceEnabled
    });
    core.savedObjects.registerType(_observability_saved_object.visualizationSavedObject);
    core.savedObjects.registerType(_observability_saved_object.searchSavedObject);
    core.savedObjects.registerType(_observability_saved_object.notebookSavedObject);
    core.capabilities.registerProvider(() => ({
      observability: {
        show: true
      }
    }));
    assistantDashboards === null || assistantDashboards === void 0 || assistantDashboards.registerMessageParser(_ppl_parser.PPLParsers);
    (0, _register_settings.registerObservabilityUISettings)(core.uiSettings);
    core.uiSettings.register({
      'observability:defaultDashboard': {
        name: 'Observability default dashboard',
        value: '',
        description: 'The default dashboard to display in Observability overview page',
        schema: _configSchema.schema.string()
      }
    });
    return {};
  }
  start(_core) {
    this.logger.debug('Observability: Started');
    return {};
  }
  stop() {}
}
exports.ObservabilityPlugin = ObservabilityPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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