#!/bin/sh

# Copyright (C) 2021-2024 Internet Systems Consortium, Inc. ("ISC")
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# Exit with error if commands exit with non-zero and if undefined variables are
# used.
set -eu

# shellcheck disable=SC2034
# SC2034: ... appears unused. Verify use (or export if used externally).
prefix="/usr"

# Include utilities based on location of this script. Check for sources first,
# so that the unexpected situations with weird paths fall on the default
# case of installed.
script_path=$(cd "$(dirname "${0}")" && pwd)
if test "${script_path}" = "/build/kea/src/kea-2.6.0/src/share/database/scripts/mysql"; then
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "/build/kea/src/kea-2.6.0/src/bin/admin/admin-utils.sh"
else
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "${prefix}/share/kea/scripts/admin-utils.sh"
fi

# Check version.
version=$(mysql_version "${@}")
if test "${version}" != "9.5"; then
    printf 'This script upgrades 9.5 to 9.6. '
    printf 'Reported version is %s. Skipping upgrade.\n' "${version}"
    exit 0
fi

# Add column only if it doesn't exist to work around the 1.9.4 leak of
# cache_threshold and cache_max_age column alters in subnet and shared network
# tables in schema version 9.5.
if ! mysql "${@}" -e 'SELECT cache_threshold FROM dhcp4_subnet LIMIT 1' &> /dev/null; then
    mysql "${@}" <<EOF
# Add new lease cache parameters.
ALTER TABLE dhcp4_subnet
    ADD COLUMN cache_threshold FLOAT DEFAULT NULL,
    ADD COLUMN cache_max_age INT(10) DEFAULT NULL;

ALTER TABLE dhcp4_shared_network
    ADD COLUMN cache_threshold FLOAT DEFAULT NULL,
    ADD COLUMN cache_max_age INT(10) DEFAULT NULL;

ALTER TABLE dhcp6_subnet
    ADD COLUMN cache_threshold FLOAT DEFAULT NULL,
    ADD COLUMN cache_max_age INT(10) DEFAULT NULL;

ALTER TABLE dhcp6_shared_network
    ADD COLUMN cache_threshold FLOAT DEFAULT NULL,
    ADD COLUMN cache_max_age INT(10) DEFAULT NULL;
EOF
fi

mysql "${@}" <<EOF
-- This line starts the schema upgrade to version 9.6.

# Add an auto-increment ID as primary key to support Percona.
ALTER TABLE logs
    ADD id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY;

# Update the schema version number.
UPDATE schema_version
    SET version = '9', minor = '6';

# This line concludes the schema upgrade to version 9.6.

EOF
