"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationInstanceBuilder = void 0;
var _uuid = require("uuid");
var _utils = require("./repository/utils");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class IntegrationInstanceBuilder {
  constructor(client) {
    _defineProperty(this, "client", void 0);
    this.client = client;
  }
  async build(integration, options) {
    const instance = await (0, _utils.deepCheck)(integration);
    if (!instance.ok) {
      return Promise.reject(instance.error);
    }
    const assets = await integration.getAssets();
    if (!assets.ok) {
      return Promise.reject(assets.error);
    }
    const remapped = this.remapIDs(this.getSavedObjectBundles(assets.value, options.workflows));
    const assets1 = this.addMDSReference(remapped, options.dataSource, options.dataSourceMDSId, options.dataSourceMDSLabel);
    const withDataSource = this.remapDataSource(assets1, options.indexPattern);
    const withSubstitutedQueries = this.substituteQueries(withDataSource, options.dataSource, options.tableName);
    const refs = await this.postAssets(withSubstitutedQueries);
    const builtInstance = await this.buildInstance(integration, refs, options);
    return builtInstance;
  }
  substituteSavedSearch(assets, dataSource, tableName) {
    if (!dataSource || !tableName) {
      return assets;
    }
    assets = assets.map(asset => {
      if (asset.type === 'search') {
        var _searchSource$query;
        const searchSourceMeta = asset.attributes.kibanaSavedObjectMeta.searchSourceJSON;
        let searchSource;
        try {
          searchSource = JSON.parse(searchSourceMeta);
        } catch (error) {
          console.error('Invalid JSON in searchSourceJSON:', error);
          return asset;
        }
        if ((_searchSource$query = searchSource.query) !== null && _searchSource$query !== void 0 && _searchSource$query.query && searchSource.query.language === 'SQL') {
          searchSource.query.query = searchSource.query.query.replaceAll('{table_name}', tableName);
        }
        if (searchSourceMeta.dataset.type === 's3glue') {
          asset.attributes.title = `${dataSource}.default.${tableName}`;
        }
        asset.attributes.kibanaSavedObjectMeta.searchSourceJSON = JSON.stringify(searchSource);
      }
      return asset;
    });
    return assets;
  }

  // If we have a data source or table specified, hunt for saved queries and update them with the
  // new DS/table.
  substituteQueries(assets, dataSource, tableName) {
    if (!dataSource) {
      return assets;
    }
    assets = assets.map(asset => {
      if (asset.type === 'observability-search') {
        const savedQuery = asset.attributes.savedQuery;
        if (!savedQuery.data_sources) {
          return asset;
        }
        const dataSources = JSON.parse(savedQuery.data_sources);
        for (const ds of dataSources) {
          if (ds.type !== 's3glue') {
            continue; // Nothing to do
          }
          // TODO is there a distinction between these where we should only set one? They're all
          // equivalent in every export I've seen.
          ds.name = dataSource;
          ds.label = dataSource;
          ds.value = dataSource;
        }
        savedQuery.data_sources = JSON.stringify(dataSources);
        if (savedQuery.query_lang === 'SQL' && tableName) {
          savedQuery.query = savedQuery.query.replaceAll('{table_name}', tableName);
        }
      }
      return asset;
    });
    return assets;
  }
  getSavedObjectBundles(assets, includeWorkflows) {
    return assets.filter(asset => {
      // At this stage we only care about installing bundles
      if (asset.type !== 'savedObjectBundle') {
        return false;
      }
      // If no workflows present: default to all workflows
      // Otherwise only install if workflow is present
      if (!asset.workflows || !includeWorkflows) {
        return true;
      }
      return includeWorkflows.some(w => {
        var _asset$workflows;
        return (_asset$workflows = asset.workflows) === null || _asset$workflows === void 0 ? void 0 : _asset$workflows.includes(w);
      });
    }).map(asset => asset.data).flat();
  }
  remapDataSource(assets, dataSource) {
    if (!dataSource) return assets;
    return assets.map(asset => {
      if (asset.type === 'index-pattern') {
        asset.attributes.title = dataSource;
      }
      return asset;
    });
  }
  remapIDs(assets) {
    const toRemap = assets.filter(asset => asset.id);
    const idMap = new Map();
    return toRemap.map(item => {
      if (!idMap.has(item.id)) {
        idMap.set(item.id, (0, _uuid.v4)());
      }
      item.id = idMap.get(item.id);
      for (let ref = 0; ref < item.references.length; ref++) {
        const refId = item.references[ref].id;
        if (!idMap.has(refId)) {
          idMap.set(refId, (0, _uuid.v4)());
        }
        item.references[ref].id = idMap.get(refId);
      }
      return item;
    });
  }
  addMDSReference(assets, dataSource, dataSourceMDSId, dataSourceMDSLabel) {
    if (!dataSource) {
      return assets;
    }
    return assets.map(asset => {
      // Check if the asset type is 'index-pattern' or if the title contains 'Timeline' visualization
      if (asset !== null && asset !== void 0 && asset.type && (asset.type === 'index-pattern' || asset.type === 'visualization' && asset.attributes.visState.type === 'timelion')) {
        const dataSourceIndex = asset.references.findIndex(ref => ref.type === 'data-source');
        if (dataSourceIndex !== -1) {
          // If a data-source reference exists, update it
          asset.references[dataSourceIndex] = {
            id: dataSourceMDSId !== null && dataSourceMDSId !== void 0 ? dataSourceMDSId : '',
            name: dataSourceMDSLabel !== null && dataSourceMDSLabel !== void 0 ? dataSourceMDSLabel : 'Local cluster',
            type: 'data-source'
          };
        } else {
          // If no data-source reference exists, add a new one
          asset.references.push({
            id: dataSourceMDSId !== null && dataSourceMDSId !== void 0 ? dataSourceMDSId : '',
            name: dataSourceMDSLabel !== null && dataSourceMDSLabel !== void 0 ? dataSourceMDSLabel : 'Local cluster',
            type: 'data-source'
          });
        }
      }
      if (asset.type === 'search') {
        var _asset$attributes;
        if (asset !== null && asset !== void 0 && (_asset$attributes = asset.attributes) !== null && _asset$attributes !== void 0 && (_asset$attributes = _asset$attributes.kibanaSavedObjectMeta) !== null && _asset$attributes !== void 0 && _asset$attributes.searchSourceJSON) {
          var _searchSourceJSON$que;
          const searchSourceJSON = JSON.parse(asset.attributes.kibanaSavedObjectMeta.searchSourceJSON);
          if (searchSourceJSON !== null && searchSourceJSON !== void 0 && (_searchSourceJSON$que = searchSourceJSON.query) !== null && _searchSourceJSON$que !== void 0 && (_searchSourceJSON$que = _searchSourceJSON$que.dataset) !== null && _searchSourceJSON$que !== void 0 && _searchSourceJSON$que.dataSource) {
            searchSourceJSON.query.dataset.dataSource.id = dataSourceMDSId !== null && dataSourceMDSId !== void 0 ? dataSourceMDSId : '';
            searchSourceJSON.query.dataset.dataSource.name = dataSourceMDSLabel !== null && dataSourceMDSLabel !== void 0 ? dataSourceMDSLabel : 'Local cluster';
            searchSourceJSON.query.dataset.dataSource.type = 'data-source';
          }
          asset.attributes.kibanaSavedObjectMeta.searchSourceJSON = JSON.stringify(searchSourceJSON);
        }
      }
      return asset;
    });
  }
  async postAssets(assets) {
    try {
      const response = await this.client.bulkCreate(assets);
      const refs = response.saved_objects.map(obj => {
        var _obj$attributes;
        return {
          assetType: obj.type,
          assetId: obj.id,
          status: 'available',
          // Assuming a successfully created object is available
          isDefaultAsset: obj.type === 'dashboard',
          // Assuming for now that dashboards are default
          description: (_obj$attributes = obj.attributes) === null || _obj$attributes === void 0 ? void 0 : _obj$attributes.title
        };
      });
      return Promise.resolve(refs);
    } catch (err) {
      return Promise.reject(err);
    }
  }
  async buildInstance(integration, refs, options) {
    const config = await integration.getConfig();
    if (!config.ok) {
      return Promise.reject(new Error('Attempted to create instance with invalid template', config.error));
    }
    const instance = {
      name: options.name,
      templateName: config.value.name,
      dataSource: options.indexPattern,
      creationDate: new Date().toISOString(),
      assets: refs
    };
    if (options.dataSourceMDSId) {
      instance.references = [{
        id: options.dataSourceMDSId || '',
        name: options.dataSourceMDSLabel || 'Local cluster',
        type: 'data-source'
      }];
    }
    return Promise.resolve(instance);
  }
}
exports.IntegrationInstanceBuilder = IntegrationInstanceBuilder;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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