"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.translateDiskSpace = exports.parseStat = exports.prepareFileFromProps = exports.parseXML = void 0;
var path_posix_1 = __importDefault(require("path-posix"));
var fast_xml_parser_1 = __importDefault(require("fast-xml-parser"));
var nested_property_1 = __importDefault(require("nested-property"));
var encode_1 = require("./encode");
var path_1 = require("./path");
var PropertyType;
(function (PropertyType) {
    PropertyType["Array"] = "array";
    PropertyType["Object"] = "object";
    PropertyType["Original"] = "original";
})(PropertyType || (PropertyType = {}));
function getPropertyOfType(obj, prop, type) {
    if (type === void 0) { type = PropertyType.Original; }
    var val = nested_property_1.default.get(obj, prop);
    if (type === "array" && Array.isArray(val) === false) {
        return [val];
    }
    else if (type === "object" && Array.isArray(val)) {
        return val[0];
    }
    return val;
}
function normaliseResponse(response) {
    var output = Object.assign({}, response);
    nested_property_1.default.set(output, "propstat", getPropertyOfType(output, "propstat", PropertyType.Object));
    nested_property_1.default.set(output, "propstat.prop", getPropertyOfType(output, "propstat.prop", PropertyType.Object));
    return output;
}
function normaliseResult(result) {
    var multistatus = result.multistatus;
    if (multistatus === "") {
        return {
            multistatus: {
                response: []
            }
        };
    }
    if (!multistatus) {
        throw new Error("Invalid response: No root multistatus found");
    }
    var output = {
        multistatus: Array.isArray(multistatus) ? multistatus[0] : multistatus
    };
    nested_property_1.default.set(output, "multistatus.response", getPropertyOfType(output, "multistatus.response", PropertyType.Array));
    nested_property_1.default.set(output, "multistatus.response", nested_property_1.default.get(output, "multistatus.response").map(function (response) { return normaliseResponse(response); }));
    return output;
}
function parseXML(xml) {
    return new Promise(function (resolve) {
        var result = fast_xml_parser_1.default.parse(xml, {
            arrayMode: false,
            ignoreNameSpace: true
            // // We don't use the processors here as decoding is done manually
            // // later on - decoding early would break some path checks.
            // attrValueProcessor: val => decodeHTMLEntities(decodeURIComponent(val)),
            // tagValueProcessor: val => decodeHTMLEntities(decodeURIComponent(val))
        });
        resolve(normaliseResult(result));
    });
}
exports.parseXML = parseXML;
function prepareFileFromProps(props, rawFilename, isDetailed) {
    if (isDetailed === void 0) { isDetailed = false; }
    // Last modified time, raw size, item type and mime
    var _a = props.getlastmodified, lastMod = _a === void 0 ? null : _a, _b = props.getcontentlength, rawSize = _b === void 0 ? "0" : _b, _c = props.resourcetype, resourceType = _c === void 0 ? null : _c, _d = props.getcontenttype, mimeType = _d === void 0 ? null : _d, _e = props.getetag, etag = _e === void 0 ? null : _e;
    var type = resourceType &&
        typeof resourceType === "object" &&
        typeof resourceType.collection !== "undefined"
        ? "directory"
        : "file";
    var filename = (0, encode_1.decodeHTMLEntities)(rawFilename);
    var stat = {
        filename: filename,
        basename: path_posix_1.default.basename(filename),
        lastmod: lastMod,
        size: parseInt(rawSize, 10),
        type: type,
        etag: typeof etag === "string" ? etag.replace(/"/g, "") : null
    };
    if (type === "file") {
        stat.mime = mimeType && typeof mimeType === "string" ? mimeType.split(";")[0] : "";
    }
    if (isDetailed) {
        stat.props = props;
    }
    return stat;
}
exports.prepareFileFromProps = prepareFileFromProps;
function parseStat(result, filename, isDetailed) {
    if (isDetailed === void 0) { isDetailed = false; }
    var responseItem = null;
    try {
        responseItem = result.multistatus.response[0];
    }
    catch (e) {
        /* ignore */
    }
    if (!responseItem) {
        throw new Error("Failed getting item stat: bad response");
    }
    var _a = responseItem.propstat, props = _a.prop, statusLine = _a.status;
    // As defined in https://tools.ietf.org/html/rfc2068#section-6.1
    var _b = statusLine.split(" ", 3), _ = _b[0], statusCodeStr = _b[1], statusText = _b[2];
    var statusCode = parseInt(statusCodeStr, 10);
    if (statusCode >= 400) {
        var err = new Error("Invalid response: ".concat(statusCode, " ").concat(statusText));
        err.status = statusCode;
        throw err;
    }
    var filePath = (0, path_1.normalisePath)(filename);
    return prepareFileFromProps(props, filePath, isDetailed);
}
exports.parseStat = parseStat;
function translateDiskSpace(value) {
    switch (value.toString()) {
        case "-3":
            return "unlimited";
        case "-2":
        /* falls-through */
        case "-1":
            // -1 is non-computed
            return "unknown";
        default:
            return parseInt(value, 10);
    }
}
exports.translateDiskSpace = translateDiskSpace;
