# SPDX-License-Identifier: BSD-2-Clause
#
# RCSid:
#	$Id: rust.mk,v 1.22 2024/11/28 21:42:36 sjg Exp $
#
#	@(#) Copyright (c) 2024, Simon J. Gerraty
#
#	This file is provided in the hope that it will
#	be of use.  There is absolutely NO WARRANTY.
#	Permission to copy, redistribute or otherwise
#	use this file is hereby granted provided that
#	the above copyright notice and this notice are
#	left intact.
#
#	Please send copies of changes and bug-fixes to:
#	sjg@crufty.net
#

##
# This makefile is used when a build includes one or more Rust projects.
#
# We first include local.rust.mk to allow for customization.
# You can get very fancy - the logic/functionality here is minimal but
# can be extended via local.rust.mk
#
# If RUST_PROJECT_DIR is not set, we will make it ${.CURDIR:C,/src.*,,}
# actually we use ${SRCTOP}/${RELDIR:C,/src.*,,} to ensure we don't
# confuse ${SRCTOP} with ${RUST_PROJECT_DIR}/src.
#
# If ${.OBJDIR} is not ${.CURDIR} we will default CARGO_TARGET_DIR
# to ${.OBJDIR}.
# 
# First, if ${.CURDIR} is a subdir of ${RUST_PROJECT_DIR} (will happen
# if an Emacs user does 'M-x compile' while visiting a src file) we
# will need to adjust ${.OBJDIR} (and hence CARGO_TARGET_DIR).
#
# We assume that RUST_CARGO will be used to build Rust projects,
# so we default RUST_CARGO_PROJECT_DIR to ${RUST_PROJECT_DIR} and
# provide a _CARGO_USE that we automatically associate with
# targets named 'cargo.*' the default is 'cargo.build'.
#
# _CARGO_USE will chdir to ${RUST_CARGO_PROJECT_DIR} and run
# ${RUST_CARGO} with ENV, FLAGS and ARGS variables derrived from
# ${.TARGET:E:tu} so in the case of 'cargo.build' we get:
# RUST_CARGO_BUILD_ENV, RUST_CARGO_BUILD_FLAGS and RUST_CARGO_BUILD_ARGS
#
# _CARGO_USE will "just work" for additional targets like
# 'cargo.test', 'cargo.clippy' etc.
# 
# If MK_META_MODE is "yes" 'cargo.build' will touch ${.TARGET}
# so the default make rules will not consider it always out-of-date.
# In META MODE, 'bmake' will know if anything changed that should
# cause the target to be re-built.
#
# If MK_STAGING_RUST is "yes" we will stage the binary we
# built to a suitable location under ${STAGE_OBJTOP}.
# 

# allow for customization
.-include <local.rust.mk>

RUST_CARGO ?= cargo
.if ${.CURDIR} == ${SRCTOP}
RELDIR ?= .
.else
RELDIR ?= ${.CURDIR:S,${SRCTOP}/,,}
.endif
.if empty(RUST_PROJECT_DIR)
# we want this set correctly from anywhere within
# using RELDIR avoids confusing ${SRCTOP} with ${RUST_PROJECT_DIR}/src
RUST_PROJECT_DIR := ${SRCTOP}/${RELDIR:C,/src.*,,}
.if ${RUST_PROJECT_DIR:T:Nsrc:N.} == ""
RUST_PROJECT_DIR := ${RUST_PROJECT_DIR:H}
.endif
.endif

.if ${.OBJDIR} != ${.CURDIR}
.if ${.CURDIR:M${RUST_PROJECT_DIR}/*} != ""
# Our .CURDIR is below RUST_PROJECT_DIR and thus our
# .OBJDIR is likely not what we want either.
# This can easily happen if in Emacs we do 'M-x compile' while
# visiting a src file.
# It is easily fixed.
__objdir := ${.OBJDIR:S,${.CURDIR:S,${RUST_PROJECT_DIR},,},,}
.OBJDIR: ${__objdir}
.endif
# tell cargo where to drop build artifacts
CARGO_TARGET_DIR ?= ${.OBJDIR}
.if !empty(OBJROOT) && exists(${OBJROOT})
CARGO_HOME_RELDIR ?= rust/cargo_home
CARGO_HOME ?= ${OBJROOT}/common/${RUST_CARGO_HOME_RELDIR}
.endif
.elif ${.CURDIR} != ${RUST_PROJECT_DIR}
.OBJDIR: ${RUST_PROJECT_DIR}
.endif
CARGO_TARGET_DIR ?= target

.if ${MK_DIRDEPS_BUILD:Uno} == "no" || ${.MAKE.LEVEL} > 0
.export CARGO_HOME CARGO_TARGET_DIR RUST_PROJECT_DIR

all: cargo.build

.if empty(RUST_PROJECT_FILES)
RUST_PROJECT_FILES != find ${RUST_PROJECT_DIR} -type f \( \
	-name '*.rs' -o \
	-name Cargo.lock -o \
	-name Cargo.toml \) | sort
.endif
RUST_CARGO_BUILD_DEPS += ${RUST_PROJECT_FILES:U}
.endif

RUST_CARGO_PROJECT_DIR ?= ${RUST_PROJECT_DIR}

_CARGO_USE:	.USEBEFORE
	@(cd ${RUST_CARGO_PROJECT_DIR} && ${RUST_CARGO_ENV} \
	${RUST_CARGO_${.TARGET:E:tu}_ENV} \
	${RUST_CARGO} ${RUST_CARGO_${.TARGET:E:tu}_FLAGS:U${RUST_CARGO_FLAGS}} \
	${.TARGET:E} ${RUST_CARGO_${.TARGET:E:tu}_ARGS})

RUST_CARGO_TARGETS += cargo.build
cargo.build: ${RUST_CARGO_BUILD_DEPS}
.if ${.OBJDIR} != ${RUST_PROJECT_DIR}
	test ! -s Cargo.lock || cp -p Cargo.lock ${RUST_CARGO_PROJECT_DIR}
.endif
	@${META_COOKIE_TOUCH}

# handle cargo.{run,test,...}
RUST_CARGO_TARGETS += ${.TARGETS:Mcargo.*}
${RUST_CARGO_TARGETS:O:u}: _CARGO_USE

.if ${MK_DEBUG_RUST:Uno} == "no" && \
	${DEBUG_RUST_DIRS:Unone:@x@${RELDIR:M$x}@} == ""
RUST_CARGO_BUILD_ARGS += --release
.endif

.if ${MK_STAGING_RUST:Uno} == "yes"
.if ${RUST_CARGO_BUILD_ARGS:U:M--release} != ""
RUST_CARGO_TARGET = release
.else
RUST_CARGO_TARGET = debug
.endif

# The binary app is all we need
.if ${RUST_PROJECT_FILES:U:T:Mmain.rs} != ""
BINDIR ?= ${prefix}/bin
PROG = ${RUST_CARGO_TARGET_DIR}/${RUST_CARGO_TARGET}/${RUST_PROJECT_DIR:T}
STAGE_DIR.rust ?= ${STAGE_OBJTOP}${RUST_BINDIR:U${BINDIR}}
STAGE_SETS += rust
stage_files.rust: ${PROG}
STAGE_TARGETS += stage_files
.endif

.endif
