# Print an optspec for argparse to handle cmd's options that are independent of any subcommand.
function __fish_sq_global_optspecs
	string join \n overwrite home= no-key-store key-store= no-cert-store cert-store= keyring= known-notation= time= trust-root= password-file= v/verbose q/quiet batch h/help
end

function __fish_sq_needs_command
	# Figure out if the current invocation already has a command.
	set -l cmd (commandline -opc)
	set -e cmd[1]
	argparse -s (__fish_sq_global_optspecs) -- $cmd 2>/dev/null
	or return
	if set -q argv[1]
		# Also print the command, so this can be used to figure out what it is.
		echo $argv[1]
		return 1
	end
	return 0
end

function __fish_sq_using_subcommand
	set -l cmd (__fish_sq_needs_command)
	test -z "$cmd"
	and return 1
	contains -- $cmd[1] $argv
end

complete -c sq -n "__fish_sq_needs_command" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_needs_command" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_needs_command" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_needs_command" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_needs_command" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_needs_command" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_needs_command" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_needs_command" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_needs_command" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_needs_command" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_needs_command" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_needs_command" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_needs_command" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_needs_command" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_needs_command" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_needs_command" -f -a "encrypt" -d 'Encrypt a message'
complete -c sq -n "__fish_sq_needs_command" -f -a "decrypt" -d 'Decrypt a message'
complete -c sq -n "__fish_sq_needs_command" -f -a "sign" -d 'Sign messages or data files'
complete -c sq -n "__fish_sq_needs_command" -f -a "verify" -d 'Verify signed messages or detached signatures'
complete -c sq -n "__fish_sq_needs_command" -f -a "inspect" -d 'Inspect data, like file(1)'
complete -c sq -n "__fish_sq_needs_command" -f -a "cert" -d 'Manage certificates'
complete -c sq -n "__fish_sq_needs_command" -f -a "key" -d 'Manage keys'
complete -c sq -n "__fish_sq_needs_command" -f -a "pki" -d 'Authenticate certs using the Web of Trust'
complete -c sq -n "__fish_sq_needs_command" -f -a "network" -d 'Retrieve and publish certificates over the network'
complete -c sq -n "__fish_sq_needs_command" -f -a "toolbox" -d 'Tools for developers, maintainers, and forensic specialists'
complete -c sq -n "__fish_sq_needs_command" -f -a "version" -d 'Detailed version and output version information'
complete -c sq -n "__fish_sq_needs_command" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l output -d 'Write to FILE or stdout if omitted' -r
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l for -d 'Use certificates with the specified fingerprint or key ID' -r
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l for-userid -d 'Use certificates with the specified user ID' -r
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l for-email -d 'Use certificates where a user ID includes the specified email address' -r
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l for-file -d 'Read certificates from PATH' -r -F
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l with-password-file -d 'File containing password to encrypt the message' -r -F
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l set-metadata-time -d 'Set time for encrypted file as metadata' -r
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l signer-file -d 'Sign the message using the key in KEY_FILE' -r -F
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l signer -d 'Sign the message using the specified key on the key store' -r
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l encrypt-for -d 'Select what kind of keys are considered for encryption.' -r -f -a "{transport\t'Protects data in transport',storage\t'Protects data at rest',universal\t'Protects data in transport and at rest'}"
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l compression -d 'Select compression scheme to use' -r -f -a "{none\t'',zip\t'',zlib\t'',bzip2\t''}"
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l binary -d 'Emit binary data'
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l with-password -d 'Prompt to add a password to encrypt with'
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l set-metadata-filename -d 'Set the filename of the encrypted file as metadata'
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l use-expired-subkey -d 'Fall back to expired encryption subkeys'
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand encrypt" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand encrypt" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand encrypt" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand encrypt" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l output -d 'Write to FILE or stdout if omitted' -r
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l signatures -d 'Set the threshold of valid signatures to N' -r
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l signer -d 'Use certificates with the specified fingerprint or key ID to verify the signatures with.  Note: signatures verified with a certificate given here are considered authenticated.  When this option is not provided, the certificate is still read from the certificate store, if it exists, but it is not implicitly considered authenticated.' -r
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l signer-file -d 'Read certificates from PATH to verify the signatures with.  Note: signatures verified with a certificate given here are considered authenticated.' -r -F
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l recipient-file -d 'Decrypt the message using the key in KEY_FILE' -r -F
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l session-key -d 'Decrypt an encrypted message using SESSION-KEY' -r
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l dump-session-key -d 'Print the session key to stderr'
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand decrypt" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand decrypt" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand decrypt" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand decrypt" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand sign" -l output -d 'Write to FILE or stdout if omitted' -r
complete -c sq -n "__fish_sq_using_subcommand sign" -l merge -d 'Merge signatures from the input and SIGNED-MESSAGE' -r -F
complete -c sq -n "__fish_sq_using_subcommand sign" -l signer-file -d 'Sign the message using the key in KEY_FILE' -r -F
complete -c sq -n "__fish_sq_using_subcommand sign" -l signer -d 'Sign the message using the specified key on the key store' -r
complete -c sq -n "__fish_sq_using_subcommand sign" -l notation -d 'Add a notation to the certification.' -r
complete -c sq -n "__fish_sq_using_subcommand sign" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand sign" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand sign" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand sign" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand sign" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand sign" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand sign" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand sign" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand sign" -l binary -d 'Emit binary data'
complete -c sq -n "__fish_sq_using_subcommand sign" -l detached -d 'Create a detached signature'
complete -c sq -n "__fish_sq_using_subcommand sign" -l cleartext-signature -d 'Create a cleartext signature'
complete -c sq -n "__fish_sq_using_subcommand sign" -l append -d 'Append a signature to existing signature'
complete -c sq -n "__fish_sq_using_subcommand sign" -l notarize -d 'Sign a message and all existing signatures'
complete -c sq -n "__fish_sq_using_subcommand sign" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand sign" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand sign" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand sign" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand sign" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand sign" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand sign" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand verify" -l output -d 'Write to FILE or stdout if omitted' -r
complete -c sq -n "__fish_sq_using_subcommand verify" -l signature-file -d 'Verify a detached signature' -r -F
complete -c sq -n "__fish_sq_using_subcommand verify" -l signatures -d 'Set the threshold of valid signatures to N' -r
complete -c sq -n "__fish_sq_using_subcommand verify" -l signer -d 'Use certificates with the specified fingerprint or key ID to verify the signatures with.  Note: signatures verified with a certificate given here are considered authenticated.  When this option is not provided, the certificate is still read from the certificate store, if it exists, but it is not implicitly considered authenticated.' -r
complete -c sq -n "__fish_sq_using_subcommand verify" -l signer-file -d 'Read certificates from PATH to verify the signatures with.  Note: signatures verified with a certificate given here are considered authenticated.' -r -F
complete -c sq -n "__fish_sq_using_subcommand verify" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand verify" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand verify" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand verify" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand verify" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand verify" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand verify" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand verify" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand verify" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand verify" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand verify" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand verify" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand verify" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand verify" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand verify" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand inspect" -l cert -d 'Inspect certificates with the specified fingerprint or key ID' -r
complete -c sq -n "__fish_sq_using_subcommand inspect" -l userid -d 'Inspect certificates with the specified user ID.  Note: User IDs are not authenticated.' -r
complete -c sq -n "__fish_sq_using_subcommand inspect" -l email -d 'Inspect certificates where a user ID includes the specified email address.  Note: User IDs are not authenticated.' -r
complete -c sq -n "__fish_sq_using_subcommand inspect" -l domain -d 'Inspect certificates where a user ID includes an email address for the specified domain.  Note: User IDs are not authenticated.' -r
complete -c sq -n "__fish_sq_using_subcommand inspect" -l grep -d 'Inspect certificates with a user ID that matches the pattern, case insensitively.  Note: User IDs are not authenticated.' -r
complete -c sq -n "__fish_sq_using_subcommand inspect" -l file -d 'Read certificates from PATH' -r -F
complete -c sq -n "__fish_sq_using_subcommand inspect" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand inspect" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand inspect" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand inspect" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand inspect" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand inspect" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand inspect" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand inspect" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand inspect" -l certifications -d 'Print third-party certifications'
complete -c sq -n "__fish_sq_using_subcommand inspect" -l dump-bad-signatures -d 'Dump signatures that are definitively bad'
complete -c sq -n "__fish_sq_using_subcommand inspect" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand inspect" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand inspect" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand inspect" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand inspect" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand inspect" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand inspect" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -f -a "import" -d 'Import certificates into the local certificate store'
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -f -a "export" -d 'Export certificates from the local certificate store'
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -f -a "list" -d 'List all authenticated bindings (User ID and certificate pairs)'
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -f -a "lint" -d 'Check certificates for issues'
complete -c sq -n "__fish_sq_using_subcommand cert; and not __fish_seen_subcommand_from import export list lint help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from import" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l output -d 'Write to FILE or stdout if omitted' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l cert -d 'Use certificates with the specified fingerprint or key ID' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l userid -d 'Use certificates with the specified user ID' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l email -d 'Use certificates where a user ID includes the specified email address' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l domain -d 'Use certificates where a user ID includes an email address for the specified domain' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l grep -d 'Use certificates with a user ID that matches the pattern, case insensitively' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l binary -d 'Emit binary data'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l local -d 'Export local (non-exportable) signatures'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l all -d 'Export all certificates'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from export" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l amount -d 'The required amount of trust' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l show-paths -d 'Show why a binding is authenticated'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l email -d 'Changes the USERID parameter to match User IDs with the specified email address'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l gossip -d 'Treats all certificates as unreliable trust roots'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l certification-network -d 'Treats the network as a certification network'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from list" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l cert -d 'Use certificates with the specified fingerprint or key ID' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l userid -d 'Use certificates with the specified user ID' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l email -d 'Use certificates where a user ID includes the specified email address' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l domain -d 'Use certificates where a user ID includes an email address for the specified domain' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l grep -d 'Use certificates with a user ID that matches the pattern, case insensitively' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l file -d 'Read certificates from PATH' -r -F
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l output -d 'Write to the specified FILE.  If not specified, and the certificate was read from the certificate store, imports the modified certificate into the cert store.  If not specified, and the certificate was read from a file, writes the modified certificate to stdout.' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l fix -d 'Attempts to fix certificates, when possible'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l export-secret-keys -d 'When fixing a certificate, the fixed certificate is exported without any secret key material.  Using this switch causes any secret key material to also be exported'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l list-keys -d 'If set, outputs a list of fingerprints, one per line, of certificates that have issues.  This output is intended for use by scripts'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l binary -d 'Emit binary data'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from lint" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from help" -f -a "import" -d 'Import certificates into the local certificate store'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from help" -f -a "export" -d 'Export certificates from the local certificate store'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from help" -f -a "list" -d 'List all authenticated bindings (User ID and certificate pairs)'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from help" -f -a "lint" -d 'Check certificates for issues'
complete -c sq -n "__fish_sq_using_subcommand cert; and __fish_seen_subcommand_from help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -f -a "list" -d 'List keys managed by the key store'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -f -a "generate" -d 'Generate a new key'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -f -a "import" -d 'Import keys into the key store'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -f -a "export" -d 'Export keys from the key store'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -f -a "delete" -d 'Delete a certificate\'s secret key material'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -f -a "password" -d 'Change the password protecting secret key material'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -f -a "expire" -d 'Change a certificate\'s expiration time'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -f -a "revoke" -d 'Revoke a certificate'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -f -a "userid" -d 'Manage User IDs'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -f -a "subkey" -d 'Manage subkeys'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -f -a "approvals" -d 'Manages certification approvals'
complete -c sq -n "__fish_sq_using_subcommand key; and not __fish_seen_subcommand_from list generate import export delete password expire revoke userid subkey approvals help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l cert -d 'List keys with the specified fingerprint or key ID' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l userid -d 'List keys with the specified user ID' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l email -d 'List keys where a user ID includes the specified email address' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l domain -d 'List keys where a user ID includes an email address for the specified domain' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l grep -d 'List keys with a user ID that matches the pattern, case insensitively' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from list" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l name -d 'Add a name as user ID to the key' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l email -d 'Add an email address as user ID to the key' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l userid -d 'Add a user ID to the key' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l cipher-suite -d 'Select the cryptographic algorithms for the key' -r -f -a "{rsa2k\t'',rsa3k\t'',rsa4k\t'',cv25519\t''}"
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l new-password-file -d 'File containing password to encrypt the secret key material' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l can-encrypt -d 'Add an encryption-capable subkey [default: universal]' -r -f -a "{transport\t'Protects data in transport',storage\t'Protects data at rest',universal\t'Protects data in transport and at rest'}"
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l output -d 'Write the key to the specified file' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l rev-cert -d 'Write the emergency revocation certificate to FILE' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l expiration -d 'Sets the expiration time' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l allow-non-canonical-userids -d 'Don\'t reject user IDs that are not in canonical form'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l no-userids -d 'Create a key without any user IDs'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l without-password -d 'Don\'t protect the secret key material with a password'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l can-sign -d 'Add a signing-capable subkey (default)'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l cannot-sign -d 'Don\'t add a signing-capable subkey'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l can-authenticate -d 'Add an authentication-capable subkey (default)'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l cannot-authenticate -d 'Don\'t add an authentication-capable subkey'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l cannot-encrypt -d 'Don\'t add an encryption-capable subkey'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from generate" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from import" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l cert -d 'Export keys with the specified fingerprint or key ID' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l userid -d 'Export keys with the specified user ID' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l email -d 'Export keys where a user ID includes the specified email address' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l domain -d 'Export keys where a user ID includes an email address for the specified domain' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l grep -d 'Export keys with a user ID that matches the pattern, case insensitively' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l output -d 'Write to FILE or stdout if omitted' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l binary -d 'Emit binary data'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from export" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l cert -d 'Delete secret key material from the key with the specified fingerprint or key ID' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l userid -d 'Delete secret key material from the key with the specified user ID' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l email -d 'Delete secret key material from the key where a user ID includes the specified email address' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l file -d 'Delete the secret key material from the key read from PATH' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l output -d 'Write the stripped certificate to the specified file' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l binary -d 'Emit binary data'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from delete" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l cert -d 'Change the password for the secret key material from the key with the specified fingerprint or key ID' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l userid -d 'Change the password for the secret key material from the key with the specified user ID' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l email -d 'Change the password for the secret key material from the key where a user ID includes the specified email address' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l file -d 'Change the password for the secret key material from the key read from PATH' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l new-password-file -d 'File containing password to encrypt the secret key material' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l output -d 'Write to FILE or stdout if omitted' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l clear-password -d 'Clear the password protecting the secret key material'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l binary -d 'Emit binary data'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from password" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l cert -d 'Change the expiration of the key with the specified fingerprint or key ID' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l userid -d 'Change the expiration of the key with the specified user ID' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l email -d 'Change the expiration of the key where a user ID includes the specified email address' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l file -d 'Change the expiration of the key read from PATH' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l output -d 'Write to FILE or stdout if omitted' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l expiration -d 'Define EXPIRATION for the key as ISO 8601 formatted string or custom duration.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l binary -d 'Emit binary data'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from expire" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l cert -d 'Revoke the key with the specified fingerprint or key ID' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l userid -d 'Revoke the key with the specified user ID' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l email -d 'Revoke the key where a user ID includes the specified email address' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l file -d 'Revoke the key read from PATH' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l revoker -d 'Use key with the specified fingerprint or key ID to create the revocation certificate.  Sign the revocation certificate using the specified key.  By default, the certificate being revoked is used.  Using this option, it is possible to create a third-party revocation.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l revoker-userid -d 'Use key with the specified user ID to create the revocation certificate.  Sign the revocation certificate using the specified key.  By default, the certificate being revoked is used.  Using this option, it is possible to create a third-party revocation.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l revoker-email -d 'Use key where a user ID includes the specified email address to create the revocation certificate.  Sign the revocation certificate using the specified key.  By default, the certificate being revoked is used.  Using this option, it is possible to create a third-party revocation.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l revoker-file -d 'Read key from PATH to create the revocation certificate.  Sign the revocation certificate using the specified key.  By default, the certificate being revoked is used.  Using this option, it is possible to create a third-party revocation.' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l reason -d 'The reason for the revocation' -r -f -a "{compromised\t'The secret key material may have been compromised.  Prefer this value if you suspect that the secret key has been leaked',superseded\t'The owner of the certificate has a new certificate.  Prefer `compromised` if the secret key material has been compromised even if the certificate is also being replaced!  You should include the fingerprint of the new certificate in the message',retired\t'The certificate should not be used anymore, and there is no replacement.  This is appropriate when someone leaves an organisation.  Prefer `compromised` if the secret key material has been compromised even if the certificate is also being retired!  You should include how to contact the owner, or who to contact instead in the message',unspecified\t'None of the other reasons apply.  OpenPGP implementations conservatively treat this type of revocation similar to a compromised key'}"
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l message -d 'A short, explanatory text' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l notation -d 'Add a notation to the certification' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l output -d 'Write to the specified FILE' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l binary -d 'Emit binary data'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from revoke" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -f -a "add" -d 'Add a user ID'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -f -a "revoke" -d 'Revoke a user ID'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from userid" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -f -a "add" -d 'Add a new subkey to a certificate'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -f -a "export" -d 'Export keys from the key store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -f -a "delete" -d 'Delete a certificate\'s secret key material'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -f -a "password" -d 'Change the password protecting secret key material'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -f -a "expire" -d 'Change a subkey\'s expiration time'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -f -a "revoke" -d 'Revoke a subkey'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -f -a "bind" -d 'Bind keys from one certificate to another'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from subkey" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -f -a "list" -d 'Lists third-party certifications'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -f -a "update" -d 'Approves of third-party certifications'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from approvals" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from help" -f -a "list" -d 'List keys managed by the key store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from help" -f -a "generate" -d 'Generate a new key'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from help" -f -a "import" -d 'Import keys into the key store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from help" -f -a "export" -d 'Export keys from the key store'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from help" -f -a "delete" -d 'Delete a certificate\'s secret key material'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from help" -f -a "password" -d 'Change the password protecting secret key material'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from help" -f -a "expire" -d 'Change a certificate\'s expiration time'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from help" -f -a "revoke" -d 'Revoke a certificate'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from help" -f -a "userid" -d 'Manage User IDs'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from help" -f -a "subkey" -d 'Manage subkeys'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from help" -f -a "approvals" -d 'Manages certification approvals'
complete -c sq -n "__fish_sq_using_subcommand key; and __fish_seen_subcommand_from help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -f -a "authenticate" -d 'Authenticate a binding'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -f -a "lookup" -d 'Lookup the certificates associated with a User ID'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -f -a "identify" -d 'Identify a certificate'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -f -a "vouch" -d 'Manage certifications'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -f -a "link" -d 'Manage authenticated certificate and User ID links'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -f -a "path" -d 'Verify the specified path'
complete -c sq -n "__fish_sq_using_subcommand pki; and not __fish_seen_subcommand_from authenticate lookup identify vouch link path help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l amount -d 'The required amount of trust' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l show-paths -d 'Show why a binding is authenticated'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l email -d 'Changes the USERID parameter to match User IDs with the specified email address'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l gossip -d 'Treats all certificates as unreliable trust roots'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l certification-network -d 'Treats the network as a certification network'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from authenticate" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l amount -d 'The required amount of trust' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l show-paths -d 'Show why a binding is authenticated'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l email -d 'Changes the USERID parameter to match User IDs with the specified email address'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l gossip -d 'Treats all certificates as unreliable trust roots'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l certification-network -d 'Treats the network as a certification network'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from lookup" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l amount -d 'The required amount of trust' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l show-paths -d 'Show why a binding is authenticated'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l gossip -d 'Treats all certificates as unreliable trust roots'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l certification-network -d 'Treats the network as a certification network'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from identify" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -f -a "certify" -d 'Certify a User ID for a Certificate'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -f -a "authorize" -d 'Mark a certificate as a trusted introducer'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from vouch" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -f -a "add" -d 'Link a certificate and a user ID'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -f -a "authorize" -d 'Make a certificate a trusted introducer'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -f -a "retract" -d 'Retract links'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -f -a "list" -d 'List links'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from link" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l amount -d 'The required amount of trust' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l certification-network -d 'Treats the network as a certification network'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from path" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from help" -f -a "authenticate" -d 'Authenticate a binding'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from help" -f -a "lookup" -d 'Lookup the certificates associated with a User ID'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from help" -f -a "identify" -d 'Identify a certificate'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from help" -f -a "vouch" -d 'Manage certifications'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from help" -f -a "link" -d 'Manage authenticated certificate and User ID links'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from help" -f -a "path" -d 'Verify the specified path'
complete -c sq -n "__fish_sq_using_subcommand pki; and __fish_seen_subcommand_from help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -f -a "search" -d 'Retrieve certificates using all supported network services'
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -f -a "keyserver" -d 'Retrieve and publishes certificates via key servers'
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -f -a "wkd" -d 'Retrieve and publishes certificates via Web Key Directories'
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -f -a "dane" -d 'Retrieve and publishes certificates via DANE'
complete -c sq -n "__fish_sq_using_subcommand network; and not __fish_seen_subcommand_from search keyserver wkd dane help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l server -d 'Set the key server to use.  Can be given multiple times.' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l output -d 'Write to FILE (or stdout when omitted) instead of importing into the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l binary -d 'Emit binary data'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l all -d 'Fetch updates for all known certificates'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from search" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l server -d 'Set the key server to use.  Can be given multiple times.' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -f -a "search" -d 'Retrieve certificates from key servers'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -f -a "publish" -d 'Publish certificates on key servers'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from keyserver" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -f -a "search" -d 'Retrieve certificates from a Web Key Directory'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -f -a "publish" -d 'Publish certificates in a Web Key Directory'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from wkd" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -f -a "search" -d 'Retrieve certificates using DANE'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -f -a "generate" -d 'Generate DANE records for the given domain and certs'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from dane" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from help" -f -a "search" -d 'Retrieve certificates using all supported network services'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from help" -f -a "keyserver" -d 'Retrieve and publishes certificates via key servers'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from help" -f -a "wkd" -d 'Retrieve and publishes certificates via Web Key Directories'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from help" -f -a "dane" -d 'Retrieve and publishes certificates via DANE'
complete -c sq -n "__fish_sq_using_subcommand network; and __fish_seen_subcommand_from help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -f -a "keyring" -d 'Manage collections of keys or certs'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -f -a "extract-cert" -d 'Convert a key to a cert'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -f -a "strip-userid" -d 'Strip a user ID'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -f -a "packet" -d 'Low-level packet manipulation'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -f -a "armor" -d 'Convert binary to ASCII'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -f -a "dearmor" -d 'Convert ASCII to binary'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and not __fish_seen_subcommand_from keyring extract-cert strip-userid packet armor dearmor help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -f -a "list" -d 'List keys in a keyring'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -f -a "split" -d 'Split a keyring into individual keys'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -f -a "merge" -d 'Merge keys or keyrings into a single keyring'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -f -a "filter" -d 'Join keys into a keyring applying a filter'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from keyring" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l output -d 'Write to FILE or stdout if omitted' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l binary -d 'Emit binary data'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from extract-cert" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l cert -d 'Strip the user ID of the certificate with the specified fingerprint or key ID' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l cert-userid -d 'Strip the user ID of the certificate with the specified user ID' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l cert-email -d 'Strip the user ID of the certificate where a user ID includes the specified email address' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l cert-file -d 'Strip the user ID from the cert read from PATH' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l output -d 'Write to the specified FILE' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l name -d 'Strip the given name user ID' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l email -d 'Strip the given email address user ID' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l userid -d 'Strip the given user IDs' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l binary -d 'Emit binary data'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from strip-userid" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -f -a "dump" -d 'List packets'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -f -a "decrypt" -d 'Unwrap an encryption container'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -f -a "split" -d 'Split a message into packets'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -f -a "join" -d 'Join packets split across files'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from packet" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l output -d 'Write to FILE or stdout if omitted' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l label -d 'Select the kind of armor header' -r -f -a "{auto\t'',message\t'',cert\t'',key\t'',sig\t'',file\t''}"
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from armor" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l output -d 'Write to FILE or stdout if omitted' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from dearmor" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from help" -f -a "keyring" -d 'Manage collections of keys or certs'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from help" -f -a "extract-cert" -d 'Convert a key to a cert'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from help" -f -a "strip-userid" -d 'Strip a user ID'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from help" -f -a "packet" -d 'Low-level packet manipulation'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from help" -f -a "armor" -d 'Convert binary to ASCII'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from help" -f -a "dearmor" -d 'Convert ASCII to binary'
complete -c sq -n "__fish_sq_using_subcommand toolbox; and __fish_seen_subcommand_from help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand version" -l home -d 'Set the home directory.' -r
complete -c sq -n "__fish_sq_using_subcommand version" -l key-store -d 'Override the key store server and its data' -r
complete -c sq -n "__fish_sq_using_subcommand version" -l cert-store -d 'Specify the location of the certificate store' -r
complete -c sq -n "__fish_sq_using_subcommand version" -l keyring -d 'Specify the location of a keyring to use' -r -F
complete -c sq -n "__fish_sq_using_subcommand version" -l known-notation -d 'Add NOTATION to the list of known notations' -r
complete -c sq -n "__fish_sq_using_subcommand version" -l time -d 'Set the reference time as ISO 8601 formatted timestamp' -r
complete -c sq -n "__fish_sq_using_subcommand version" -l trust-root -d 'Consider the specified certificate to be a trust root' -r
complete -c sq -n "__fish_sq_using_subcommand version" -l password-file -d 'Seed the password cache with the specified password' -r -F
complete -c sq -n "__fish_sq_using_subcommand version" -l overwrite -d 'Overwrite existing files'
complete -c sq -n "__fish_sq_using_subcommand version" -l no-key-store -d 'Disable the use of the key store.'
complete -c sq -n "__fish_sq_using_subcommand version" -l no-cert-store -d 'Disable the use of a certificate store'
complete -c sq -n "__fish_sq_using_subcommand version" -s v -l verbose -d 'Be more verbose.'
complete -c sq -n "__fish_sq_using_subcommand version" -s q -l quiet -d 'Be more quiet.'
complete -c sq -n "__fish_sq_using_subcommand version" -l batch -d 'Prevents any kind of prompting'
complete -c sq -n "__fish_sq_using_subcommand version" -s h -l help -d 'Print help (see more with \'--help\')'
complete -c sq -n "__fish_sq_using_subcommand help; and not __fish_seen_subcommand_from encrypt decrypt sign verify inspect cert key pki network toolbox version help" -f -a "encrypt" -d 'Encrypt a message'
complete -c sq -n "__fish_sq_using_subcommand help; and not __fish_seen_subcommand_from encrypt decrypt sign verify inspect cert key pki network toolbox version help" -f -a "decrypt" -d 'Decrypt a message'
complete -c sq -n "__fish_sq_using_subcommand help; and not __fish_seen_subcommand_from encrypt decrypt sign verify inspect cert key pki network toolbox version help" -f -a "sign" -d 'Sign messages or data files'
complete -c sq -n "__fish_sq_using_subcommand help; and not __fish_seen_subcommand_from encrypt decrypt sign verify inspect cert key pki network toolbox version help" -f -a "verify" -d 'Verify signed messages or detached signatures'
complete -c sq -n "__fish_sq_using_subcommand help; and not __fish_seen_subcommand_from encrypt decrypt sign verify inspect cert key pki network toolbox version help" -f -a "inspect" -d 'Inspect data, like file(1)'
complete -c sq -n "__fish_sq_using_subcommand help; and not __fish_seen_subcommand_from encrypt decrypt sign verify inspect cert key pki network toolbox version help" -f -a "cert" -d 'Manage certificates'
complete -c sq -n "__fish_sq_using_subcommand help; and not __fish_seen_subcommand_from encrypt decrypt sign verify inspect cert key pki network toolbox version help" -f -a "key" -d 'Manage keys'
complete -c sq -n "__fish_sq_using_subcommand help; and not __fish_seen_subcommand_from encrypt decrypt sign verify inspect cert key pki network toolbox version help" -f -a "pki" -d 'Authenticate certs using the Web of Trust'
complete -c sq -n "__fish_sq_using_subcommand help; and not __fish_seen_subcommand_from encrypt decrypt sign verify inspect cert key pki network toolbox version help" -f -a "network" -d 'Retrieve and publish certificates over the network'
complete -c sq -n "__fish_sq_using_subcommand help; and not __fish_seen_subcommand_from encrypt decrypt sign verify inspect cert key pki network toolbox version help" -f -a "toolbox" -d 'Tools for developers, maintainers, and forensic specialists'
complete -c sq -n "__fish_sq_using_subcommand help; and not __fish_seen_subcommand_from encrypt decrypt sign verify inspect cert key pki network toolbox version help" -f -a "version" -d 'Detailed version and output version information'
complete -c sq -n "__fish_sq_using_subcommand help; and not __fish_seen_subcommand_from encrypt decrypt sign verify inspect cert key pki network toolbox version help" -f -a "help" -d 'Print this message or the help of the given subcommand(s)'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from cert" -f -a "import" -d 'Import certificates into the local certificate store'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from cert" -f -a "export" -d 'Export certificates from the local certificate store'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from cert" -f -a "list" -d 'List all authenticated bindings (User ID and certificate pairs)'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from cert" -f -a "lint" -d 'Check certificates for issues'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from key" -f -a "list" -d 'List keys managed by the key store'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from key" -f -a "generate" -d 'Generate a new key'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from key" -f -a "import" -d 'Import keys into the key store'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from key" -f -a "export" -d 'Export keys from the key store'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from key" -f -a "delete" -d 'Delete a certificate\'s secret key material'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from key" -f -a "password" -d 'Change the password protecting secret key material'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from key" -f -a "expire" -d 'Change a certificate\'s expiration time'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from key" -f -a "revoke" -d 'Revoke a certificate'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from key" -f -a "userid" -d 'Manage User IDs'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from key" -f -a "subkey" -d 'Manage subkeys'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from key" -f -a "approvals" -d 'Manages certification approvals'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from pki" -f -a "authenticate" -d 'Authenticate a binding'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from pki" -f -a "lookup" -d 'Lookup the certificates associated with a User ID'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from pki" -f -a "identify" -d 'Identify a certificate'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from pki" -f -a "vouch" -d 'Manage certifications'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from pki" -f -a "link" -d 'Manage authenticated certificate and User ID links'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from pki" -f -a "path" -d 'Verify the specified path'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from network" -f -a "search" -d 'Retrieve certificates using all supported network services'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from network" -f -a "keyserver" -d 'Retrieve and publishes certificates via key servers'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from network" -f -a "wkd" -d 'Retrieve and publishes certificates via Web Key Directories'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from network" -f -a "dane" -d 'Retrieve and publishes certificates via DANE'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from toolbox" -f -a "keyring" -d 'Manage collections of keys or certs'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from toolbox" -f -a "extract-cert" -d 'Convert a key to a cert'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from toolbox" -f -a "strip-userid" -d 'Strip a user ID'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from toolbox" -f -a "packet" -d 'Low-level packet manipulation'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from toolbox" -f -a "armor" -d 'Convert binary to ASCII'
complete -c sq -n "__fish_sq_using_subcommand help; and __fish_seen_subcommand_from toolbox" -f -a "dearmor" -d 'Convert ASCII to binary'
