"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _constants = require("../../utils/constants");
var _helpers = require("./utils/helpers");
var _MDSEnabledClientService = require("./MDSEnabledClientService");
var _constants2 = require("./utils/constants");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class MonitorService extends _MDSEnabledClientService.MDSEnabledClientService {
  constructor(...args) {
    super(...args);
    _defineProperty(this, "createMonitor", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const createResponse = await client('alerting.createMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "createWorkflow", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const createResponse = await client('alerting.createWorkflow', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const response = await client('alerting.deleteMonitor', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const response = await client('alerting.deleteWorkflow', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          headers: _constants2.DEFAULT_HEADERS
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getMonitor', params);
        let monitor = _lodash.default.get(getResponse, 'monitor', null);
        const version = _lodash.default.get(getResponse, '_version', null);
        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);
        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);
        const associated_workflows = _lodash.default.get(getResponse, 'associated_workflows', null);
        if (monitor) {
          const client = this.getClientBasedOnDataSource(context, req);
          const aggsParams = {
            index: _constants.INDEX.ALL_ALERTS,
            body: {
              size: 0,
              query: {
                bool: {
                  must: {
                    term: {
                      monitor_id: id
                    }
                  }
                }
              },
              aggs: {
                active_count: {
                  terms: {
                    field: 'state'
                  }
                },
                '24_hour_count': {
                  date_range: {
                    field: 'start_time',
                    ranges: [{
                      from: 'now-24h/h'
                    }]
                  }
                }
              }
            }
          };
          const searchResponse = await client('alerting.getMonitors', aggsParams);
          const dayCount = _lodash.default.get(searchResponse, 'aggregations.24_hour_count.buckets.0.doc_count', 0);
          const activeBuckets = _lodash.default.get(searchResponse, 'aggregations.active_count.buckets', []);
          const activeCount = activeBuckets.reduce((acc, curr) => curr.key === 'ACTIVE' ? curr.doc_count : acc, 0);
          if (associated_workflows) {
            monitor = {
              ...monitor,
              associated_workflows,
              associatedCompositeMonitorCnt: associated_workflows.length
            };
          }
          monitor = {
            ...monitor,
            item_type: monitor.workflow_type || monitor.monitor_type,
            id,
            version
          };
          return res.ok({
            body: {
              ok: true,
              resp: monitor,
              activeCount,
              dayCount,
              version,
              ifSeqNo,
              ifPrimaryTerm
            }
          });
        } else {
          return res.ok({
            body: {
              ok: false
            }
          });
        }
      } catch (err) {
        console.error('Alerting - MonitorService - getMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getWorkflow', params);
        let workflow = _lodash.default.get(getResponse, 'workflow', null);
        const version = _lodash.default.get(getResponse, '_version', null);
        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);
        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);
        workflow.monitor_type = workflow.workflow_type;
        workflow = {
          ...workflow,
          item_type: workflow.workflow_type,
          id,
          version
        };
        return res.ok({
          body: {
            ok: true,
            resp: workflow,
            activeCount: 0,
            dayCount: 0,
            version,
            ifSeqNo,
            ifPrimaryTerm
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - getWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "updateMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body,
          refresh: 'wait_for'
        };
        const {
          type
        } = req.body;

        // TODO DRAFT: Are we sure we need to include ifSeqNo and ifPrimaryTerm from the UI side when updating monitors?
        const {
          ifSeqNo,
          ifPrimaryTerm
        } = req.query;
        if (ifSeqNo && ifPrimaryTerm) {
          params.if_seq_no = ifSeqNo;
          params.if_primary_term = ifPrimaryTerm;
        }
        const client = this.getClientBasedOnDataSource(context, req);
        const updateResponse = await client(`alerting.${type === 'workflow' ? 'updateWorkflow' : 'updateMonitor'}`, params);
        const {
          _version,
          _id
        } = updateResponse;
        return res.ok({
          body: {
            ok: true,
            version: _version,
            id: _id
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - updateMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getMonitors", async (context, req, res) => {
      try {
        const {
          from,
          size,
          search,
          sortDirection,
          sortField,
          state,
          monitorIds
        } = req.query;
        let must = {
          match_all: {}
        };
        if (search.trim()) {
          // This is an expensive wildcard query to match monitor names such as: "This is a long monitor name"
          // search query => "long monit"
          // This is acceptable because we will never allow more than 1,000 monitors
          must = {
            query_string: {
              default_field: 'monitor.name',
              default_operator: 'AND',
              query: `*${search.trim().split(' ').join('* *')}*`
            }
          };
        }
        const should = [];
        const mustList = [must];
        if (monitorIds !== undefined) {
          mustList.push({
            terms: {
              _id: Array.isArray(monitorIds) ? monitorIds : [monitorIds]
            }
          });
        } else if (monitorIds === 'empty') {
          mustList.push({
            terms: {
              _id: []
            }
          });
        }
        if (state !== 'all') {
          const enabled = state === 'enabled';
          should.push({
            term: {
              'monitor.enabled': enabled
            }
          });
          should.push({
            term: {
              'workflow.enabled': enabled
            }
          });
        }
        const monitorSorts = {
          name: 'monitor.name.keyword'
        };
        const monitorSortPageData = {
          size: 1000
        };
        if (monitorSorts[sortField]) {
          monitorSortPageData.sort = [{
            [monitorSorts[sortField]]: sortDirection
          }];
          monitorSortPageData.size = _lodash.default.defaultTo(size, 1000);
          monitorSortPageData.from = _lodash.default.defaultTo(from, 0);
        }
        const params = {
          body: {
            seq_no_primary_term: true,
            version: true,
            ...monitorSortPageData,
            query: {
              bool: {
                should,
                minimum_should_match: state !== 'all' ? 1 : 0,
                must: mustList
              }
            },
            aggregations: {
              associated_composite_monitors: {
                nested: {
                  path: 'workflow.inputs.composite_input.sequence.delegates'
                },
                aggs: {
                  monitor_ids: {
                    terms: {
                      field: 'workflow.inputs.composite_input.sequence.delegates.monitor_id'
                    }
                  }
                }
              }
            }
          }
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getMonitors', params);
        const totalMonitors = _lodash.default.get(getResponse, 'hits.total.value', 0);
        const monitorKeyValueTuples = _lodash.default.get(getResponse, 'hits.hits', []).map(result => {
          const {
            _id: id,
            _version: version,
            _seq_no: ifSeqNo,
            _primary_term: ifPrimaryTerm,
            _source
          } = result;
          const monitor = _source.monitor ? _source.monitor : _source;
          monitor['item_type'] = monitor.workflow_type || monitor.monitor_type;
          const {
            name,
            enabled,
            item_type
          } = monitor;
          return [id, {
            id,
            version,
            ifSeqNo,
            ifPrimaryTerm,
            name,
            enabled,
            item_type,
            monitor
          }];
        }, {});
        const monitorMap = new Map(monitorKeyValueTuples);
        const associatedCompositeMonitorCountMap = {};
        _lodash.default.get(getResponse, 'aggregations.associated_composite_monitors.monitor_ids.buckets', []).forEach(({
          key,
          doc_count
        }) => {
          associatedCompositeMonitorCountMap[key] = doc_count;
        });
        const monitorIdsOutput = [...monitorMap.keys()];
        const aggsOrderData = {};
        const aggsSorts = {
          active: 'active',
          acknowledged: 'acknowledged',
          errors: 'errors',
          ignored: 'ignored',
          lastNotificationTime: 'last_notification_time'
        };
        if (aggsSorts[sortField]) {
          aggsOrderData.order = {
            [aggsSorts[sortField]]: sortDirection
          };
        }
        const aggsParams = {
          index: _constants.INDEX.ALL_ALERTS,
          body: {
            size: 0,
            query: {
              terms: {
                monitor_id: monitorIdsOutput
              }
            },
            aggregations: {
              uniq_monitor_ids: {
                terms: {
                  field: 'monitor_id',
                  ...aggsOrderData,
                  size: from + size
                },
                aggregations: {
                  active: {
                    filter: {
                      term: {
                        state: 'ACTIVE'
                      }
                    }
                  },
                  acknowledged: {
                    filter: {
                      term: {
                        state: 'ACKNOWLEDGED'
                      }
                    }
                  },
                  errors: {
                    filter: {
                      term: {
                        state: 'ERROR'
                      }
                    }
                  },
                  ignored: {
                    filter: {
                      bool: {
                        filter: {
                          term: {
                            state: 'COMPLETED'
                          }
                        },
                        must_not: {
                          exists: {
                            field: 'acknowledged_time'
                          }
                        }
                      }
                    }
                  },
                  last_notification_time: {
                    max: {
                      field: 'last_notification_time'
                    }
                  },
                  latest_alert: {
                    top_hits: {
                      size: 1,
                      sort: [{
                        start_time: {
                          order: 'desc'
                        }
                      }],
                      _source: {
                        includes: ['last_notification_time', 'trigger_name']
                      }
                    }
                  }
                }
              }
            }
          }
        };
        const esAggsResponse = await client('alerting.getMonitors', aggsParams);
        const buckets = _lodash.default.get(esAggsResponse, 'aggregations.uniq_monitor_ids.buckets', []).map(bucket => {
          const {
            key: id,
            last_notification_time: {
              value: lastNotificationTime
            },
            ignored: {
              doc_count: ignored
            },
            acknowledged: {
              doc_count: acknowledged
            },
            active: {
              doc_count: active
            },
            errors: {
              doc_count: errors
            },
            latest_alert: {
              hits: {
                hits: [{
                  _source: {
                    trigger_name: latestAlert
                  }
                }]
              }
            }
          } = bucket;
          const monitor = monitorMap.get(id);
          monitorMap.delete(id);
          return {
            ...monitor,
            id,
            lastNotificationTime,
            ignored,
            latestAlert,
            acknowledged,
            active,
            errors,
            currentTime: Date.now(),
            associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[id] || 0
          };
        });
        const unusedMonitors = [...monitorMap.values()].map(monitor => ({
          ...monitor,
          lastNotificationTime: null,
          ignored: 0,
          active: 0,
          acknowledged: 0,
          errors: 0,
          latestAlert: '--',
          currentTime: Date.now(),
          associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[monitor.id] || 0
        }));
        let results = _lodash.default.orderBy(buckets.concat(unusedMonitors), [sortField], [sortDirection]);
        // If we sorted on monitor name then we already applied from/size to the first query to limit what we're aggregating over
        // Therefore we do not need to apply from/size to this result set
        // If we sorted on aggregations, then this is our in memory pagination
        if (!monitorSorts[sortField]) {
          results = results.slice(from, from + size);
        }
        return res.ok({
          body: {
            ok: true,
            monitors: results,
            totalMonitors
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - getMonitors', err);
        if ((0, _helpers.isIndexNotFoundError)(err)) {
          return res.ok({
            body: {
              ok: false,
              resp: {
                totalMonitors: 0,
                monitors: []
              }
            }
          });
        }
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "acknowledgeAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const acknowledgeResponse = await client('alerting.acknowledgeAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "acknowledgeChainedAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id,
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, this.dataSourceEnabled, req, this.esDriver);
        const acknowledgeResponse = await client('alerting.acknowledgeChainedAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeChainedAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "executeMonitor", async (context, req, res) => {
      try {
        const {
          dryrun = 'true'
        } = req.query;
        const params = {
          body: req.body,
          dryrun
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const executeResponse = await client('alerting.executeMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: executeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - executeMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    //TODO: This is temporarily a pass through call which needs to be deprecated
    _defineProperty(this, "searchMonitors", async (context, req, res) => {
      try {
        const {
          query,
          index,
          size
        } = req.body;
        const params = {
          index,
          size,
          body: query
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const results = await client('alerting.getMonitors', params);
        return res.ok({
          body: {
            ok: true,
            resp: results
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - searchMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
  }
}
exports.default = MonitorService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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