package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.40.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.40.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`MP`!````P!X```````!``````````,@!`0``
M`````````$``.``)`$``&@`9``8````$````0`````````!``````````$``
M````````^`$```````#X`0````````@``````````P````0````X`@``````
M`#@"````````.`(````````;`````````!L``````````0`````````!````
M!0```````````````````````````````````'B@````````>*``````````
M``$```````$````&````</L```````!P^P$``````'#[`0``````&`4`````
M``!`!0```````````0```````@````8```"`^P```````(#[`0``````@/L!
M```````@`@```````"`"````````"``````````$````!````%0"````````
M5`(```````!4`@```````$0`````````1``````````$`````````%#E=&0$
M````P)@```````#`F````````,"8````````#`$````````,`0````````0`
M````````4>5T9`8`````````````````````````````````````````````
M````````````$`````````!2Y71D!````'#[````````</L!``````!P^P$`
M`````)`$````````D`0````````!`````````"]L:6(O;&0M;&EN=7@M86%R
M8V@V-"YS;RXQ```$````%`````,```!'3E4`TH#`Y"X<)[B`.BXG40DH4*%@
M!QH$````$`````$```!'3E4```````,````'`````````#L```!)````!```
M``@````A$!``!<`$A"@"$`6`AD"$@HD`!@D@!(B(@!!!N@`""`````!)````
M2P```$P```````````````````!.`````````$\```!0````40``````````
M````4@```%,`````````````````````````5````%4```!6`````````%<`
M``````````````````!8````60```%H```````````````````!;````7```
M``````!>`````````%\`````````8```````````````````````````````
M`````&$```````````````````!B`````````&,```````````````````!E
M````?JF<FUGOEWEK"4/6=E0\E"'_Y`X-.I<+P^4_+(D'<-/9<5@<K4OCP&L:
MGE#59-1\F4_#FR_%F[8S8MOM:W^:?`&O-.B[XY)\LP&+;T+ZUX=#1=7L9Z_,
M9A6,)<RCX/M)S]Y'+>-'Q@'@IP(*;\7?'[[:5QO[H6Q@````````````````
M```````````````````````````#``L`R!L````````````````````````#
M`!8````"`````````````````'$````2`````````````````````````,T!
M```2`````````````````````````'L````2````````````````````````
M`'L!```2`````````````````````````)P!```2````````````````````
M`````#4"```2`````````````````````````-0!```2````````````````
M`````````(<````2``````````````````````````$````@````````````
M`````````````!8"```2``````````````````````````\"```2````````
M`````````````````+(#```2`````````````````````````+D````2````
M`````````````````````/4"```1`````````````````````````&$````2
M`````````````````````````*P"```2`````````````````````````.<`
M```2`````````````````````````!\"```2````````````````````````
M`)<!```2`````````````````````````,4!```2````````````````````
M`````*0!```2`````````````````````````'D"```B````````````````
M`````````!`````@`````````````````````````-P````2````````````
M`````````````"P````@`````````````````````````+4!```2````````
M`````````````````-$"```2`````````````````````````%("```2````
M`````````````````````*4"```2`````````````````````````&L"```1
M`````````````````````````,L"```2`````````````````````````%,`
M```2`````````````````````````"<!```6````````````````````````
M``0!```2`````````````````````````%L!```2````````````````````
M`````.(!```2`````````````````````````*@````2````````````````
M`````````/4````2`````````````````````````)X"```2````````````
M`````````````+H"```2`````````````````````````.,"```2````````
M`````````````````.@!```2`````````````````````````$,"```2````
M`````````````````````,T````2`````````````````````````(8!```2
M```````````````````````````"```2`````````````````````````)H`
M```2`````````````````````````"`!```2````````````````````````
M`$\!```2`````````````````````````%$"```2````````````````````
M`````+,"```2`````````````````````````)8"```2````````````````
M`````````.T!```2`````````````````````````*X!```2````````````
M`````````````#H!```2`````````````````````````(@"```2````````
M`````````````````!0!```2`````````````````````````/H!```2````
M`````````````````````'("```2`````````````````````````$8````1
M`````````````````````````'8````2`````````````````````````,(`
M```2`````````````````````````&P!```2````````````````````````
M`%D"```2`````````````````````````-D"```1````````````````````
M`````+X!```2`````````````````````````-L!```2````````````````
M`````````"<"```2`````````````````````````-L"```1````````````
M`````````````(\"```2`````````````````````````*`#```2``T`\#4`
M```````P``````````0$```0`!<`L``"`````````````````-8#```0`!8`
M```"`````````````````.,#```0`!<`B``"`````````````````%@$```2
M``T`!"D```````#T`@```````-@#```@`!8````"`````````````````/P#
M```2``T`H"<````````T`````````&4$```1`!8`@``"```````(````````
M`!@$```0`!<`B``"`````````````````,<#```1``\`(%<````````$````
M`````)8#```2``T`(%4```````#<`0```````%`#```2``T`"#\```````"@
M`@```````&\$```2``T`H#,```````"H`````````#(#```2``T``"D`````
M```$`````````-P#```2``T`P!X````````T`````````&8"```2``T`0!P`
M``````!T`@````````T$```0`!<`L``"`````````````````"T$```0`!<`
ML``"`````````````````&8#```2``T`(#,```````"``````````/$#```2
M``T`Z%,````````P`0```````#\$```0`!8`B``"`````````````````(H#
M```2``T`0"P```````"\`0```````"0$```2``T`H%,```````!(````````
M`'<$```2``T``"X````````8!0```````#($```2``T`:#<```````"@!P``
M`````$,#```2``T``"P````````\`````````'0#```2``T`X"<````````<
M`0```````+L#```2``T`J$$````````8`0````````P$```0`!<`L``"````
M`````````````$8$```2``T`(#8```````!(`0````````!?7V=M;VY?<W1A
M<G1?7P!?251-7V1E<F5G:7-T97)434-L;VYE5&%B;&4`7TE435]R96=I<W1E
M<E1-0VQO;F5486)L90!03%]D;U]U;F1U;7``4&5R;%]S>7-?=&5R;0!097)L
M7V=V7V9E=&-H<'8`<&5R;%]F<F5E`%!E<FQ?9V5T7W-V`%!E<FQ?871F;W)K
M7W5N;&]C:P!P97)L7V1E<W1R=6-T`%!E<FQ?;F5W6%-?9FQA9W,`<&5R;%]R
M=6X`4&5R;%]N97=84P!097)L7W-V7W-E='!V;@!P97)L7V%L;&]C`%!E<FQ?
M<W9?<V5T<'8`<&5R;%]C;VYS=')U8W0`8F]O=%]$>6YA3&]A9&5R`%!E<FQ?
M<V%F97-Y<VUA;&QO8P!03%]C=7)R96YT7V-O;G1E>'0`4&5R;%]C<F]A:U]N
M;V-O;G1E>'0`4&5R;%]M9U]S970`4&5R;%]A=&9O<FM?;&]C:P!097)L7W-Y
M<U]I;FET,P!P97)L7W!A<G-E`%]?<W1A8VM?8VAK7V9A:6P`97AI=`!L<V5E
M:S8T`')E861D:7(V-`!S=')D=7``8VQO<V5D:7(`;W!E;C8T`'-T<G)C:'(`
M=6YL:6YK`'-T<G1O:P!S=')L96X`<FUD:7(`<F5A9`!?7VUE;6-P>5]C:&L`
M;6MD:7(`7U]V9G!R:6YT9E]C:&L`9V5T<&ED`')E861L:6YK`')E86QL;V,`
M7U]S<')I;G1F7V-H:P!?7VUE;6UO=F5?8VAK`%]?9G!R:6YT9E]C:&L`;'-T
M870V-`!?7VQI8F-?<W1A<G1?;6%I;@!S=&1E<G(`;65M8VAR`%]?8WAA7V9I
M;F%L:7IE`'-T<F-H<@!M96US970`;W!E;F1I<@!G971U:60`86-C97-S`&UE
M;6-P>0!S=')C;7``7U]E<G)N;U]L;V-A=&EO;@!A8F]R=`!S=')N8VUP`%]?
M96YV:7)O;@!?7W)E9VES=&5R7V%T9F]R:P!?7W-T86-K7V-H:U]G=6%R9`!L
M:6)P97)L+G-O`&QI8F,N<V\N-@!L9"UL:6YU>"UA87)C:#8T+G-O+C$`<&%R
M7V-U<G)E;G1?97AE8P!P87)?8F%S96YA;64`6%-?26YT97)N86QS7U!!4E]"
M3T]4`'!A<E]E;G9?8VQE86X`<&%R7V-U<G)E;G1?97AE8U]P<F]C`'!A<E]D
M:7)N86UE`'-H85]F:6YA;`!G971?=7-E<FYA;65?9G)O;5]G971P=W5I9`!P
M87)?8VQE86YU<`!?24]?<W1D:6Y?=7-E9`!?7V1A=&%?<W1A<G0`7U]B<W-?
M<W1A<G1?7P!S:&%?=7!D871E`'AS7VEN:70`7U]E;F1?7P!?7V)S<U]E;F1?
M7P!?7V)S<U]S=&%R=`!S:&%?:6YI=`!?96YD`'!A<E]M:W1M<&1I<@!?961A
M=&$`<&%R7W-E='5P7VQI8G!A=&@`<&%R7V9I;F1P<F]G`&UY7W!A<E]P;`!P
M87)?9&EE`'!A<E]I;FET7V5N=@!'3$E"0U\R+C$W`$=,24)#7S(N,S0`1TQ)
M0D-?,BXS,P`O=7-R+VQI8B]P97)L-2\U+C0P+V-O<F5?<&5R;"]#3U)%````
M``````$``@`!``$``@`"``(``0`!``(``@`"``$``P`!``(``0`"``(``@`"
M``(``0`!``$``@`"``0``@`"``(``0`!``$``0`"``$``0`"``(``@`"``(`
M`0`"``(``0`"``$`!``"``(``@`"``$``@`!``(``@`!``(``0`!``4``@`"
M``(``@`"``(``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!``$````!``$`'`,``!`````@````EY&6
M!@```P"$!`````````$``P`2`P``$`````````"TD98&```%`(\$```0````
MLY&6!@``!`":!```$````)>1E@8```(`A`0```````!P^P$```````,$````
M````X!\```````!X^P$```````,$````````C!\```````!8_P$```````,$
M`````````%<```````#P_P$```````,$````````0!P````````(``(`````
M``,$````````"``"```````@``(```````,$`````````%L````````H``(`
M``````,$````````D%<````````X``(```````,$````````@%@```````!`
M``(```````,$````````X%H```````!(``(```````,$````````Z%H`````
M``!0``(```````,$````````\%H```````!8``(```````,$````````^%H`
M``````!H``(```````,$````````R%H```````!P``(```````,$````````
MT%H```````"```(```````,$````````"%L```````#@_0$```````$$```#
M``````````````#H_0$```````$$```$``````````````#P_0$```````$$
M```%``````````````#X_0$```````$$```&````````````````_@$`````
M``$$```'```````````````(_@$```````$$```(```````````````0_@$`
M``````$$```)```````````````8_@$```````$$```*```````````````@
M_@$```````$$```+```````````````H_@$```````$$```,````````````
M```P_@$```````$$```-```````````````X_@$```````$$```.````````
M``````!`_@$```````$$```/``````````````!(_@$```````$$```0````
M``````````!0_@$```````$$```1``````````````!8_@$```````$$```2
M``````````````!@_@$```````$$```3``````````````!H_@$```````$$
M```4``````````````!P_@$```````$$```5``````````````!X_@$`````
M``$$```6``````````````"`_@$```````$$```7``````````````"(_@$`
M``````$$```8``````````````"0_@$```````$$```9``````````````"8
M_@$```````$$```:``````````````"@_@$```````$$```;````````````
M``"H_@$```````$$```<``````````````"P_@$```````$$```=````````
M``````"X_@$```````$$```>``````````````#`_@$```````$$```?````
M``````````#(_@$```````$$```@``````````````#0_@$```````$$```B
M``````````````#8_@$```````8$```C``````````````#@_@$```````$$
M```D``````````````#H_@$```````$$```E``````````````#P_@$`````
M``$$```F``````````````#X_@$```````$$```G````````````````_P$`
M``````$$```H```````````````(_P$```````$$```I```````````````0
M_P$```````$$```J```````````````8_P$```````$$```L````````````
M```@_P$```````$$```M```````````````H_P$```````$$```N````````
M```````P_P$```````$$```O```````````````X_P$```````$$```P````
M``````````!`_P$```````$$```Q``````````````!(_P$```````$$```R
M``````````````!0_P$```````$$```S``````````````!@_P$```````$$
M```T``````````````!H_P$```````$$```U``````````````!P_P$`````
M``$$```V``````````````!X_P$```````$$```W``````````````"`_P$`
M``````$$```X``````````````"(_P$```````$$```Y``````````````"0
M_P$```````$$```Z``````````````"8_P$```````$$```[````````````
M``"@_P$```````$$```\``````````````"H_P$```````$$```]````````
M``````"P_P$```````$$```^``````````````"X_P$```````$$```_````
M``````````#`_P$```````$$``!```````````````#(_P$```````$$``!!
M``````````````#0_P$```````$$``!$``````````````#8_P$```````$$
M``!%``````````````#@_P$```````$$``!&``````````````#H_P$`````
M``$$``!'``````````````#X_P$```````$$``!(``````````````"X_0$`
M``````($```+``````````````#`_0$```````($```A``````````````#(
M_0$```````($```K``````````````#0_0$```````($``!"````````````
M```?(`/5_7N_J?T#`)'(``"4_7O!J,`#7];P>[^I\```T!':1OD0PC:1(`(?
MUA\@`]4?(`/5'R`#U?```-`1WD;Y$.(VD2`"'];P``#0$>)&^1`"-Y$@`A_6
M\```T!'F1OD0(C>1(`(?UO```-`1ZD;Y$$(WD2`"'];_@P'1XP``T&,D1_G]
M>P.I_<,`D>*'`*GB(P"1X4,`D>`?`+G@<P"1\U,$J?5;!:ED`$#YY!<`^00`
M@-+C``#08^1'^6``/];B``#00@Q'^>```-``=$?YX0,"JN,``-!CK$?Y8``_
MUN```-``V$?Y``!`.<`*`#;@``#0`&Q'^4'0.]7S``#P(FA@^/0?0+E!:%(Y
M@!(`$90.`!$A!``R06@2.>$``-`AS$?Y`'Q]DR``/];C`P"JY@M`^>$``/!B
M0@*1X!]`N25`0/DD``"0QP!`^83@*I%G$`"I(0``D"$`*Y%#!`#Y900!J1\$
M`'$-"P!4(0"`TA\@`]4?(`/5)`P`$<5X8?@A!`"19=@D^!\``6ML__]4``P`
M$75"`I%_V"#XY0``T*7\1OD!``"PH`Y`^>(#%"HAH!R1!`"`TJ``/]9`!@`T
M=4("D?:3`)'A``#0(:!'^:`.0/D@`#_6]`,`*N$#%JH@``"0`$`@D90``)1@
M!0"T<T("D>$``-`A\$;Y8`Y`^2``/];@``#0`&A'^0``/];@``#0`"1'^>(7
M0/D!`$#Y0@`!ZP$`@-+!!0!4_7M#J>`#%"KS4T2I]5M%J?^#`9'``U_6X```
MT`!,1_GS``#P```_UF)"`I%`#`#YH`,`M.$``-`A@$?Y(``_UN```-``;$?Y
M0=`[U2)H8/A?6"0YH/__%Z`.0/GA``#0(2!'^2``/];+__\7H`I`^6```+30
M"`"4T___%^$#%JH@``"0`&`>D6$``)2@"@#YH/G_M,@(`)3+__\7@`"`4K+_
M_Q?A``#0(3A'^2``@%(@`#_6X```T`"81_D``#_6'R`#U1\@`]4?(`/5'R`#
MU1T`@-(>`(#2Y0,`JN$#0/GB(P"1Y@,`D>```-``^$?Y`P"`T@0`@-)1__^7
M2/__E^```-``$$?Y0```M$#__Q?``U_6'R`#U1\@`]4?(`/5'R`#U1\@`]4?
M(`/5X```\``@`I'A``#P(2`"D3\``.O```!4X0``T"%(1_EA``"T\`,!J@`"
M'];``U_6X```\``@`I'A``#P(2`"D2$``,LB_'_300R!BR'\09/!``"TX@``
MT$)01_EB``"T\`,"J@`"'];``U_6_7N^J?T#`)'S"P#Y\P``\&`B0CE``0`W
MX0``T"%$1_F!``"TX```\``$0/D@`#_6V?__ER``@%)@(@(Y\PM`^?U[PJC`
M`U_6'R`#U1\@`]4?(`/5W/__%Q\@`]4?(`/5'R`#U1\@`]4?(`/5'R`#U1\@
M`]7]>[NIX@``L$+T1_G]`P"1\U,!J?DC`/E9`$#YN08`M/5;`JGV`P&J%0"`
M4O=C`ZGW`P"J``!`.1_T`'$$&$!Z(0,`5#,#0/D3`@"T^```L!A;1_FU?D"3
M]`,9JA\@`]7B`Q6JX0,7JN`#$ZH``S_6@```-6!J=3@?]`!Q(`(`5)..0/CS
M_O^U]5M"J?=C0ZG@`Q.J^2-`^?-30:G]>\6HP`-?UO4#%ZJ@'D`X'_0`<008
M0'JA__]4M0(72^+__Q>4`AG+M08`D7,"%8N4_D.3U`(`N>`#$ZKY(T#Y]5M"
MJ?=C0ZGS4T&I_7O%J,`#7]83`(#2Y___%Q\@`]4?(`/5_\,"T>(``+!")$?Y
MX2,`D?U[":G]0P*1\U,`^?,#`*I#`$#YXT<`^0,`@-+B``"P0K!'^4``/]9`
M`@`UX1M`N0(`E%(A#!02/Q!`<2000GK``0!4X0``L"$D1_GC1T#Y(@!`^6,`
M`NL"`(#2X0$`5/-30/G]>TFI_\,"D<`#7]8``(!2]/__%^(``+!"8$?YX`,3
MJD$`@%)``#_6'P``<>`7GQKL__\7X```L`"81_D``#_6'R`#U1\@`]4?(`/5
M_T,#T>$``+`A)$?Y_7L)J?U#`I$B`$#YXD<`^0(`@-+```"T\U,*J?,#`*H!
M`$`Y@0$`-?-32JG@``"P`"1'^>)'0/D!`$#Y0@`!ZP$`@-*!"P!4_7M)J?]#
M`Y'``U_6X0``L"&X1_D@`#_6]`,`JB#^_[3W8PRI]P``L/="1_GU6PNIX`,4
MJN`"/]8`!P"T`4Q`.15,`)$_N`!Q80``5*$&0#D!__\T`$Q`.1^X`'%@!P!4
M^```L!CO1_G@`Q.J``,_UO8#`*K@`Q6J``,_UN$``+`AI$?YU@(`"]8*`!'6
M?D"3X`,6JB``/];F``"PQO!'^>4#%:KB`Q:J]0,`JD$`@%+D`Q.J`P``\&.@
M')'``#_6X@``L$)<1_GA(P"1X`,5JD``/]8?!``QH```5.`;0+D`#!02'Q!`
M<<`#`%3A``"P(?1&^>`#%:H@`#_6X0``L"'<1_G@`Q6J(``_UN`#%*K@`C_6
M0/G_M>$``+`A5$?YX`,4JB``/];A``"P(7A'^>`#$ZH@`#_6\U-*J?5;2ZGW
M8TRIJ/__%Z`&0#D?N`!Q@?C_5*`*0#G`]O\TP?__%^`#%:J2__^7Y?__%^``
M`+``F$?Y\U,*J?5;"ZGW8PRI```_UA\@`]4?(`/5_\,!T?U[`:G]0P"1\U,"
MJ?0#`:KS`P"J]5L#J?=C!*GY:P6I^W,&J2``0#GA``"P(21'^1_T`''W``"P
M]^Y'^2(`0/GB!P#Y`@"`TI06E)K@`Q2JX`(_UO8#`*KA$P"1X`,3JO_^_Y?@
M"`"T]0,`JN`"/];?`@#KJ0<`5/@'@+GY``"P.?='^?L``+![IT?Y&/-]TV`"
M0#D?]`!Q!!A`>B`-`%3D`Q.J@QQ`.'_T`'%D&$!ZH?__5(0`$\N$?$"3-0-`
M^<`*`)$```2+8`,_UJ!J./C@`@"T(0-`^>`#$ZHC:'CX`11`.&$``#D_]`!Q
M)!A`>@`!`%0?(`/5'R`#U0(40#AB'``X7_0`<4080'J!__]4H0>`4@``@-)A
M%``X@FI@.&)H(#@`!`"1HO__->```+``)$?YX@=`^0$`0/E"``'K`0"`TF$(
M`%3]>T&I\U-"J?5;0ZGW8T2I^6M%J?MS1JG_PP&1P`-?U@$`@-*":F$XHFHA
M."$$`)&B__\U[/__%_D``+`Y]T?Y%0"`4CP#0/F``T#YH`4`M.`#'*H!C$#X
MX@,5*K4&`!&A__^U0@P`$5=\?9/A``#0()!`N:`"`#7[``"P>Z='^2(`@%+@
M`Q>J(I``N6`#/];Z`P"J@/K_M.0``+"$!$?YN'Y]D^,#%ZKA`QRJX@,8JH``
M/]8Z`P#YH`8`$?4'`+E?VR#XH?__%^(``+!"-$?YX0,7JN`#'*I``#_6(`,`
M^?H#`*H`^/^T^P``L'NG1_FX?GV3\?__%P0`@-*>__\7%P*`TMK__Q?@``"P
M`)A'^0``/];]>[VI#0*"TOT#`)'S4P&I]5L"J?]C+<O_`P+Y\P,`JN,``+!C
M)$?Y]2,`D?0#`:KB_X'28`!`^>`'"/D``(#2X0,5JN,``+!C%$?Y````\`#`
M')%@`#_6'P``<<T``%0$!`!1X@,`*J#*9#@@`P`UY`(`->```+``;$?Y0]`[
MU>(#%*KA`Q.J8&A@^.,``+!C,$?Y8``_UN```+``)$?YX@=(^0$`0/E"``'K
M`0"`TF$#`%0-`H+2_V,MB_-30:GU6T*I_7O#J,`#7];B`P0JXP``L&/41_E6
M?$"3X@,6JN`#%:KA!8!28``_UD#\_[3@``"P`&Q'^430.]7C`Q:JX@,5JN$#
M$ZJ`:&#XY```L(241_F``#_6X/__%^```+``F$?Y```_UOU[OJDC``#P`0``
M\/T#`)%CX"&1\U,!J530.]7S``"P<VY'^>(``+!"<$?Y(0`=D>0``+"$X$?Y
M@&IS^(``/]:`:G/X!```\/-30:F$P"J1_7O"J.,#!*KP``"P$'Y'^0(``+`!
M``#P0B`\D2&`'9$%`(!2``(?UN```+``;$?Y1-`[U>(``+!"<$?Y(P``\/``
M`+`0XD?Y8^`AD8!H8/@!``#P(0`=D0`"']8?(`/5'R`#U1\@`]7]>[VI#02$
MTOT#`)'S4P&I]5L"J?]C+<O_`P+YX```L``D1_GS(P"1%`"`T@$`0/GA#Q#Y
M`0"`TN```+``&$?Y```_UN0#`"KF``"PQO!'^2(`@M)!`(!2X`,3J@4``/`#
M``#PI>`=D6,`'I'``#_6XP``L&,41_D``H+2]F,@BP(`@M+@`Q.JX0,6JF``
M/]9``O@WX0``L"&D1_D5!``1\P,`JK5^0)/@`Q6J(``_UO0#`*H@`0"TY```
ML(2\1_ES?D"3XP,5JN$#%JKB`Q.J@``_UI]J,SC@``"P`"1'^>(/4/D!`$#Y
M0@`!ZP$`@-(!`0!4#02$TO]C+8O@`Q2J\U-!J?5;0JG]>\.HP`-?UN```+``
MF$?Y```_UA\@`]6X__\7_7NZJ0T6@M+]`P"1\U,!J?5;`JG_8RW+_P,"^?0#
M`*KB``"P0B1'^?4#`:KADP"10`!`^>!7"/D``(#2````\`!@'I&N_?^7\P,`
MJN(``+!"R$?YX`,4JN$%@%)``#_68`(`M.$#%*H```#P`*`>D8O^_Y?@`Q2J
MX0``L"$D1_GC5TCY(@!`^6,``NL"`(#2`1(`5`T6@M+_8RV+\U-!J?5;0JG]
M>\:HP`-?UN```+``P$?Y]W,(^?<``+#W"D?Y^'<(^?N#"/GA`P"J_(<(^>`#
M%:KA#P#Y%@``\"``/];!XAZ1X`(_UOP#`*J`#@"TX```L`!<1_GXHP*1^P``
ML'OS1_G@!P#Y]0``L+7N1_GY>PCYX```L`"T1_G@"P#Y^G\(^8`#0#G`!P`T
MTP``M.(+0/GA`QRJX`,3JD``/]:`!0`TX`,<JJ`"/]8"!`#1^0,5JH(#`HOZ
M`P"JGP,"ZZ,``%0*```47_0?.)\#`NN```!40`!`.1^\`'%@__]4X`,<JJ`"
M/];Z`P"JX`,4JB`#/]8``!J+``0`D1_\/_$H`P!4`@""TD$`@%+D`QRJY@,4
MJN`#&*H%``#P`P``\*4`'Y%C(!^18`,_UN('0/GAHP"1X`,8JD``/]:@```U
MX#M`N0`,%!(?($!Q0`,`5,'B'I$``(#2X`(_UOP#`*H@^?^U]W-(^?AW2/GY
M>TCY^G](^?N#2/G\ATCYDO__%S,$`+1@`D`Y'[@`<>```%0$``#P.@"`TOD`
M`+`Y[T?YG$`>D=+__Q=@!D`Y(/W_-/C__Q?B``"P0F!'^>`#&*HA`(!20``_
MUD#\_S7A`QBJ````\`"@'I$)_O^7X0]`^>`#&*H@`#_6]W-(^?AW2/GY>TCY
M^G](^?N#2/G\ATCY=O__%P0``/`Z`(#2^0``L#GO1_F<0!Z1M?__%_=S2/GX
M=TCY^X-(^?R'2/EF__\7X```L`"81_GW<PCY^'<(^?E["/GZ?PCY^X,(^?R'
M"/D``#_6'R`#U1\@`]7]>[ZIX06`4N(``+!"/$?Y_0,`D?,+`/GS`P"J0``_
MUA\``/%@!H":\PM`^?U[PJCP``"P$,)'^0`"']8?(`/5_7N]J0T"@M+]`P"1
M_V,MR_\#`OGA``"P(21'^2(`0/GB!PCY`@"`TD`'`+3U&PCY]0,`J@$`0#FA
M!@`TX0``L"'L1_GS$PCY]!<(^2``/]8"!`"1\P,`JE\$0/'(!P!4XP``L&,L
M1_GT(P"1X0,5JN`#%*I@`#_68`8`T8`"`(L?`!3KJ```5`<``!0?]!\X'P`4
MZX```%0!`$`Y/[P`<6#__U3B``"P0CQ'^>`#%*KA!8!20``_UB`%`+0?`!3K
M:`0`5.$``+`AP$?YX`,4JB``/];A``"P(21'^>,'2/DB`$#Y8P`"ZP(`@-)A
M!0!4\Q-(^0T"@M+T%TCY]1M(^?]C+8O]>\.HP`-?UO4;2/G@``"P`"1'^>('
M2/D!`$#Y0@`!ZP$`@-+A`P!4\```L!#"1_D```#P`$`>D0T"@M+_8RV+_7O#
MJ``"']8?```YW?__%P``@-+?__\7X```L``D1_GB!TCY`0!`^4(``>L!`(#2
M(0$`5/,32/D```#P]!=(^0!`'I'U&TCY\```L!#"1_GJ__\7X```L`"81_D`
M`#_6X```L`"81_GS$PCY]!<(^?4;"/D``#_6'R`#U?^#`='@``"P`"1'^?U[
M`:G]0P"1\U,"J10``/#S$P"1]5L#J91"'Y'W8P2I^2L`^?4``+"U]D?Y`0!`
M^>$'`/D!`(#2X0,3JN`#%*IM_/^7X`$`M*("0/GC!X"Y0`P#BP$$0/E!>"/X
MX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,4JE_\_Y=@_O^U]0``L+7V1_D4
M``#PE((?D>$#$ZK@`Q2J5_S_E^`!`+2B`D#YXP>`N4`,`XL!!$#Y07@C^.'^
M_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#%*I)_/^78/[_M?4``+"U]D?Y%```
M\)2B'Y'A`Q.JX`,4JD'\_Y<``@"TH@)`^>,'@+E`#`.+`01`^4%X(_CA_O^T
M'R`#U0$(0/D!!`#Y`"``D:'__[7A`Q.JX`,4JC+\_Y=`_O^U]0``L+7V1_D4
M``#PE.(?D>$#$ZK@`Q2J*OS_ER`"`+2B`D#YXP>`N4`,`XL!!$#Y07@C^.'^
M_[0?(`/5'R`#U0$(0/D!!`#Y`"``D:'__[7A`Q.JX`,4JAK\_Y<@_O^U]@``
ML-;V1_D4``#PE0(@D>$#$ZK@`Q6J$OS_ER`"`+3"`D#YXP>`N4`,`XL!!$#Y
M07@C^.'^_[0?(`/5'R`#U0$(0/D!!`#Y`"``D:'__[7A`Q.JX`,5J@+\_Y<@
M_O^U]@``D-;V1_D5``#0M4(@D>$#$ZK@`Q6J^OO_ER`"`+3"`D#YXP>`N4`,
M`XL!!$#Y07@C^.'^_[0?(`/5'R`#U0$(0/D!!`#Y`"``D:'__[7A`Q.JX`,5
MJNK[_Y<@_O^U]P``D/?V1_D5``#0MF(>D>$#$ZK@`Q:JXOO_ER`"`+3B`D#Y
MXP>`N4`,`XL!!$#Y07@C^.'^_[0?(`/5'R`#U0$(0/D!!`#Y`"``D:'__[7A
M`Q.JX`,6JM+[_Y<@_O^U^```D!CW1_D6``#0UX(@D>$#$ZK@`Q>JROO_ER`"
M`+0"`T#YXP>`N4`,`XL!!$#Y07@C^.'^_[0?(`/5'R`#U0$(0/D!!`#Y`"``
MD:'__[7A`Q.JX`,7JKK[_Y<@_O^U^0``D#GW1_D7``#0^,(@D>$#$ZK@`QBJ
MLOO_ER`"`+0B`T#YXP>`N4`,`XL!!$#Y07@C^.'^_[0?(`/5'R`#U0$(0/D!
M!`#Y`"``D:'__[7A`Q.JX`,8JJ+[_Y<@_O^U^0``D#GW1_D8``#0&`,AD>$#
M$ZK@`QBJFOO_ER`"`+0B`T#YXP>`N4`,`XL!!$#Y07@C^.'^_[0?(`/5'R`#
MU0$(0/D!!`#Y`"``D:'__[7A`Q.JX`,8JHK[_Y<@_O^U^0``D#GW1_D8``#0
M&*,>D>$#$ZK@`QBJ@OO_ER`"`+0B`T#YXP>`N4`,`XL!!$#Y07@C^.'^_[0?
M(`/5'R`#U0$(0/D!!`#Y`"``D:'__[7A`Q.JX`,8JG+[_Y<@_O^UX0,3J@``
M`-``0"&1;?O_E^$#`*I@``"TX,(@D5'\_Y?A`Q.J````T`"@(9%E^_^7X0,`
MJH```+0```#0```BD4C\_Y?A`Q.J````T`!`(I%<^_^7X0,`JH`"`+2@8AZ1
M0/S_E^```)``)$?YX@=`^0$`0/E"``'K`0"`TH$"`%3]>T&I@`(@D?5;0ZD!
M``#0\U-"J2'@(I'W8T2I^2M`^?^#`9$O_/\7X0,3J@```-``@"*10_O_E^$#
M`*I`_?^TP((@D2?\_Y?G__\7X```D`"81_D``#_6'R`#U1\@`]7_@P#1````
MT.(``)!")$?YX1,`D?U[`:G]0P"1`(`@D4,`0/GC!P#Y`P"`TBW[_Y?@`0"T
M``!`.1_``'$$&$!ZX`>?&N$``)`A)$?YXP=`^2(`0/EC``+K`@"`TL$``%3]
M>T&I_X,`D<`#7]8``(!2]?__%^```)``F$?Y```_UO\#!='L!H`2Z0``D"DE
M1_GM`P21_7L'J?W#`9'K#X`2Z@``D$IE1_GABQ"IX@,`JN.3$:E!`(!2XP,`
MD>6;$JGGGP#YX`<$K>(/!:WD%P:MYA\'K20!0/GD-P#Y!`"`TN0#!9'DDP*I
M[1\`^>PO""E``4#Y_H/"//^#PSSD``"0A)Q'^?X#@#W_!X`]_H.$//^#A3R`
M`#_6X!^`4N$``)`A.$?Y(``_UOU[O*G]`P"1\U,!J?5;`JGW8P.I_T-`T?\#
M`OG_0P#1_P,"^>1#0)'C``"08R1'^0$`@-+T``"0E`)'^64`0/F%!`#Y!0"`
MTO,#`"I"`(!2@`(_UA@$`-'@`QCK`3]`D@`\0)(@1(#:&`,`ZX0&`%3U``"0
MM8Y'^?8#`)'W``"PX0,8JN`#$RH"`(!2@`(_UA\$`+'@!@!4`@&`TN$#%JK@
M`Q,J(@"@\J`"/]8?!``QH`0`5`!\0),?'`#QJ0,`5``@`-'``@"KHP``5!D`
M`!0`!`#1'P`6Z\,"`%0!`$#YX@I`^5\``>M!__]4```6RP``&(OA``"0(21'
M^>-#0)%D!$#Y(@!`^80``NL"`(#2800`5/]#`)'_0T"1\U-!J?5;0JGW8T.I
M_7O$J,`#7]880T#1'T-`L4'Z_U0``("2[?__%^```)``)$?YXD-`D4,$0/D!
M`$#Y8P`!ZP$`@-+A`0!4````T`!`(Y%[__^7X```D``D1_GB0T"10P1`^0$`
M0/EC``'K`0"`TH$``%0```#0```CD7#__Y?@``"0`)A'^0``/];@``"0`(1'
M^?U[OZG]`P"1```_UN$``)`A'$?Y(``_UD```+0``$#Y_7O!J,`#7];_0P'1
MX0``D"$D1_G]>P&I_4,`D?-3`JGT`P"J]5L#J14``-"V@B.1(@!`^>('`/D"
M`(#2X`,6JN$3`)%I^O^7@```M/,#`*H``$`YX`$`->```)``)$?YX@=`^0$`
M0/E"``'K`0"`TB$&`%3]>T&IX0,4JJ""(Y'S4T*I]5M#J?]#`9$_^_\7]R,`
M^>`#%*KW``"0]^Y'^>`"/];U`P"JX`,3JN`"/];A``"0(:1'^;4"`(NU"@"1
MX`,5JB``/];G``"0Y_!'^>8#$ZKB`Q6J00"`4N0#%*KS`P"J!0``T`,``-"E
MX!Z18R`?D>``/];@``"0`"1'^>('0/D!`$#Y0@`!ZP$`@-*A`0!4_7M!J>$#
M$ZK@`Q:J\U-"J?5;0ZGW(T#Y_T,!D1;[_Q?@``"0`)A'^?<C`/D``#_6X```
MD`"81_D``#_6_P,'T>(``+!"0`"1X\,!D42@`)']>Q:I_8,%D>$``)`A)$?Y
MX`\`^4@D0:GS4Q>I\P,!D4:<1:GU6QBI]&,!D?=C&:GX\P"17H#"/"``0/G@
MKP#Y``"`TIW\0*WH)P2I````T$400/GFGP6IX0,8JD0T0/E^=`"M`&`>D64*
M`/F$"@#Y?PB`/03Z_Y>```"T`0!`.?4#`*H!*@`UX```D`"$1_GY:QJI^W,;
MJ>`7`/D``#_6X0``D"$<1_D@`#_6P!,`M!L`0/F;$P"T^@``D%KO1_G@`QNJ
M0`,_UA?X?]/W!@"1]0``D+6F1_G@`Q>J^0,7JJ`"/];@$P#Y9`-`.?P#`*K$
M*``T]@``D-;R1_D4``#0E.(CD>(#%ZK@`QRJ00"`4N,#%*K``C_6G`L`D3\#
M%^O@"@#1(2.7FF0?0#@W`!?+]PH`D?\"`>OW(H&:]P(!R_<"`(L$_O\U````
MT/?C`9$``"*1`P``%."&0/A``@"TX0,8JLCY_Y?T`P"J8/__M`$`0#DA__\T
M`_K_E^#^_S3A``"0(<!'^>`#%*H@`#_6]P,`J@`"`+4?(`/5'R`#U1\@`]7T
M``"0E,)'^7<"0/DW`0"TX`)`.2`+`#3@`Q>J\?G_EP`*`#5W!D#Y<R(`D3?_
M_[7@`Q>J0`,_UOD30/GS`P"J'```T.`#&:I``S_6=`(`"Y02$!&4?D"3X`,4
MJJ`"/];G`QFJY`,7JOD``)`YTT?YX@,4JH4#'Y$&``#0`P``T,8`))%C("21
M]P,`JD$`@%+``C_6X`,7J@$X@%(@`S_6'P0`,>$``%3@``"0`(A'^0``/]8%
M`$"YOT0`<8$9`%3B``"00K!'^>&#`I'@`Q>J0``_UO,#`"J@```UX+-`N0`,
M%!(?$$!Q0`D`5.$/0/D"``#0X```D`!D1_GD`Q>J(P!`^4)@)9$``$#Y00"`
M4N4``)"ED$?YH``_UA4`@-+@``"0`"1'^>*O0/D!`$#Y0@`!ZP$`@-*!)`!4
M^6M:J>`#%:K[<UNI_7M6J?-35ZGU6UBI]V-9J?\#!Y'``U_6X`,7JG,B`)&`
M`C_6]P,`JL#T_[2O__\7%P"`TJW__Q?U``"0M<)'^8`"0/D@`0"TX0,8JE/Y
M_Y=@``"T`0!`.4$"`#6`!D#YE"(`D2#__[7U``"0M:9'^:`!@-*Y`8#2]P,9
MJOH``)!:[T?YH`(_U@8``-#\`P"JV\`CD60*@%+@$P#Y6___%Z`"/]:4(@"1
M^P,`JJ#\_[3Z``"06N]'^4`#/]87^'_3]P8`D4C__Q?@%T#Y^[M`N0``/]9_
M`P!K8?;_5."S0+D`(``2'P`'<>'U_U3@`Q2JH`(_UN$#&*KU`P"J````T`#`
M)I$E^?^7X0,`JN`0`+3@#T#Y``!`^6'[_Y?[`P"J0!``M.`#&ZI``S_6X@,`
MJA\,`'%-`@!4XP``D&.T1_D`?$"3X@\"J0`0`-$!``#08`,`BR'@)I%@`#_6
M``$`->(30/E?$`!Q(!4`5$(4`%%@RV(X'[P`<:`4`%3A`QBJ&```T`"#()$$
M^?^7H```M```0#D?P`!Q!!A`>@$!`%3B``"00NA'^>`#&ZH!`(!20``_UOL#
M`"K@"P`U^P``D'L;1_D`@R"1`0``T"'@(I':^?^7F`,?D1H``-!@`S_6Y@,`
M*N4#&*KD`Q>JX@,4JN`#%:I'PRJ1`P``T$$`@%)CH">1P`(_U@```-``X">1
M_```D)R+1_G@#P#Y$@``%(`#/]8``$"Y'T0`<6$"`%1@`S_6<P8`$>,/0/E!
MPRJ1X0,`^>8#`"KG`Q,JY0,8JN0#%ZKB`Q2JX`,5JD$`@%+``C_6X`,5J@$X
M@%(@`S_6'P0`,6#]_U3A``"0(=Q'^>`#%ZH@`#_6X0,5J@```-``8!Z1J?G_
ME^`#%:I'_O^73O__%^$/0/GD`Q>JX```D`!D1_D"``#0(P!`^4)@))$``$#Y
M00"`4N8``)#&D$?YP``_UD#__Q<X_O^7X```D``D1_GBKT#Y`0!`^4(``>L!
M`(#2P0P`5/U[5JG@`Q6J\U-7J?5;6*GW8UFI_P,'D?```)`0PD?Y``(?UO8`
M`)#6\D?YSO[_%^`/0/D;`$#Y??__%ZO]_Y?D`P"J`/3_M^,``)!C`$?YXQ,"
MJ0$H`-'Z``"06H]'^0(`@%+@`QLJ8``_UN&#!)'@`QLJP@"`TN$/`/E``S_6
M'Q@`\>070/GACT&IH08`5.(C0;D`8(A2(&BH<E\``&LA\?]4(0A`>0"IB%(_
M``!KH?#_5('(`-$"`(!2X`,;*O.C!)%@`#_6X`,;*N$#$ZH"!8#20`,_UA^@
M`/$A!`!4Y@,3JH4#'Y'B`Q2JY`,7JN`#%:H'``#0`P``T.?`*I%C8">100"`
M4O]#!3G``C_6G/__%^`/0/D"!$#Y0NO_M.`#`JKB#P#Y0`,_UA\,`/&IZO]4
MX@]`^0`0`-$!``#0(0`GD4```(OB%T#Y0``_UA\``''B#T#Y>Q."FDK__Q?@
M``"0`"1'^>*O0/D!`$#Y0@`!ZP$`@-*!``!4````T``@)Y$O_?^7X```D`"8
M1_D``#_6X```D`"81_GY:QJI^W,;J0``/];_0P'1````T.(``)!")$?Y```@
MD?U[`:G]0P"1\U,"J530.]7S``"0<VY'^?5;`ZGV$P"1X0,6JO=C!*F5:G/X
M0P!`^>,'`/D#`(#2HL)".:+&`CDI^/^7@```M`$`0#D_Q`!QH`T`5*3[_Y>5
M:G/XX`,5JO<``)#W*D?Y`0``T&,`@%(A0"B1(@"`4N`"/];U`P"JH`D`M*$*
M0/D"``#0@&IS^$)@*)$A`$#Y^```D!@S1_D``S_6H`I`^0$`0/D@#$"YP`NP
M-X!J<_@!``#08P"`4B&`*)$B`(!2X`(_UO<``+#U`P"J8`$`M.$#%JH```#0
M`*`>D0#X_Y>A"D#YX@,`JB$`0/G@"`"T@&IS^``#/];#``#P8_A&^2(`@%*5
M:G/X`0``L"&@*)'@`Q6J8``_U@(``+#A`P"J0@`ID>`#%:H``S_6@FIS^/1"
M`I'A`Q:J````L`!@'I%?Q`(YYO?_EX`*`/GS`P"J0`,`M,(``/!"T$?Y`3B`
M4N`#$ZI``#_6`00`$3\$`'$$B%%Z808`5,```/``)$?YX@=`^0$`0/E"``'K
M`0"`TB$%`%3]>T&I\U-"J?5;0ZGW8T2I_T,!D<`#7];8``#P&#-'^;[__Q>`
M!D#YHOW_EX`*`/GS`P"J8/S_M<```/``)$?YX@=`^0$`0/E"``'K`0"`TH$"
M`%3!``#P(<1'^0```+``("F1(``_U@`$0#F@\O\TDO__%^)"`I'@`P&J001`
M^2$`0/DQ^?^7M___%X!J<_C"``#P0JA'^4``/]:?__\7P```\`"81_D``#_6
MP0``\"$D1_GC!T#Y(@!`^6,``NL"`(#2X?[_5/="`I'C`P`JQ```\(3$1_D`
M``"PX09`^>(#$ZH`P"F1(0!`^8``/];_PP#1P0``\"$D1_G]>P&I_4,`D?-3
M`JGS`P"J````L"(`0/GB!P#Y`@"`T@"`()'A$P"1B??_ER`!`+0``$`Y'\``
M<0080'K@!Y\:?P(`\>$'GQH_``!J@0$`5,```/``)$?YX@=`^0$`0/E"``'K
M`0"`TL$$`%3]>T&I\U-"J?_#`)'``U_68`)`.8#^_S3@`Q.J@?K_E\(``/!"
M/$?Y]`,`JN$%@%)``#_6'P``\<$``/`AP$?Y@`:`FB``/];#``#P8UA'^0$`
M`+""`(#2(0`DD6``/]8@_/\UP```\``D1_GB!T#Y`0!`^4(``>L!`(#2P0``
M5/U[0:G@`Q.J\U-"J?_#`)'$]_\7P```\`"81_D``#_6_7N_J0'@`)$$X`&1
MPP``\&,D1_G]`P"1%00`3_]#"M&6Y`!O90!`^>5'`?D%`(#2]^4`;^(#`)%:
MY`!OXP,`D3M\0&T]!,$\>SN?#K\'8&^^!UAOO`=(;WL+("[]5TA/_U=83]X?
M.DYX.Y5.O1\V3O\?-T[>'[Q.OQ^_3OD?ODYXC)],/P`$Z\']_U3Y'\`]0J`)
MD?CE`&_AHP&1__M`K?WS0:W[ZT*M(4``D2"`V#PA`-\\]1\\;M\?ODX@'"!N
MOA^]3IT?O$Y\'[M.%1PU;EL?NDXZ'[E.M@9A;[5604_5'K5.N1XX3CF`@#Q?
M``'KP?W_5`8D0*DE,X_2!R!!J450J_+A-T"I)'2=T@H00/G"_%O3`P$'RD(4
M!JIC``F**_U"TV,`",HA``6+(0`*BV)`(HM"``$+:WD)JFY]0)*M`06+0?Q;
MT\P!!\HA%`*JC`$&BB$`#8N,`0?*[4-!J</\0M.,`0B+8W@&JB$`#`MK`0/*
M:P$"BD_\0M,L_%O3:P$.RJT!!8N,%0&JC`$-BVL!!XN+`0L+XGD"JF]\0))C
M``+*8P`!BFW]6],,`@6+8P`/RJT5"ZIC``Z+K0$,BRS\0M/N$T#YK0$#"X%Y
M`:HDVZWRK/U;TT,``<K.`06+C!4-JD)\0)*,`0Z+8P`+BN[#0JEC``+*8P`/
MBV_]0M.#`0,+ZWD+JB]\0)(A``O*;/Q;TR$`#8K.`06+C!4#JB$`#\J,`0Z+
M(0`"BZ+]0M.!`0$+0G@-JFP!`LIN?4"2C`$#BBW\6],+`@6+C`$.RJT5`:J,
M`0^+K0$+BV_\0M/P'T#YJP$,"^-Y`ZI/?$"20@`#RFS]6]-"``&*#0(%BXP5
M"ZI"``_*C`$-BT(`#HLM_$+3@@$""_`C0/FA>0&J;``!RFY\0)*,`0N*3?Q;
MTP,"!8N,`0[*K14"JHP!#XNM`0.+;_U"T_`G0/FC`0P+ZWD+JB]\0))L_%O3
M(0`+R@T"!8LA``**C!4#JB$`#\J,`0V+3?Q"TZ)Y`JHA``Z+@0$!"VP!`LIN
M?4"2C`$#BHP!#LHM_%O3C`$/BZT5`:KP/T6I"P(%BW#\0M,#>@.JK0$+BU!\
M0))"``/*JP$,"T(``8I"`!#*[0$%BVS]6]-"``Z+[D=&J8P5"ZJ,`0V++?Q"
MTX(!`@NA>0&J;``!RF]\0)*,`0N*3?Q;T\,!!8N,`0_*K14"JHP!$(NM`0.+
M8_U"TZX!#`MK>`NJ(P`+RC!\0))C``**P?U;TRP"!8LA%`ZJ8P`0RB$`#(MC
M``^+3/Q"TR$``PN">0*JXT='J6P!`LIO?4"2C`$.BBW\6].,`0_*K14!JHP!
M$(M0?$"28P`%BZT!`XO#_4+3K0$,"V-X#JI"``/*+`(%BZO]6]-"``&*:Q4-
MJD(`$,IK`0R+0@`/BV\!`@LL_$+3XD=(J8%Y`:IK``'*;GQ`DFL!#8KL_5O3
M:P$.RHP5#ZIK`1"+,'Q`DD(`!8N,`0*+HOU"TXP!"PM">`VJ(0`"RBL"!8N#
M_5O3(0`/BF,4#*HA`!#*(0`.BV,`"XMM``$+X?U"T^Y+0/DA>`^J2P`!RJ/]
M6]-"?$"2:P$,BF,4#:IK`0+*S@$%BW(!$(O#02.+B_U"TW$`$@ML>0RJ[D]`
M^2L`#,HO_EO3(7Q`DN\5$:IK`0V*:P$!RL4!!8M^`0*+HD`OBT\`'@NE_4+3
M[E-`^:UX#:KP_5O38P`2BQ`6#ZJ2`0W*4@(#2L,!!(M2`@&+84`PBS``$@LC
M_D+3ZU=`^65X$:H._EO3HP$%RLX5$*I"`!Z+:P$$BV(``DI,0"R+8D$NBU$`
M#`OC_4+3ZUM`^6-X#ZHN_EO3KP`#RLX5$:HA`!*+X0$!2FL!!(LM0"V+84$N
MBRL`#0L/_D+3[E]`^?!Y$*IO_5O30@`,B^\5"ZIL`!#*@@$"2LX!!(M%0"6+
MPD$OBUX`!0LL_D+3[F-`^8QY$:K/_UO3(0`-B^\5'JH-`@S*H0$!2LT!!(LC
M0".+H4$OBS$``PMM_4+3[F=`^:MY"ZHO_EO30@`%B^\5$:J%`0O*H@`"2LX!
M!(M-0#"+PD$OBU(`#0O%_T+3[FM`^:5X'JI/_EO3(0`#B^\5$JIC`07*80`!
M2LX!!(LL0"R+P4$OBS``#`LC_D+3[F]`^6-X$:H/_EO30@`-B^\5$*JM``/*
MH@$"2LX!!(M+0"N+PD$OBU$`"PM-_D+3[G-`^:UY$JHO_EO3(0`,B^\5$:IL
M``W*@0$!2LX!!(LE0"6+P4$OBS(`!0L,_D+3[G=`^8QY$*I/_EO30@`+B^\5
M$JJK`0S*8@$"2LX!!(M#0".+PD$OBU```PLK_D+3[GM`^6MY$:H/_EO3(0`%
MB^\5$*J%`0O*H0`!2LX!!(LM0"V+P4$OBS$`#0M%_D+3[G]`^:5X$JHO_EO3
M0@`#B^\5$:IC`07*8@`"2LX!!(M,0"R+PD$OBTX`#`L#_D+3[X-`^6-X$*K0
M_5O3(0`-BQ`6#JJM``/*H0$!2NT!!(LK0"N+H4$PBS``"PLO_D+3[8=`^?%Y
M$:H/_EO30@`,B^\5$*IL`!'*@@$"2JP!!(M%0"6+@D$OBTP`!0O/_4+3[8M`
M^>YY#JJ/_5O3(0`+B^\5#*HK`@[*80$!2JL!!(LC0".+84$OBRL``PL/_D+3
M[8]`^?!Y$*K2`1#*;_U;TT(`!8OE%0NJ0@("2JT!!(M/0#&+@OU"TTQX#*JE
M026+L0`/"R$``XL#`@S*I0`/B^V30/EC``%*8?U"TR+^6],K>`NJ0A01JH\!
M"\JM`02+[P$%2J)!(HO^03"+<D`NB^V_4JE.`!(+(_Y"T\']6]-C>!&J(10.
MJJ4!!(M-`!*+H4`ABV(!`\HP`!X+0@`-2NT!!(M"0"R+!?Y;TR$`'HNE%!"J
MI0`-BZP``@O"_4+3[9]`^4)X#JIN``+*A?U;T\$!`4JM`02+(4`KBZ04#*H+
M_D+3A``-BW!Y$*J/``$+Y:-`^4L`$*H!`@(*:P$,BFL!`:KD_5O3@9N7TH04
M#ZIAX['RI0`!BV-!(XN$``6+A?U"TXT``PODIT#YK'@,J@L"#*J%`1`*:P$/
MBJ/]6]-K`06JA``!BV,4#:IB02*+8P`$B^7]0M/DJT#YKW@/JFX``@N"`0^J
MXP$,"D(`#8K+_5O30@`#JFL5#JJ#``&+14`PBVL!`XNQ_4+38@$%"^.O0/DM
M>@VJZP$-JJ0!#PIK`0Z*1?Q;TV,``8MK`02JI10"JF1!+(NE``.+R_U"T^.S
M0/EN>0ZJK``$"Z0!#JK%`0T*A``"BHO]6].$``6J:Q4,JF4``8M0_$+3@T`O
MBVL!!8L$>@*JY;=`^7`!`PO+`02J@@`."FL!#(H#_EO3I0`!BVL!`JIC%!"J
M8D$MBV,`!8N+_4+3Y;M`^6QY#*IO``(+@@`,JH,!!`I"`!"*Z_U;TT(``ZIK
M%0^JHP`!BPW^0M-%0"Z+:P$#BZ)Y$*KCOT#Y;0$%"XL!`JI.``P*:P$/BJ7]
M6]-K`0ZJ8P`!BZ44#:ID022+I0`#B^O]0M/CPT#Y;WD/JJX`!`M$``^JY0$"
M"H0`#8K+_5O3A``%JFL5#JIE``&+:P$%BX-`+(NP_4+3;`$#"^3'0/D->@VJ
MZP$-JJ4!#PIK`0Z*@_U;TVL!!:J$``&+8Q0,JL7]0M.N>`ZJ8D$BBV,`!(O%
M`0T*8P`""Z(!#JI"``R*D/U"TT(`!:IK_%O314`OBVL5`ZKD/UFI#'H,JH0`
M`8OO`0&+:P$$BX0!#@IB`04+RP$,JFL!`XI%_%O3:P$$JJ44`JID02V+I0`/
MBVO\0M/MTT#YKP`$"V5Y`ZI0_$+3A`$%JJ,`#`J$``**Z_U;TZT!`8N$``.J
M:Q4/JH-`+HMK`0V+!'H"JFX!`PNK``2J@@`%"FL!#XIK`0*JP_U;TV)!+(MC
M%`ZJ[;-:J>O]0M.M`0&+C`$!BV,`#8MM``(+8WD/JH(``ZIP``0*0@`.BJO]
M6]//_4+30@`0JFL5#:I%0"6+:P$,B^)Y#JIO`04+:P`"JDX``PIK`0V*:P$.
MJN7]6]-D022+I10/JNRO6ZF,`0&+:P$!BZ4`#(NL_4+3K@`$"XUY#:I%``VJ
ML`$""J4`#XK$_5O3I0`0JNS]0M.$%`ZJCWD/JJ-`(XN$``N+Y>=`^8P``PND
M`0^JZP$-"H0`#HJ#_5O3A``+JJ4``8N"0"*+8Q0,JL3]0M-C``6+A7@.JFL`
M`@OC`06JK@`/"F,`#(IB_5O38P`.JD(4"ZIR0"V+@_U"T^0W7:EL>`RJC@$%
M"H0``8NA`0&+@D`BBZ0`#*I0`!(+A``+BH0`#JH#_EO3D4`OBV,4$*HC0".+
M8?U"TV0`$0LK>`NJCP$+RD$`$HN._%O38P`1B\(5!*KN`0%*Q4$EB\$ZF-+M
M.UZI04RY\@_^0M/P>1"J<0$0RB,"`TJM`0&+S@$!BZ)!(HML0"R+30`%"T(`
M!8NO_5O3[Q4-JL-!+XN/_$+3?@`,"^1Y!*KN^T#YS_];TP4"!,KO%1ZJH@`"
M2L4!`8M+0"N+HD`OBU(`"PNE_4+3[O]`^:UX#:I/_EO3A0`-RN\5$JIC``R+
MS@$!BZ,``TIE0#"+PT$OBW$`!0O,_T+3[@-!^8QY'JHO_EO30@`+B^\5$:JK
M`0S*8@$"2LX!`8M$0"2+PD$OBU``!`M+_D+3[@=!^6MY$JH/_EO38P`%B^\5
M$*J%`0O*HP`#2LX!`8MM0"V+PT$OBW(`#0LE_D+3[@M!^:5X$:I/_EO30@`$
MB^\5$JID`07*@@`"2LX!`8M,0"R+PD$OBU$`#`L$_D+3[@]!^81X$*HO_EO3
M8P`-B^\5$:JM``3*HP$#2LX!`8MK0"N+PT$OBW``"PM-_D+3[A-!^:UY$JH/
M_EO30@`,B^\5$*J,``W*@@$"2LX!`8M%0"6+PD$OBU(`!0LL_D+3[A=!^8QY
M$:I/_EO38P`+B^\5$JJK`0S*8P$#2LX!`8MD0"2+PT$OBW$`!`L+_D+3[AM!
M^6MY$*HO_EO30@`%B^\5$:J%`0O*H@`"2LX!`8M-0"V+PD$OBU``#0M%_D+3
M[A]!^:5X$JH/_EO38P`$B^\5$*ID`07*@P`#2LX!`8ML0"R+PT$OBW(`#`LD
M_D+3[B-!^81X$:I/_EO30@`-B^\5$JJM``3*H@$"2LT!`8M+0"N+HD$OBU$`
M"PL-_D+3[B=!^;!Y$*HO_EO38P`,B^\5$:J,`!#*@P$#2LP!`8ME0"6+@T$O
MBWX`!0M-_D+3["M!^:UY$JK._UO30@`+B\X5'JH+`@W*8@$"2HL!`8M$0"2+
M8D$NBTP`!`LK_D+3[R]!^7%Y$:J._5O38P`%B\X5#*JE`1'*I0`#2N,!`8NR
M0#"+;D`NB\4!$@O+_T+3XS-!^6MY'JJO_%O3,`(+RD(`!(OD%06J`@("2F,`
M`8M>0"V+8D`DBT,`'@N$_4+3[S=!^8QX#*ID_%O3<`$,RLX!$HN$%`.J$`(.
M2NX!`8L10C&+Q$$DB^\[0?F.`!$+K?Q"TX0`$8NM>06J10`>BY`!#<K"_5O3
M$`(%2D(4#JKE`0&+"T(KBT(`!8ME_$+3[S]!^4(`"PNC>`.JS$`LBZ4!`\I+
M_%O39A4"JJ0`!$KA`0&+Q?U"TX0``8O&``R+H7@.JJ4!"@O*``#02B5'^4(`
M"0LA``<+8P`("X0`!@L$"`"I`0P!J040`/GB1T'Y00%`^4(``>L!`(#2@0``
M5/]#"I']>\&HP`-?UL```-``F$?Y```_UA\@`]4?(`/5P0``T"&D1_G]>[^I
M`!"`TOT#`)$@`#_6'Q@`^2(``)`A``"0'W@`N5THPCTB``"0/B3"/5\LPCT>
M=`"M'PB`/?U[P:C``U_6_7N[J4)\0)/]`P"1\U,!J?0#`*KU6P*I]V,#J?<#
M`JKX`P&J^6L$J06,0JD`>$"YI`P""[\`!.MCE(.:8G1"BX2*`JG`!``UV0``
MT#DO1_F5X@"1__X`<<T"`%3V?@93\P,8JMH&`%'69AI3%@,6BQ\@`]4?(`/5
M?7)`K>`#%*I_>D&M<P(!D;UR`*V_>@&MDOO_EW\"%NL!__]4&`,!D0`(@%+W
M`@%1]QH:2UACH)OA`QBJX`,5JN)^0),@`S_6EWH`N?-30:GU6T*I]V-#J?EK
M1*G]>\6HP`-?UA,(@%)S`@!+?P(7:Y7B`)'9``#0.2]'^7/2EQJ@PB"+=GY`
MD^(#%JH@`S_6@'I`N6`"``N`>@"Y'P`!<6']_U3W`A-+&`,6B^`#%*IL^_^7
MRO__%Q\@`]4?(`/5_7N\J>,/@!+]`P"1\U,!J?0#`*KS`P&J]5L"J3;@`)'O
M&P#]-8!"J:(B`U,/`&>>000`$2!\0)/#RB(XP`(`BS_@`''-"P!4PP``T&/\
M1_D""(!20@`!2P$`@%)"?$"38``_UN`#$ZI.^_^7'P0`3W^"@SS?_@"M?S8`
M^?H%<'__!6A_^P5X?[X"9Y[@`Q.JW`=H?]T'<']?!P-NW@=X?W\'!6[_!0=N
MGP<);K\'"V[?!PUNOQX/3G\Z`/TY^_^780)`^>`#$ZK0``#0$-Y'^2'\6-.!
M`@`Y80)`^2'\4-.!!@`Y80)`^2'\2-.!"@`Y80)`^8$.`#EA!D#Y(?Q8TX$2
M`#EA!D#Y(?Q0TX$6`#EA!D#Y(?Q(TX$:`#EA!D#Y@1X`.6$*0/DA_%C3@2(`
M.6$*0/DA_%#3@28`.6$*0/DA_$C3@2H`.6$*0/F!+@`Y80Y`^2'\6-.!,@`Y
M80Y`^2'\4-.!-@`Y80Y`^2'\2-.!.@`Y80Y`^8$^`#EA$D#Y(?Q8TX%"`#EA
M$D#Y(?Q0TX%&`#EA$D#Y(?Q(TX%*`#EA$D#Y@4X`.>\;0/WS4T&I]5M"J?U[
MQ*@``A_6PP``T&/\1_D"!X!20@`!2P$`@%)"?$"38``_UJG__Q<?(`/5PP``
M\&,$0/E&\?\7'R`#U?U[OZG]`P"1_7O!J,`#7]8!``(``````"5S+R5S````
M+W!R;V,O<V5L9B]E>&4``$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@``
M````26YT97)N86QS.CI005(Z.D)/3U0`````97AE```````O<')O8R\E:2\E
M<P``````+@````````!005)?5$5-4```````````4$%27U!23T=.04U%````
M`#H`````````+P`````````E<R5S)7,``%!%4DPU3$E"``````````!015),
M3$E"`%!%4DPU3U!4``````````!015),24\``%!!4E])3DE424%,25I%1`!0
M05)?4U!!5TY%1```````4$%27T-,14%.`````````%!!4E]$14)51P``````
M``!005)?0T%#2$4`````````4$%27T=,3T)!3%]$14)51P``````````4$%2
M7T=,3T)!3%]435!$25(`````````4$%27U1-4$1)4@```````%!!4E]'3$]"
M04Q?5$5-4`!005)?1TQ/0D%,7T-,14%.```````````Q`````````&QS965K
M(&9A:6QE9`````!R96%D(&9A:6QE9```````3$1?3$E"4D%265]0051(`%-9
M4U1%30``)3`R>`````!P87(M`````"5S)7,E<R5S```````````E<SH@8W)E
M871I;VX@;V8@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N
M;STE:2D*````````)7,Z('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&ES('5N
M<V%F92`H<&QE87-E(')E;6]V92!I="!A;F0@<F5T<GD@>6]U<B!O<&5R871I
M;VXI"@```````%!!5$@`````<&%R;``````N<&%R`````'-H;W)T(')E860`
M```````E<R5S8V%C:&4M)7,E<P``)7,E<W1E;7`M)74E<P```"5S)7-T96UP
M+25U+25U)7,``````````!@`````````<&5R;``````P`````````%!!4CHZ
M4&%C:V5R.CI615)324].`````#$N,#8S````56YA8FQE('1O(&-R96%T92!C
M86-H92!D:7)E8W1O<GD``````````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E
M(&-A8VAE('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H`````
M`````"UE````````+2T```````!54T52`````%5315).04U%``````````!4
M35!$25(``%1%35!$25(`5$5-4`````!435```````"]T;7``````979A;"`G
M97AE8R`O=7-R+V)I;B]P97)L("`M4R`D,"`D>S$K(B1`(GTG"FEF(#`[(",@
M;F]T(')U;FYI;F<@=6YD97(@<V]M92!S:&5L;`IP86-K86=E(%]?<&%R7W!L
M.PIM>2`H)%!!4E]-04=)0RP@)$9)3$5?;V9F<V5T7W-I>F4L("1C86-H95]N
M86UE7W-I>F4I.PI"14=)3B!["B1005)?34%'24,@/2`B7&Y005(N<&U<;B([
M"B1&24Q%7V]F9G-E=%]S:7IE(#T@-#L@("`C('!A8VLH(DXB*0HD8V%C:&5?
M;F%M95]S:7IE(#T@-#`["B102UI)4%]-04=)0R`](")02UPP,#-<,#`T(CL*
M?0IS=6(@9FEN9%]P87)?;6%G:6,*>PIM>2`H)&9H*2`]($!?.PIM>2`D8VAU
M;FM?<VEZ92`](#8T("H@,3`R-#L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M<R`D
M9F@["FUY("1P;W,@/2`H)'-I>F4M,2D@+2`H)'-I>F4M,2D@)2`D8VAU;FM?
M<VEZ93L@("`@("`C($Y/5$4Z("1P;W,@:7,@82!M=6QT:7!L92!O9B`D8VAU
M;FM?<VEZ90IW:&EL92`H)'!O<R`^/2`P*2!["G-E96L@)&9H+"`D<&]S+"`P
M.PIR96%D("1F:"P@)&)U9BP@)&-H=6YK7W-I>F4@*R!L96YG=&@H)%!!4E]-
M04=)0RD["FEF("@H;7D@)&D@/2!R:6YD97@H)&)U9BP@)%!!4E]-04=)0RDI
M(#X](#`I('L*<F5T=7)N("1P;W,@*R`D:3L*?0HD<&]S("T]("1C:'5N:U]S
M:7IE.PI]"G)E='5R;B`M,3L*?0IM>2`H)'!A<E]T96UP+"`D<')O9VYA;64L
M($!T;7!F:6QE+"`E36]D=6QE0V%C:&4I.PI%3D0@>R!I9B`H)$5.5GM005)?
M0TQ%04Y]*2!["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI"
M87-E;F%M93L*<F5Q=6ER92!&:6QE.CI3<&5C.PIM>2`D=&]P9&ER(#T@1FEL
M93HZ0F%S96YA;64Z.F1I<FYA;64H)'!A<E]T96UP*3L*;W5T<RAQ<5M296UO
M=FEN9R!F:6QE<R!I;B`B)'!A<E]T96UP(ETI.PI&:6QE.CI&:6YD.CIF:6YD
M9&5P=&@H<W5B('L@*"`M9"`I(#\@<FUD:7(@.B!U;FQI;FL@?2P@)'!A<E]T
M96UP*3L*<FUD:7(@)'!A<E]T96UP.PII9B`H+60@)'!A<E]T96UP("8F("1>
M3R!N92`G35-7:6XS,B<I('L*;7D@)'1M<"`](&YE=R!&:6QE.CI496UP*`I4
M14U03$%412`]/B`G=&UP6%A86%@G+`I$25(@/3X@1FEL93HZ0F%S96YA;64Z
M.F1I<FYA;64H)'1O<&1I<BDL"E-51D9)6"`]/B`G+F-M9"<L"E5.3$E.2R`]
M/B`P+`HI.PIM>2`D9FEL96YA;64@/2`D=&UP+3YF:6QE;F%M93L*<')I;G0@
M)'1M<"`\/"(N+BXB.PIX/3$[('=H:6QE(%L@7"1X("UL="`Q,"!=.R!D;PIR
M;2`M<F8@)R1P87)?=&5M<"<*:68@6R!<(2`M9"`G)'!A<E]T96UP)R!=.R!T
M:&5N"F)R96%K"F9I"G-L965P(#$*>#U@97AP<B!<)'@@*R`Q8`ID;VYE"G)M
M("<D9FEL96YA;64G"BXN+@IC;&]S92`D=&UP.PIC:&UO9"`P-S`P+"`D9FEL
M96YA;64["FUY("1C;60@/2`B)&9I;&5N86UE(#XO9&5V+VYU;&P@,CXF,2`F
M(CL*<WES=&5M*"1C;60I.PIO=71S*'%Q6U-P87=N960@8F%C:V=R;W5N9"!P
M<F]C97-S('1O('!E<F9O<FT@8VQE86YU<#H@)&9I;&5N86UE72D["GT*?2!]
M"D)%1TE.('L*26YT97)N86QS.CI005(Z.D)/3U0H*2!I9B!D969I;F5D("9)
M;G1E<FYA;',Z.E!!4CHZ0D]/5#L*979A;"!["E]P87)?:6YI=%]E;G8H*3L*
M;7D@)'%U:65T(#T@(21%3E9[4$%27T1%0E5'?3L*;7D@)7-Y<R`]("@*<&%T
M:%]S97`@("`@/3X@*"1>3R`]?B`O7DU35VEN+R`_("<[)R`Z("<Z)RDL"E]E
M>&4@("`@("`@(#T^("@D7D\@/7X@+UXH/SI-4U=I;GQ/4S)\8WEG=VEN*2\@
M/R`G+F5X92<@.B`G)RDL"E]D96QI;2`@("`@(#T^("@D7D\@/7X@+UY-4U=I
M;GQ/4S(O(#\@)UQ<)R`Z("<O)RDL"BD["E]S971?<')O9VYA;64H*3L*;W5T
M<RAQ<5M<)'!R;V=N86UE(#T@(B1P<F]G;F%M92)=*3L*7W-E=%]P87)?=&5M
M<"@I.PIO=71S*'%Q6UPD14Y6>U!!4E]414U0?2`]("(D14Y6>U!!4E]414U0
M?2)=*3L*;7D@*"1S=&%R=%]P;W,L("1S=&%R=%]O9E]&24Q%7W-E8W1I;VXI
M.PI-04=)0SH@>PIL;V-A;"`D4TE'>U]?5T%23E]??2`]('-U8B![?3L*=6YL
M97-S("AO<&5N(%]&2"P@)SPZ<F%W)RP@)'!R;V=N86UE*2!["F]U=',H<7%;
M0V%N)W0@<F5A9"!F<F]M(&9I;&4@(B1P<F]G;F%M92)=*3L@(",@9&]N)W0@
M=7-E("0A(&AE<F4@87,@:70@<F5Q=6ER97,@17)R;F\N<&T*;&%S="!-04=)
M0SL*?0IM>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H*E]&2"D["FEF
M("@D;6%G:6-?<&]S(#P@,"D@>PIO=71S*'%Q6T-A;B=T(&9I;F0@;6%G:6,@
M<W1R:6YG("(D4$%27TU!1TE#(B!I;B!F:6QE("(D<')O9VYA;64B72D["FQA
M<W0@34%'24,["GT*;W5T<R@B1F]U;F0@4$%2(&UA9VEC(&%T('!O<VET:6]N
M("1M86=I8U]P;W,B*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O
M9F9S971?<VEZ92`M(&QE;F=T:"@B7#!#04-(12(I+"`P.PIR96%D(%]&2"P@
M)&)U9BP@;&5N9W1H*")<,$-!0TA%(BD["FEF("@D8G5F(&YE(")<,$-!0TA%
M(B`F)B`D8G5F(&YE(")<,$-,14%.(BD@>PIO=71S*").;R!C86-H92!M87)K
M97(@9F]U;F0B*3L*;&%S="!-04=)0SL*?0IO=71S*'%Q6T-A8VAE(&UA<FME
M<B`B)&)U9B(@9F]U;F1=*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,
M15]O9F9S971?<VEZ92P@,#L*<F5A9"!?1D@L("1B=68L("1&24Q%7V]F9G-E
M=%]S:7IE.PIM>2`D;V9F<V5T(#T@=6YP86-K*").(BP@)&)U9BD["F]U=',H
M(D]F9G-E="!F<F]M('-T87)T(&]F($9)3$5S(&ES("1O9F9S970B*3L*<V5E
M:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ92`M("1O9F9S
M970L(#`["B1S=&%R=%]O9E]&24Q%7W-E8W1I;VX@/2!T96QL(%]&2#L*;7D@
M)7)E<75I<F5?;&ES=#L*<F5A9"!?1D@L("1B=68L(#0[("`@("`@("`@("`@
M("`@("`@("`@("`@("`@(R!R96%D('1H92!F:7)S="`B1DE,12(*=VAI;&4@
M*"1B=68@97$@(D9)3$4B*2!["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&
M2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FUY("1F=6QL;F%M92`]("1B
M=68["F]U=',H<7%;56YP86-K:6YG($9)3$4@(B1F=6QL;F%M92(N+BY=*3L*
M;7D@)&-R8R`]("@@)&9U;&QN86UE(#U^('-\7BA;82UF7&1=>SA]*2]\?"`I
M(#\@)#$@.B!U;F1E9CL*;7D@*"1B87-E;F%M92P@)&5X="D@/2`H)&)U9B`]
M?B!M?"@_.BXJ+RD_*"XJ*2A<+BXJ*7PI.PIR96%D(%]&2"P@)&)U9BP@-#L*
M<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PII9B`H9&5F:6YE
M9"@D97AT*2!A;F0@)&5X="`A?B`O7"XH/SIP;7QP;'QI>'QA;"DD+VDI('L*
M;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1C<F,D97AT(BP@)&)U9BP@,#<U
M-2D["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9U;&QN86UE?2`]("1F:6QE
M;F%M93L*)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9FEL96YA;65](#T@)&9U
M;&QN86UE.PI]"F5L<V4@>PHD36]D=6QE0V%C:&5[)&9U;&QN86UE?2`]('L*
M8G5F(#T^("1B=68L"F-R8R`]/B`D8W)C+`IN86UE(#T^("1F=6QL;F%M92P*
M?3L*)')E<75I<F5?;&ES='LD9G5L;&YA;65]*RL["GT*<F5A9"!?1D@L("1B
M=68L(#0["GT*;&]C86P@0$E.0R`]("AS=6(@>PIM>2`H)'-E;&8L("1M;V1U
M;&4I(#T@0%\["G)E='5R;B!I9B!R968@)&UO9'5L92!O<B`A)&UO9'5L93L*
M;7D@)&EN9F\@/2`D36]D=6QE0V%C:&5[)&UO9'5L97T["G)E='5R;B!U;FQE
M<W,@)&EN9F\["F1E;&5T92`D<F5Q=6ER95]L:7-T>R1M;V1U;&5].PHD24Y#
M>R1M;V1U;&5](#T@(B]L;V%D97(O)&EN9F\O)&UO9'5L92(["FEF("@D14Y6
M>U!!4E]#3$5!3GT@86YD(&1E9FEN960H)DE/.CI&:6QE.CIN97<I*2!["FUY
M("1F:"`]($E/.CI&:6QE+3YN97=?=&UP9FEL92!O<B!D:64@(D-A;B=T(&-R
M96%T92!T96UP(&9I;&4Z("0A(CL*)&9H+3YB:6YM;V1E*"D["B1F:"T^<')I
M;G0H)&EN9F\M/GMB=69]*3L*)&9H+3YS965K*#`L(#`I.PIR971U<FX@)&9H
M.PI]"F5L<V4@>PIM>2`D9FEL96YA;64@/2!?<V%V95]A<R@B)&EN9F\M/GMC
M<F-]+G!M(BP@)&EN9F\M/GMB=69]*3L*)&EN9F\M/GMF:6QE?2`]("1F:6QE
M;F%M93L*;W!E;B!M>2`D9F@L("<\.G)A=R<L("1F:6QE;F%M92!O<B!D:64@
M<7%;0V%N)W0@<F5A9"`B)&9I;&5N86UE(CH@)"%=.PIR971U<FX@)&9H.PI]
M"F1I92`B0F]O='-T<F%P<&EN9R!F86EL960Z(&-A;B=T(&9I;F0@;6]D=6QE
M("1M;V1U;&4A(CL*?2P@0$E.0RD["G)E<75I<F4@6%-,;V%D97(["G)E<75I
M<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%
M>'!O<G1E<CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H
M*3L*<F5Q=6ER92!)3SHZ1FEL93L*=VAI;&4@*&UY("1F:6QE;F%M92`]("AS
M;W)T(&ME>7,@)7)E<75I<F5?;&ES="E;,%TI('L*=6YL97-S("@D24Y#>R1F
M:6QE;F%M97T@;W(@)&9I;&5N86UE(#U^("]"4T1004XO*2!["FEF("@D9FEL
M96YA;64@/7X@+UPN<&UC/R0O:2D@>PIR97%U:7)E("1F:6QE;F%M93L*?0IE
M;'-E('L*9&\@)&9I;&5N86UE('5N;&5S<R`D9FEL96YA;64@/7X@+W-I=&5C
M=7-T;VUI>F5<+G!L)"\["GT*?0ID96QE=&4@)')E<75I<F5?;&ES='LD9FEL
M96YA;65].PI]"FEF("@D8G5F(&YE("102UI)4%]-04=)0RD@>PIO=71S*'%Q
M6TYO('II<"!F;W5N9"!A9G1E<B!&24Q%('-E8W1I;VX@:6X@9FEL92`B)'!R
M;V=N86UE(ETI.PIL87-T($U!1TE#(#L*?0HD<W1A<G1?<&]S(#T@*'1E;&P@
M7T9(*2`M(#0[("`@("`@("`@("`@("`@(",@<W1A<G0@;V8@>FEP"GT*;7D@
M0'!A<E]A<F=S.PIM>2`H)&]U="P@)&)U;F1L92P@)&QO9V9H+"`D8V%C:&5?
M;F%M92D["F1E;&5T92`D14Y6>U!!4E]!4%!?4D554T5].R`C('-A;FET:7IE
M("A2155312!M87D@8F4@82!S96-U<FET>2!P<F]B;&5M*0II9B`H(21S=&%R
M=%]P;W,@;W(@*"1!4D=66S!=(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF
M="DI('L*;7D@)61I<W1?8VUD(#T@<7<H"G`@("!B;&EB7W1O7W!A<@II("`@
M:6YS=&%L;%]P87(*=2`@('5N:6YS=&%L;%]P87(*<R`@('-I9VY?<&%R"G8@
M("!V97)I9GE?<&%R"BD["FEF("A`05)'5B!A;F0@)$%21U9;,%T@97$@)RTM
M<F5U<V4G*2!["G-H:69T($!!4D=6.PHD14Y6>U!!4E]!4%!?4D554T5](#T@
M<VAI9G0@0$%21U8["GT*96QS92![(",@;F]R;6%L('!A<FP@8F5H879I;W5R
M"FUY($!A9&1?=&]?:6YC.PIW:&EL92`H0$%21U8I('L*)$%21U9;,%T@/7X@
M+UXM*%M!24U/0DQB<7!I=7-4=ETI*"XJ*2\@;W(@;&%S=#L*:68@*"0Q(&5Q
M("=))RD@>PIP=7-H($!A9&1?=&]?:6YC+"`D,CL*?0IE;'-I9B`H)#$@97$@
M)TTG*2!["F5V86P@(G5S92`D,B(["GT*96QS:68@*"0Q(&5Q("=!)RD@>PIU
M;G-H:69T($!P87)?87)G<RP@)#(["GT*96QS:68@*"0Q(&5Q("=/)RD@>PHD
M;W5T(#T@)#(["GT*96QS:68@*"0Q(&5Q("=B)RD@>PHD8G5N9&QE(#T@)W-I
M=&4G.PI]"F5L<VEF("@D,2!E<2`G0B<I('L*)&)U;F1L92`]("=A;&PG.PI]
M"F5L<VEF("@D,2!E<2`G<2<I('L*)'%U:65T(#T@,3L*?0IE;'-I9B`H)#$@
M97$@)TPG*2!["F]P96X@)&QO9V9H+"`B/CXB+"`D,B!O<B!D:64@<7%;0V%N
M)W0@;W!E;B!L;V<@9FEL92`B)#(B.B`D(5T["GT*96QS:68@*"0Q(&5Q("=4
M)RD@>PHD8V%C:&5?;F%M92`]("0R.PI]"G-H:69T*$!!4D=6*3L*:68@*&UY
M("1C;60@/2`D9&ES=%]C;61[)#%]*2!["F1E;&5T92`D14Y6>R=005)?5$5-
M4"=].PII;FET7VEN8R@I.PIR97%U:7)E(%!!4CHZ1&ES=#L*)GLB4$%2.CI$
M:7-T.CHD8VUD(GTH*2!U;FQE<W,@0$%21U8["B9[(E!!4CHZ1&ES=#HZ)&-M
M9")]*"1?*2!F;W(@0$%21U8["F5X:70["GT*?0IU;G-H:69T($!)3D,L($!A
M9&1?=&]?:6YC.PI]"GT*:68@*"1O=70I('L*>PIR97%U:7)E($E/.CI&:6QE
M.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*
M?0IM>2`D<&%R(#T@<VAI9G0H0$%21U8I.PIM>2`D>FEP.PII9B`H9&5F:6YE
M9"`D<&%R*2!["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q
M<5M#86XG="!F:6YD('!A<B!F:6QE("(D<&%R(CH@)"%=.PIB;&5S<R@D9F@L
M("=)3SHZ1FEL92<I.PHD>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II
M<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ
M6FEP.CI!6E]/2R@I("D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R
M8VAI=F4@(B1P87(B73L*?0IM>2`E;65T85]P87(@/2!D;R!["FEF("@D>FEP
M(&%N9"!M>2`D;65T82`]("1Z:7`M/F-O;G1E;G1S*"=-151!+GEM;"<I*2![
M"B1M971A(#U^(',O+BI><&%R.B0O+VUS.PHD;65T82`]?B!S+UY<4RXJ+R]M
M<SL*)&UE=&$@/7X@+UX@("A;7CI=*RDZ("@N*RDD+VUG.PI]"GT["FEF("AD
M969I;F5D("1P87(I('L*;W!E;B!M>2`D<&@L("<\.G)A=R<L("1P87(@;W(@
M9&EE('%Q6T-A;B=T(')E860@<&%R(&9I;&4@(B1P87(B.B`D(5T["FUY("1B
M=68["G)E860@)'!H+"`D8G5F+"`T.PID:64@<7%;(B1P87(B(&ES(&YO="!A
M('!A<B!F:6QE72!U;FQE<W,@)&)U9B!E<2`D4$M:25!?34%'24,["F-L;W-E
M("1P:#L*?0I#<F5A=&50871H*"1O=70I(#L*;7D@)&9H(#T@24\Z.D9I;&4M
M/FYE=R@*)&]U="P*24\Z.D9I;&4Z.D]?0U)%050H*2!\($E/.CI&:6QE.CI/
M7U)$5U(H*2!\($E/.CI&:6QE.CI/7U1254Y#*"DL"C`W-S<L"BD@;W(@9&EE
M('%Q6T-A;B=T(&-R96%T92!F:6QE("(D;W5T(CH@)"%=.PHD9F@M/F)I;FUO
M9&4H*3L*<V5E:R!?1D@L(#`L(#`["FUY("1L;V%D97(["FEF("AD969I;F5D
M("1S=&%R=%]O9E]&24Q%7W-E8W1I;VXI('L*<F5A9"!?1D@L("1L;V%D97(L
M("1S=&%R=%]O9E]&24Q%7W-E8W1I;VX["GT@96QS92!["FQO8V%L("0O(#T@
M=6YD968["B1L;V%D97(@/2`\7T9(/CL*?0II9B`H(21%3E9[4$%27U9%4D)!
M5$E-?2!A;F0@)&QO861E<B`]?B`O7B@_.B,A?%Q`<F5M*2\I('L*<F5Q=6ER
M92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["E!!4CHZ1FEL=&5R.CI0;V13=')I
M<"T^87!P;'DH7"1L;V%D97(L("0P*3L*?0HD9F@M/G!R:6YT*"1L;V%D97(I
M"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG(&QO861E<B!T;R`B)&]U="(Z("0A
M73L*:68@*"1B=6YD;&4I('L*<F5Q=6ER92!005(Z.DAE879Y.PI005(Z.DAE
M879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["FEN:71?:6YC*"D["G)E<75I<F5?
M;6]D=6QE<R@I.PIM>2!`:6YC(#T@9W)E<"![("$O0E-$4$%.+R!]"F=R97`@
M>PHH)&)U;F1L92!N92`G<VET92<I(&]R"B@D7R!N92`D0V]N9FEG.CI#;VYF
M:6=[87)C:&QI8F5X<'T@86YD"B1?(&YE("1#;VYF:6<Z.D-O;F9I9WMP<FEV
M;&EB97AP?2D["GT@0$E.0SL*<WPO*WPO?&<L('-\+R1\?"!F;W)E86-H($!I
M;F,["FUY("5F:6QE<SL*)&9I;&5S>R1??2LK(&9O<B!`1'EN84QO861E<CHZ
M9&Q?<VAA<F5D7V]B:F5C=',L('9A;'5E<R`E24Y#.PIM>2`D;&EB7V5X="`]
M("1#;VYF:6<Z.D-O;F9I9WML:6)?97AT?3L@("`@("`@("`C(%A86"!L:6)?
M97AT('9S(&1L97AT(#\*;7D@)7=R:71T96X["F9O<F5A8V@@;7D@)&ME>2`H
M<V]R="!K97ES("5F:6QE<RD@>PIM>2`H)&9I;&4L("1N86UE*3L*:68@*&1E
M9FEN960H;7D@)&9C(#T@)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD:V5Y?2DI
M('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)&9C*3L*?0IE;'-E('L*9F]R
M96%C:"!M>2`D9&ER("A`:6YC*2!["FEF("@D:V5Y(#U^(&U\7EQ1)&1I<EQ%
M+R@N*BDD?&DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)#$I.PIL87-T
M.PI]"FEF("@D:V5Y(#U^(&U\7B]L;V%D97(O6UXO72LO*"XJ*21\*2!["FEF
M("AM>2`D<F5F(#T@)$UO9'5L94-A8VAE>R0Q?2D@>PHH)&9I;&4L("1N86UE
M*2`]("@D<F5F+"`D,2D["FQA<W0["GT*:68@*"UF("(D9&ER+R0Q(BD@>PHH
M)&9I;&4L("1N86UE*2`]("@B)&1I<B\D,2(L("0Q*3L*;&%S=#L*?0I]"GT*
M?0IN97AT('5N;&5S<R!D969I;F5D("1N86UE.PIN97AT(&EF("1W<FET=&5N
M>R1N86UE?2LK.PIN97AT(&EF("%R968H)&9I;&4I(&%N9"`D9FEL92`]?B`O
M7"Y<421L:6)?97AT7$4D+VD["F]U=',H<W!R:6YT9BAQ<5M086-K:6YG($9)
M3$4@(B5S(BXN+ETL(')E9B`D9FEL92`_("1F:6QE+3Y[;F%M97T@.B`D9FEL
M92DI.PIM>2`D8V]N=&5N=#L*:68@*')E9B@D9FEL92DI('L*)&-O;G1E;G0@
M/2`D9FEL92T^>V)U9GT["GT*96QS92!["FQO8V%L("0O(#T@=6YD968["F]P
M96X@;7D@)'1H+"`G/#IR87<G+"`D9FEL92!O<B!D:64@<7%;0V%N)W0@<F5A
M9"`B)&9I;&4B.B`D(5T["B1C;VYT96YT(#T@/"1T:#X["F-L;W-E("1T:#L*
M4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&-O;G1E;G0L("(\96UB
M961D960^+R1N86UE(BD*:68@(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&YA
M;64@/7X@+UPN*#\Z<&U\:7A\86PI)"]I.PI005(Z.D9I;'1E<CHZ4&%T8VA#
M;VYT96YT+3YN97<M/F%P<&QY*%PD8V]N=&5N="P@)&9I;&4L("1N86UE*3L*
M?0HD9F@M/G!R:6YT*")&24Q%(BP*<&%C:R@G3B<L(&QE;F=T:"@D;F%M92D@
M*R`Y*2P*<W!R:6YT9B@B)3`X>"\E<R(L($%R8VAI=F4Z.EII<#HZ8V]M<'5T
M94-20S,R*"1C;VYT96YT*2P@)&YA;64I+`IP86-K*"=.)RP@;&5N9W1H*"1C
M;VYT96YT*2DL"B1C;VYT96YT*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!E
M;6)E9&1E9"!&24Q%('1O("(D;W5T(CH@)"%=.PIO=71S*'%Q6U=R:71T96X@
M87,@(B1N86UE(ETI.PI]"GT*:68@*"1Z:7`I('L*)'II<"T^=W)I=&54;T9I
M;&5(86YD;&4H)&9H*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+"F]R(&1I92!Q
M<5M%<G)O<B!W<FET:6YG('II<"!P87)T(&]F("(D;W5T(ET["GT*:68@*"1M
M971A7W!A<GMC;&5A;GTI('L*)&9H+3YP<FEN="@B7#!#3$5!3B(I.PI]"F5L
M<V4@>PII9B`H(61E9FEN960@)&-A8VAE7VYA;64I('L*;7D@)&-T>"`]($1I
M9V5S=#HZ4TA!+3YN97<H,2D["G-E96L@)&9H+"`P+"`P.PHD8W1X+3YA9&1F
M:6QE*"1F:"D["G-E96L@)&9H+"`P+"`R.PHD8V%C:&5?;F%M92`]("1C='@M
M/FAE>&1I9V5S=#L*?0HD8V%C:&5?;F%M92`]('-U8G-T<B`D8V%C:&5?;F%M
M92P@,"P@)&-A8VAE7VYA;65?<VEZ93L*)&-A8VAE7VYA;64@+CT@(EPP(B!X
M("@D8V%C:&5?;F%M95]S:7IE("T@;&5N9W1H("1C86-H95]N86UE*3L*)&9H
M+3YP<FEN="@D8V%C:&5?;F%M92P*(EPP0T%#2$4B*3L*?0IM>2`D;V9F<V5T
M(#T@)&9H+3YT96QL("T@;&5N9W1H*"1L;V%D97(I.PHD9F@M/G!R:6YT*'!A
M8VLH)TXG+"`D;V9F<V5T*2P*)%!!4E]-04=)0RD["B1F:"T^8VQO<V4@;W(@
M9&EE('%Q6T5R<F]R('=R:71I;F<@=')A:6QE<B!O9B`B)&]U="(Z("0A73L*
M8VAM;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I["FQA<W0@=6YL97-S(&1E9FEN
M960@)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M92@I.PIR97%U:7)E(%!!4CL*
M4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PI["G)E<75I<F4@1FEL
M93HZ1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI::7`["GT*;7D@)&9H(#T@24\Z
M.D9I;&4M/FYE=SL@("`@("`@("`@("`@("`@("`@("`@("`@("`@(",@07)C
M:&EV93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/.CI(86YD;&4*)&9H+3YF9&]P
M96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE('%Q6V9D;W!E;B@I(&9A:6QE
M9#H@)"%=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@M<R!?1D@I.PIM
M>2`D>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@D>FEP+3YR96%D1G)O;49I
M;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]($%R8VAI=F4Z.EII<#HZ05I?
M3TLH*2D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P
M<F]G;F%M92)=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@V-"`J(#$P
M,C0I.PIP=7-H($!005(Z.DQI8D-A8VAE+"`D>FEP.PHD4$%2.CI,:6)#86-H
M97LD<')O9VYA;65](#T@)'II<#L*;W5T<R@B17AT<F%C=&EN9R!Z:7`N+BXB
M*3L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'TI('L@(R!S:&]U;&0@8F4@
M<V5T(&%T('1H:7,@<&]I;G0A"F9O<F5A8V@@;7D@)&UE;6)E<B`H("1Z:7`M
M/FUE;6)E<G,@*2!["FYE>'0@:68@)&UE;6)E<BT^:7-$:7)E8W1O<GD["FUY
M("1M96UB97)?;F%M92`]("1M96UB97(M/F9I;&5.86UE.PIN97AT('5N;&5S
M<R`D;65M8F5R7VYA;64@/7X@;7L*7@HO/W-H;&EB+PHH/SHD0V]N9FEG.CI#
M;VYF:6=[87)C:&YA;65]+RD_"BA;7B]=*RD*)`I]>#L*;7D@)&5X=')A8W1?
M;F%M92`]("0Q.PIM>2`D9&5S=%]N86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL
M92@D14Y6>U!!4E]414U0?2P@)&5X=')A8W1?;F%M92D["FEF("@M9B`D9&5S
M=%]N86UE("8F("US(%\@/3T@)&UE;6)E<BT^=6YC;VUP<F5S<V5D4VEZ92@I
M*2!["F]U=',H<7%;4VMI<'!I;F<@(B1M96UB97)?;F%M92(@<VEN8V4@:70@
M86QR96%D>2!E>&ES=',@870@(B1D97-T7VYA;64B72D["GT@96QS92!["F]U
M=',H<7%;17AT<F%C=&EN9R`B)&UE;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M
M92(N+BY=*3L**"1M96UB97(M/F5X=')A8W14;T9I;&5.86UE9"@D9&5S=%]N
M86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O
M<B!E>'1R86-T:6YG('II<"!M96UB97(@=&\@(B1D97-T7VYA;64B73L*8VAM
M;V0H,#4U-2P@)&1E<W1?;F%M92D@:68@)%Y/(&5Q(")H<'5X(CL*?0I]"GT*
M?0IU;FQE<W,@*"1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97TI('L*9&EE(#P\
M("(N(B!U;FQE<W,@0$%21U8["E5S86=E.B`D,"!;("U!;&EB+G!A<B!=(%L@
M+4ED:7(@72!;("U-;6]D=6QE(%T@6R!S<F,N<&%R(%T@6R!P<F]G<F%M+G!L
M(%T*)#`@6R`M0GPM8B!=(%LM3V]U=&9I;&5=('-R8RYP87(*+@HD14Y6>U!!
M4E]04D]'3D%-17T@/2`D<')O9VYA;64@/2`D,"`]('-H:69T*$!!4D=6*3L*
M?0IS=6(@0W)E871E4&%T:"!["FUY("@D;F%M92D@/2!`7SL*<F5Q=6ER92!&
M:6QE.CI"87-E;F%M93L*;7D@*"1B87-E;F%M92P@)'!A=&@L("1E>'0I(#T@
M1FEL93HZ0F%S96YA;64Z.F9I;&5P87)S92@D;F%M92P@*"=<+BXJ)RDI.PIR
M97%U:7)E($9I;&4Z.E!A=&@["D9I;&4Z.E!A=&@Z.FUK<&%T:"@D<&%T:"D@
M=6YL97-S*"UE("1P871H*3L@(R!M:W!A=&@@9&EE<R!W:71H(&5R<F]R"GT*
M<W5B(')E<75I<F5?;6]D=6QE<R!["G)E<75I<F4@;&EB.PIR97%U:7)E($1Y
M;F%,;V%D97(["G)E<75I<F4@:6YT96=E<CL*<F5Q=6ER92!S=')I8W0["G)E
M<75I<F4@=V%R;FEN9W,["G)E<75I<F4@=F%R<SL*<F5Q=6ER92!#87)P.PIR
M97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5R<FYO.PIR97%U:7)E($5X
M<&]R=&5R.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CL*<F5Q=6ER92!&8VYT
M;#L*<F5Q=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.E-P96,["G)E
M<75I<F4@6%-,;V%D97(["G)E<75I<F4@0V]N9FEG.PIR97%U:7)E($E/.CI(
M86YD;&4["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@0V]M<')E<W,Z.EIL
M:6(["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!
M.PIR97%U:7)E(%!!4CL*<F5Q=6ER92!005(Z.DAE879Y.PIR97%U:7)E(%!!
M4CHZ1&ES=#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["G)E<75I
M<F4@4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N=#L*<F5Q=6ER92!A='1R:6)U
M=&5S.PIE=F%L('L@<F5Q=6ER92!#=V0@?3L*979A;"![(')E<75I<F4@5VEN
M,S(@?3L*979A;"![(')E<75I<F4@4V-A;&%R.CI5=&EL('T["F5V86P@>R!R
M97%U:7)E($%R8VAI=F4Z.E5N>FEP.CI"=7)S="!].PIE=F%L('L@<F5Q=6ER
M92!4:64Z.DAA<V@Z.DYA;65D0V%P='5R92!].PIE=F%L('L@<F5Q=6ER92!0
M97)L24\[(')E<75I<F4@4&5R;$E/.CIS8V%L87(@?3L*979A;"![(')E<75I
M<F4@=71F."!].PI]"G-U8B!?<V5T7W!A<E]T96UP('L*:68@*&1E9FEN960@
M)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI+RD@
M>PHD<&%R7W1E;7`@/2`D,3L*<F5T=7)N.PI]"F9O<F5A8V@@;7D@)'!A=&@@
M*`HH;6%P("1%3E9[)%]]+"!Q=R@@4$%27U1-4$1)4B!435!$25(@5$5-4$1)
M4B!414U0(%1-4"`I*2P*<7<H($,Z7%Q414U0("]T;7`@+B`I"BD@>PIN97AT
M('5N;&5S<R!D969I;F5D("1P871H(&%N9"`M9"`D<&%T:"!A;F0@+7<@)'!A
M=&@["FUY("1U<V5R;F%M93L*;7D@)'!W=6ED.PIE=F%L('LH)'!W=6ED*2`]
M(&=E='!W=6ED*"0^*2!I9B!D969I;F5D("0^.WT["FEF("@@9&5F:6YE9"@F
M5VEN,S(Z.DQO9VEN3F%M92D@*2!["B1U<V5R;F%M92`]("97:6XS,CHZ3&]G
M:6Y.86UE.PI]"F5L<VEF("AD969I;F5D("1P=W5I9"D@>PHD=7-E<FYA;64@
M/2`D<'=U:60["GT*96QS92!["B1U<V5R;F%M92`]("1%3E9[55-%4DY!345]
M('Q\("1%3E9[55-%4GT@?'P@)U-94U1%32<["GT*;7D@)'-T;7!D:7(@/2`B
M)'!A=&@D<WES>U]D96QI;7UP87(M(BYU;G!A8VLH(D@J(BP@)'5S97)N86UE
M*3L*;6MD:7(@)'-T;7!D:7(L(#`W-34["FUY("1C86-H95]D:7(["FEF("@D
M14Y6>U!!4E]#3$5!3GTI('L*)&-A8VAE7V1I<B`](")T96UP+20D(CL*?0IE
M;'-E('L*;W!E;B!M>2`D9F@L("(\.G)A=R(L("1P<F]G;F%M92!O<B!D:64@
M<7%;0V%N)W0@<F5A9"`B)'!R;V=N86UE(CH@)"%=.PII9B`H*&UY("1M86=I
M8U]P;W,@/2!F:6YD7W!A<E]M86=I8R@D9F@I*2`^/2`P*2!["G-E96L@)&9H
M+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP
M0T%#2$4B*2P@,#L*;7D@)&)U9CL*<F5A9"`D9F@L("1B=68L(&QE;F=T:"@B
M7#!#04-(12(I.PII9B`H)&)U9B!E<2`B7#!#04-(12(I('L*<V5E:R`D9F@L
M("1M86=I8U]P;W,*+2`D1DE,15]O9F9S971?<VEZ90HM(&QE;F=T:"@B7#!#
M04-(12(I"BT@)&-A8VAE7VYA;65?<VEZ92P@,#L*<F5A9"`D9F@L("1B=68L
M("1C86-H95]N86UE7W-I>F4["B1B=68@/7X@<R]<,"\O9SL*)&-A8VAE7V1I
M<B`](")C86-H92TD8G5F(CL*?0I]"F-L;W-E("1F:#L*?0II9B`H(21C86-H
M95]D:7(I('L*)&-A8VAE7V1I<B`](")T96UP+20D(CL*)$5.5GM005)?0TQ%
M04Y](#T@,3L*?0HD<W1M<&1I<B`N/2`B)'-Y<WM?9&5L:6U])&-A8VAE7V1I
M<B(["FUK9&ER("1S=&UP9&ER+"`P-S4U.PHD14Y6>U!!4E]414U0?2`]("1S
M=&UP9&ER.PIL87-T.PI]"B1P87)?=&5M<"`]("0Q(&EF("1%3E9[4$%27U1%
M35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\["GT*<W5B(%]S879E
M7V%S('L*;7D@*"1N86UE+"`D8V]N=&5N=',L("1M;V1E*2`]($!?.PIM>2`D
M9G5L;&YA;64@/2`B)'!A<E]T96UP+R1N86UE(CL*=6YL97-S("@M92`D9G5L
M;&YA;64I('L*;7D@)'1E;7!N86UE(#T@(B1F=6QL;F%M92XD)"(["FUY("1F
M:#L**&]P96X@)&9H+"`G/CIR87<G+"`D=&5M<&YA;64I"B8F("AP<FEN="`D
M9F@@)&-O;G1E;G1S*0HF)B`H8VQO<V4@)&9H*0IO<B!D:64@<7%;17)R;W(@
M=W)I=&EN9R`B)'1E;7!N86UE(CH@)"%=.PIC:&UO9"`D;6]D92P@)'1E;7!N
M86UE(&EF(&1E9FEN960@)&UO9&4["G)E;F%M92@D=&5M<&YA;64L("1F=6QL
M;F%M92D@;W(@=6YL:6YK*"1T96UP;F%M92D["GT*<F5T=7)N("1F=6QL;F%M
M93L*?0IS=6(@7W-E=%]P<F]G;F%M92!["FEF("AD969I;F5D("1%3E9[4$%2
M7U!23T=.04U%?2!A;F0@)$5.5GM005)?4%)/1TY!345](#U^("\H+BLI+RD@
M>PHD<')O9VYA;64@/2`D,3L*?0HD<')O9VYA;64@?'P]("0P.PII9B`H)$5.
M5GM005)?5$5-4'T@86YD(&EN9&5X*"1P<F]G;F%M92P@)$5.5GM005)?5$5-
M4'TI(#X](#`I('L*)'!R;V=N86UE(#T@<W5B<W1R*"1P<F]G;F%M92P@<FEN
M9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`K(#$I.PI]"FEF("@A)$5.
M5GM005)?4%)/1TY!345](&]R(&EN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L
M:6U]*2`^/2`P*2!["FEF("AO<&5N(&UY("1F:"P@)SPG+"`D<')O9VYA;64I
M('L*<F5T=7)N(&EF("US("1F:#L*?0II9B`H+7,@(B1P<F]G;F%M921S>7-[
M7V5X97TB*2!["B1P<F]G;F%M92`N/2`D<WES>U]E>&5].PIR971U<FX["GT*
M?0IF;W)E86-H(&UY("1D:7(@*'-P;&ET("]<421S>7-[<&%T:%]S97!]7$4O
M+"`D14Y6>U!!5$A]*2!["FYE>'0@:68@97AI<W1S("1%3E9[4$%27U1%35!]
M(&%N9"`D9&ER(&5Q("1%3E9[4$%27U1%35!].PHD9&ER(#U^(',O7%$D<WES
M>U]D96QI;7U<120O+SL**"@D<')O9VYA;64@/2`B)&1I<B1S>7-[7V1E;&EM
M?21P<F]G;F%M921S>7-[7V5X97TB*2P@;&%S="D*:68@+7,@(B1D:7(D<WES
M>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](CL**"@D<')O9VYA;64@/2`B
M)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M92(I+"!L87-T*0II9B`M<R`B)&1I
M<B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*?0IS=6(@7V9I>%]P<F]G;F%M
M92!["B0P(#T@)'!R;V=N86UE('Q\/2`D14Y6>U!!4E]04D]'3D%-17T["FEF
M("AI;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@/"`P*2!["B1P<F]G
M;F%M92`]("(N)'-Y<WM?9&5L:6U])'!R;V=N86UE(CL*?0IM>2`D<'=D(#T@
M*&1E9FEN960@)D-W9#HZ9V5T8W=D*2`_($-W9#HZ9V5T8W=D*"D*.B`H*&1E
M9FEN960@)E=I;C,R.CI'971#=V0I(#\@5VEN,S(Z.D=E=$-W9"@I(#H@8'!W
M9&`I.PIC:&]M<"@D<'=D*3L*)'!R;V=N86UE(#U^(',O7B@_/5PN7"X_7%$D
M<WES>U]D96QI;7U<12DO)'!W9"1S>7-[7V1E;&EM?2\["B1%3E9[4$%27U!2
M3T=.04U%?2`]("1P<F]G;F%M93L*?0IS=6(@7W!A<E]I;FET7V5N=B!["FEF
M("@@)$5.5GM005)?24Y)5$E!3$E:141]*RL@/3T@,2`I('L*<F5T=7)N.PI]
M(&5L<V4@>PHD14Y6>U!!4E])3DE424%,25I%1'T@/2`R.PI]"F9O<B`H<7<H
M(%-005=.140@5$5-4"!#3$5!3B!$14)51R!#04-(12!04D]'3D%-12`I("D@
M>PID96QE=&4@)$5.5GLB4$%27R1?(GT["GT*9F]R("AQ=R\@5$U01$E2(%1%
M35`@0TQ%04X@1$5"54<@+RD@>PHD14Y6>R)005)?)%\B?2`]("1%3E9[(E!!
M4E]'3$]"04Q?)%\B?2!I9B!E>&ES=',@)$5.5GLB4$%27T=,3T)!3%\D7R)]
M.PI]"FEF("@D14Y6>U!!4E]414U0?2D@>PID96QE=&4@)$5.5GM005)?0TQ%
M04Y].PI]"GT*<W5B(&]U=',@>PIR971U<FX@:68@)'%U:65T.PII9B`H)&QO
M9V9H*2!["G!R:6YT("1L;V=F:"`B0%]<;B(["GT*96QS92!["G!R:6YT(")`
M7UQN(CL*?0I]"G-U8B!I;FET7VEN8R!["G)E<75I<F4@0V]N9FEG.PIP=7-H
M($!)3D,L(&=R97`@9&5F:6YE9"P@;6%P("1#;VYF:6<Z.D-O;F9I9WLD7WTL
M('%W*`IA<F-H;&EB97AP('!R:79L:6)E>'`@<VET96%R8VAE>'`@<VET96QI
M8F5X<`IV96YD;W)A<F-H97AP('9E;F1O<FQI8F5X<`HI.PI]"G!A8VMA9V4@
M;6%I;CL*<F5Q=6ER92!005(["G5N<VAI9G0@0$E.0RP@7"9005(Z.F9I;F1?
M<&%R.PI005(M/FEM<&]R="A`<&%R7V%R9W,I.PID:64@<7%;<&%R+G!L.B!#
M86XG="!O<&5N('!E<FP@<V-R:7!T("(D<')O9VYA;64B.B!.;R!S=6-H(&9I
M;&4@;W(@9&ER96-T;W)Y7&Y="G5N;&5S<R`M92`D<')O9VYA;64["F1O("1P
M<F]G;F%M93L*0T]213HZ97AI="@D,2D@:68@*"1`(#U^+UY?5$M?15A)5%]<
M*"A<9"LI7"DO*3L*9&EE("1`(&EF("1`.PI].PHD.CI?7T524D]2(#T@)$`@
M:68@)$`["GT*0T]213HZ97AI="@D,2D@:68@*"0Z.E]?15)23U(@/7XO7E]4
M2U]%6$E47UPH*%QD*RE<*2\I.PID:64@)#HZ7U]%4E)/4B!I9B`D.CI?7T52
M4D]2.PHQ.PI?7T5.1%]?"@!P97)L>'-I+F,``````````$-!0TA%```!(T5G
M`````(FKS>\`````_MRZF`````!V5#(0`````/#ATL,````````````````!
M&P,[#`$``"````"`@___L`8```"&__\D`0``8(;__S@!``"0AO__3`$``,R&
M__]@`0``((?__X0!``!`A___F`$``$"(__\``@```(G__R@"``#@BO__B`(`
M`$"-___,`@``:([__P`#``#@CO__)`,``""/__\X`P``0)#__VP#``!$D/__
M@`,``$"3__\`!```@)/__R0$``!`E?__D`0``&":___,!```X)K___`$``"(
MF___#`4``#"=__]0!0``8)W__W`%``"HGO__P`4``$BF__]$!@``Z*C__WP&
M````JO__X`8``."Z__\,!P``*+O__RP'``!@O/__:`<``$"^__^@!P``````
M`!```````````7I2``1X'@$;#!\`$````!@```#4A/__-`````!!!QX0````
M+````""%__\P`````````!````!`````/(7__SP`````````(````%0```!D
MA?__2`````!!#B"=!)X#0I,"3M[=TPX`````$````'@```"4A?__!```````
M``!D````C````*"%___X`````$$.4)T*G@E%DPB4!YD"0Y8%E09#F`.7!%?6
MU4'8UT3>W=G3U`X`00Y0DPB4!Y4&E@67!)@#F0*="IX)3];50=C70M[=V=/4
M#@!!#E"3")0'F0*="IX)`"0```#T````.(;__[0`````00ZP`42=!)X#0I,"
M6`K3W=X.`$$+``!<````'`$``-"&___8`0```$$.T`%#G0B>!T:4!9,&1-33
M2=W>#@!!#M`!DP:4!9T(G@=&F`&7`D.6`Y4$`D,*U--!UM5!V-=!"TG3U-76
MU]A#E`63!D&6`Y4$09@!EP)`````?`$``%"(__]@`@```$$.<$&=#)X+0I,*
ME`E&E0B6!Y<&F`69!)H#FP*<`0)0"MO<V=K7V-76T]3=W@X`00L``#````#`
M`0``;(K__R@!````00XPG0:>!44.P""3!)0#E0*6`6D*#C!#WMW5UM/4#@!!
M"P`@````]`$``&"+__]X`````$$.()T$G@-%DP*4`4_>W=/4#@`0````&`(`
M`+2+__\T`````````#`````L`@``X(O__QP!````00XPG0:>!44.T$"3!)0#
ME0*6`7D*#C!$WMW5UM/4#@!!"P`0````8`(``,R,__\$`````````'P```!T
M`@``O(S___0"````00Y@G0R>"T4.D"*3"I0)E0B6!V$*#F!#WMW5UM/4#@!!
M"T.7!D.8!4&;`D*<`5&9!$2:`WD*UT'80=E!VD';0=Q!"UL*UT'80=E!VD';
M0=Q!"T;9VD'70=A!VT'<1)<&09@%09D$09H#09L"09P!(````/0"```XC___
M/`````!!#B"=!)X#19,"1M[=TPX`````:````!@#``!4C___O`$```!!#C"=
M!IX%0P[`($B5`D:3!$&4`VK30M1!U4$.,$'>W0X`00[`()4"G0:>!4'530XP
M0=[=#@!!#L`@DP24`Y4"G0:>!4P*TT+40M5#"T/3U-5#DP1!E`-!E0(`.```
M`(0#``"HD/__&`4```!!#F!#G0J>"4*3")0'1I4&E@67!)@#F0(#+0$*V=?8
MU=;3U-W>#@!!"P``(````,`#``",E?__@`````!!#B!%G0*>`50*W=X.`$$+
M````&````.0#``#HE?__J`````!!#L`"19T:GAD``$``````!```=);__Z@!
M````00Y`G0B>!T4.P(`$DP:4!94$E@.7`I@!0@[0@`1^"@[`@`1!#D!$WMW7
MV-76T]0.`$$+'````$0$``#8E___,`````!##A"=`IX!2-[=#@````!,````
M9`0``.B7__](`0```$$.4$.=")X'0I,&E`5"E026`UD*U=;3U-W>#@!!"T&7
M`F;70=76T]3=W@X`00Y0DP:4!94$E@.=")X'0Y<"`(````"T!```X)C__Z`'
M````00[``T6=#)X+1I,*E`E%E0B6!Y<&F`59F@.9!$&<`9L"`I,*VME"W-M%
MU]C5UM/4W=X.`$$+`K#9VMO<3M?8U=;3U-W>#@!##L`#DPJ4"94(E@>7!I@%
MF02:`YL"G`&=#)X+`ES9VMO<0YH#F01!G`&;`C0````X!0``_)___Z`"````
M00Y019T(G@="DP:4!425!)8#0Y<"F`$"8@K7V-76T]3=W@X`00L`,````'`%
M``!DHO__&`$```!!#C!#G02>`T*3`I0!6PK3U-W>#@!!"V`*T]3=W@X`00L`
M`"P```"D!0``R'S__W0"````00Y@0YT&G@5(DP24`Y4"E@$"9@K5UM/4W=X.
M`$$+`"@```#4!0``&*/__]@0````00X0G0*>`4<.H`4#*00*#A!!WMT.`$$+
M````'``````&``#,L___2`````!##A"=`IX!3M[=#@`````X````(`8``/2S
M__\P`0```$$.4)T*G@E#DPB4!T.5!I8%EP28`T.9`IH!;`K>W=G:U]C5UM/4
M#@!!"P`T````7`8``/"T___<`0```$$.0)T(G@=#DP:4!4.5!)8#0@5/`@)E
M"M[=U=;3U`9/#@!!"P```!````"4!@``F+;__PP`````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````X!\```````",'P````````$`````````!P,`
M```````!`````````!(#`````````0`````````<`P```````!T`````````
MI00````````,`````````,@;````````#0`````````,5P```````!D`````
M````</L!```````;``````````@`````````&@````````!X^P$``````!P`
M````````"`````````#U_O]O`````)@"````````!0````````#8#0``````
M``8`````````,`0````````*`````````,@$````````"P`````````8````
M`````!4````````````````````#`````````*#]`0```````@````````!@
M`````````!0`````````!P`````````7`````````&@;````````!P``````
M``#0$P````````@`````````F`<````````)`````````!@`````````'@``
M```````(`````````/O__V\``````0``"`````#^__]O`````'`3````````
M____;P`````"`````````/#__V\`````H!(```````#Y__]O``````\`````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````#@&P```````.`;````````X!L```````#@&P``
M`````(#[`0``````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````!7````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M0!P`````````````````````````````"``"```````*4$%2+G!M"@``````
M`````%L```````"05P``````````````````@%@```````#@6@```````.A:
M````````\%H```````#X6@``````````````````R%H```````#06@``````
M````````````"%L```````!'0T,Z("A'3E4I(#$T+C$N,2`R,#(T,#4P-P!'
M0T,Z("AC<F]S<W1O;VPM3D<@,2XR-BXP+C<R7S@Q,#`R,60I(#$T+C$N,2`R
M,#(T,#4P-P``+G-H<W1R=&%B`"YI;G1E<G``+FYO=&4N9VYU+F)U:6QD+6ED
M`"YN;W1E+D%"22UT86<`+F=N=2YH87-H`"YD>6YS>6T`+F1Y;G-T<@`N9VYU
M+G9E<G-I;VX`+F=N=2YV97)S:6]N7W(`+G)E;&$N9'EN`"YR96QA+G!L=``N
M:6YI=``N=&5X=``N9FEN:0`N<F]D871A`"YE:%]F<F%M95]H9'(`+F5H7V9R
M86UE`"YI;FET7V%R<F%Y`"YF:6YI7V%R<F%Y`"YD>6YA;6EC`"YG;W0`+F1A
M=&$`+F)S<P`N8V]M;65N=```````````````````````````````````````
M```````````````````````````````````````````````````````+````
M`0````(`````````.`(````````X`@```````!L````````````````````!
M````````````````````$P````<````"`````````%0"````````5`(`````
M```D````````````````````!````````````````````"8````'`````@``
M``````!X`@```````'@"````````(`````````````````````0`````````
M```````````T````]O__;P(`````````F`(```````"8`@```````)0!````
M````!0`````````(````````````````````/@````L````"`````````#`$
M````````,`0```````"H"0````````8````#````"``````````8````````
M`$8````#`````@````````#8#0```````-@-````````R`0`````````````
M``````$```````````````````!.````____;P(`````````H!(```````"@
M$@```````,X`````````!0`````````"``````````(`````````6P```/[_
M_V\"`````````'`3````````<!,```````!@``````````8````"````"```
M`````````````````&H````$`````@````````#0$P```````-`3````````
MF`<````````%``````````@`````````&`````````!T````!````$(`````
M````:!L```````!H&P```````&``````````!0```!4````(`````````!@`
M````````?@````$````&`````````,@;````````R!L````````8````````
M````````````!````````````````````'D````!````!@````````#@&P``
M`````.`;````````8````````````````````!````````````````````"$
M`````0````8`````````0!P```````!`'````````,PZ````````````````
M``!`````````````````````B@````$````&``````````Q7````````#%<`
M```````4````````````````````!````````````````````)`````!````
M`@`````````@5P```````"!7````````H$$``````````````````!``````
M``````````````"8`````0````(`````````P)@```````#`F`````````P!
M```````````````````$````````````````````I@````$````"````````
M`-"9````````T)D```````"H!@``````````````````"```````````````
M`````+`````.`````P````````!P^P$``````'#[````````"```````````
M``````````@`````````"`````````"\````#P````,`````````>/L!````
M``!X^P````````@````````````````````(``````````@`````````R```
M``8````#`````````(#[`0``````@/L````````@`@````````8`````````
M"``````````0`````````-$````!`````P````````"@_0$``````*#]````
M````8`(```````````````````@`````````"`````````#6`````0````,`
M```````````"``````````$``````(@````````````````````(````````
M````````````W`````@````#`````````(@``@``````B``!```````H````
M````````````````"````````````````````.$````!````,```````````
M`````````(@``0``````40````````````````````$``````````0``````
M```!`````P````````````````````````#9``$``````.H`````````````
5```````!````````````````````
