(function () {
// Licensed under the Apache License, Version 2.0 (the "License"); you may not
// use this file except in compliance with the License. You may obtain a copy of
// the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
// License for the specific language governing permissions and limitations under
// the License.
//
// Based on the normalizeFunction which can be
// found here:
//
//  https://github.com/dmunch/couch-chakra/blob/master/js/normalizeFunction.js

function rewriteFunInt(source) {
    return source;
}
// Licensed under the Apache License, Version 2.0 (the "License"); you may not
// use this file except in compliance with the License. You may obtain a copy of
// the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
// License for the specific language governing permissions and limitations under
// the License.

var Dreyfus = (function() {

  var index_results = []; // holds temporary emitted values during index

  function handleIndexError(err, doc) {
    if (err == "fatal_error") {
      throw(["error", "map_runtime_error", "function raised 'fatal_error'"]);
    } else if (err[0] == "fatal") {
      throw(err);
    }
    var message = "function raised exception " + errstr(err);
    if (doc) message += " with doc._id " + doc._id;
    log(message);
  };

  return {
    index: function(name, value, options) {
      if (typeof name !== 'string') {
        throw({name: 'TypeError', message: 'name must be a string not ' + typeof name});
      }
      if (name.substring(0, 1) === '_') {
        throw({name: 'ReservedName', message: 'name must not start with an underscore'});
      }
      if (typeof value !== 'string' && typeof value !== 'number' && typeof value !== 'boolean') {
        throw({name: 'TypeError', message: 'value must be a string, a number or boolean not ' + typeof value});
      }
      if (options && typeof options !== 'object') {
        throw({name: 'TypeError', message: 'options must be an object not ' + typeof options});
      }
      index_results.push([name, value, options || {}]);
    },

    indexDoc: function(doc) {
      Couch.recursivelySeal(doc);
      var buf = [];
      for (var fun in State.funs) {
        index_results = [];
        try {
          State.funs[fun](doc);
          buf.push(index_results);
        } catch (err) {
          handleIndexError(err, doc);
          buf.push([]);
        }
      }
      print(JSON.stringify(buf));
    }

  }
})();
// Licensed under the Apache License, Version 2.0 (the "License"); you may not
// use this file except in compliance with the License. You may obtain a copy of
// the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
// License for the specific language governing permissions and limitations under
// the License.

var Nouveau = (function () {

  var index_results = []; // holds temporary emitted values during index

  function handleIndexError(err, doc) {
    if (err == "fatal_error") {
      throw (["error", "map_runtime_error", "function raised 'fatal_error'"]);
    } else if (err[0] == "fatal") {
      throw (err);
    }
    var message = "function raised exception " + err.toSource();
    if (doc) message += " with doc._id " + doc._id;
    log(message);
  };

  function assertType(name, expected, actual) {
    if (typeof actual !== expected) {
      throw ({ name: 'TypeError', message: 'type of ' + name + ' must be a ' + expected + ' not ' + typeof actual });
    }
  };

  function rejectReservedName(name) {
    if (name.substring(0, 1) === '_') {
      throw ({ name: 'ReservedName', message: 'name must not start with an underscore' });
    }
  };

  return {
    index: function (doc) {
      var type = arguments[0];
      var name = arguments[1];

      assertType('type', 'string', type);
      assertType('name', 'string', name);

      rejectReservedName(name);

      switch (type) {
        case 'double':
        case 'string':
          var value = arguments[2];
          var options = arguments[3] || {};
          assertType('value', type == 'double' ? 'number' : 'string', value);
          index_results.push({
            '@type': type,
            'name': name,
            'value': value,
            'store': options.store,
            'facet': options.facet
          });
          break;
        case 'text':
          var value = arguments[2];
          var options = arguments[3] || {};
          assertType('value', 'string', value);
          index_results.push({
            '@type': type,
            'name': name,
            'value': value,
            'store': options.store
          });
          break;
        case 'stored':
          var value = arguments[2];
          if (typeof value != 'number' && typeof value != 'string') {
            throw ({ name: 'TypeError', message: 'type of ' + value + ' must be a string or number' });
          }
          index_results.push({
            '@type': type,
            'name': name,
            'value': value
          });
          break;
        default:
          throw ({ name: 'TypeError', message: type + ' not supported' });
      }
    },

    indexDoc: function (doc) {
      Couch.recursivelySeal(doc);
      var buf = [];
      for (var fun in State.funs) {
        index_results = [];
        try {
          State.funs[fun](doc);
          buf.push(index_results);
        } catch (err) {
          handleIndexError(err, doc);
          buf.push([]);
        }
      }
      print(JSON.stringify(buf));
    }

  }
})();
// Licensed under the Apache License, Version 2.0 (the "License"); you may not
// use this file except in compliance with the License. You may obtain a copy of
// the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
// License for the specific language governing permissions and limitations under
// the License.

var Filter = (function() {

  var view_emit = false;

  return {
      emit : function(key, value) {
        view_emit = true;
      },
      filter : function(fun, ddoc, args) {
        var results = [];
        var docs = args[0];
        var req = args[1];
        for (var i=0; i < docs.length; i++) {
          results.push((fun.apply(ddoc, [docs[i], req]) && true) || false);
        };
        respond([true, results]);
      },
      filter_view : function(fun, ddoc, args) {
        var results = [];
        var docs = args[0];
        for (var i=0; i < docs.length; i++) {
          view_emit = false;
          fun(docs[i]);
          results.push((view_emit && true) || false);
        };
        respond([true, results]);
      }
    }
})();
// mimeparse.js
//
// This module provides basic functions for handling mime-types. It can
// handle matching mime-types against a list of media-ranges. See section
// 14.1 of the HTTP specification [RFC 2616] for a complete explanation.
//
//   http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.1
//
// A port to JavaScript of Joe Gregorio's MIME-Type Parser:
//
//   http://code.google.com/p/mimeparse/
//
// Ported by J. Chris Anderson <jchris@apache.org>, targeting the Spidermonkey runtime.
//
// To run the tests, open mimeparse-js-test.html in a browser.
// Ported from version 0.1.2
// Comments are mostly excerpted from the original.

var Mimeparse = (function() {
  // private helpers
  function strip(string) {
    return string.replace(/^\s+/, '').replace(/\s+$/, '');
  };

  function parseRanges(ranges) {
    var parsedRanges = [], rangeParts = ranges.split(",");
    for (var i=0; i < rangeParts.length; i++) {
      parsedRanges.push(publicMethods.parseMediaRange(rangeParts[i]));
    };
    return parsedRanges;
  };

  var publicMethods = {
    // Carves up a mime-type and returns an Array of the
    //  [type, subtype, params] where "params" is a Hash of all
    //  the parameters for the media range.
    //
    // For example, the media range "application/xhtml;q=0.5" would
    //  get parsed into:
    //
    // ["application", "xhtml", { "q" : "0.5" }]
    parseMimeType : function(mimeType) {
      var fullType, typeParts, params = {}, parts = mimeType.split(';');
      for (var i=0; i < parts.length; i++) {
        var p = parts[i].split('=');
        if (p.length == 2) {
          params[strip(p[0])] = strip(p[1]);
        }
      };
      fullType = parts[0].replace(/^\s+/, '').replace(/\s+$/, '');
      if (fullType == '*') fullType = '*/*';
      typeParts = fullType.split('/');
      return [typeParts[0], typeParts[1], params];
    },

    // Carves up a media range and returns an Array of the
    //  [type, subtype, params] where "params" is a Object with
    //  all the parameters for the media range.
    //
    // For example, the media range "application/*;q=0.5" would
    //  get parsed into:
    //
    // ["application", "*", { "q" : "0.5" }]
    //
    // In addition this function also guarantees that there
    //  is a value for "q" in the params dictionary, filling it
    //  in with a proper default if necessary.
    parseMediaRange : function(range) {
      var q, parsedType = this.parseMimeType(range);
      if (!parsedType[2]['q']) {
        parsedType[2]['q'] = '1';
      } else {
        q = parseFloat(parsedType[2]['q']);
        if (isNaN(q)) {
          parsedType[2]['q'] = '1';
        } else if (q > 1 || q < 0) {
          parsedType[2]['q'] = '1';
        }
      }
      return parsedType;
    },

    // Find the best match for a given mime-type against
    // a list of media_ranges that have already been
    // parsed by parseMediaRange(). Returns an array of
    // the fitness value and the value of the 'q' quality
    // parameter of the best match, or (-1, 0) if no match
    // was found. Just as for qualityParsed(), 'parsed_ranges'
    // must be a list of parsed media ranges.
    fitnessAndQualityParsed : function(mimeType, parsedRanges) {
      var bestFitness = -1, bestFitQ = 0, target = this.parseMediaRange(mimeType);
      var targetType = target[0], targetSubtype = target[1], targetParams = target[2];

      for (var i=0; i < parsedRanges.length; i++) {
        var parsed = parsedRanges[i];
        var type = parsed[0], subtype = parsed[1], params = parsed[2];
        if ((type == targetType || type == "*" || targetType == "*") &&
          (subtype == targetSubtype || subtype == "*" || targetSubtype == "*")) {
          var matchCount = 0;
          for (var param in targetParams) {
            if (param != 'q' && params[param] && params[param] == targetParams[param]) {
              matchCount += 1;
            }
          }

          var fitness = (type == targetType) ? 100 : 0;
          fitness += (subtype == targetSubtype) ? 10 : 0;
          fitness += matchCount;

          if (fitness > bestFitness) {
            bestFitness = fitness;
            bestFitQ = params["q"];
          }
        }
      };
      return [bestFitness, parseFloat(bestFitQ)];
    },

    // Find the best match for a given mime-type against
    // a list of media_ranges that have already been
    // parsed by parseMediaRange(). Returns the
    // 'q' quality parameter of the best match, 0 if no
    // match was found. This function bahaves the same as quality()
    // except that 'parsedRanges' must be a list of
    // parsed media ranges.
    qualityParsed : function(mimeType, parsedRanges) {
      return this.fitnessAndQualityParsed(mimeType, parsedRanges)[1];
    },

    // Returns the quality 'q' of a mime-type when compared
    // against the media-ranges in ranges. For example:
    //
    // >>> Mimeparse.quality('text/html','text/*;q=0.3, text/html;q=0.7, text/html;level=1, text/html;level=2;q=0.4, */*;q=0.5')
    // 0.7
    quality : function(mimeType, ranges) {
      return this.qualityParsed(mimeType, parseRanges(ranges));
    },

    // Takes a list of supported mime-types and finds the best
    // match for all the media-ranges listed in header. The value of
    // header must be a string that conforms to the format of the
    // HTTP Accept: header. The value of 'supported' is a list of
    // mime-types.
    //
    // >>> bestMatch(['application/xbel+xml', 'text/xml'], 'text/*;q=0.5,*/*; q=0.1')
    // 'text/xml'
    bestMatch : function(supported, header) {
      var parsedHeader = parseRanges(header);
      var weighted = [];
      for (var i=0; i < supported.length; i++) {
        weighted.push([publicMethods.fitnessAndQualityParsed(supported[i], parsedHeader), i, supported[i]]);
      };
      weighted.sort();
      return weighted[weighted.length-1][0][1] ? weighted[weighted.length-1][2] : '';
    }
  };
  return publicMethods;
})();
// Licensed under the Apache License, Version 2.0 (the "License"); you may not
// use this file except in compliance with the License. You may obtain a copy of
// the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
// License for the specific language governing permissions and limitations under
// the License.


var Mime = (function() {
  // registerType(name, mime-type, mime-type, ...)
  // 
  // Available in query server sandbox. TODO: The list is cleared on reset.
  // This registers a particular name with the set of mimetypes it can handle.
  // Whoever registers last wins.
  // 
  // Example: 
  // registerType("html", "text/html; charset=utf-8");

  var mimesByKey = {};
  var keysByMime = {};
  function registerType() {
    var mimes = [], key = arguments[0];
    for (var i=1; i < arguments.length; i++) {
      mimes.push(arguments[i]);
    };
    mimesByKey[key] = mimes;
    for (var i=0; i < mimes.length; i++) {
      keysByMime[mimes[i]] = key;
    };
  }

  // Some default types
  // Ported from Ruby on Rails
  // Build list of Mime types for HTTP responses
  // http://www.iana.org/assignments/media-types/
  // http://dev.rubyonrails.org/svn/rails/trunk/actionpack/lib/action_controller/mime_types.rb

  registerType("all", "*/*");
  registerType("text", "text/plain; charset=utf-8", "txt");
  registerType("html", "text/html; charset=utf-8");
  registerType("xhtml", "application/xhtml+xml", "xhtml");
  registerType("xml", "application/xml", "text/xml", "application/x-xml");
  registerType("js", "text/javascript", "application/javascript", "application/x-javascript");
  registerType("css", "text/css");
  registerType("ics", "text/calendar");
  registerType("csv", "text/csv");
  registerType("rss", "application/rss+xml");
  registerType("atom", "application/atom+xml");
  registerType("yaml", "application/x-yaml", "text/yaml");
  // just like Rails
  registerType("multipart_form", "multipart/form-data");
  registerType("url_encoded_form", "application/x-www-form-urlencoded");
  // http://www.ietf.org/rfc/rfc4627.txt
  registerType("json", "application/json", "text/x-json");


  var providesUsed = false;
  var mimeFuns = [];
  var responseContentType = null;

  function provides(type, fun) {
    providesUsed = true;
    mimeFuns.push([type, fun]);
  };

  function resetProvides() {
    // set globals
    providesUsed = false;
    mimeFuns = [];
    responseContentType = null;
  };

  function runProvides(req, ddoc) {
    var supportedMimes = [], bestFun, bestKey = null, accept = req.headers["Accept"];
    if (req.query && req.query.format) {
      bestKey = req.query.format;
      responseContentType = mimesByKey[bestKey][0];
    } else if (accept) {
      // log("using accept header: "+accept);
      mimeFuns.reverse().forEach(function(mimeFun) {
        var mimeKey = mimeFun[0];
        if (mimesByKey[mimeKey]) {
          supportedMimes = supportedMimes.concat(mimesByKey[mimeKey]);
        }
      });
      responseContentType = Mimeparse.bestMatch(supportedMimes, accept);
      bestKey = keysByMime[responseContentType];
    } else {
      // just do the first one
      bestKey = mimeFuns[0][0];
      responseContentType = mimesByKey[bestKey][0];
    }

    if (bestKey) {
      for (var i=0; i < mimeFuns.length; i++) {
        if (mimeFuns[i][0] == bestKey) {
          bestFun = mimeFuns[i][1];
          break;
        }
      };
    };

    if (bestFun) {
      return bestFun.call(ddoc);
    } else {
      var supportedTypes = mimeFuns.map(function(mf) {
        return mimesByKey[mf[0]].join(', ') || mf[0];
      });
      throw(["error","not_acceptable",
        "Content-Type "+(accept||bestKey)+" not supported, try one of: "+supportedTypes.join(', ')]);
    }
  };


  return {
    registerType : registerType,
    provides : provides,
    resetProvides : resetProvides,
    runProvides : runProvides,
    providesUsed : function () {
      return providesUsed;
    },
    responseContentType : function () {
      return responseContentType;
    }
  };
})();




////
////  Render dispatcher
////
////
////
////

var Render = (function() {
  var new_header = false;
  var chunks = [];
  
  
  //  Start chunks
  var startResp = {};
  function start(resp) {
    startResp = resp || {};
    new_header = true;
  };

  function sendStart() {
    startResp = applyContentType((startResp || {}), Mime.responseContentType());
    respond(["start", chunks, startResp]);
    chunks = [];
    startResp = {};
    new_header = false;
  }

  function applyContentType(resp, responseContentType) {
    resp["headers"] = resp["headers"] || {};
    if (responseContentType) {
      resp["headers"]["Content-Type"] = resp["headers"]["Content-Type"] || responseContentType;
    }
    return resp;
  }

  function send(chunk) {
    chunks.push(chunk.toString());
  };

  function blowChunks(label) {
    if (new_header) {
      respond([label||"chunks", chunks, startResp]);
      new_header = false;
    }
    else {
      respond([label||"chunks", chunks]);
    }
    chunks = [];
  };

  var gotRow = false, lastRow = false;
  function getRow() {
    if (lastRow) return null;
    if (!gotRow) {
      gotRow = true;
      sendStart();
    } else {
      blowChunks();
    }
    var json = JSON.parse(readline());
    if (json[0] == "list_end") {
      lastRow = true;
      return null;
    }
    if (json[0] != "list_row") {
      throw(["fatal", "list_error", "not a row '" + json[0] + "'"]);
    }
    return json[1];
  };

  
  function maybeWrapResponse(resp) {
    var type = typeof resp;
    if ((type == "string") || (type == "xml")) {
      return {body:resp};
    } else {
      return resp;
    }
  };

  // from http://javascript.crockford.com/remedial.html
  function typeOf(value) {
    var s = typeof value;
    if (s === 'object') {
      if (value) {
        if (value instanceof Array) {
          s = 'array';
        }
      } else {
        s = 'null';
      }
    }
    return s;
  };

  function isDocRequestPath(info) {
    var path = info.path;
    return path.length > 5;
  };

  function runShow(fun, ddoc, args) {
    try {
      resetList();
      Mime.resetProvides();
      var resp = fun.apply(ddoc, args) || {};
      resp = maybeWrapResponse(resp);

      // handle list() style API
      if (chunks.length && chunks.length > 0) {
        resp.headers = resp.headers || {};
        for(var header in startResp) {
          resp.headers[header] = startResp[header];
        }
        resp.body = chunks.join("") + (resp.body || "");
        resetList();
      }

      if (Mime.providesUsed()) {
        var provided_resp = Mime.runProvides(args[1], ddoc) || {};
        provided_resp = maybeWrapResponse(provided_resp);
        resp.body = (resp.body || "") + chunks.join("");
        resp.body += provided_resp.body || "";
        resp = applyContentType(resp, Mime.responseContentType());
        resetList();
      }

      var type = typeOf(resp);
      if (type == 'object' || type == 'string') {
        respond(["resp", maybeWrapResponse(resp)]);
      } else {
        throw(["error", "render_error", "undefined response from show function"]);      
      }
    } catch(e) {
      if (args[0] === null && isDocRequestPath(args[1])) {
        throw(["error", "not_found", "document not found"]);
      } else {
        renderError(e, fun.toString());
      }
    }
  };

  function runUpdate(fun, ddoc, args) {
    try {
      var method = args[1].method;
      // for analytics logging applications you might want to remove the next line
      if (method == "GET") throw(["error","method_not_allowed","Update functions do not allow GET"]);
      var result = fun.apply(ddoc, args);
      var doc = result[0];
      var resp = result[1];
      var type = typeOf(resp);
      if (type == 'object' || type == 'string') {
        respond(["up", doc, maybeWrapResponse(resp)]);
      } else {
        throw(["error", "render_error", "undefined response from update function"]);      
      }
    } catch(e) {
      renderError(e, fun.toString());
    }
  };

  function resetList() {
    gotRow = false;
    lastRow = false;
    chunks = [];
    startResp = {};
    new_header = false;
  };

  function runList(listFun, ddoc, args) {
    try {
      Mime.resetProvides();
      resetList();
      var head = args[0];
      var req = args[1];
      var tail = listFun.apply(ddoc, args);

      if (Mime.providesUsed()) {
        tail = Mime.runProvides(req, ddoc);
      }
      if (!gotRow) getRow();
      if (typeof tail != "undefined") {
        chunks.push(tail);
      }
      blowChunks("end");
    } catch(e) {
      renderError(e, listFun.toString());
    }
  };

  function runRewrite(fun, ddoc, args) {
      var result;
      try {
        result = fun.apply(ddoc, args);
      } catch(error) {
        renderError(error, fun.toString(), "rewrite_error");
      }

      if (!result) {
        respond(["no_dispatch_rule"]);
        return;
      }

      if (typeof result === "string") {
        result = {path: result, method: args[0].method};
      }
      respond(["ok", result]);
  }

  function renderError(e, funSrc, errType) {
    if (e.error && e.reason || e[0] == "error" || e[0] == "fatal") {
      throw(e);
    } else {
      var logMessage = "function raised error: " +
                        errstr(e) + " \n" +
                       "stacktrace: " + e.stack;
      log(logMessage);
      throw(["error", errType || "render_error", logMessage]);
    }
  };

  function escapeHTML(string) {
    return string && string.replace(/&/g, "&amp;")
                 .replace(/</g, "&lt;")
                 .replace(/>/g, "&gt;");
  };

  
  return {
    start : start,
    send : send,
    getRow : getRow,
    show : function(fun, ddoc, args) {
      // var showFun = Couch.compileFunction(funSrc);
      runShow(fun, ddoc, args);
    },
    update : function(fun, ddoc, args) {
      // var upFun = Couch.compileFunction(funSrc);
      runUpdate(fun, ddoc, args);
    },
    list : function(fun, ddoc, args) {
      runList(fun, ddoc, args);
    },
    rewrite : function(fun, ddoc, args) {
      runRewrite(fun, ddoc, args);
    }
  };
})();

// send = Render.send;
// getRow = Render.getRow;
// start = Render.start;

// unused. this will be handled in the Erlang side of things.
// function htmlRenderError(e, funSrc) {
//   var msg = ["<html><body><h1>Render Error</h1>",
//     "<p>JavaScript function raised error: ",
//     e.toString(),
//     "</p><h2>Stacktrace:</h2><code><pre>",
//     escapeHTML(e.stack),
//     "</pre></code><h2>Function source:</h2><code><pre>",
//     escapeHTML(funSrc),
//     "</pre></code></body></html>"].join('');
//   return {body:msg};
// };
// Licensed under the Apache License, Version 2.0 (the "License"); you may not
// use this file except in compliance with the License. You may obtain a copy of
// the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
// License for the specific language governing permissions and limitations under
// the License.

var State = {
  reset : function(config) {
    // clear the globals and run gc
    State.funs = [];
    State.lib = null;
    State.query_config = config || {};
    gc();
    print("true"); // indicates success
  },
  addFun : function(newFun, option) {
    // Compile to a function and add it to funs array
    switch (option) {
      case 'nouveau':
        var sandbox = create_nouveau_sandbox();
        break;
      default:
        var sandbox = create_dreyfus_sandbox();
        break;
    }
    State.funs.push(Couch.compileFunction(newFun, {views : {lib : State.lib}}, undefined, sandbox));
    print("true");
  },
  addLib : function(lib) {
    State.lib = lib;
    print("true");
  }
};
// Licensed under the Apache License, Version 2.0 (the "License"); you may not
// use this file except in compliance with the License. You may obtain a copy of
// the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
// License for the specific language governing permissions and limitations under
// the License.

var resolveModule = function(names, mod, root) {
  if (names.length == 0) {
    if (typeof mod.current != "string") {
      throw ["error","invalid_require_path",
        'Must require a JavaScript string, not: '+(typeof mod.current)];
    }
    return {
      current : mod.current,
      parent : mod.parent,
      id : mod.id,
      exports : {}
    };
  }
  // we need to traverse the path
  var n = names.shift();
  if (n == '..') {
    if (!(mod.parent && mod.parent.parent)) {
      throw ["error", "invalid_require_path", 'Object has no parent '+JSON.stringify(mod.current)];
    }
    return resolveModule(names, {
      id : mod.id.slice(0, mod.id.lastIndexOf('/')),
      parent : mod.parent.parent,
      current : mod.parent.current
    });
  } else if (n == '.') {
    if (!mod.parent) {
      throw ["error", "invalid_require_path", 'Object has no parent '+JSON.stringify(mod.current)];
    }
    return resolveModule(names, {
      parent : mod.parent,
      current : mod.current,
      id : mod.id
    });
  } else if (root) {
    mod = {current : root};
  }
  if (mod.current[n] === undefined) {
    throw ["error", "invalid_require_path", 'Object has no property "'+n+'". '+JSON.stringify(mod.current)];
  }
  return resolveModule(names, {
    current : mod.current[n],
    parent : mod,
    id : mod.id ? mod.id + '/' + n : n
  });
};

var Couch = {
  // moving this away from global so we can move to json2.js later
  compileFunction : function(source, ddoc, name, sandbox) {
    if (!source) throw(["error","not_found","missing function"]);

    var functionObject = null;
    var sandbox = sandbox || create_sandbox();

    var require = function(name, module) {
      module = module || {};
      var newModule = resolveModule(name.split('/'), module.parent, ddoc);
      if (!ddoc._module_cache.hasOwnProperty(newModule.id)) {
        // create empty exports object before executing the module,
        // stops circular requires from filling the stack
        ddoc._module_cache[newModule.id] = {};
        var s = "(function (module, exports, require) { " + newModule.current + "\n });";
        try {
          var func = sandbox ? evalcx(s, sandbox, newModule.id) : eval(s);
          func.apply(sandbox, [newModule, newModule.exports, function(name) {
            return require(name, newModule);
          }]);
        } catch(e) {
          throw [
            "error",
            "compilation_error",
            "Module require('" +name+ "') raised error " + errstr(e)
          ];
        }
        ddoc._module_cache[newModule.id] = newModule.exports;
      }
      return ddoc._module_cache[newModule.id];
    };

    if (ddoc) {
      sandbox.require = require;
      if (!ddoc._module_cache) ddoc._module_cache = {};
    }

    try {
      if(typeof CoffeeScript === "undefined") {
        var rewrittenFun = rewriteFunInt(source);
        functionObject = evalcx(rewrittenFun, sandbox, name);
      } else {
        var transpiled = CoffeeScript.compile(source, {bare: true});
        functionObject = evalcx(transpiled, sandbox, name);
      }
    } catch (err) {
      throw([
        "error",
        "compilation_error",
        errstr(err) + " (" + source + ")"
      ]);
    };
    if (typeof(functionObject) == "function") {
      return functionObject;
    } else {
      throw(["error","compilation_error",
        "Expression does not eval to a function. (" + source.toString() + ")"]);
    };
  },
  recursivelySeal : function(obj) {
    seal(obj);
    for (var propname in obj) {
      if (typeof obj[propname] == "object") {
        arguments.callee(obj[propname]);
      }
    }
  }
};

function errstr(e) {
  // toSource() is a Spidermonkey "special"
  return (e.toSource ? e.toSource() : e.toString());
};

// prints the object as JSON, and rescues and logs any JSON.stringify() related errors
function respond(obj) {
  try {
    print(JSON.stringify(obj));
  } catch(e) {
    log("Error converting object to JSON: " + e.toString());
    log("error on obj: "+ obj.toString());
  }
};

function log(message) {
  // idea: query_server_config option for log level
  if (typeof message == "xml") {
    message = message.toXMLString();
  } else if (typeof message != "string") {
    message = JSON.stringify(message);
  }
  respond(["log", String(message)]);
};

function isArray(obj) {
  return toString.call(obj) === "[object Array]";
}
// Licensed under the Apache License, Version 2.0 (the "License"); you may not
// use this file except in compliance with the License. You may obtain a copy of
// the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
// License for the specific language governing permissions and limitations under
// the License.

var Validate = {
  validate : function(fun, ddoc, args) {
    try {
      fun.apply(ddoc, args);
      respond(1);
    } catch (error) {
      if (error.name && error.stack) {
        throw error;
      }
      respond(error);
    }
  }
};
// Licensed under the Apache License, Version 2.0 (the "License"); you may not
// use this file except in compliance with the License. You may obtain a copy of
// the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
// License for the specific language governing permissions and limitations under
// the License.



var Views = (function() {

  var map_results = []; // holds temporary emitted values during doc map

  function runReduce(reduceFuns, keys, values, rereduce) {
    var code_size = 0;
    for (var i in reduceFuns) {
      var fun_body =  reduceFuns[i];
      code_size += fun_body.length;
      reduceFuns[i] = Couch.compileFunction(fun_body);
    };
    var reductions = new Array(reduceFuns.length);
    for(var i = 0; i < reduceFuns.length; i++) {
      try {
        reductions[i] = reduceFuns[i](keys, values, rereduce);
      } catch (err) {
        handleViewError(err);
        // if the error is not fatal, ignore the results and continue
        reductions[i] = null;
      }
    };
    var reduce_line = JSON.stringify(reductions);
    var reduce_length = reduce_line.length;
    var input_length =  State.line_length - code_size
    // TODO make reduce_limit config into a number
    if (State.query_config && State.query_config.reduce_limit &&
          reduce_length > 4096 && ((reduce_length * 2) > input_length)) {
      var log_message = [
          "Reduce output must shrink more rapidly:",
          "input size:", input_length,
          "output size:", reduce_length
      ].join(" ");
      if (State.query_config.reduce_limit === "log") {
          log("reduce_overflow_error: " + log_message);
          print("[true," + reduce_line + "]");
      } else {
          throw(["error", "reduce_overflow_error", log_message]);
      };
    } else {
      print("[true," + reduce_line + "]");
    }
  };

  function handleViewError(err, doc) {
    if (err == "fatal_error") {
      // Only if it's a "fatal_error" do we exit. What's a fatal error?
      // That's for the query to decide.
      //
      // This will make it possible for queries to completely error out,
      // by catching their own local exception and rethrowing a
      // fatal_error. But by default if they don't do error handling we
      // just eat the exception and carry on.
      //
      // In this case we abort map processing but don't destroy the 
      // JavaScript process. If you need to destroy the JavaScript 
      // process, throw the error form matched by the block below.
      throw(["error", "map_runtime_error", "function raised 'fatal_error'"]);
    } else if (err[0] == "fatal") {
      // Throwing errors of the form ["fatal","error_key","reason"]
      // will kill the OS process. This is not normally what you want.
      throw(err);
    } else if (err.name == "InternalError") {
      throw(["fatal", err.name, err.message]);
    }
    var message = "function raised exception " + errstr(err);
    if (doc) message += " with doc._id " + doc._id;
    log(message);
  };

  return {
    // view helper functions
    emit : function(key, value) {
      map_results.push([key, value]);
    },
    sum : function(values) {
      var rv = 0;
      for (var i in values) {
        rv += values[i];
      }
      return rv;
    },
    reduce : function(reduceFuns, kvs) {
      var keys = new Array(kvs.length);
      var values = new Array(kvs.length);
      for(var i = 0; i < kvs.length; i++) {
          keys[i] = kvs[i][0];
          values[i] = kvs[i][1];
      }
      runReduce(reduceFuns, keys, values, false);
    },
    rereduce : function(reduceFuns, values) {
      runReduce(reduceFuns, null, values, true);
    },
    mapDoc : function(doc) {
      // Compute all the map functions against the document.
      //
      // Each function can output multiple key/value pairs for each document.
      //
      // Example output of map_doc after three functions set by add_fun cmds:
      // [
      //  [["Key","Value"]],                    <- fun 1 returned 1 key value
      //  [],                                   <- fun 2 returned 0 key values
      //  [["Key1","Value1"],["Key2","Value2"]] <- fun 3 returned 2 key values
      // ]
      //

      Couch.recursivelySeal(doc);

      var buf = [];
      for (var fun in State.funs) {
        map_results = [];
        try {
          State.funs[fun](doc);
          buf.push(map_results);
        } catch (err) {
          handleViewError(err, doc);
          // If the error is not fatal, we treat the doc as if it
          // did not emit anything, by buffering an empty array.
          buf.push([]);
        }
      }
      print(JSON.stringify(buf));
    }
  };
})();
/**
 * CoffeeScript Compiler v1.10.0
 * http://coffeescript.org
 *
 * Copyright 2011, Jeremy Ashkenas
 * Released under the MIT License
 */
(function(root){var CoffeeScript=function(){function require(e){return require[e]}return require["./helpers"]=function(){var e={},t={exports:e};return function(){var t,n,i,r,s,o;e.starts=function(e,t,n){return t===e.substr(n,t.length)},e.ends=function(e,t,n){var i;return i=t.length,t===e.substr(e.length-i-(n||0),i)},e.repeat=s=function(e,t){var n;for(n="";t>0;)1&t&&(n+=e),t>>>=1,e+=e;return n},e.compact=function(e){var t,n,i,r;for(r=[],t=0,i=e.length;i>t;t++)n=e[t],n&&r.push(n);return r},e.count=function(e,t){var n,i;if(n=i=0,!t.length)return 1/0;for(;i=1+e.indexOf(t,i);)n++;return n},e.merge=function(e,t){return n(n({},e),t)},n=e.extend=function(e,t){var n,i;for(n in t)i=t[n],e[n]=i;return e},e.flatten=i=function(e){var t,n,r,s;for(n=[],r=0,s=e.length;s>r;r++)t=e[r],"[object Array]"===Object.prototype.toString.call(t)?n=n.concat(i(t)):n.push(t);return n},e.del=function(e,t){var n;return n=e[t],delete e[t],n},e.some=null!=(r=Array.prototype.some)?r:function(e){var t,n,i;for(n=0,i=this.length;i>n;n++)if(t=this[n],e(t))return!0;return!1},e.invertLiterate=function(e){var t,n,i;return i=!0,n=function(){var n,r,s,o;for(s=e.split("\n"),o=[],n=0,r=s.length;r>n;n++)t=s[n],i&&/^([ ]{4}|[ ]{0,3}\t)/.test(t)?o.push(t):(i=/^\s*$/.test(t))?o.push(t):o.push("# "+t);return o}(),n.join("\n")},t=function(e,t){return t?{first_line:e.first_line,first_column:e.first_column,last_line:t.last_line,last_column:t.last_column}:e},e.addLocationDataFn=function(e,n){return function(i){return"object"==typeof i&&i.updateLocationDataIfMissing&&i.updateLocationDataIfMissing(t(e,n)),i}},e.locationDataToString=function(e){var t;return"2"in e&&"first_line"in e[2]?t=e[2]:"first_line"in e&&(t=e),t?t.first_line+1+":"+(t.first_column+1)+"-"+(t.last_line+1+":"+(t.last_column+1)):"No location data"},e.baseFileName=function(e,t,n){var i,r;return null==t&&(t=!1),null==n&&(n=!1),r=n?/\\|\//:/\//,i=e.split(r),e=i[i.length-1],t&&e.indexOf(".")>=0?(i=e.split("."),i.pop(),"coffee"===i[i.length-1]&&i.length>1&&i.pop(),i.join(".")):e},e.isCoffee=function(e){return/\.((lit)?coffee|coffee\.md)$/.test(e)},e.isLiterate=function(e){return/\.(litcoffee|coffee\.md)$/.test(e)},e.throwSyntaxError=function(e,t){var n;throw n=new SyntaxError(e),n.location=t,n.toString=o,n.stack=""+n,n},e.updateSyntaxError=function(e,t,n){return e.toString===o&&(e.code||(e.code=t),e.filename||(e.filename=n),e.stack=""+e),e},o=function(){var e,t,n,i,r,o,a,c,l,h,u,p,d,f,m;return this.code&&this.location?(u=this.location,a=u.first_line,o=u.first_column,l=u.last_line,c=u.last_column,null==l&&(l=a),null==c&&(c=o),r=this.filename||"[stdin]",e=this.code.split("\n")[a],m=o,i=a===l?c+1:e.length,h=e.slice(0,m).replace(/[^\s]/g," ")+s("^",i-m),"undefined"!=typeof process&&null!==process&&(n=(null!=(p=process.stdout)?p.isTTY:void 0)&&!(null!=(d=process.env)?d.NODE_DISABLE_COLORS:void 0)),(null!=(f=this.colorful)?f:n)&&(t=function(e){return"[1;31m"+e+"[0m"},e=e.slice(0,m)+t(e.slice(m,i))+e.slice(i),h=t(h)),r+":"+(a+1)+":"+(o+1)+": error: "+this.message+"\n"+e+"\n"+h):Error.prototype.toString.call(this)},e.nameWhitespaceCharacter=function(e){switch(e){case" ":return"space";case"\n":return"newline";case"\r":return"carriage return";case"	":return"tab";default:return e}}}.call(this),t.exports}(),require["./rewriter"]=function(){var e={},t={exports:e};return function(){var t,n,i,r,s,o,a,c,l,h,u,p,d,f,m,g,v,b,y,k=[].indexOf||function(e){for(var t=0,n=this.length;n>t;t++)if(t in this&&this[t]===e)return t;return-1},w=[].slice;for(f=function(e,t,n){var i;return i=[e,t],i.generated=!0,n&&(i.origin=n),i},e.Rewriter=function(){function e(){}return e.prototype.rewrite=function(e){return this.tokens=e,this.removeLeadingNewlines(),this.closeOpenCalls(),this.closeOpenIndexes(),this.normalizeLines(),this.tagPostfixConditionals(),this.addImplicitBracesAndParens(),this.addLocationDataToGeneratedTokens(),this.tokens},e.prototype.scanTokens=function(e){var t,n,i;for(i=this.tokens,t=0;n=i[t];)t+=e.call(this,n,t,i);return!0},e.prototype.detectEnd=function(e,t,n){var i,o,a,c,l;for(l=this.tokens,i=0;c=l[e];){if(0===i&&t.call(this,c,e))return n.call(this,c,e);if(!c||0>i)return n.call(this,c,e-1);o=c[0],k.call(s,o)>=0?i+=1:(a=c[0],k.call(r,a)>=0&&(i-=1)),e+=1}return e-1},e.prototype.removeLeadingNewlines=function(){var e,t,n,i,r;for(i=this.tokens,e=t=0,n=i.length;n>t&&(r=i[e][0],"TERMINATOR"===r);e=++t);return e?this.tokens.splice(0,e):void 0},e.prototype.closeOpenCalls=function(){var e,t;return t=function(e,t){var n;return")"===(n=e[0])||"CALL_END"===n||"OUTDENT"===e[0]&&")"===this.tag(t-1)},e=function(e,t){return this.tokens["OUTDENT"===e[0]?t-1:t][0]="CALL_END"},this.scanTokens(function(n,i){return"CALL_START"===n[0]&&this.detectEnd(i+1,t,e),1})},e.prototype.closeOpenIndexes=function(){var e,t;return t=function(e){var t;return"]"===(t=e[0])||"INDEX_END"===t},e=function(e){return e[0]="INDEX_END"},this.scanTokens(function(n,i){return"INDEX_START"===n[0]&&this.detectEnd(i+1,t,e),1})},e.prototype.indexOfTag=function(){var e,t,n,i,r,s,o;for(t=arguments[0],r=arguments.length>=2?w.call(arguments,1):[],e=0,n=i=0,s=r.length;s>=0?s>i:i>s;n=s>=0?++i:--i){for(;"HERECOMMENT"===this.tag(t+n+e);)e+=2;if(null!=r[n]&&("string"==typeof r[n]&&(r[n]=[r[n]]),o=this.tag(t+n+e),0>k.call(r[n],o)))return-1}return t+n+e-1},e.prototype.looksObjectish=function(e){var t,n;return this.indexOfTag(e,"@",null,":")>-1||this.indexOfTag(e,null,":")>-1?!0:(n=this.indexOfTag(e,s),n>-1&&(t=null,this.detectEnd(n+1,function(e){var t;return t=e[0],k.call(r,t)>=0},function(e,n){return t=n}),":"===this.tag(t+1))?!0:!1)},e.prototype.findTagsBackwards=function(e,t){var n,i,o,a,c,l,h;for(n=[];e>=0&&(n.length||(a=this.tag(e),0>k.call(t,a)&&(c=this.tag(e),0>k.call(s,c)||this.tokens[e].generated)&&(l=this.tag(e),0>k.call(u,l))));)i=this.tag(e),k.call(r,i)>=0&&n.push(this.tag(e)),o=this.tag(e),k.call(s,o)>=0&&n.length&&n.pop(),e-=1;return h=this.tag(e),k.call(t,h)>=0},e.prototype.addImplicitBracesAndParens=function(){var e,t;return e=[],t=null,this.scanTokens(function(i,h,p){var d,m,g,v,b,y,w,T,C,F,E,N,L,x,S,D,R,A,I,_,O,$,j,M,B,V,P,U;if(U=i[0],E=(N=h>0?p[h-1]:[])[0],C=(p.length-1>h?p[h+1]:[])[0],j=function(){return e[e.length-1]},M=h,g=function(e){return h-M+e},v=function(){var e,t;return null!=(e=j())?null!=(t=e[2])?t.ours:void 0:void 0},b=function(){var e;return v()&&"("===(null!=(e=j())?e[0]:void 0)},w=function(){var e;return v()&&"{"===(null!=(e=j())?e[0]:void 0)},y=function(){var e;return v&&"CONTROL"===(null!=(e=j())?e[0]:void 0)},B=function(t){var n;return n=null!=t?t:h,e.push(["(",n,{ours:!0}]),p.splice(n,0,f("CALL_START","(")),null==t?h+=1:void 0},d=function(){return e.pop(),p.splice(h,0,f("CALL_END",")",["","end of input",i[2]])),h+=1},V=function(t,n){var r,s;return null==n&&(n=!0),r=null!=t?t:h,e.push(["{",r,{sameLine:!0,startsLine:n,ours:!0}]),s=new String("{"),s.generated=!0,p.splice(r,0,f("{",s,i)),null==t?h+=1:void 0},m=function(t){return t=null!=t?t:h,e.pop(),p.splice(t,0,f("}","}",i)),h+=1},b()&&("IF"===U||"TRY"===U||"FINALLY"===U||"CATCH"===U||"CLASS"===U||"SWITCH"===U))return e.push(["CONTROL",h,{ours:!0}]),g(1);if("INDENT"===U&&v()){if("=>"!==E&&"->"!==E&&"["!==E&&"("!==E&&","!==E&&"{"!==E&&"TRY"!==E&&"ELSE"!==E&&"="!==E)for(;b();)d();return y()&&e.pop(),e.push([U,h]),g(1)}if(k.call(s,U)>=0)return e.push([U,h]),g(1);if(k.call(r,U)>=0){for(;v();)b()?d():w()?m():e.pop();t=e.pop()}if((k.call(c,U)>=0&&i.spaced||"?"===U&&h>0&&!p[h-1].spaced)&&(k.call(o,C)>=0||k.call(l,C)>=0&&!(null!=(L=p[h+1])?L.spaced:void 0)&&!(null!=(x=p[h+1])?x.newLine:void 0)))return"?"===U&&(U=i[0]="FUNC_EXIST"),B(h+1),g(2);if(k.call(c,U)>=0&&this.indexOfTag(h+1,"INDENT")>-1&&this.looksObjectish(h+2)&&!this.findTagsBackwards(h,["CLASS","EXTENDS","IF","CATCH","SWITCH","LEADING_WHEN","FOR","WHILE","UNTIL"]))return B(h+1),e.push(["INDENT",h+2]),g(3);if(":"===U){for(I=function(){var e;switch(!1){case e=this.tag(h-1),0>k.call(r,e):return t[1];case"@"!==this.tag(h-2):return h-2;default:return h-1}}.call(this);"HERECOMMENT"===this.tag(I-2);)I-=2;return this.insideForDeclaration="FOR"===C,P=0===I||(S=this.tag(I-1),k.call(u,S)>=0)||p[I-1].newLine,j()&&(D=j(),$=D[0],O=D[1],("{"===$||"INDENT"===$&&"{"===this.tag(O-1))&&(P||","===this.tag(I-1)||"{"===this.tag(I-1)))?g(1):(V(I,!!P),g(2))}if(w()&&k.call(u,U)>=0&&(j()[2].sameLine=!1),T="OUTDENT"===E||N.newLine,k.call(a,U)>=0||k.call(n,U)>=0&&T)for(;v();)if(R=j(),$=R[0],O=R[1],A=R[2],_=A.sameLine,P=A.startsLine,b()&&","!==E)d();else if(w()&&!this.insideForDeclaration&&_&&"TERMINATOR"!==U&&":"!==E)m();else{if(!w()||"TERMINATOR"!==U||","===E||P&&this.looksObjectish(h+1))break;if("HERECOMMENT"===C)return g(1);m()}if(!(","!==U||this.looksObjectish(h+1)||!w()||this.insideForDeclaration||"TERMINATOR"===C&&this.looksObjectish(h+2)))for(F="OUTDENT"===C?1:0;w();)m(h+F);return g(1)})},e.prototype.addLocationDataToGeneratedTokens=function(){return this.scanTokens(function(e,t,n){var i,r,s,o,a,c;return e[2]?1:e.generated||e.explicit?("{"===e[0]&&(s=null!=(a=n[t+1])?a[2]:void 0)?(r=s.first_line,i=s.first_column):(o=null!=(c=n[t-1])?c[2]:void 0)?(r=o.last_line,i=o.last_column):r=i=0,e[2]={first_line:r,first_column:i,last_line:r,last_column:i},1):1})},e.prototype.normalizeLines=function(){var e,t,r,s,o;return o=r=s=null,t=function(e,t){var r,s,a,c;return";"!==e[1]&&(r=e[0],k.call(p,r)>=0)&&!("TERMINATOR"===e[0]&&(s=this.tag(t+1),k.call(i,s)>=0))&&!("ELSE"===e[0]&&"THEN"!==o)&&!!("CATCH"!==(a=e[0])&&"FINALLY"!==a||"->"!==o&&"=>"!==o)||(c=e[0],k.call(n,c)>=0&&this.tokens[t-1].newLine)},e=function(e,t){return this.tokens.splice(","===this.tag(t-1)?t-1:t,0,s)},this.scanTokens(function(n,a,c){var l,h,u,p,f,m;if(m=n[0],"TERMINATOR"===m){if("ELSE"===this.tag(a+1)&&"OUTDENT"!==this.tag(a-1))return c.splice.apply(c,[a,1].concat(w.call(this.indentation()))),1;if(u=this.tag(a+1),k.call(i,u)>=0)return c.splice(a,1),0}if("CATCH"===m)for(l=h=1;2>=h;l=++h)if("OUTDENT"===(p=this.tag(a+l))||"TERMINATOR"===p||"FINALLY"===p)return c.splice.apply(c,[a+l,0].concat(w.call(this.indentation()))),2+l;return k.call(d,m)>=0&&"INDENT"!==this.tag(a+1)&&("ELSE"!==m||"IF"!==this.tag(a+1))?(o=m,f=this.indentation(c[a]),r=f[0],s=f[1],"THEN"===o&&(r.fromThen=!0),c.splice(a+1,0,r),this.detectEnd(a+2,t,e),"THEN"===m&&c.splice(a,1),1):1})},e.prototype.tagPostfixConditionals=function(){var e,t,n;return n=null,t=function(e,t){var n,i;return i=e[0],n=this.tokens[t-1][0],"TERMINATOR"===i||"INDENT"===i&&0>k.call(d,n)},e=function(e){return"INDENT"!==e[0]||e.generated&&!e.fromThen?n[0]="POST_"+n[0]:void 0},this.scanTokens(function(i,r){return"IF"!==i[0]?1:(n=i,this.detectEnd(r+1,t,e),1)})},e.prototype.indentation=function(e){var t,n;return t=["INDENT",2],n=["OUTDENT",2],e?(t.generated=n.generated=!0,t.origin=n.origin=e):t.explicit=n.explicit=!0,[t,n]},e.prototype.generate=f,e.prototype.tag=function(e){var t;return null!=(t=this.tokens[e])?t[0]:void 0},e}(),t=[["(",")"],["[","]"],["{","}"],["INDENT","OUTDENT"],["CALL_START","CALL_END"],["PARAM_START","PARAM_END"],["INDEX_START","INDEX_END"],["STRING_START","STRING_END"],["REGEX_START","REGEX_END"]],e.INVERSES=h={},s=[],r=[],m=0,v=t.length;v>m;m++)b=t[m],g=b[0],y=b[1],s.push(h[y]=g),r.push(h[g]=y);i=["CATCH","THEN","ELSE","FINALLY"].concat(r),c=["IDENTIFIER","SUPER",")","CALL_END","]","INDEX_END","@","THIS"],o=["IDENTIFIER","NUMBER","STRING","STRING_START","JS","REGEX","REGEX_START","NEW","PARAM_START","CLASS","IF","TRY","SWITCH","THIS","BOOL","NULL","UNDEFINED","UNARY","YIELD","UNARY_MATH","SUPER","THROW","@","->","=>","[","(","{","--","++"],l=["+","-"],a=["POST_IF","FOR","WHILE","UNTIL","WHEN","BY","LOOP","TERMINATOR"],d=["ELSE","->","=>","TRY","FINALLY","THEN"],p=["TERMINATOR","CATCH","FINALLY","ELSE","OUTDENT","LEADING_WHEN"],u=["TERMINATOR","INDENT","OUTDENT"],n=[".","?.","::","?::"]}.call(this),t.exports}(),require["./lexer"]=function(){var e={},t={exports:e};return function(){var t,n,i,r,s,o,a,c,l,h,u,p,d,f,m,g,v,b,y,k,w,T,C,F,E,N,L,x,S,D,R,A,I,_,O,$,j,M,B,V,P,U,G,H,q,X,W,Y,K,z,J,Q,Z,et,tt,nt,it,rt,st,ot,at,ct,lt,ht,ut=[].indexOf||function(e){for(var t=0,n=this.length;n>t;t++)if(t in this&&this[t]===e)return t;return-1};ot=require("./rewriter"),P=ot.Rewriter,w=ot.INVERSES,at=require("./helpers"),nt=at.count,lt=at.starts,tt=at.compact,ct=at.repeat,it=at.invertLiterate,st=at.locationDataToString,ht=at.throwSyntaxError,e.Lexer=S=function(){function e(){}return e.prototype.tokenize=function(e,t){var n,i,r,s;for(null==t&&(t={}),this.literate=t.literate,this.indent=0,this.baseIndent=0,this.indebt=0,this.outdebt=0,this.indents=[],this.ends=[],this.tokens=[],this.seenFor=!1,this.chunkLine=t.line||0,this.chunkColumn=t.column||0,e=this.clean(e),r=0;this.chunk=e.slice(r);)if(n=this.identifierToken()||this.commentToken()||this.whitespaceToken()||this.lineToken()||this.stringToken()||this.numberToken()||this.regexToken()||this.jsToken()||this.literalToken(),s=this.getLineAndColumnFromChunk(n),this.chunkLine=s[0],this.chunkColumn=s[1],r+=n,t.untilBalanced&&0===this.ends.length)return{tokens:this.tokens,index:r};return this.closeIndentation(),(i=this.ends.pop())&&this.error("missing "+i.tag,i.origin[2]),t.rewrite===!1?this.tokens:(new P).rewrite(this.tokens)},e.prototype.clean=function(e){return e.charCodeAt(0)===t&&(e=e.slice(1)),e=e.replace(/\r/g,"").replace(z,""),et.test(e)&&(e="\n"+e,this.chunkLine--),this.literate&&(e=it(e)),e},e.prototype.identifierToken=function(){var e,t,n,i,r,c,l,h,u,p,d,f,m,g,b,y;return(h=v.exec(this.chunk))?(l=h[0],r=h[1],t=h[2],c=r.length,u=void 0,"own"===r&&"FOR"===this.tag()?(this.token("OWN",r),r.length):"from"===r&&"YIELD"===this.tag()?(this.token("FROM",r),r.length):(d=this.tokens,p=d[d.length-1],i=t||null!=p&&("."===(f=p[0])||"?."===f||"::"===f||"?::"===f||!p.spaced&&"@"===p[0]),b="IDENTIFIER",!i&&(ut.call(F,r)>=0||ut.call(a,r)>=0)&&(b=r.toUpperCase(),"WHEN"===b&&(m=this.tag(),ut.call(N,m)>=0)?b="LEADING_WHEN":"FOR"===b?this.seenFor=!0:"UNLESS"===b?b="IF":ut.call(J,b)>=0?b="UNARY":ut.call(B,b)>=0&&("INSTANCEOF"!==b&&this.seenFor?(b="FOR"+b,this.seenFor=!1):(b="RELATION","!"===this.value()&&(u=this.tokens.pop(),r="!"+r)))),ut.call(C,r)>=0&&(i?(b="IDENTIFIER",r=new String(r),r.reserved=!0):ut.call(V,r)>=0&&this.error("reserved word '"+r+"'",{length:r.length})),i||(ut.call(s,r)>=0&&(e=r,r=o[r]),b=function(){switch(r){case"!":return"UNARY";case"==":case"!=":return"COMPARE";case"&&":case"||":return"LOGIC";case"true":case"false":return"BOOL";case"break":case"continue":return"STATEMENT";default:return b}}()),y=this.token(b,r,0,c),e&&(y.origin=[b,e,y[2]]),y.variable=!i,u&&(g=[u[2].first_line,u[2].first_column],y[2].first_line=g[0],y[2].first_column=g[1]),t&&(n=l.lastIndexOf(":"),this.token(":",":",n,t.length)),l.length)):0},e.prototype.numberToken=function(){var e,t,n,i,r;return(n=I.exec(this.chunk))?(i=n[0],t=i.length,/^0[BOX]/.test(i)?this.error("radix prefix in '"+i+"' must be lowercase",{offset:1}):/E/.test(i)&&!/^0x/.test(i)?this.error("exponential notation in '"+i+"' must be indicated with a lowercase 'e'",{offset:i.indexOf("E")}):/^0\d*[89]/.test(i)?this.error("decimal literal '"+i+"' must not be prefixed with '0'",{length:t}):/^0\d+/.test(i)&&this.error("octal literal '"+i+"' must be prefixed with '0o'",{length:t}),(r=/^0o([0-7]+)/.exec(i))&&(i="0x"+parseInt(r[1],8).toString(16)),(e=/^0b([01]+)/.exec(i))&&(i="0x"+parseInt(e[1],2).toString(16)),this.token("NUMBER",i,0,t),t):0},e.prototype.stringToken=function(){var e,t,n,i,r,s,o,a,c,l,h,u,m,g,v,b;if(h=(Y.exec(this.chunk)||[])[0],!h)return 0;if(g=function(){switch(h){case"'":return W;case'"':return q;case"'''":return f;case'"""':return p}}(),s=3===h.length,u=this.matchWithInterpolations(g,h),b=u.tokens,r=u.index,e=b.length-1,n=h.charAt(0),s){for(a=null,i=function(){var e,t,n;for(n=[],o=e=0,t=b.length;t>e;o=++e)v=b[o],"NEOSTRING"===v[0]&&n.push(v[1]);return n}().join("#{}");l=d.exec(i);)t=l[1],(null===a||(m=t.length)>0&&a.length>m)&&(a=t);a&&(c=RegExp("^"+a,"gm")),this.mergeInterpolationTokens(b,{delimiter:n},function(t){return function(n,i){return n=t.formatString(n),0===i&&(n=n.replace(E,"")),i===e&&(n=n.replace(K,"")),c&&(n=n.replace(c,"")),n}}(this))}else this.mergeInterpolationTokens(b,{delimiter:n},function(t){return function(n,i){return n=t.formatString(n),n=n.replace(G,function(t,r){return 0===i&&0===r||i===e&&r+t.length===n.length?"":" "})}}(this));return r},e.prototype.commentToken=function(){var e,t,n;return(n=this.chunk.match(c))?(e=n[0],t=n[1],t&&((n=u.exec(e))&&this.error("block comments cannot contain "+n[0],{offset:n.index,length:n[0].length}),t.indexOf("\n")>=0&&(t=t.replace(RegExp("\\n"+ct(" ",this.indent),"g"),"\n")),this.token("HERECOMMENT",t,0,e.length)),e.length):0},e.prototype.jsToken=function(){var e,t;return"`"===this.chunk.charAt(0)&&(e=T.exec(this.chunk))?(this.token("JS",(t=e[0]).slice(1,-1),0,t.length),t.length):0},e.prototype.regexToken=function(){var e,t,n,r,s,o,a,c,l,h,u,p,d;switch(!1){case!(o=M.exec(this.chunk)):this.error("regular expressions cannot begin with "+o[2],{offset:o.index+o[1].length});break;case!(o=this.matchWithInterpolations(m,"///")):d=o.tokens,s=o.index;break;case!(o=$.exec(this.chunk)):if(p=o[0],e=o[1],t=o[2],this.validateEscapes(e,{isRegex:!0,offsetInChunk:1}),s=p.length,l=this.tokens,c=l[l.length-1],c)if(c.spaced&&(h=c[0],ut.call(i,h)>=0)){if(!t||O.test(p))return 0}else if(u=c[0],ut.call(A,u)>=0)return 0;t||this.error("missing / (unclosed regex)");break;default:return 0}switch(r=j.exec(this.chunk.slice(s))[0],n=s+r.length,a=this.makeToken("REGEX",null,0,n),!1){case!!Z.test(r):this.error("invalid regular expression flags "+r,{offset:s,length:r.length});break;case!(p||1===d.length):null==e&&(e=this.formatHeregex(d[0][1])),this.token("REGEX",""+this.makeDelimitedLiteral(e,{delimiter:"/"})+r,0,n,a);break;default:this.token("REGEX_START","(",0,0,a),this.token("IDENTIFIER","RegExp",0,0),this.token("CALL_START","(",0,0),this.mergeInterpolationTokens(d,{delimiter:'"',"double":!0},this.formatHeregex),r&&(this.token(",",",",s,0),this.token("STRING",'"'+r+'"',s,r.length)),this.token(")",")",n,0),this.token("REGEX_END",")",n,0)}return n},e.prototype.lineToken=function(){var e,t,n,i,r;if(!(n=R.exec(this.chunk)))return 0;if(t=n[0],this.seenFor=!1,r=t.length-1-t.lastIndexOf("\n"),i=this.unfinished(),r-this.indebt===this.indent)return i?this.suppressNewlines():this.newlineToken(0),t.length;if(r>this.indent){if(i)return this.indebt=r-this.indent,this.suppressNewlines(),t.length;if(!this.tokens.length)return this.baseIndent=this.indent=r,t.length;e=r-this.indent+this.outdebt,this.token("INDENT",e,t.length-r,r),this.indents.push(e),this.ends.push({tag:"OUTDENT"}),this.outdebt=this.indebt=0,this.indent=r}else this.baseIndent>r?this.error("missing indentation",{offset:t.length}):(this.indebt=0,this.outdentToken(this.indent-r,i,t.length));return t.length},e.prototype.outdentToken=function(e,t,n){var i,r,s,o;for(i=this.indent-e;e>0;)s=this.indents[this.indents.length-1],s?s===this.outdebt?(e-=this.outdebt,this.outdebt=0):this.outdebt>s?(this.outdebt-=s,e-=s):(r=this.indents.pop()+this.outdebt,n&&(o=this.chunk[n],ut.call(b,o)>=0)&&(i-=r-e,e=r),this.outdebt=0,this.pair("OUTDENT"),this.token("OUTDENT",e,0,n),e-=r):e=0;for(r&&(this.outdebt-=e);";"===this.value();)this.tokens.pop();return"TERMINATOR"===this.tag()||t||this.token("TERMINATOR","\n",n,0),this.indent=i,this},e.prototype.whitespaceToken=function(){var e,t,n,i;return(e=et.exec(this.chunk))||(t="\n"===this.chunk.charAt(0))?(i=this.tokens,n=i[i.length-1],n&&(n[e?"spaced":"newLine"]=!0),e?e[0].length:0):0},e.prototype.newlineToken=function(e){for(;";"===this.value();)this.tokens.pop();return"TERMINATOR"!==this.tag()&&this.token("TERMINATOR","\n",e,0),this},e.prototype.suppressNewlines=function(){return"\\"===this.value()&&this.tokens.pop(),this},e.prototype.literalToken=function(){var e,t,n,s,o,a,c,u,p,d;if((e=_.exec(this.chunk))?(d=e[0],r.test(d)&&this.tagParameters()):d=this.chunk.charAt(0),u=d,n=this.tokens,t=n[n.length-1],"="===d&&t&&(!t[1].reserved&&(s=t[1],ut.call(C,s)>=0)&&(t.origin&&(t=t.origin),this.error("reserved word '"+t[1]+"' can't be assigned",t[2])),"||"===(o=t[1])||"&&"===o))return t[0]="COMPOUND_ASSIGN",t[1]+="=",d.length;if(";"===d)this.seenFor=!1,u="TERMINATOR";else if(ut.call(D,d)>=0)u="MATH";else if(ut.call(l,d)>=0)u="COMPARE";else if(ut.call(h,d)>=0)u="COMPOUND_ASSIGN";else if(ut.call(J,d)>=0)u="UNARY";else if(ut.call(Q,d)>=0)u="UNARY_MATH";else if(ut.call(U,d)>=0)u="SHIFT";else if(ut.call(x,d)>=0||"?"===d&&(null!=t?t.spaced:void 0))u="LOGIC";else if(t&&!t.spaced)if("("===d&&(a=t[0],ut.call(i,a)>=0))"?"===t[0]&&(t[0]="FUNC_EXIST"),u="CALL_START";else if("["===d&&(c=t[0],ut.call(y,c)>=0))switch(u="INDEX_START",t[0]){case"?":t[0]="INDEX_SOAK"}switch(p=this.makeToken(u,d),d){case"(":case"{":case"[":this.ends.push({tag:w[d],origin:p});break;case")":case"}":case"]":this.pair(d)}return this.tokens.push(p),d.length},e.prototype.tagParameters=function(){var e,t,n,i;if(")"!==this.tag())return this;for(t=[],i=this.tokens,e=i.length,i[--e][0]="PARAM_END";n=i[--e];)switch(n[0]){case")":t.push(n);break;case"(":case"CALL_START":if(!t.length)return"("===n[0]?(n[0]="PARAM_START",this):this;t.pop()}return this},e.prototype.closeIndentation=function(){return this.outdentToken(this.indent)},e.prototype.matchWithInterpolations=function(t,n){var i,r,s,o,a,c,l,h,u,p,d,f,m,g,v;if(v=[],h=n.length,this.chunk.slice(0,h)!==n)return null;for(m=this.chunk.slice(h);;){if(g=t.exec(m)[0],this.validateEscapes(g,{isRegex:"/"===n.charAt(0),offsetInChunk:h}),v.push(this.makeToken("NEOSTRING",g,h)),m=m.slice(g.length),h+=g.length,"#{"!==m.slice(0,2))break;p=this.getLineAndColumnFromChunk(h+1),c=p[0],r=p[1],d=(new e).tokenize(m.slice(1),{line:c,column:r,untilBalanced:!0}),l=d.tokens,o=d.index,o+=1,u=l[0],i=l[l.length-1],u[0]=u[1]="(",i[0]=i[1]=")",i.origin=["","end of interpolation",i[2]],"TERMINATOR"===(null!=(f=l[1])?f[0]:void 0)&&l.splice(1,1),v.push(["TOKENS",l]),m=m.slice(o),h+=o}return m.slice(0,n.length)!==n&&this.error("missing "+n,{length:n.length}),s=v[0],a=v[v.length-1],s[2].first_column-=n.length,a[2].last_column+=n.length,0===a[1].length&&(a[2].last_column-=1),{tokens:v,index:h+n.length}},e.prototype.mergeInterpolationTokens=function(e,t,n){var i,r,s,o,a,c,l,h,u,p,d,f,m,g,v,b;for(e.length>1&&(u=this.token("STRING_START","(",0,0)),s=this.tokens.length,o=a=0,l=e.length;l>a;o=++a){switch(g=e[o],m=g[0],b=g[1],m){case"TOKENS":if(2===b.length)continue;h=b[0],v=b;break;case"NEOSTRING":if(i=n(g[1],o),0===i.length){if(0!==o)continue;r=this.tokens.length}2===o&&null!=r&&this.tokens.splice(r,2),g[0]="STRING",g[1]=this.makeDelimitedLiteral(i,t),h=g,v=[g]}this.tokens.length>s&&(p=this.token("+","+"),p[2]={first_line:h[2].first_line,first_column:h[2].first_column,last_line:h[2].first_line,last_column:h[2].first_column}),(d=this.tokens).push.apply(d,v)}return u?(c=e[e.length-1],u.origin=["STRING",null,{first_line:u[2].first_line,first_column:u[2].first_column,last_line:c[2].last_line,last_column:c[2].last_column}],f=this.token("STRING_END",")"),f[2]={first_line:c[2].last_line,first_column:c[2].last_column,last_line:c[2].last_line,last_column:c[2].last_column}):void 0},e.prototype.pair=function(e){var t,n,i,r,s;return i=this.ends,n=i[i.length-1],e!==(s=null!=n?n.tag:void 0)?("OUTDENT"!==s&&this.error("unmatched "+e),r=this.indents,t=r[r.length-1],this.outdentToken(t,!0),this.pair(e)):this.ends.pop()},e.prototype.getLineAndColumnFromChunk=function(e){var t,n,i,r,s;return 0===e?[this.chunkLine,this.chunkColumn]:(s=e>=this.chunk.length?this.chunk:this.chunk.slice(0,+(e-1)+1||9e9),i=nt(s,"\n"),t=this.chunkColumn,i>0?(r=s.split("\n"),n=r[r.length-1],t=n.length):t+=s.length,[this.chunkLine+i,t])},e.prototype.makeToken=function(e,t,n,i){var r,s,o,a,c;return null==n&&(n=0),null==i&&(i=t.length),s={},o=this.getLineAndColumnFromChunk(n),s.first_line=o[0],s.first_column=o[1],r=Math.max(0,i-1),a=this.getLineAndColumnFromChunk(n+r),s.last_line=a[0],s.last_column=a[1],c=[e,t,s]},e.prototype.token=function(e,t,n,i,r){var s;return s=this.makeToken(e,t,n,i),r&&(s.origin=r),this.tokens.push(s),s},e.prototype.tag=function(){var e,t;return e=this.tokens,t=e[e.length-1],null!=t?t[0]:void 0},e.prototype.value=function(){var e,t;return e=this.tokens,t=e[e.length-1],null!=t?t[1]:void 0},e.prototype.unfinished=function(){var e;return L.test(this.chunk)||"\\"===(e=this.tag())||"."===e||"?."===e||"?::"===e||"UNARY"===e||"MATH"===e||"UNARY_MATH"===e||"+"===e||"-"===e||"YIELD"===e||"**"===e||"SHIFT"===e||"RELATION"===e||"COMPARE"===e||"LOGIC"===e||"THROW"===e||"EXTENDS"===e},e.prototype.formatString=function(e){return e.replace(X,"$1")},e.prototype.formatHeregex=function(e){return e.replace(g,"$1$2")},e.prototype.validateEscapes=function(e,t){var n,i,r,s,o,a,c,l;return null==t&&(t={}),s=k.exec(e),!s||(s[0],n=s[1],a=s[2],i=s[3],l=s[4],t.isRegex&&a&&"0"!==a.charAt(0))?void 0:(o=a?"octal escape sequences are not allowed":"invalid escape sequence",r="\\"+(a||i||l),this.error(o+" "+r,{offset:(null!=(c=t.offsetInChunk)?c:0)+s.index+n.length,length:r.length}))},e.prototype.makeDelimitedLiteral=function(e,t){var n;return null==t&&(t={}),""===e&&"/"===t.delimiter&&(e="(?:)"),n=RegExp("(\\\\\\\\)|(\\\\0(?=[1-7]))|\\\\?("+t.delimiter+")|\\\\?(?:(\\n)|(\\r)|(\\u2028)|(\\u2029))|(\\\\.)","g"),e=e.replace(n,function(e,n,i,r,s,o,a,c,l){switch(!1){case!n:return t.double?n+n:n;case!i:return"\\x00";case!r:return"\\"+r;case!s:return"\\n";case!o:return"\\r";case!a:return"\\u2028";case!c:return"\\u2029";case!l:return t.double?"\\"+l:l}}),""+t.delimiter+e+t.delimiter},e.prototype.error=function(e,t){var n,i,r,s,o,a;return null==t&&(t={}),r="first_line"in t?t:(o=this.getLineAndColumnFromChunk(null!=(s=t.offset)?s:0),i=o[0],n=o[1],o,{first_line:i,first_column:n,last_column:n+(null!=(a=t.length)?a:1)-1}),ht(e,r)},e}(),F=["true","false","null","this","new","delete","typeof","in","instanceof","return","throw","break","continue","debugger","yield","if","else","switch","for","while","do","try","catch","finally","class","extends","super"],a=["undefined","then","unless","until","loop","of","by","when"],o={and:"&&",or:"||",is:"==",isnt:"!=",not:"!",yes:"true",no:"false",on:"true",off:"false"},s=function(){var e;e=[];for(rt in o)e.push(rt);return e}(),a=a.concat(s),V=["case","default","function","var","void","with","const","let","enum","export","import","native","implements","interface","package","private","protected","public","static"],H=["arguments","eval","yield*"],C=F.concat(V).concat(H),e.RESERVED=V.concat(F).concat(a).concat(H),e.STRICT_PROSCRIBED=H,t=65279,v=/^(?!\d)((?:(?!\s)[$\w\x7f-\uffff])+)([^\n\S]*:(?!:))?/,I=/^0b[01]+|^0o[0-7]+|^0x[\da-f]+|^\d*\.?\d+(?:e[+-]?\d+)?/i,_=/^(?:[-=]>|[-+*\/%<>&|^!?=]=|>>>=?|([-+:])\1|([&|<>*\/%])\2=?|\?(\.|::)|\.{2,3})/,et=/^[^\n\S]+/,c=/^###([^#][\s\S]*?)(?:###[^\n\S]*|###$)|^(?:\s*#(?!##[^#]).*)+/,r=/^[-=]>/,R=/^(?:\n[^\n\S]*)+/,T=/^`[^\\`]*(?:\\.[^\\`]*)*`/,Y=/^(?:'''|"""|'|")/,W=/^(?:[^\\']|\\[\s\S])*/,q=/^(?:[^\\"#]|\\[\s\S]|\#(?!\{))*/,f=/^(?:[^\\']|\\[\s\S]|'(?!''))*/,p=/^(?:[^\\"#]|\\[\s\S]|"(?!"")|\#(?!\{))*/,X=/((?:\\\\)+)|\\[^\S\n]*\n\s*/g,G=/\s*\n\s*/g,d=/\n+([^\n\S]*)(?=\S)/g,$=/^\/(?!\/)((?:[^[\/\n\\]|\\[^\n]|\[(?:\\[^\n]|[^\]\n\\])*\])*)(\/)?/,j=/^\w*/,Z=/^(?!.*(.).*\1)[imgy]*$/,m=/^(?:[^\\\/#]|\\[\s\S]|\/(?!\/\/)|\#(?!\{))*/,g=/((?:\\\\)+)|\\(\s)|\s+(?:#.*)?/g,M=/^(\/|\/{3}\s*)(\*)/,O=/^\/=?\s/,u=/\*\//,L=/^\s*(?:,|\??\.(?![.\d])|::)/,k=/((?:^|[^\\])(?:\\\\)*)\\(?:(0[0-7]|[1-7])|(x(?![\da-fA-F]{2}).{0,2})|(u(?![\da-fA-F]{4}).{0,4}))/,E=/^[^\n\S]*\n/,K=/\n[^\n\S]*$/,z=/\s+$/,h=["-=","+=","/=","*=","%=","||=","&&=","?=","<<=",">>=",">>>=","&=","^=","|=","**=","//=","%%="],J=["NEW","TYPEOF","DELETE","DO"],Q=["!","~"],x=["&&","||","&","|","^"],U=["<<",">>",">>>"],l=["==","!=","<",">","<=",">="],D=["*","/","%","//","%%"],B=["IN","OF","INSTANCEOF"],n=["TRUE","FALSE"],i=["IDENTIFIER",")","]","?","@","THIS","SUPER"],y=i.concat(["NUMBER","STRING","STRING_END","REGEX","REGEX_END","BOOL","NULL","UNDEFINED","}","::"]),A=y.concat(["++","--"]),N=["INDENT","OUTDENT","TERMINATOR"],b=[")","}","]"]}.call(this),t.exports}(),require["./parser"]=function(){var e={},t={exports:e},n=function(){function e(){this.yy={}}var t=function(e,t,n,i){for(n=n||{},i=e.length;i--;n[e[i]]=t);return n},n=[1,20],i=[1,75],r=[1,71],s=[1,76],o=[1,77],a=[1,73],c=[1,74],l=[1,50],h=[1,52],u=[1,53],p=[1,54],d=[1,55],f=[1,45],m=[1,46],g=[1,27],v=[1,60],b=[1,61],y=[1,70],k=[1,43],w=[1,26],T=[1,58],C=[1,59],F=[1,57],E=[1,38],N=[1,44],L=[1,56],x=[1,65],S=[1,66],D=[1,67],R=[1,68],A=[1,42],I=[1,64],_=[1,29],O=[1,30],$=[1,31],j=[1,32],M=[1,33],B=[1,34],V=[1,35],P=[1,78],U=[1,6,26,34,109],G=[1,88],H=[1,81],q=[1,80],X=[1,79],W=[1,82],Y=[1,83],K=[1,84],z=[1,85],J=[1,86],Q=[1,87],Z=[1,91],et=[1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,118,133,136,137,142,143,144,145,146,147,148],tt=[1,97],nt=[1,98],it=[1,99],rt=[1,100],st=[1,102],ot=[1,103],at=[1,96],ct=[2,115],lt=[1,6,25,26,34,56,61,64,73,74,75,76,78,80,81,85,91,92,93,98,100,109,111,112,113,117,118,133,136,137,142,143,144,145,146,147,148],ht=[2,82],ut=[1,108],pt=[2,61],dt=[1,112],ft=[1,117],mt=[1,118],gt=[1,120],vt=[1,6,25,26,34,46,56,61,64,73,74,75,76,78,80,81,85,91,92,93,98,100,109,111,112,113,117,118,133,136,137,142,143,144,145,146,147,148],bt=[2,79],yt=[1,6,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,118,133,136,137,142,143,144,145,146,147,148],kt=[1,155],wt=[1,157],Tt=[1,152],Ct=[1,6,25,26,34,46,56,61,64,73,74,75,76,78,80,81,85,87,91,92,93,98,100,109,111,112,113,117,118,133,136,137,140,141,142,143,144,145,146,147,148,149],Ft=[2,98],Et=[1,6,25,26,34,49,56,61,64,73,74,75,76,78,80,81,85,91,92,93,98,100,109,111,112,113,117,118,133,136,137,142,143,144,145,146,147,148],Nt=[1,6,25,26,34,46,49,56,61,64,73,74,75,76,78,80,81,85,87,91,92,93,98,100,109,111,112,113,117,118,124,125,133,136,137,140,141,142,143,144,145,146,147,148,149],Lt=[1,207],xt=[1,206],St=[1,6,25,26,34,38,56,61,64,73,74,75,76,78,80,81,85,91,92,93,98,100,109,111,112,113,117,118,133,136,137,142,143,144,145,146,147,148],Dt=[2,59],Rt=[1,217],At=[6,25,26,56,61],It=[6,25,26,46,56,61,64],_t=[1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,118,133,136,137,143,145,146,147,148],Ot=[1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,118,133],$t=[73,74,75,76,78,81,91,92],jt=[1,236],Mt=[2,136],Bt=[1,6,25,26,34,46,56,61,64,73,74,75,76,78,80,81,85,91,92,93,98,100,109,111,112,113,117,118,124,125,133,136,137,142,143,144,145,146,147,148],Vt=[1,245],Pt=[6,25,26,61,93,98],Ut=[1,6,25,26,34,56,61,64,80,85,93,98,100,109,118,133],Gt=[1,6,25,26,34,56,61,64,80,85,93,98,100,109,112,118,133],Ht=[124,125],qt=[61,124,125],Xt=[1,256],Wt=[6,25,26,61,85],Yt=[6,25,26,49,61,85],Kt=[6,25,26,46,49,61,85],zt=[1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,118,133,136,137,145,146,147,148],Jt=[11,28,30,32,33,36,37,40,41,42,43,44,52,53,54,58,59,80,83,86,90,95,96,97,103,107,108,111,113,115,117,126,132,134,135,136,137,138,140,141],Qt=[2,125],Zt=[6,25,26],en=[2,60],tn=[1,270],nn=[1,271],rn=[1,6,25,26,34,56,61,64,80,85,93,98,100,105,106,109,111,112,113,117,118,128,130,133,136,137,142,143,144,145,146,147,148],sn=[26,128,130],on=[1,6,26,34,56,61,64,80,85,93,98,100,109,112,118,133],an=[2,74],cn=[1,293],ln=[1,294],hn=[1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,118,128,133,136,137,142,143,144,145,146,147,148],un=[1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,113,117,118,133],pn=[1,305],dn=[1,306],fn=[6,25,26,61],mn=[1,6,25,26,34,56,61,64,80,85,93,98,100,105,109,111,112,113,117,118,133,136,137,142,143,144,145,146,147,148],gn=[25,61],vn={trace:function(){},yy:{},symbols_:{error:2,Root:3,Body:4,Line:5,TERMINATOR:6,Expression:7,Statement:8,Return:9,Comment:10,STATEMENT:11,Value:12,Invocation:13,Code:14,Operation:15,Assign:16,If:17,Try:18,While:19,For:20,Switch:21,Class:22,Throw:23,Block:24,INDENT:25,OUTDENT:26,Identifier:27,IDENTIFIER:28,AlphaNumeric:29,NUMBER:30,String:31,STRING:32,STRING_START:33,STRING_END:34,Regex:35,REGEX:36,REGEX_START:37,REGEX_END:38,Literal:39,JS:40,DEBUGGER:41,UNDEFINED:42,NULL:43,BOOL:44,Assignable:45,"=":46,AssignObj:47,ObjAssignable:48,":":49,SimpleObjAssignable:50,ThisProperty:51,RETURN:52,HERECOMMENT:53,PARAM_START:54,ParamList:55,PARAM_END:56,FuncGlyph:57,"->":58,"=>":59,OptComma:60,",":61,Param:62,ParamVar:63,"...":64,Array:65,Object:66,Splat:67,SimpleAssignable:68,Accessor:69,Parenthetical:70,Range:71,This:72,".":73,"?.":74,"::":75,"?::":76,Index:77,INDEX_START:78,IndexValue:79,INDEX_END:80,INDEX_SOAK:81,Slice:82,"{":83,AssignList:84,"}":85,CLASS:86,EXTENDS:87,OptFuncExist:88,Arguments:89,SUPER:90,FUNC_EXIST:91,CALL_START:92,CALL_END:93,ArgList:94,THIS:95,"@":96,"[":97,"]":98,RangeDots:99,"..":100,Arg:101,SimpleArgs:102,TRY:103,Catch:104,FINALLY:105,CATCH:106,THROW:107,"(":108,")":109,WhileSource:110,WHILE:111,WHEN:112,UNTIL:113,Loop:114,LOOP:115,ForBody:116,FOR:117,BY:118,ForStart:119,ForSource:120,ForVariables:121,OWN:122,ForValue:123,FORIN:124,FOROF:125,SWITCH:126,Whens:127,ELSE:128,When:129,LEADING_WHEN:130,IfBlock:131,IF:132,POST_IF:133,UNARY:134,UNARY_MATH:135,"-":136,"+":137,YIELD:138,FROM:139,"--":140,"++":141,"?":142,MATH:143,"**":144,SHIFT:145,COMPARE:146,LOGIC:147,RELATION:148,COMPOUND_ASSIGN:149,$accept:0,$end:1},terminals_:{2:"error",6:"TERMINATOR",11:"STATEMENT",25:"INDENT",26:"OUTDENT",28:"IDENTIFIER",30:"NUMBER",32:"STRING",33:"STRING_START",34:"STRING_END",36:"REGEX",37:"REGEX_START",38:"REGEX_END",40:"JS",41:"DEBUGGER",42:"UNDEFINED",43:"NULL",44:"BOOL",46:"=",49:":",52:"RETURN",53:"HERECOMMENT",54:"PARAM_START",56:"PARAM_END",58:"->",59:"=>",61:",",64:"...",73:".",74:"?.",75:"::",76:"?::",78:"INDEX_START",80:"INDEX_END",81:"INDEX_SOAK",83:"{",85:"}",86:"CLASS",87:"EXTENDS",90:"SUPER",91:"FUNC_EXIST",92:"CALL_START",93:"CALL_END",95:"THIS",96:"@",97:"[",98:"]",100:"..",103:"TRY",105:"FINALLY",106:"CATCH",107:"THROW",108:"(",109:")",111:"WHILE",112:"WHEN",113:"UNTIL",115:"LOOP",117:"FOR",118:"BY",122:"OWN",124:"FORIN",125:"FOROF",126:"SWITCH",128:"ELSE",130:"LEADING_WHEN",132:"IF",133:"POST_IF",134:"UNARY",135:"UNARY_MATH",136:"-",137:"+",138:"YIELD",139:"FROM",140:"--",141:"++",142:"?",143:"MATH",144:"**",145:"SHIFT",146:"COMPARE",147:"LOGIC",148:"RELATION",149:"COMPOUND_ASSIGN"},productions_:[0,[3,0],[3,1],[4,1],[4,3],[4,2],[5,1],[5,1],[8,1],[8,1],[8,1],[7,1],[7,1],[7,1],[7,1],[7,1],[7,1],[7,1],[7,1],[7,1],[7,1],[7,1],[7,1],[24,2],[24,3],[27,1],[29,1],[29,1],[31,1],[31,3],[35,1],[35,3],[39,1],[39,1],[39,1],[39,1],[39,1],[39,1],[39,1],[16,3],[16,4],[16,5],[47,1],[47,3],[47,5],[47,3],[47,5],[47,1],[50,1],[50,1],[48,1],[48,1],[9,2],[9,1],[10,1],[14,5],[14,2],[57,1],[57,1],[60,0],[60,1],[55,0],[55,1],[55,3],[55,4],[55,6],[62,1],[62,2],[62,3],[62,1],[63,1],[63,1],[63,1],[63,1],[67,2],[68,1],[68,2],[68,2],[68,1],[45,1],[45,1],[45,1],[12,1],[12,1],[12,1],[12,1],[12,1],[69,2],[69,2],[69,2],[69,2],[69,1],[69,1],[77,3],[77,2],[79,1],[79,1],[66,4],[84,0],[84,1],[84,3],[84,4],[84,6],[22,1],[22,2],[22,3],[22,4],[22,2],[22,3],[22,4],[22,5],[13,3],[13,3],[13,1],[13,2],[88,0],[88,1],[89,2],[89,4],[72,1],[72,1],[51,2],[65,2],[65,4],[99,1],[99,1],[71,5],[82,3],[82,2],[82,2],[82,1],[94,1],[94,3],[94,4],[94,4],[94,6],[101,1],[101,1],[101,1],[102,1],[102,3],[18,2],[18,3],[18,4],[18,5],[104,3],[104,3],[104,2],[23,2],[70,3],[70,5],[110,2],[110,4],[110,2],[110,4],[19,2],[19,2],[19,2],[19,1],[114,2],[114,2],[20,2],[20,2],[20,2],[116,2],[116,4],[116,2],[119,2],[119,3],[123,1],[123,1],[123,1],[123,1],[121,1],[121,3],[120,2],[120,2],[120,4],[120,4],[120,4],[120,6],[120,6],[21,5],[21,7],[21,4],[21,6],[127,1],[127,2],[129,3],[129,4],[131,3],[131,5],[17,1],[17,3],[17,3],[17,3],[15,2],[15,2],[15,2],[15,2],[15,2],[15,2],[15,3],[15,2],[15,2],[15,2],[15,2],[15,2],[15,3],[15,3],[15,3],[15,3],[15,3],[15,3],[15,3],[15,3],[15,3],[15,5],[15,4],[15,3]],performAction:function(e,t,n,i,r,s,o){var a=s.length-1;
switch(r){case 1:return this.$=i.addLocationDataFn(o[a],o[a])(new i.Block);case 2:return this.$=s[a];case 3:this.$=i.addLocationDataFn(o[a],o[a])(i.Block.wrap([s[a]]));break;case 4:this.$=i.addLocationDataFn(o[a-2],o[a])(s[a-2].push(s[a]));break;case 5:this.$=s[a-1];break;case 6:case 7:case 8:case 9:case 11:case 12:case 13:case 14:case 15:case 16:case 17:case 18:case 19:case 20:case 21:case 22:case 27:case 32:case 34:case 47:case 48:case 49:case 50:case 51:case 59:case 60:case 70:case 71:case 72:case 73:case 78:case 79:case 82:case 86:case 92:case 136:case 137:case 139:case 169:case 170:case 186:case 192:this.$=s[a];break;case 10:case 25:case 26:case 28:case 30:case 33:case 35:this.$=i.addLocationDataFn(o[a],o[a])(new i.Literal(s[a]));break;case 23:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Block);break;case 24:case 31:case 93:this.$=i.addLocationDataFn(o[a-2],o[a])(s[a-1]);break;case 29:case 149:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Parens(s[a-1]));break;case 36:this.$=i.addLocationDataFn(o[a],o[a])(new i.Undefined);break;case 37:this.$=i.addLocationDataFn(o[a],o[a])(new i.Null);break;case 38:this.$=i.addLocationDataFn(o[a],o[a])(new i.Bool(s[a]));break;case 39:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Assign(s[a-2],s[a]));break;case 40:this.$=i.addLocationDataFn(o[a-3],o[a])(new i.Assign(s[a-3],s[a]));break;case 41:this.$=i.addLocationDataFn(o[a-4],o[a])(new i.Assign(s[a-4],s[a-1]));break;case 42:case 75:case 80:case 81:case 83:case 84:case 85:case 171:case 172:this.$=i.addLocationDataFn(o[a],o[a])(new i.Value(s[a]));break;case 43:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Assign(i.addLocationDataFn(o[a-2])(new i.Value(s[a-2])),s[a],"object",{operatorToken:i.addLocationDataFn(o[a-1])(new i.Literal(s[a-1]))}));break;case 44:this.$=i.addLocationDataFn(o[a-4],o[a])(new i.Assign(i.addLocationDataFn(o[a-4])(new i.Value(s[a-4])),s[a-1],"object",{operatorToken:i.addLocationDataFn(o[a-3])(new i.Literal(s[a-3]))}));break;case 45:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Assign(i.addLocationDataFn(o[a-2])(new i.Value(s[a-2])),s[a],null,{operatorToken:i.addLocationDataFn(o[a-1])(new i.Literal(s[a-1]))}));break;case 46:this.$=i.addLocationDataFn(o[a-4],o[a])(new i.Assign(i.addLocationDataFn(o[a-4])(new i.Value(s[a-4])),s[a-1],null,{operatorToken:i.addLocationDataFn(o[a-3])(new i.Literal(s[a-3]))}));break;case 52:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Return(s[a]));break;case 53:this.$=i.addLocationDataFn(o[a],o[a])(new i.Return);break;case 54:this.$=i.addLocationDataFn(o[a],o[a])(new i.Comment(s[a]));break;case 55:this.$=i.addLocationDataFn(o[a-4],o[a])(new i.Code(s[a-3],s[a],s[a-1]));break;case 56:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Code([],s[a],s[a-1]));break;case 57:this.$=i.addLocationDataFn(o[a],o[a])("func");break;case 58:this.$=i.addLocationDataFn(o[a],o[a])("boundfunc");break;case 61:case 98:this.$=i.addLocationDataFn(o[a],o[a])([]);break;case 62:case 99:case 131:case 173:this.$=i.addLocationDataFn(o[a],o[a])([s[a]]);break;case 63:case 100:case 132:this.$=i.addLocationDataFn(o[a-2],o[a])(s[a-2].concat(s[a]));break;case 64:case 101:case 133:this.$=i.addLocationDataFn(o[a-3],o[a])(s[a-3].concat(s[a]));break;case 65:case 102:case 135:this.$=i.addLocationDataFn(o[a-5],o[a])(s[a-5].concat(s[a-2]));break;case 66:this.$=i.addLocationDataFn(o[a],o[a])(new i.Param(s[a]));break;case 67:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Param(s[a-1],null,!0));break;case 68:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Param(s[a-2],s[a]));break;case 69:case 138:this.$=i.addLocationDataFn(o[a],o[a])(new i.Expansion);break;case 74:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Splat(s[a-1]));break;case 76:this.$=i.addLocationDataFn(o[a-1],o[a])(s[a-1].add(s[a]));break;case 77:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Value(s[a-1],[].concat(s[a])));break;case 87:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Access(s[a]));break;case 88:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Access(s[a],"soak"));break;case 89:this.$=i.addLocationDataFn(o[a-1],o[a])([i.addLocationDataFn(o[a-1])(new i.Access(new i.Literal("prototype"))),i.addLocationDataFn(o[a])(new i.Access(s[a]))]);break;case 90:this.$=i.addLocationDataFn(o[a-1],o[a])([i.addLocationDataFn(o[a-1])(new i.Access(new i.Literal("prototype"),"soak")),i.addLocationDataFn(o[a])(new i.Access(s[a]))]);break;case 91:this.$=i.addLocationDataFn(o[a],o[a])(new i.Access(new i.Literal("prototype")));break;case 94:this.$=i.addLocationDataFn(o[a-1],o[a])(i.extend(s[a],{soak:!0}));break;case 95:this.$=i.addLocationDataFn(o[a],o[a])(new i.Index(s[a]));break;case 96:this.$=i.addLocationDataFn(o[a],o[a])(new i.Slice(s[a]));break;case 97:this.$=i.addLocationDataFn(o[a-3],o[a])(new i.Obj(s[a-2],s[a-3].generated));break;case 103:this.$=i.addLocationDataFn(o[a],o[a])(new i.Class);break;case 104:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Class(null,null,s[a]));break;case 105:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Class(null,s[a]));break;case 106:this.$=i.addLocationDataFn(o[a-3],o[a])(new i.Class(null,s[a-1],s[a]));break;case 107:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Class(s[a]));break;case 108:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Class(s[a-1],null,s[a]));break;case 109:this.$=i.addLocationDataFn(o[a-3],o[a])(new i.Class(s[a-2],s[a]));break;case 110:this.$=i.addLocationDataFn(o[a-4],o[a])(new i.Class(s[a-3],s[a-1],s[a]));break;case 111:case 112:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Call(s[a-2],s[a],s[a-1]));break;case 113:this.$=i.addLocationDataFn(o[a],o[a])(new i.Call("super",[new i.Splat(new i.Literal("arguments"))]));break;case 114:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Call("super",s[a]));break;case 115:this.$=i.addLocationDataFn(o[a],o[a])(!1);break;case 116:this.$=i.addLocationDataFn(o[a],o[a])(!0);break;case 117:this.$=i.addLocationDataFn(o[a-1],o[a])([]);break;case 118:case 134:this.$=i.addLocationDataFn(o[a-3],o[a])(s[a-2]);break;case 119:case 120:this.$=i.addLocationDataFn(o[a],o[a])(new i.Value(new i.Literal("this")));break;case 121:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Value(i.addLocationDataFn(o[a-1])(new i.Literal("this")),[i.addLocationDataFn(o[a])(new i.Access(s[a]))],"this"));break;case 122:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Arr([]));break;case 123:this.$=i.addLocationDataFn(o[a-3],o[a])(new i.Arr(s[a-2]));break;case 124:this.$=i.addLocationDataFn(o[a],o[a])("inclusive");break;case 125:this.$=i.addLocationDataFn(o[a],o[a])("exclusive");break;case 126:this.$=i.addLocationDataFn(o[a-4],o[a])(new i.Range(s[a-3],s[a-1],s[a-2]));break;case 127:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Range(s[a-2],s[a],s[a-1]));break;case 128:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Range(s[a-1],null,s[a]));break;case 129:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Range(null,s[a],s[a-1]));break;case 130:this.$=i.addLocationDataFn(o[a],o[a])(new i.Range(null,null,s[a]));break;case 140:this.$=i.addLocationDataFn(o[a-2],o[a])([].concat(s[a-2],s[a]));break;case 141:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Try(s[a]));break;case 142:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Try(s[a-1],s[a][0],s[a][1]));break;case 143:this.$=i.addLocationDataFn(o[a-3],o[a])(new i.Try(s[a-2],null,null,s[a]));break;case 144:this.$=i.addLocationDataFn(o[a-4],o[a])(new i.Try(s[a-3],s[a-2][0],s[a-2][1],s[a]));break;case 145:this.$=i.addLocationDataFn(o[a-2],o[a])([s[a-1],s[a]]);break;case 146:this.$=i.addLocationDataFn(o[a-2],o[a])([i.addLocationDataFn(o[a-1])(new i.Value(s[a-1])),s[a]]);break;case 147:this.$=i.addLocationDataFn(o[a-1],o[a])([null,s[a]]);break;case 148:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Throw(s[a]));break;case 150:this.$=i.addLocationDataFn(o[a-4],o[a])(new i.Parens(s[a-2]));break;case 151:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.While(s[a]));break;case 152:this.$=i.addLocationDataFn(o[a-3],o[a])(new i.While(s[a-2],{guard:s[a]}));break;case 153:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.While(s[a],{invert:!0}));break;case 154:this.$=i.addLocationDataFn(o[a-3],o[a])(new i.While(s[a-2],{invert:!0,guard:s[a]}));break;case 155:this.$=i.addLocationDataFn(o[a-1],o[a])(s[a-1].addBody(s[a]));break;case 156:case 157:this.$=i.addLocationDataFn(o[a-1],o[a])(s[a].addBody(i.addLocationDataFn(o[a-1])(i.Block.wrap([s[a-1]]))));break;case 158:this.$=i.addLocationDataFn(o[a],o[a])(s[a]);break;case 159:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.While(i.addLocationDataFn(o[a-1])(new i.Literal("true"))).addBody(s[a]));break;case 160:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.While(i.addLocationDataFn(o[a-1])(new i.Literal("true"))).addBody(i.addLocationDataFn(o[a])(i.Block.wrap([s[a]]))));break;case 161:case 162:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.For(s[a-1],s[a]));break;case 163:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.For(s[a],s[a-1]));break;case 164:this.$=i.addLocationDataFn(o[a-1],o[a])({source:i.addLocationDataFn(o[a])(new i.Value(s[a]))});break;case 165:this.$=i.addLocationDataFn(o[a-3],o[a])({source:i.addLocationDataFn(o[a-2])(new i.Value(s[a-2])),step:s[a]});break;case 166:this.$=i.addLocationDataFn(o[a-1],o[a])(function(){return s[a].own=s[a-1].own,s[a].name=s[a-1][0],s[a].index=s[a-1][1],s[a]}());break;case 167:this.$=i.addLocationDataFn(o[a-1],o[a])(s[a]);break;case 168:this.$=i.addLocationDataFn(o[a-2],o[a])(function(){return s[a].own=!0,s[a]}());break;case 174:this.$=i.addLocationDataFn(o[a-2],o[a])([s[a-2],s[a]]);break;case 175:this.$=i.addLocationDataFn(o[a-1],o[a])({source:s[a]});break;case 176:this.$=i.addLocationDataFn(o[a-1],o[a])({source:s[a],object:!0});break;case 177:this.$=i.addLocationDataFn(o[a-3],o[a])({source:s[a-2],guard:s[a]});break;case 178:this.$=i.addLocationDataFn(o[a-3],o[a])({source:s[a-2],guard:s[a],object:!0});break;case 179:this.$=i.addLocationDataFn(o[a-3],o[a])({source:s[a-2],step:s[a]});break;case 180:this.$=i.addLocationDataFn(o[a-5],o[a])({source:s[a-4],guard:s[a-2],step:s[a]});break;case 181:this.$=i.addLocationDataFn(o[a-5],o[a])({source:s[a-4],step:s[a-2],guard:s[a]});break;case 182:this.$=i.addLocationDataFn(o[a-4],o[a])(new i.Switch(s[a-3],s[a-1]));break;case 183:this.$=i.addLocationDataFn(o[a-6],o[a])(new i.Switch(s[a-5],s[a-3],s[a-1]));break;case 184:this.$=i.addLocationDataFn(o[a-3],o[a])(new i.Switch(null,s[a-1]));break;case 185:this.$=i.addLocationDataFn(o[a-5],o[a])(new i.Switch(null,s[a-3],s[a-1]));break;case 187:this.$=i.addLocationDataFn(o[a-1],o[a])(s[a-1].concat(s[a]));break;case 188:this.$=i.addLocationDataFn(o[a-2],o[a])([[s[a-1],s[a]]]);break;case 189:this.$=i.addLocationDataFn(o[a-3],o[a])([[s[a-2],s[a-1]]]);break;case 190:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.If(s[a-1],s[a],{type:s[a-2]}));break;case 191:this.$=i.addLocationDataFn(o[a-4],o[a])(s[a-4].addElse(i.addLocationDataFn(o[a-2],o[a])(new i.If(s[a-1],s[a],{type:s[a-2]}))));break;case 193:this.$=i.addLocationDataFn(o[a-2],o[a])(s[a-2].addElse(s[a]));break;case 194:case 195:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.If(s[a],i.addLocationDataFn(o[a-2])(i.Block.wrap([s[a-2]])),{type:s[a-1],statement:!0}));break;case 196:case 197:case 200:case 201:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Op(s[a-1],s[a]));break;case 198:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Op("-",s[a]));break;case 199:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Op("+",s[a]));break;case 202:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Op(s[a-2].concat(s[a-1]),s[a]));break;case 203:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Op("--",s[a]));break;case 204:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Op("++",s[a]));break;case 205:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Op("--",s[a-1],null,!0));break;case 206:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Op("++",s[a-1],null,!0));break;case 207:this.$=i.addLocationDataFn(o[a-1],o[a])(new i.Existence(s[a-1]));break;case 208:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Op("+",s[a-2],s[a]));break;case 209:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Op("-",s[a-2],s[a]));break;case 210:case 211:case 212:case 213:case 214:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Op(s[a-1],s[a-2],s[a]));break;case 215:this.$=i.addLocationDataFn(o[a-2],o[a])(function(){return"!"===s[a-1].charAt(0)?new i.Op(s[a-1].slice(1),s[a-2],s[a]).invert():new i.Op(s[a-1],s[a-2],s[a])}());break;case 216:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Assign(s[a-2],s[a],s[a-1]));break;case 217:this.$=i.addLocationDataFn(o[a-4],o[a])(new i.Assign(s[a-4],s[a-1],s[a-3]));break;case 218:this.$=i.addLocationDataFn(o[a-3],o[a])(new i.Assign(s[a-3],s[a],s[a-2]));break;case 219:this.$=i.addLocationDataFn(o[a-2],o[a])(new i.Extends(s[a-2],s[a]))}},table:[{1:[2,1],3:1,4:2,5:3,7:4,8:5,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{1:[3]},{1:[2,2],6:P},t(U,[2,3]),t(U,[2,6],{119:69,110:89,116:90,111:x,113:S,117:R,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(U,[2,7],{119:69,110:92,116:93,111:x,113:S,117:R,133:Z}),t(et,[2,11],{88:94,69:95,77:101,73:tt,74:nt,75:it,76:rt,78:st,81:ot,91:at,92:ct}),t(et,[2,12],{77:101,88:104,69:105,73:tt,74:nt,75:it,76:rt,78:st,81:ot,91:at,92:ct}),t(et,[2,13]),t(et,[2,14]),t(et,[2,15]),t(et,[2,16]),t(et,[2,17]),t(et,[2,18]),t(et,[2,19]),t(et,[2,20]),t(et,[2,21]),t(et,[2,22]),t(et,[2,8]),t(et,[2,9]),t(et,[2,10]),t(lt,ht,{46:[1,106]}),t(lt,[2,83]),t(lt,[2,84]),t(lt,[2,85]),t(lt,[2,86]),t([1,6,25,26,34,38,56,61,64,73,74,75,76,78,80,81,85,91,93,98,100,109,111,112,113,117,118,133,136,137,142,143,144,145,146,147,148],[2,113],{89:107,92:ut}),t([6,25,56,61],pt,{55:109,62:110,63:111,27:113,51:114,65:115,66:116,28:i,64:dt,83:y,96:ft,97:mt}),{24:119,25:gt},{7:121,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:123,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:124,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:125,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:127,8:126,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,139:[1,128],140:B,141:V},{12:130,13:131,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:132,51:63,65:47,66:48,68:129,70:23,71:24,72:25,83:y,90:w,95:T,96:C,97:F,108:L},{12:130,13:131,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:132,51:63,65:47,66:48,68:133,70:23,71:24,72:25,83:y,90:w,95:T,96:C,97:F,108:L},t(vt,bt,{87:[1,137],140:[1,134],141:[1,135],149:[1,136]}),t(et,[2,192],{128:[1,138]}),{24:139,25:gt},{24:140,25:gt},t(et,[2,158]),{24:141,25:gt},{7:142,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,25:[1,143],27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(yt,[2,103],{39:22,70:23,71:24,72:25,65:47,66:48,29:49,35:51,27:62,51:63,31:72,12:130,13:131,45:132,24:144,68:146,25:gt,28:i,30:r,32:s,33:o,36:a,37:c,40:l,41:h,42:u,43:p,44:d,83:y,87:[1,145],90:w,95:T,96:C,97:F,108:L}),{7:147,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t([1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,118,133,142,143,144,145,146,147,148],[2,53],{12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,9:18,10:19,45:21,39:22,70:23,71:24,72:25,57:28,68:36,131:37,110:39,114:40,116:41,65:47,66:48,29:49,35:51,27:62,51:63,119:69,31:72,8:122,7:148,11:n,28:i,30:r,32:s,33:o,36:a,37:c,40:l,41:h,42:u,43:p,44:d,52:f,53:m,54:g,58:v,59:b,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,115:D,126:A,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V}),t(et,[2,54]),t(vt,[2,80]),t(vt,[2,81]),t(lt,[2,32]),t(lt,[2,33]),t(lt,[2,34]),t(lt,[2,35]),t(lt,[2,36]),t(lt,[2,37]),t(lt,[2,38]),{4:149,5:3,7:4,8:5,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,25:[1,150],27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:151,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,25:kt,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,64:wt,65:47,66:48,67:156,68:36,70:23,71:24,72:25,83:y,86:k,90:w,94:153,95:T,96:C,97:F,98:Tt,101:154,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(lt,[2,119]),t(lt,[2,120],{27:158,28:i}),{25:[2,57]},{25:[2,58]},t(Ct,[2,75]),t(Ct,[2,78]),{7:159,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:160,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:161,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:163,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,24:162,25:gt,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{27:168,28:i,51:169,65:170,66:171,71:164,83:y,96:ft,97:F,121:165,122:[1,166],123:167},{120:172,124:[1,173],125:[1,174]},t([6,25,61,85],Ft,{31:72,84:175,47:176,48:177,50:178,10:179,29:180,27:181,51:182,28:i,30:r,32:s,33:o,53:m,96:ft}),t(Et,[2,26]),t(Et,[2,27]),t(lt,[2,30]),{12:130,13:183,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:132,51:63,65:47,66:48,68:184,70:23,71:24,72:25,83:y,90:w,95:T,96:C,97:F,108:L},t(Nt,[2,25]),t(Et,[2,28]),{4:185,5:3,7:4,8:5,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(U,[2,5],{7:4,8:5,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,9:18,10:19,45:21,39:22,70:23,71:24,72:25,57:28,68:36,131:37,110:39,114:40,116:41,65:47,66:48,29:49,35:51,27:62,51:63,119:69,31:72,5:186,11:n,28:i,30:r,32:s,33:o,36:a,37:c,40:l,41:h,42:u,43:p,44:d,52:f,53:m,54:g,58:v,59:b,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,111:x,113:S,115:D,117:R,126:A,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V}),t(et,[2,207]),{7:187,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:188,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:189,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:190,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:191,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:192,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:193,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:194,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:195,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(et,[2,157]),t(et,[2,162]),{7:196,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(et,[2,156]),t(et,[2,161]),{89:197,92:ut},t(Ct,[2,76]),{92:[2,116]},{27:198,28:i},{27:199,28:i},t(Ct,[2,91],{27:200,28:i}),{27:201,28:i},t(Ct,[2,92]),{7:203,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,64:Lt,65:47,66:48,68:36,70:23,71:24,72:25,79:202,82:204,83:y,86:k,90:w,95:T,96:C,97:F,99:205,100:xt,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{77:208,78:st,81:ot},{89:209,92:ut},t(Ct,[2,77]),{6:[1,211],7:210,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,25:[1,212],27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(St,[2,114]),{7:215,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,25:kt,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,64:wt,65:47,66:48,67:156,68:36,70:23,71:24,72:25,83:y,86:k,90:w,93:[1,213],94:214,95:T,96:C,97:F,101:154,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t([6,25],Dt,{60:218,56:[1,216],61:Rt}),t(At,[2,62]),t(At,[2,66],{46:[1,220],64:[1,219]}),t(At,[2,69]),t(It,[2,70]),t(It,[2,71]),t(It,[2,72]),t(It,[2,73]),{27:158,28:i},{7:215,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,25:kt,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,64:wt,65:47,66:48,67:156,68:36,70:23,71:24,72:25,83:y,86:k,90:w,94:153,95:T,96:C,97:F,98:Tt,101:154,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(et,[2,56]),{4:222,5:3,7:4,8:5,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,26:[1,221],27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t([1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,118,133,136,137,143,144,145,146,147,148],[2,196],{119:69,110:89,116:90,142:X}),{110:92,111:x,113:S,116:93,117:R,119:69,133:Z},t(_t,[2,197],{119:69,110:89,116:90,142:X,144:Y}),t(_t,[2,198],{119:69,110:89,116:90,142:X,144:Y}),t(_t,[2,199],{119:69,110:89,116:90,142:X,144:Y}),t(et,[2,200],{119:69,110:92,116:93}),t(Ot,[2,201],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),{7:223,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(et,[2,203],{73:bt,74:bt,75:bt,76:bt,78:bt,81:bt,91:bt,92:bt}),{69:95,73:tt,74:nt,75:it,76:rt,77:101,78:st,81:ot,88:94,91:at,92:ct},{69:105,73:tt,74:nt,75:it,76:rt,77:101,78:st,81:ot,88:104,91:at,92:ct},t($t,ht),t(et,[2,204],{73:bt,74:bt,75:bt,76:bt,78:bt,81:bt,91:bt,92:bt}),t(et,[2,205]),t(et,[2,206]),{6:[1,226],7:224,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,25:[1,225],27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:227,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{24:228,25:gt,132:[1,229]},t(et,[2,141],{104:230,105:[1,231],106:[1,232]}),t(et,[2,155]),t(et,[2,163]),{25:[1,233],110:89,111:x,113:S,116:90,117:R,119:69,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q},{127:234,129:235,130:jt},t(et,[2,104]),{7:237,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(yt,[2,107],{24:238,25:gt,73:bt,74:bt,75:bt,76:bt,78:bt,81:bt,91:bt,92:bt,87:[1,239]}),t(Ot,[2,148],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(Ot,[2,52],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),{6:P,109:[1,240]},{4:241,5:3,7:4,8:5,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t([6,25,61,98],Mt,{119:69,110:89,116:90,99:242,64:[1,243],100:xt,111:x,113:S,117:R,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(Bt,[2,122]),t([6,25,98],Dt,{60:244,61:Vt}),t(Pt,[2,131]),{7:215,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,25:kt,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,64:wt,65:47,66:48,67:156,68:36,70:23,71:24,72:25,83:y,86:k,90:w,94:246,95:T,96:C,97:F,101:154,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(Pt,[2,137]),t(Pt,[2,138]),t(Nt,[2,121]),{24:247,25:gt,110:89,111:x,113:S,116:90,117:R,119:69,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q},t(Ut,[2,151],{119:69,110:89,116:90,111:x,112:[1,248],113:S,117:R,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(Ut,[2,153],{119:69,110:89,116:90,111:x,112:[1,249],113:S,117:R,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(et,[2,159]),t(Gt,[2,160],{119:69,110:89,116:90,111:x,113:S,117:R,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t([1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,133,136,137,142,143,144,145,146,147,148],[2,164],{118:[1,250]}),t(Ht,[2,167]),{27:168,28:i,51:169,65:170,66:171,83:y,96:ft,97:mt,121:251,123:167},t(Ht,[2,173],{61:[1,252]}),t(qt,[2,169]),t(qt,[2,170]),t(qt,[2,171]),t(qt,[2,172]),t(et,[2,166]),{7:253,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:254,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t([6,25,85],Dt,{60:255,61:Xt}),t(Wt,[2,99]),t(Wt,[2,42],{49:[1,257]}),t(Yt,[2,50],{46:[1,258]}),t(Wt,[2,47]),t(Yt,[2,51]),t(Kt,[2,48]),t(Kt,[2,49]),{38:[1,259],69:105,73:tt,74:nt,75:it,76:rt,77:101,78:st,81:ot,88:104,91:at,92:ct},t($t,bt),{6:P,34:[1,260]},t(U,[2,4]),t(zt,[2,208],{119:69,110:89,116:90,142:X,143:W,144:Y}),t(zt,[2,209],{119:69,110:89,116:90,142:X,143:W,144:Y}),t(_t,[2,210],{119:69,110:89,116:90,142:X,144:Y}),t(_t,[2,211],{119:69,110:89,116:90,142:X,144:Y}),t([1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,118,133,145,146,147,148],[2,212],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y}),t([1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,118,133,146,147],[2,213],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,148:Q}),t([1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,118,133,147],[2,214],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,148:Q}),t([1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,118,133,146,147,148],[2,215],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K}),t(Gt,[2,195],{119:69,110:89,116:90,111:x,113:S,117:R,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(Gt,[2,194],{119:69,110:89,116:90,111:x,113:S,117:R,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(St,[2,111]),t(Ct,[2,87]),t(Ct,[2,88]),t(Ct,[2,89]),t(Ct,[2,90]),{80:[1,261]},{64:Lt,80:[2,95],99:262,100:xt,110:89,111:x,113:S,116:90,117:R,119:69,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q},{80:[2,96]},{7:263,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,80:[2,130],83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(Jt,[2,124]),t(Jt,Qt),t(Ct,[2,94]),t(St,[2,112]),t(Ot,[2,39],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),{7:264,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:265,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(St,[2,117]),t([6,25,93],Dt,{60:266,61:Vt}),t(Pt,Mt,{119:69,110:89,116:90,64:[1,267],111:x,113:S,117:R,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),{57:268,58:v,59:b},t(Zt,en,{63:111,27:113,51:114,65:115,66:116,62:269,28:i,64:dt,83:y,96:ft,97:mt}),{6:tn,25:nn},t(At,[2,67]),{7:272,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(rn,[2,23]),{6:P,26:[1,273]},t(Ot,[2,202],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(Ot,[2,216],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),{7:274,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:275,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(Ot,[2,219],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(et,[2,193]),{7:276,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(et,[2,142],{105:[1,277]}),{24:278,25:gt},{24:281,25:gt,27:279,28:i,66:280,83:y},{127:282,129:235,130:jt},{26:[1,283],128:[1,284],129:285,130:jt},t(sn,[2,186]),{7:287,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,102:286,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(on,[2,105],{119:69,110:89,116:90,24:288,25:gt,111:x,113:S,117:R,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(et,[2,108]),{7:289,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(lt,[2,149]),{6:P,26:[1,290]},{7:291,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t([11,28,30,32,33,36,37,40,41,42,43,44,52,53,54,58,59,83,86,90,95,96,97,103,107,108,111,113,115,117,126,132,134,135,136,137,138,140,141],Qt,{6:an,25:an,61:an,98:an}),{6:cn,25:ln,98:[1,292]},t([6,25,26,93,98],en,{12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,9:18,10:19,45:21,39:22,70:23,71:24,72:25,57:28,68:36,131:37,110:39,114:40,116:41,65:47,66:48,29:49,35:51,27:62,51:63,119:69,31:72,8:122,67:156,7:215,101:295,11:n,28:i,30:r,32:s,33:o,36:a,37:c,40:l,41:h,42:u,43:p,44:d,52:f,53:m,54:g,58:v,59:b,64:wt,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,111:x,113:S,115:D,117:R,126:A,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V}),t(Zt,Dt,{60:296,61:Vt}),t(hn,[2,190]),{7:297,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:298,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:299,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(Ht,[2,168]),{27:168,28:i,51:169,65:170,66:171,83:y,96:ft,97:mt,123:300},t([1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,113,117,133],[2,175],{119:69,110:89,116:90,112:[1,301],118:[1,302],136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(un,[2,176],{119:69,110:89,116:90,112:[1,303],136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),{6:pn,25:dn,85:[1,304]},t([6,25,26,85],en,{31:72,48:177,50:178,10:179,29:180,27:181,51:182,47:307,28:i,30:r,32:s,33:o,53:m,96:ft}),{7:308,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,25:[1,309],27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:310,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,25:[1,311],27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(lt,[2,31]),t(Et,[2,29]),t(Ct,[2,93]),{7:312,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,80:[2,128],83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{80:[2,129],110:89,111:x,113:S,116:90,117:R,119:69,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q},t(Ot,[2,40],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),{26:[1,313],110:89,111:x,113:S,116:90,117:R,119:69,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q},{6:cn,25:ln,93:[1,314]},t(Pt,an),{24:315,25:gt},t(At,[2,63]),{27:113,28:i,51:114,62:316,63:111,64:dt,65:115,66:116,83:y,96:ft,97:mt},t(fn,pt,{62:110,63:111,27:113,51:114,65:115,66:116,55:317,28:i,64:dt,83:y,96:ft,97:mt}),t(At,[2,68],{119:69,110:89,116:90,111:x,113:S,117:R,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(rn,[2,24]),{26:[1,318],110:89,111:x,113:S,116:90,117:R,119:69,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q},t(Ot,[2,218],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),{24:319,25:gt,110:89,111:x,113:S,116:90,117:R,119:69,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q},{24:320,25:gt},t(et,[2,143]),{24:321,25:gt},{24:322,25:gt},t(mn,[2,147]),{26:[1,323],128:[1,324],129:285,130:jt},t(et,[2,184]),{24:325,25:gt},t(sn,[2,187]),{24:326,25:gt,61:[1,327]},t(gn,[2,139],{119:69,110:89,116:90,111:x,113:S,117:R,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(et,[2,106]),t(on,[2,109],{119:69,110:89,116:90,24:328,25:gt,111:x,113:S,117:R,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),{109:[1,329]},{98:[1,330],110:89,111:x,113:S,116:90,117:R,119:69,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q},t(Bt,[2,123]),{7:215,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,64:wt,65:47,66:48,67:156,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,101:331,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:215,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,25:kt,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,64:wt,65:47,66:48,67:156,68:36,70:23,71:24,72:25,83:y,86:k,90:w,94:332,95:T,96:C,97:F,101:154,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(Pt,[2,132]),{6:cn,25:ln,26:[1,333]},t(Gt,[2,152],{119:69,110:89,116:90,111:x,113:S,117:R,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(Gt,[2,154],{119:69,110:89,116:90,111:x,113:S,117:R,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(Gt,[2,165],{119:69,110:89,116:90,111:x,113:S,117:R,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(Ht,[2,174]),{7:334,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:335,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:336,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(Bt,[2,97]),{10:179,27:181,28:i,29:180,30:r,31:72,32:s,33:o,47:337,48:177,50:178,51:182,53:m,96:ft},t(fn,Ft,{31:72,47:176,48:177,50:178,10:179,29:180,27:181,51:182,84:338,28:i,30:r,32:s,33:o,53:m,96:ft}),t(Wt,[2,100]),t(Wt,[2,43],{119:69,110:89,116:90,111:x,113:S,117:R,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),{7:339,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(Wt,[2,45],{119:69,110:89,116:90,111:x,113:S,117:R,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),{7:340,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{80:[2,127],110:89,111:x,113:S,116:90,117:R,119:69,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q},t(et,[2,41]),t(St,[2,118]),t(et,[2,55]),t(At,[2,64]),t(Zt,Dt,{60:341,61:Rt}),t(et,[2,217]),t(hn,[2,191]),t(et,[2,144]),t(mn,[2,145]),t(mn,[2,146]),t(et,[2,182]),{24:342,25:gt},{26:[1,343]},t(sn,[2,188],{6:[1,344]}),{7:345,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},t(et,[2,110]),t(lt,[2,150]),t(lt,[2,126]),t(Pt,[2,133]),t(Zt,Dt,{60:346,61:Vt}),t(Pt,[2,134]),t([1,6,25,26,34,56,61,64,80,85,93,98,100,109,111,112,113,117,133],[2,177],{119:69,110:89,116:90,118:[1,347],136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(un,[2,179],{119:69,110:89,116:90,112:[1,348],136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(Ot,[2,178],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(Wt,[2,101]),t(Zt,Dt,{60:349,61:Xt}),{26:[1,350],110:89,111:x,113:S,116:90,117:R,119:69,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q},{26:[1,351],110:89,111:x,113:S,116:90,117:R,119:69,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q},{6:tn,25:nn,26:[1,352]},{26:[1,353]},t(et,[2,185]),t(sn,[2,189]),t(gn,[2,140],{119:69,110:89,116:90,111:x,113:S,117:R,133:G,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),{6:cn,25:ln,26:[1,354]},{7:355,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{7:356,8:122,9:18,10:19,11:n,12:6,13:7,14:8,15:9,16:10,17:11,18:12,19:13,20:14,21:15,22:16,23:17,27:62,28:i,29:49,30:r,31:72,32:s,33:o,35:51,36:a,37:c,39:22,40:l,41:h,42:u,43:p,44:d,45:21,51:63,52:f,53:m,54:g,57:28,58:v,59:b,65:47,66:48,68:36,70:23,71:24,72:25,83:y,86:k,90:w,95:T,96:C,97:F,103:E,107:N,108:L,110:39,111:x,113:S,114:40,115:D,116:41,117:R,119:69,126:A,131:37,132:I,134:_,135:O,136:$,137:j,138:M,140:B,141:V},{6:pn,25:dn,26:[1,357]},t(Wt,[2,44]),t(Wt,[2,46]),t(At,[2,65]),t(et,[2,183]),t(Pt,[2,135]),t(Ot,[2,180],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(Ot,[2,181],{119:69,110:89,116:90,136:H,137:q,142:X,143:W,144:Y,145:K,146:z,147:J,148:Q}),t(Wt,[2,102])],defaultActions:{60:[2,57],61:[2,58],96:[2,116],204:[2,96]},parseError:function(e,t){if(!t.recoverable)throw Error(e);
this.trace(e)},parse:function(e){function t(){var e;return e=f.lex()||p,"number"!=typeof e&&(e=n.symbols_[e]||e),e}var n=this,i=[0],r=[null],s=[],o=this.table,a="",c=0,l=0,h=0,u=2,p=1,d=s.slice.call(arguments,1),f=Object.create(this.lexer),m={yy:{}};for(var g in this.yy)Object.prototype.hasOwnProperty.call(this.yy,g)&&(m.yy[g]=this.yy[g]);f.setInput(e,m.yy),m.yy.lexer=f,m.yy.parser=this,f.yylloc===void 0&&(f.yylloc={});var v=f.yylloc;s.push(v);var b=f.options&&f.options.ranges;this.parseError="function"==typeof m.yy.parseError?m.yy.parseError:Object.getPrototypeOf(this).parseError;for(var y,k,w,T,C,F,E,N,L,x={};;){if(w=i[i.length-1],this.defaultActions[w]?T=this.defaultActions[w]:((null===y||y===void 0)&&(y=t()),T=o[w]&&o[w][y]),T===void 0||!T.length||!T[0]){var S="";L=[];for(F in o[w])this.terminals_[F]&&F>u&&L.push("'"+this.terminals_[F]+"'");S=f.showPosition?"Parse error on line "+(c+1)+":\n"+f.showPosition()+"\nExpecting "+L.join(", ")+", got '"+(this.terminals_[y]||y)+"'":"Parse error on line "+(c+1)+": Unexpected "+(y==p?"end of input":"'"+(this.terminals_[y]||y)+"'"),this.parseError(S,{text:f.match,token:this.terminals_[y]||y,line:f.yylineno,loc:v,expected:L})}if(T[0]instanceof Array&&T.length>1)throw Error("Parse Error: multiple actions possible at state: "+w+", token: "+y);switch(T[0]){case 1:i.push(y),r.push(f.yytext),s.push(f.yylloc),i.push(T[1]),y=null,k?(y=k,k=null):(l=f.yyleng,a=f.yytext,c=f.yylineno,v=f.yylloc,h>0&&h--);break;case 2:if(E=this.productions_[T[1]][1],x.$=r[r.length-E],x._$={first_line:s[s.length-(E||1)].first_line,last_line:s[s.length-1].last_line,first_column:s[s.length-(E||1)].first_column,last_column:s[s.length-1].last_column},b&&(x._$.range=[s[s.length-(E||1)].range[0],s[s.length-1].range[1]]),C=this.performAction.apply(x,[a,l,c,m.yy,T[1],r,s].concat(d)),C!==void 0)return C;E&&(i=i.slice(0,2*-1*E),r=r.slice(0,-1*E),s=s.slice(0,-1*E)),i.push(this.productions_[T[1]][0]),r.push(x.$),s.push(x._$),N=o[i[i.length-2]][i[i.length-1]],i.push(N);break;case 3:return!0}}return!0}};return e.prototype=vn,vn.Parser=e,new e}();return require!==void 0&&e!==void 0&&(e.parser=n,e.Parser=n.Parser,e.parse=function(){return n.parse.apply(n,arguments)},e.main=function(t){t[1]||(console.log("Usage: "+t[0]+" FILE"),process.exit(1));var n=require("fs").readFileSync(require("path").normalize(t[1]),"utf8");return e.parser.parse(n)},t!==void 0&&require.main===t&&e.main(process.argv.slice(1))),t.exports}(),require["./scope"]=function(){var e={},t={exports:e};return function(){var t,n=[].indexOf||function(e){for(var t=0,n=this.length;n>t;t++)if(t in this&&this[t]===e)return t;return-1};e.Scope=t=function(){function e(e,t,n,i){var r,s;this.parent=e,this.expressions=t,this.method=n,this.referencedVars=i,this.variables=[{name:"arguments",type:"arguments"}],this.positions={},this.parent||(this.utilities={}),this.root=null!=(r=null!=(s=this.parent)?s.root:void 0)?r:this}return e.prototype.add=function(e,t,n){return this.shared&&!n?this.parent.add(e,t,n):Object.prototype.hasOwnProperty.call(this.positions,e)?this.variables[this.positions[e]].type=t:this.positions[e]=this.variables.push({name:e,type:t})-1},e.prototype.namedMethod=function(){var e;return(null!=(e=this.method)?e.name:void 0)||!this.parent?this.method:this.parent.namedMethod()},e.prototype.find=function(e){return this.check(e)?!0:(this.add(e,"var"),!1)},e.prototype.parameter=function(e){return this.shared&&this.parent.check(e,!0)?void 0:this.add(e,"param")},e.prototype.check=function(e){var t;return!!(this.type(e)||(null!=(t=this.parent)?t.check(e):void 0))},e.prototype.temporary=function(e,t,n){return null==n&&(n=!1),n?(t+parseInt(e,36)).toString(36).replace(/\d/g,"a"):e+(t||"")},e.prototype.type=function(e){var t,n,i,r;for(i=this.variables,t=0,n=i.length;n>t;t++)if(r=i[t],r.name===e)return r.type;return null},e.prototype.freeVariable=function(e,t){var i,r,s;for(null==t&&(t={}),i=0;;){if(s=this.temporary(e,i,t.single),!(this.check(s)||n.call(this.root.referencedVars,s)>=0))break;i++}return(null!=(r=t.reserve)?r:!0)&&this.add(s,"var",!0),s},e.prototype.assign=function(e,t){return this.add(e,{value:t,assigned:!0},!0),this.hasAssignments=!0},e.prototype.hasDeclarations=function(){return!!this.declaredVariables().length},e.prototype.declaredVariables=function(){var e;return function(){var t,n,i,r;for(i=this.variables,r=[],t=0,n=i.length;n>t;t++)e=i[t],"var"===e.type&&r.push(e.name);return r}.call(this).sort()},e.prototype.assignedVariables=function(){var e,t,n,i,r;for(n=this.variables,i=[],e=0,t=n.length;t>e;e++)r=n[e],r.type.assigned&&i.push(r.name+" = "+r.type.value);return i},e}()}.call(this),t.exports}(),require["./nodes"]=function(){var e={},t={exports:e};return function(){var t,n,i,r,s,o,a,c,l,h,u,p,d,f,m,g,v,b,y,k,w,T,C,F,E,N,L,x,S,D,R,A,I,_,O,$,j,M,B,V,P,U,G,H,q,X,W,Y,K,z,J,Q,Z,et,tt,nt,it,rt,st,ot,at,ct,lt,ht,ut,pt,dt,ft,mt,gt,vt,bt,yt,kt=function(e,t){function n(){this.constructor=e}for(var i in t)wt.call(t,i)&&(e[i]=t[i]);return n.prototype=t.prototype,e.prototype=new n,e.__super__=t.prototype,e},wt={}.hasOwnProperty,Tt=[].indexOf||function(e){for(var t=0,n=this.length;n>t;t++)if(t in this&&this[t]===e)return t;return-1},Ct=[].slice;Error.stackTraceLimit=1/0,P=require("./scope").Scope,dt=require("./lexer"),$=dt.RESERVED,V=dt.STRICT_PROSCRIBED,ft=require("./helpers"),et=ft.compact,rt=ft.flatten,it=ft.extend,ht=ft.merge,tt=ft.del,gt=ft.starts,nt=ft.ends,mt=ft.some,Z=ft.addLocationDataFn,lt=ft.locationDataToString,vt=ft.throwSyntaxError,e.extend=it,e.addLocationDataFn=Z,Q=function(){return!0},D=function(){return!1},X=function(){return this},S=function(){return this.negated=!this.negated,this},e.CodeFragment=l=function(){function e(e,t){var n;this.code=""+t,this.locationData=null!=e?e.locationData:void 0,this.type=(null!=e?null!=(n=e.constructor)?n.name:void 0:void 0)||"unknown"}return e.prototype.toString=function(){return""+this.code+(this.locationData?": "+lt(this.locationData):"")},e}(),st=function(e){var t;return function(){var n,i,r;for(r=[],n=0,i=e.length;i>n;n++)t=e[n],r.push(t.code);return r}().join("")},e.Base=r=function(){function e(){}return e.prototype.compile=function(e,t){return st(this.compileToFragments(e,t))},e.prototype.compileToFragments=function(e,t){var n;return e=it({},e),t&&(e.level=t),n=this.unfoldSoak(e)||this,n.tab=e.indent,e.level!==L&&n.isStatement(e)?n.compileClosure(e):n.compileNode(e)},e.prototype.compileClosure=function(e){var n,i,r,a,l,h,u;return(a=this.jumps())&&a.error("cannot use a pure statement in an expression"),e.sharedScope=!0,r=new c([],s.wrap([this])),n=[],((i=this.contains(at))||this.contains(ct))&&(n=[new x("this")],i?(l="apply",n.push(new x("arguments"))):l="call",r=new z(r,[new t(new x(l))])),h=new o(r,n).compileNode(e),(r.isGenerator||(null!=(u=r.base)?u.isGenerator:void 0))&&(h.unshift(this.makeCode("(yield* ")),h.push(this.makeCode(")"))),h},e.prototype.cache=function(e,t,n){var r,s,o;return r=null!=n?n(this):this.isComplex(),r?(s=new x(e.scope.freeVariable("ref")),o=new i(s,this),t?[o.compileToFragments(e,t),[this.makeCode(s.value)]]:[o,s]):(s=t?this.compileToFragments(e,t):this,[s,s])},e.prototype.cacheToCodeFragments=function(e){return[st(e[0]),st(e[1])]},e.prototype.makeReturn=function(e){var t;return t=this.unwrapAll(),e?new o(new x(e+".push"),[t]):new M(t)},e.prototype.contains=function(e){var t;return t=void 0,this.traverseChildren(!1,function(n){return e(n)?(t=n,!1):void 0}),t},e.prototype.lastNonComment=function(e){var t;for(t=e.length;t--;)if(!(e[t]instanceof h))return e[t];return null},e.prototype.toString=function(e,t){var n;return null==e&&(e=""),null==t&&(t=this.constructor.name),n="\n"+e+t,this.soak&&(n+="?"),this.eachChild(function(t){return n+=t.toString(e+q)}),n},e.prototype.eachChild=function(e){var t,n,i,r,s,o,a,c;if(!this.children)return this;for(a=this.children,i=0,s=a.length;s>i;i++)if(t=a[i],this[t])for(c=rt([this[t]]),r=0,o=c.length;o>r;r++)if(n=c[r],e(n)===!1)return this;return this},e.prototype.traverseChildren=function(e,t){return this.eachChild(function(n){var i;return i=t(n),i!==!1?n.traverseChildren(e,t):void 0})},e.prototype.invert=function(){return new I("!",this)},e.prototype.unwrapAll=function(){var e;for(e=this;e!==(e=e.unwrap()););return e},e.prototype.children=[],e.prototype.isStatement=D,e.prototype.jumps=D,e.prototype.isComplex=Q,e.prototype.isChainable=D,e.prototype.isAssignable=D,e.prototype.unwrap=X,e.prototype.unfoldSoak=D,e.prototype.assigns=D,e.prototype.updateLocationDataIfMissing=function(e){return this.locationData?this:(this.locationData=e,this.eachChild(function(t){return t.updateLocationDataIfMissing(e)}))},e.prototype.error=function(e){return vt(e,this.locationData)},e.prototype.makeCode=function(e){return new l(this,e)},e.prototype.wrapInBraces=function(e){return[].concat(this.makeCode("("),e,this.makeCode(")"))},e.prototype.joinFragmentArrays=function(e,t){var n,i,r,s,o;for(n=[],r=s=0,o=e.length;o>s;r=++s)i=e[r],r&&n.push(this.makeCode(t)),n=n.concat(i);return n},e}(),e.Block=s=function(e){function t(e){this.expressions=et(rt(e||[]))}return kt(t,e),t.prototype.children=["expressions"],t.prototype.push=function(e){return this.expressions.push(e),this},t.prototype.pop=function(){return this.expressions.pop()},t.prototype.unshift=function(e){return this.expressions.unshift(e),this},t.prototype.unwrap=function(){return 1===this.expressions.length?this.expressions[0]:this},t.prototype.isEmpty=function(){return!this.expressions.length},t.prototype.isStatement=function(e){var t,n,i,r;for(r=this.expressions,n=0,i=r.length;i>n;n++)if(t=r[n],t.isStatement(e))return!0;return!1},t.prototype.jumps=function(e){var t,n,i,r,s;for(s=this.expressions,n=0,r=s.length;r>n;n++)if(t=s[n],i=t.jumps(e))return i},t.prototype.makeReturn=function(e){var t,n;for(n=this.expressions.length;n--;)if(t=this.expressions[n],!(t instanceof h)){this.expressions[n]=t.makeReturn(e),t instanceof M&&!t.expression&&this.expressions.splice(n,1);break}return this},t.prototype.compileToFragments=function(e,n){return null==e&&(e={}),e.scope?t.__super__.compileToFragments.call(this,e,n):this.compileRoot(e)},t.prototype.compileNode=function(e){var n,i,r,s,o,a,c,l,h;for(this.tab=e.indent,h=e.level===L,i=[],l=this.expressions,s=o=0,a=l.length;a>o;s=++o)c=l[s],c=c.unwrapAll(),c=c.unfoldSoak(e)||c,c instanceof t?i.push(c.compileNode(e)):h?(c.front=!0,r=c.compileToFragments(e),c.isStatement(e)||(r.unshift(this.makeCode(""+this.tab)),r.push(this.makeCode(";"))),i.push(r)):i.push(c.compileToFragments(e,F));return h?this.spaced?[].concat(this.joinFragmentArrays(i,"\n\n"),this.makeCode("\n")):this.joinFragmentArrays(i,"\n"):(n=i.length?this.joinFragmentArrays(i,", "):[this.makeCode("void 0")],i.length>1&&e.level>=F?this.wrapInBraces(n):n)},t.prototype.compileRoot=function(e){var t,n,i,r,s,o,a,c,l,u,p;for(e.indent=e.bare?"":q,e.level=L,this.spaced=!0,e.scope=new P(null,this,null,null!=(l=e.referencedVars)?l:[]),u=e.locals||[],r=0,s=u.length;s>r;r++)o=u[r],e.scope.parameter(o);return a=[],e.bare||(c=function(){var e,n,r,s;for(r=this.expressions,s=[],i=e=0,n=r.length;n>e&&(t=r[i],t.unwrap()instanceof h);i=++e)s.push(t);return s}.call(this),p=this.expressions.slice(c.length),this.expressions=c,c.length&&(a=this.compileNode(ht(e,{indent:""})),a.push(this.makeCode("\n"))),this.expressions=p),n=this.compileWithDeclarations(e),e.bare?n:[].concat(a,this.makeCode("(function() {\n"),n,this.makeCode("\n}).call(this);\n"))},t.prototype.compileWithDeclarations=function(e){var t,n,i,r,s,o,a,c,l,u,p,d,f,m;for(r=[],c=[],l=this.expressions,s=o=0,a=l.length;a>o&&(i=l[s],i=i.unwrap(),i instanceof h||i instanceof x);s=++o);return e=ht(e,{level:L}),s&&(d=this.expressions.splice(s,9e9),u=[this.spaced,!1],m=u[0],this.spaced=u[1],p=[this.compileNode(e),m],r=p[0],this.spaced=p[1],this.expressions=d),c=this.compileNode(e),f=e.scope,f.expressions===this&&(n=e.scope.hasDeclarations(),t=f.hasAssignments,n||t?(s&&r.push(this.makeCode("\n")),r.push(this.makeCode(this.tab+"var ")),n&&r.push(this.makeCode(f.declaredVariables().join(", "))),t&&(n&&r.push(this.makeCode(",\n"+(this.tab+q))),r.push(this.makeCode(f.assignedVariables().join(",\n"+(this.tab+q))))),r.push(this.makeCode(";\n"+(this.spaced?"\n":"")))):r.length&&c.length&&r.push(this.makeCode("\n"))),r.concat(c)},t.wrap=function(e){return 1===e.length&&e[0]instanceof t?e[0]:new t(e)},t}(r),e.Literal=x=function(e){function t(e){this.value=e}return kt(t,e),t.prototype.makeReturn=function(){return this.isStatement()?this:t.__super__.makeReturn.apply(this,arguments)},t.prototype.isAssignable=function(){return g.test(this.value)},t.prototype.isStatement=function(){var e;return"break"===(e=this.value)||"continue"===e||"debugger"===e},t.prototype.isComplex=D,t.prototype.assigns=function(e){return e===this.value},t.prototype.jumps=function(e){return"break"!==this.value||(null!=e?e.loop:void 0)||(null!=e?e.block:void 0)?"continue"!==this.value||(null!=e?e.loop:void 0)?void 0:this:this},t.prototype.compileNode=function(e){var t,n,i;return n="this"===this.value?(null!=(i=e.scope.method)?i.bound:void 0)?e.scope.method.context:this.value:this.value.reserved?'"'+this.value+'"':this.value,t=this.isStatement()?""+this.tab+n+";":n,[this.makeCode(t)]},t.prototype.toString=function(){return' "'+this.value+'"'},t}(r),e.Undefined=function(e){function t(){return t.__super__.constructor.apply(this,arguments)}return kt(t,e),t.prototype.isAssignable=D,t.prototype.isComplex=D,t.prototype.compileNode=function(e){return[this.makeCode(e.level>=T?"(void 0)":"void 0")]},t}(r),e.Null=function(e){function t(){return t.__super__.constructor.apply(this,arguments)}return kt(t,e),t.prototype.isAssignable=D,t.prototype.isComplex=D,t.prototype.compileNode=function(){return[this.makeCode("null")]},t}(r),e.Bool=function(e){function t(e){this.val=e}return kt(t,e),t.prototype.isAssignable=D,t.prototype.isComplex=D,t.prototype.compileNode=function(){return[this.makeCode(this.val)]},t}(r),e.Return=M=function(e){function t(e){this.expression=e}return kt(t,e),t.prototype.children=["expression"],t.prototype.isStatement=Q,t.prototype.makeReturn=X,t.prototype.jumps=X,t.prototype.compileToFragments=function(e,n){var i,r;return i=null!=(r=this.expression)?r.makeReturn():void 0,!i||i instanceof t?t.__super__.compileToFragments.call(this,e,n):i.compileToFragments(e,n)},t.prototype.compileNode=function(e){var t,n,i;return t=[],n=null!=(i=this.expression)?"function"==typeof i.isYieldReturn?i.isYieldReturn():void 0:void 0,n||t.push(this.makeCode(this.tab+("return"+(this.expression?" ":"")))),this.expression&&(t=t.concat(this.expression.compileToFragments(e,N))),n||t.push(this.makeCode(";")),t},t}(r),e.Value=z=function(e){function t(e,n,i){return!n&&e instanceof t?e:(this.base=e,this.properties=n||[],i&&(this[i]=!0),this)}return kt(t,e),t.prototype.children=["base","properties"],t.prototype.add=function(e){return this.properties=this.properties.concat(e),this},t.prototype.hasProperties=function(){return!!this.properties.length},t.prototype.bareLiteral=function(e){return!this.properties.length&&this.base instanceof e},t.prototype.isArray=function(){return this.bareLiteral(n)},t.prototype.isRange=function(){return this.bareLiteral(j)},t.prototype.isComplex=function(){return this.hasProperties()||this.base.isComplex()},t.prototype.isAssignable=function(){return this.hasProperties()||this.base.isAssignable()},t.prototype.isSimpleNumber=function(){return this.bareLiteral(x)&&B.test(this.base.value)},t.prototype.isString=function(){return this.bareLiteral(x)&&b.test(this.base.value)},t.prototype.isRegex=function(){return this.bareLiteral(x)&&v.test(this.base.value)},t.prototype.isAtomic=function(){var e,t,n,i;for(i=this.properties.concat(this.base),e=0,t=i.length;t>e;e++)if(n=i[e],n.soak||n instanceof o)return!1;return!0},t.prototype.isNotCallable=function(){return this.isSimpleNumber()||this.isString()||this.isRegex()||this.isArray()||this.isRange()||this.isSplice()||this.isObject()},t.prototype.isStatement=function(e){return!this.properties.length&&this.base.isStatement(e)},t.prototype.assigns=function(e){return!this.properties.length&&this.base.assigns(e)},t.prototype.jumps=function(e){return!this.properties.length&&this.base.jumps(e)},t.prototype.isObject=function(e){return this.properties.length?!1:this.base instanceof A&&(!e||this.base.generated)},t.prototype.isSplice=function(){var e,t;return t=this.properties,e=t[t.length-1],e instanceof U},t.prototype.looksStatic=function(e){var t;return this.base.value===e&&1===this.properties.length&&"prototype"!==(null!=(t=this.properties[0].name)?t.value:void 0)},t.prototype.unwrap=function(){return this.properties.length?this:this.base},t.prototype.cacheReference=function(e){var n,r,s,o,a;return a=this.properties,s=a[a.length-1],2>this.properties.length&&!this.base.isComplex()&&!(null!=s?s.isComplex():void 0)?[this,this]:(n=new t(this.base,this.properties.slice(0,-1)),n.isComplex()&&(r=new x(e.scope.freeVariable("base")),n=new t(new O(new i(r,n)))),s?(s.isComplex()&&(o=new x(e.scope.freeVariable("name")),s=new w(new i(o,s.index)),o=new w(o)),[n.add(s),new t(r||n.base,[o||s])]):[n,r])},t.prototype.compileNode=function(e){var t,n,i,r,s;for(this.base.front=this.front,s=this.properties,t=this.base.compileToFragments(e,s.length?T:null),(this.base instanceof O||s.length)&&B.test(st(t))&&t.push(this.makeCode(".")),n=0,i=s.length;i>n;n++)r=s[n],t.push.apply(t,r.compileToFragments(e));return t},t.prototype.unfoldSoak=function(e){return null!=this.unfoldedSoak?this.unfoldedSoak:this.unfoldedSoak=function(n){return function(){var r,s,o,a,c,l,h,p,d,f;if(o=n.base.unfoldSoak(e))return(p=o.body.properties).push.apply(p,n.properties),o;for(d=n.properties,s=a=0,c=d.length;c>a;s=++a)if(l=d[s],l.soak)return l.soak=!1,r=new t(n.base,n.properties.slice(0,s)),f=new t(n.base,n.properties.slice(s)),r.isComplex()&&(h=new x(e.scope.freeVariable("ref")),r=new O(new i(h,r)),f.base=h),new y(new u(r),f,{soak:!0});return!1}}(this)()},t}(r),e.Comment=h=function(e){function t(e){this.comment=e}return kt(t,e),t.prototype.isStatement=Q,t.prototype.makeReturn=X,t.prototype.compileNode=function(e,t){var n,i;return i=this.comment.replace(/^(\s*)#(?=\s)/gm,"$1 *"),n="/*"+ut(i,this.tab)+(Tt.call(i,"\n")>=0?"\n"+this.tab:"")+" */",(t||e.level)===L&&(n=e.indent+n),[this.makeCode("\n"),this.makeCode(n)]},t}(r),e.Call=o=function(e){function n(e,t,n){this.args=null!=t?t:[],this.soak=n,this.isNew=!1,this.isSuper="super"===e,this.variable=this.isSuper?null:e,e instanceof z&&e.isNotCallable()&&e.error("literal is not a function")}return kt(n,e),n.prototype.children=["variable","args"],n.prototype.newInstance=function(){var e,t;return e=(null!=(t=this.variable)?t.base:void 0)||this.variable,e instanceof n&&!e.isNew?e.newInstance():this.isNew=!0,this},n.prototype.superReference=function(e){var n,r,s,o,a,c,l,h;return a=e.scope.namedMethod(),(null!=a?a.klass:void 0)?(o=a.klass,c=a.name,h=a.variable,o.isComplex()&&(s=new x(e.scope.parent.freeVariable("base")),r=new z(new O(new i(s,o))),h.base=r,h.properties.splice(0,o.properties.length)),(c.isComplex()||c instanceof w&&c.index.isAssignable())&&(l=new x(e.scope.parent.freeVariable("name")),c=new w(new i(l,c.index)),h.properties.pop(),h.properties.push(c)),n=[new t(new x("__super__"))],a["static"]&&n.push(new t(new x("constructor"))),n.push(null!=l?new w(l):c),new z(null!=s?s:o,n).compile(e)):(null!=a?a.ctor:void 0)?a.name+".__super__.constructor":this.error("cannot call super outside of an instance method.")},n.prototype.superThis=function(e){var t;return t=e.scope.method,t&&!t.klass&&t.context||"this"},n.prototype.unfoldSoak=function(e){var t,i,r,s,o,a,c,l,h;if(this.soak){if(this.variable){if(i=bt(e,this,"variable"))return i;c=new z(this.variable).cacheReference(e),s=c[0],h=c[1]}else s=new x(this.superReference(e)),h=new z(s);return h=new n(h,this.args),h.isNew=this.isNew,s=new x("typeof "+s.compile(e)+' === "function"'),new y(s,new z(h),{soak:!0})}for(t=this,a=[];;)if(t.variable instanceof n)a.push(t),t=t.variable;else{if(!(t.variable instanceof z))break;if(a.push(t),!((t=t.variable.base)instanceof n))break}for(l=a.reverse(),r=0,o=l.length;o>r;r++)t=l[r],i&&(t.variable instanceof n?t.variable=i:t.variable.base=i),i=bt(e,t,"variable");return i},n.prototype.compileNode=function(e){var t,n,i,r,s,o,a,c,l,h;if(null!=(l=this.variable)&&(l.front=this.front),r=G.compileSplattedArray(e,this.args,!0),r.length)return this.compileSplat(e,r);for(i=[],h=this.args,n=o=0,a=h.length;a>o;n=++o)t=h[n],n&&i.push(this.makeCode(", ")),i.push.apply(i,t.compileToFragments(e,F));return s=[],this.isSuper?(c=this.superReference(e)+(".call("+this.superThis(e)),i.length&&(c+=", "),s.push(this.makeCode(c))):(this.isNew&&s.push(this.makeCode("new ")),s.push.apply(s,this.variable.compileToFragments(e,T)),s.push(this.makeCode("("))),s.push.apply(s,i),s.push(this.makeCode(")")),s},n.prototype.compileSplat=function(e,t){var n,i,r,s,o,a;return this.isSuper?[].concat(this.makeCode(this.superReference(e)+".apply("+this.superThis(e)+", "),t,this.makeCode(")")):this.isNew?(s=this.tab+q,[].concat(this.makeCode("(function(func, args, ctor) {\n"+s+"ctor.prototype = func.prototype;\n"+s+"var child = new ctor, result = func.apply(child, args);\n"+s+"return Object(result) === result ? result : child;\n"+this.tab+"})("),this.variable.compileToFragments(e,F),this.makeCode(", "),t,this.makeCode(", function(){})"))):(n=[],i=new z(this.variable),(o=i.properties.pop())&&i.isComplex()?(a=e.scope.freeVariable("ref"),n=n.concat(this.makeCode("("+a+" = "),i.compileToFragments(e,F),this.makeCode(")"),o.compileToFragments(e))):(r=i.compileToFragments(e,T),B.test(st(r))&&(r=this.wrapInBraces(r)),o?(a=st(r),r.push.apply(r,o.compileToFragments(e))):a="null",n=n.concat(r)),n=n.concat(this.makeCode(".apply("+a+", "),t,this.makeCode(")")))},n}(r),e.Extends=d=function(e){function t(e,t){this.child=e,this.parent=t}return kt(t,e),t.prototype.children=["child","parent"],t.prototype.compileToFragments=function(e){return new o(new z(new x(yt("extend",e))),[this.child,this.parent]).compileToFragments(e)},t}(r),e.Access=t=function(e){function t(e,t){this.name=e,this.name.asKey=!0,this.soak="soak"===t}return kt(t,e),t.prototype.children=["name"],t.prototype.compileToFragments=function(e){var t;return t=this.name.compileToFragments(e),g.test(st(t))?t.unshift(this.makeCode(".")):(t.unshift(this.makeCode("[")),t.push(this.makeCode("]"))),t},t.prototype.isComplex=D,t}(r),e.Index=w=function(e){function t(e){this.index=e}return kt(t,e),t.prototype.children=["index"],t.prototype.compileToFragments=function(e){return[].concat(this.makeCode("["),this.index.compileToFragments(e,N),this.makeCode("]"))},t.prototype.isComplex=function(){return this.index.isComplex()},t}(r),e.Range=j=function(e){function t(e,t,n){this.from=e,this.to=t,this.exclusive="exclusive"===n,this.equals=this.exclusive?"":"="}return kt(t,e),t.prototype.children=["from","to"],t.prototype.compileVariables=function(e){var t,n,i,r,s,o;return e=ht(e,{top:!0}),t=tt(e,"isComplex"),n=this.cacheToCodeFragments(this.from.cache(e,F,t)),this.fromC=n[0],this.fromVar=n[1],i=this.cacheToCodeFragments(this.to.cache(e,F,t)),this.toC=i[0],this.toVar=i[1],(o=tt(e,"step"))&&(r=this.cacheToCodeFragments(o.cache(e,F,t)),this.step=r[0],this.stepVar=r[1]),s=[this.fromVar.match(R),this.toVar.match(R)],this.fromNum=s[0],this.toNum=s[1],this.stepVar?this.stepNum=this.stepVar.match(R):void 0},t.prototype.compileNode=function(e){var t,n,i,r,s,o,a,c,l,h,u,p,d,f;return this.fromVar||this.compileVariables(e),e.index?(a=this.fromNum&&this.toNum,s=tt(e,"index"),o=tt(e,"name"),l=o&&o!==s,f=s+" = "+this.fromC,this.toC!==this.toVar&&(f+=", "+this.toC),this.step!==this.stepVar&&(f+=", "+this.step),h=[s+" <"+this.equals,s+" >"+this.equals],c=h[0],r=h[1],n=this.stepNum?pt(this.stepNum[0])>0?c+" "+this.toVar:r+" "+this.toVar:a?(u=[pt(this.fromNum[0]),pt(this.toNum[0])],i=u[0],d=u[1],u,d>=i?c+" "+d:r+" "+d):(t=this.stepVar?this.stepVar+" > 0":this.fromVar+" <= "+this.toVar,t+" ? "+c+" "+this.toVar+" : "+r+" "+this.toVar),p=this.stepVar?s+" += "+this.stepVar:a?l?d>=i?"++"+s:"--"+s:d>=i?s+"++":s+"--":l?t+" ? ++"+s+" : --"+s:t+" ? "+s+"++ : "+s+"--",l&&(f=o+" = "+f),l&&(p=o+" = "+p),[this.makeCode(f+"; "+n+"; "+p)]):this.compileArray(e)},t.prototype.compileArray=function(e){var t,n,i,r,s,o,a,c,l,h,u,p,d;return this.fromNum&&this.toNum&&20>=Math.abs(this.fromNum-this.toNum)?(l=function(){p=[];for(var e=h=+this.fromNum,t=+this.toNum;t>=h?t>=e:e>=t;t>=h?e++:e--)p.push(e);return p}.apply(this),this.exclusive&&l.pop(),[this.makeCode("["+l.join(", ")+"]")]):(o=this.tab+q,s=e.scope.freeVariable("i",{single:!0}),u=e.scope.freeVariable("results"),c="\n"+o+u+" = [];",this.fromNum&&this.toNum?(e.index=s,n=st(this.compileNode(e))):(d=s+" = "+this.fromC+(this.toC!==this.toVar?", "+this.toC:""),i=this.fromVar+" <= "+this.toVar,n="var "+d+"; "+i+" ? "+s+" <"+this.equals+" "+this.toVar+" : "+s+" >"+this.equals+" "+this.toVar+"; "+i+" ? "+s+"++ : "+s+"--"),a="{ "+u+".push("+s+"); }\n"+o+"return "+u+";\n"+e.indent,r=function(e){return null!=e?e.contains(at):void 0},(r(this.from)||r(this.to))&&(t=", arguments"),[this.makeCode("(function() {"+c+"\n"+o+"for ("+n+")"+a+"}).apply(this"+(null!=t?t:"")+")")])},t}(r),e.Slice=U=function(e){function t(e){this.range=e,t.__super__.constructor.call(this)}return kt(t,e),t.prototype.children=["range"],t.prototype.compileNode=function(e){var t,n,i,r,s,o,a;return s=this.range,o=s.to,i=s.from,r=i&&i.compileToFragments(e,N)||[this.makeCode("0")],o&&(t=o.compileToFragments(e,N),n=st(t),(this.range.exclusive||-1!==+n)&&(a=", "+(this.range.exclusive?n:B.test(n)?""+(+n+1):(t=o.compileToFragments(e,T),"+"+st(t)+" + 1 || 9e9")))),[this.makeCode(".slice("+st(r)+(a||"")+")")]},t}(r),e.Obj=A=function(e){function n(e,t){this.generated=null!=t?t:!1,this.objects=this.properties=e||[]}return kt(n,e),n.prototype.children=["properties"],n.prototype.compileNode=function(e){var n,r,s,o,a,c,l,u,p,d,f,m,g,v,b,y,k,w,T,C,F;if(T=this.properties,this.generated)for(l=0,g=T.length;g>l;l++)y=T[l],y instanceof z&&y.error("cannot have an implicit value in an implicit object");for(r=p=0,v=T.length;v>p&&(w=T[r],!((w.variable||w).base instanceof O));r=++p);for(s=T.length>r,a=e.indent+=q,m=this.lastNonComment(this.properties),n=[],s&&(k=e.scope.freeVariable("obj"),n.push(this.makeCode("(\n"+a+k+" = "))),n.push(this.makeCode("{"+(0===T.length||0===r?"}":"\n"))),o=f=0,b=T.length;b>f;o=++f)w=T[o],o===r&&(0!==o&&n.push(this.makeCode("\n"+a+"}")),n.push(this.makeCode(",\n"))),u=o===T.length-1||o===r-1?"":w===m||w instanceof h?"\n":",\n",c=w instanceof h?"":a,s&&r>o&&(c+=q),w instanceof i&&("object"!==w.context&&w.operatorToken.error("unexpected "+w.operatorToken.value),w.variable instanceof z&&w.variable.hasProperties()&&w.variable.error("invalid object key")),w instanceof z&&w["this"]&&(w=new i(w.properties[0].name,w,"object")),w instanceof h||(r>o?(w instanceof i||(w=new i(w,w,"object")),(w.variable.base||w.variable).asKey=!0):(w instanceof i?(d=w.variable,F=w.value):(C=w.base.cache(e),d=C[0],F=C[1]),w=new i(new z(new x(k),[new t(d)]),F))),c&&n.push(this.makeCode(c)),n.push.apply(n,w.compileToFragments(e,L)),u&&n.push(this.makeCode(u));return s?n.push(this.makeCode(",\n"+a+k+"\n"+this.tab+")")):0!==T.length&&n.push(this.makeCode("\n"+this.tab+"}")),this.front&&!s?this.wrapInBraces(n):n},n.prototype.assigns=function(e){var t,n,i,r;for(r=this.properties,t=0,n=r.length;n>t;t++)if(i=r[t],i.assigns(e))return!0;return!1},n}(r),e.Arr=n=function(e){function t(e){this.objects=e||[]}return kt(t,e),t.prototype.children=["objects"],t.prototype.compileNode=function(e){var t,n,i,r,s,o,a;if(!this.objects.length)return[this.makeCode("[]")];if(e.indent+=q,t=G.compileSplattedArray(e,this.objects),t.length)return t;for(t=[],n=function(){var t,n,i,r;for(i=this.objects,r=[],t=0,n=i.length;n>t;t++)a=i[t],r.push(a.compileToFragments(e,F));return r}.call(this),r=s=0,o=n.length;o>s;r=++s)i=n[r],r&&t.push(this.makeCode(", ")),t.push.apply(t,i);return st(t).indexOf("\n")>=0?(t.unshift(this.makeCode("[\n"+e.indent)),t.push(this.makeCode("\n"+this.tab+"]"))):(t.unshift(this.makeCode("[")),t.push(this.makeCode("]"))),t},t.prototype.assigns=function(e){var t,n,i,r;for(r=this.objects,t=0,n=r.length;n>t;t++)if(i=r[t],i.assigns(e))return!0;return!1},t}(r),e.Class=a=function(e){function n(e,t,n){this.variable=e,this.parent=t,this.body=null!=n?n:new s,this.boundFuncs=[],this.body.classBody=!0}return kt(n,e),n.prototype.children=["variable","parent","body"],n.prototype.determineName=function(){var e,n,i;return this.variable?(n=this.variable.properties,i=n[n.length-1],e=i?i instanceof t&&i.name.value:this.variable.base.value,Tt.call(V,e)>=0&&this.variable.error("class variable name may not be "+e),e&&(e=g.test(e)&&e)):null},n.prototype.setContext=function(e){return this.body.traverseChildren(!1,function(t){return t.classBody?!1:t instanceof x&&"this"===t.value?t.value=e:t instanceof c&&t.bound?t.context=e:void 0})},n.prototype.addBoundFunctions=function(e){var n,i,r,s,o;for(o=this.boundFuncs,i=0,r=o.length;r>i;i++)n=o[i],s=new z(new x("this"),[new t(n)]).compile(e),this.ctor.body.unshift(new x(s+" = "+yt("bind",e)+"("+s+", this)"))},n.prototype.addProperties=function(e,n,r){var s,o,a,l,h,u;return u=e.base.properties.slice(0),l=function(){var e;for(e=[];o=u.shift();)o instanceof i&&(a=o.variable.base,delete o.context,h=o.value,"constructor"===a.value?(this.ctor&&o.error("cannot define more than one constructor in a class"),h.bound&&o.error("cannot define a constructor as a bound function"),h instanceof c?o=this.ctor=h:(this.externalCtor=r.classScope.freeVariable("class"),o=new i(new x(this.externalCtor),h))):o.variable["this"]?h["static"]=!0:(s=a.isComplex()?new w(a):new t(a),o.variable=new z(new x(n),[new t(new x("prototype")),s]),h instanceof c&&h.bound&&(this.boundFuncs.push(a),h.bound=!1))),e.push(o);return e}.call(this),et(l)},n.prototype.walkBody=function(e,t){return this.traverseChildren(!1,function(r){return function(o){var a,c,l,h,u,p,d;if(a=!0,o instanceof n)return!1;if(o instanceof s){for(d=c=o.expressions,l=h=0,u=d.length;u>h;l=++h)p=d[l],p instanceof i&&p.variable.looksStatic(e)?p.value["static"]=!0:p instanceof z&&p.isObject(!0)&&(a=!1,c[l]=r.addProperties(p,e,t));o.expressions=c=rt(c)}return a&&!(o instanceof n)}}(this))},n.prototype.hoistDirectivePrologue=function(){var e,t,n;for(t=0,e=this.body.expressions;(n=e[t])&&n instanceof h||n instanceof z&&n.isString();)++t;return this.directives=e.splice(0,t)},n.prototype.ensureConstructor=function(e){return this.ctor||(this.ctor=new c,this.externalCtor?this.ctor.body.push(new x(this.externalCtor+".apply(this, arguments)")):this.parent&&this.ctor.body.push(new x(e+".__super__.constructor.apply(this, arguments)")),this.ctor.body.makeReturn(),this.body.expressions.unshift(this.ctor)),this.ctor.ctor=this.ctor.name=e,this.ctor.klass=null,this.ctor.noReturn=!0},n.prototype.compileNode=function(e){var t,n,r,a,l,h,u,p,f;return(a=this.body.jumps())&&a.error("Class bodies cannot contain pure statements"),(n=this.body.contains(at))&&n.error("Class bodies shouldn't reference arguments"),u=this.determineName()||"_Class",u.reserved&&(u="_"+u),h=new x(u),r=new c([],s.wrap([this.body])),t=[],e.classScope=r.makeScope(e.scope),this.hoistDirectivePrologue(),this.setContext(u),this.walkBody(u,e),this.ensureConstructor(u),this.addBoundFunctions(e),this.body.spaced=!0,this.body.expressions.push(h),this.parent&&(f=new x(e.classScope.freeVariable("superClass",{reserve:!1})),this.body.expressions.unshift(new d(h,f)),r.params.push(new _(f)),t.push(this.parent)),(p=this.body.expressions).unshift.apply(p,this.directives),l=new O(new o(r,t)),this.variable&&(l=new i(this.variable,l)),l.compileToFragments(e)},n}(r),e.Assign=i=function(e){function n(e,t,n,i){var r,s,o;this.variable=e,this.value=t,this.context=n,null==i&&(i={}),this.param=i.param,this.subpattern=i.subpattern,this.operatorToken=i.operatorToken,o=s=this.variable.unwrapAll().value,r=Tt.call(V,o)>=0,r&&"object"!==this.context&&this.variable.error('variable name may not be "'+s+'"')
}return kt(n,e),n.prototype.children=["variable","value"],n.prototype.isStatement=function(e){return(null!=e?e.level:void 0)===L&&null!=this.context&&Tt.call(this.context,"?")>=0},n.prototype.assigns=function(e){return this["object"===this.context?"value":"variable"].assigns(e)},n.prototype.unfoldSoak=function(e){return bt(e,this,"variable")},n.prototype.compileNode=function(e){var t,n,i,r,s,o,a,l,h,u,p,d,f,m;if(i=this.variable instanceof z){if(this.variable.isArray()||this.variable.isObject())return this.compilePatternMatch(e);if(this.variable.isSplice())return this.compileSplice(e);if("||="===(l=this.context)||"&&="===l||"?="===l)return this.compileConditional(e);if("**="===(h=this.context)||"//="===h||"%%="===h)return this.compileSpecialMath(e)}return this.value instanceof c&&(this.value["static"]?(this.value.klass=this.variable.base,this.value.name=this.variable.properties[0],this.value.variable=this.variable):(null!=(u=this.variable.properties)?u.length:void 0)>=2&&(p=this.variable.properties,o=p.length>=3?Ct.call(p,0,r=p.length-2):(r=0,[]),a=p[r++],s=p[r++],"prototype"===(null!=(d=a.name)?d.value:void 0)&&(this.value.klass=new z(this.variable.base,o),this.value.name=s,this.value.variable=this.variable))),this.context||(m=this.variable.unwrapAll(),m.isAssignable()||this.variable.error('"'+this.variable.compile(e)+'" cannot be assigned'),("function"==typeof m.hasProperties?m.hasProperties():void 0)||(this.param?e.scope.add(m.value,"var"):e.scope.find(m.value))),f=this.value.compileToFragments(e,F),i&&this.variable.base instanceof A&&(this.variable.front=!0),n=this.variable.compileToFragments(e,F),"object"===this.context?n.concat(this.makeCode(": "),f):(t=n.concat(this.makeCode(" "+(this.context||"=")+" "),f),F>=e.level?t:this.wrapInBraces(t))},n.prototype.compilePatternMatch=function(e){var i,r,s,o,a,c,l,h,u,d,f,m,v,b,y,k,T,C,N,S,D,R,A,_,O,j,M,B;if(_=e.level===L,j=this.value,y=this.variable.base.objects,!(k=y.length))return s=j.compileToFragments(e),e.level>=E?this.wrapInBraces(s):s;if(b=y[0],1===k&&b instanceof p&&b.error("Destructuring assignment has no target"),u=this.variable.isObject(),_&&1===k&&!(b instanceof G))return o=null,b instanceof n&&"object"===b.context?(C=b,N=C.variable,h=N.base,b=C.value,b instanceof n&&(o=b.value,b=b.variable)):(b instanceof n&&(o=b.value,b=b.variable),h=u?b["this"]?b.properties[0].name:b:new x(0)),i=g.test(h.unwrap().value),j=new z(j),j.properties.push(new(i?t:w)(h)),S=b.unwrap().value,Tt.call($,S)>=0&&b.error("assignment to a reserved word: "+b.compile(e)),o&&(j=new I("?",j,o)),new n(b,j,null,{param:this.param}).compileToFragments(e,L);for(M=j.compileToFragments(e,F),B=st(M),r=[],a=!1,(!g.test(B)||this.variable.assigns(B))&&(r.push([this.makeCode((T=e.scope.freeVariable("ref"))+" = ")].concat(Ct.call(M))),M=[this.makeCode(T)],B=T),l=f=0,m=y.length;m>f;l=++f){if(b=y[l],h=l,!a&&b instanceof G)v=b.name.unwrap().value,b=b.unwrap(),O=k+" <= "+B+".length ? "+yt("slice",e)+".call("+B+", "+l,(A=k-l-1)?(d=e.scope.freeVariable("i",{single:!0}),O+=", "+d+" = "+B+".length - "+A+") : ("+d+" = "+l+", [])"):O+=") : []",O=new x(O),a=d+"++";else{if(!a&&b instanceof p){(A=k-l-1)&&(1===A?a=B+".length - 1":(d=e.scope.freeVariable("i",{single:!0}),O=new x(d+" = "+B+".length - "+A),a=d+"++",r.push(O.compileToFragments(e,F))));continue}(b instanceof G||b instanceof p)&&b.error("multiple splats/expansions are disallowed in an assignment"),o=null,b instanceof n&&"object"===b.context?(D=b,R=D.variable,h=R.base,b=D.value,b instanceof n&&(o=b.value,b=b.variable)):(b instanceof n&&(o=b.value,b=b.variable),h=u?b["this"]?b.properties[0].name:b:new x(a||h)),v=b.unwrap().value,i=g.test(h.unwrap().value),O=new z(new x(B),[new(i?t:w)(h)]),o&&(O=new I("?",O,o))}null!=v&&Tt.call($,v)>=0&&b.error("assignment to a reserved word: "+b.compile(e)),r.push(new n(b,O,null,{param:this.param,subpattern:!0}).compileToFragments(e,F))}return _||this.subpattern||r.push(M),c=this.joinFragmentArrays(r,", "),F>e.level?c:this.wrapInBraces(c)},n.prototype.compileConditional=function(e){var t,i,r,s;return r=this.variable.cacheReference(e),i=r[0],s=r[1],!i.properties.length&&i.base instanceof x&&"this"!==i.base.value&&!e.scope.check(i.base.value)&&this.variable.error('the variable "'+i.base.value+"\" can't be assigned with "+this.context+" because it has not been declared before"),Tt.call(this.context,"?")>=0?(e.isExistentialEquals=!0,new y(new u(i),s,{type:"if"}).addElse(new n(s,this.value,"=")).compileToFragments(e)):(t=new I(this.context.slice(0,-1),i,new n(s,this.value,"=")).compileToFragments(e),F>=e.level?t:this.wrapInBraces(t))},n.prototype.compileSpecialMath=function(e){var t,i,r;return i=this.variable.cacheReference(e),t=i[0],r=i[1],new n(t,new I(this.context.slice(0,-1),r,this.value)).compileToFragments(e)},n.prototype.compileSplice=function(e){var t,n,i,r,s,o,a,c,l,h,u,p;return a=this.variable.properties.pop().range,i=a.from,h=a.to,n=a.exclusive,o=this.variable.compile(e),i?(c=this.cacheToCodeFragments(i.cache(e,E)),r=c[0],s=c[1]):r=s="0",h?i instanceof z&&i.isSimpleNumber()&&h instanceof z&&h.isSimpleNumber()?(h=h.compile(e)-s,n||(h+=1)):(h=h.compile(e,T)+" - "+s,n||(h+=" + 1")):h="9e9",l=this.value.cache(e,F),u=l[0],p=l[1],t=[].concat(this.makeCode("[].splice.apply("+o+", ["+r+", "+h+"].concat("),u,this.makeCode(")), "),p),e.level>L?this.wrapInBraces(t):t},n}(r),e.Code=c=function(e){function t(e,t,n){this.params=e||[],this.body=t||new s,this.bound="boundfunc"===n,this.isGenerator=!!this.body.contains(function(e){var t;return e instanceof I&&("yield"===(t=e.operator)||"yield*"===t)})}return kt(t,e),t.prototype.children=["params","body"],t.prototype.isStatement=function(){return!!this.ctor},t.prototype.jumps=D,t.prototype.makeScope=function(e){return new P(e,this.body,this)},t.prototype.compileNode=function(e){var r,a,c,l,h,u,d,f,m,g,v,b,k,w,C,F,E,N,L,S,D,R,A,O,$,j,M,B,V,P,U,G,H;if(this.bound&&(null!=(A=e.scope.method)?A.bound:void 0)&&(this.context=e.scope.method.context),this.bound&&!this.context)return this.context="_this",H=new t([new _(new x(this.context))],new s([this])),a=new o(H,[new x("this")]),a.updateLocationDataIfMissing(this.locationData),a.compileNode(e);for(e.scope=tt(e,"classScope")||this.makeScope(e.scope),e.scope.shared=tt(e,"sharedScope"),e.indent+=q,delete e.bare,delete e.isExistentialEquals,L=[],l=[],O=this.params,u=0,m=O.length;m>u;u++)N=O[u],N instanceof p||e.scope.parameter(N.asReference(e));for($=this.params,d=0,g=$.length;g>d;d++)if(N=$[d],N.splat||N instanceof p){for(j=this.params,f=0,v=j.length;v>f;f++)E=j[f],E instanceof p||!E.name.value||e.scope.add(E.name.value,"var",!0);V=new i(new z(new n(function(){var t,n,i,r;for(i=this.params,r=[],n=0,t=i.length;t>n;n++)E=i[n],r.push(E.asReference(e));return r}.call(this))),new z(new x("arguments")));break}for(M=this.params,F=0,b=M.length;b>F;F++)N=M[F],N.isComplex()?(U=R=N.asReference(e),N.value&&(U=new I("?",R,N.value)),l.push(new i(new z(N.name),U,"=",{param:!0}))):(R=N,N.value&&(C=new x(R.name.value+" == null"),U=new i(new z(N.name),N.value,"="),l.push(new y(C,U)))),V||L.push(R);for(G=this.body.isEmpty(),V&&l.unshift(V),l.length&&(B=this.body.expressions).unshift.apply(B,l),h=S=0,k=L.length;k>S;h=++S)E=L[h],L[h]=E.compileToFragments(e),e.scope.parameter(st(L[h]));for(P=[],this.eachParamName(function(e,t){return Tt.call(P,e)>=0&&t.error("multiple parameters named "+e),P.push(e)}),G||this.noReturn||this.body.makeReturn(),c="function",this.isGenerator&&(c+="*"),this.ctor&&(c+=" "+this.name),c+="(",r=[this.makeCode(c)],h=D=0,w=L.length;w>D;h=++D)E=L[h],h&&r.push(this.makeCode(", ")),r.push.apply(r,E);return r.push(this.makeCode(") {")),this.body.isEmpty()||(r=r.concat(this.makeCode("\n"),this.body.compileWithDeclarations(e),this.makeCode("\n"+this.tab))),r.push(this.makeCode("}")),this.ctor?[this.makeCode(this.tab)].concat(Ct.call(r)):this.front||e.level>=T?this.wrapInBraces(r):r},t.prototype.eachParamName=function(e){var t,n,i,r,s;for(r=this.params,s=[],t=0,n=r.length;n>t;t++)i=r[t],s.push(i.eachName(e));return s},t.prototype.traverseChildren=function(e,n){return e?t.__super__.traverseChildren.call(this,e,n):void 0},t}(r),e.Param=_=function(e){function t(e,t,n){var i,r,s;this.name=e,this.value=t,this.splat=n,r=i=this.name.unwrapAll().value,Tt.call(V,r)>=0&&this.name.error('parameter name "'+i+'" is not allowed'),this.name instanceof A&&this.name.generated&&(s=this.name.objects[0].operatorToken,s.error("unexpected "+s.value))}return kt(t,e),t.prototype.children=["name","value"],t.prototype.compileToFragments=function(e){return this.name.compileToFragments(e,F)},t.prototype.asReference=function(e){var t,n;return this.reference?this.reference:(n=this.name,n["this"]?(t=n.properties[0].name.value,t.reserved&&(t="_"+t),n=new x(e.scope.freeVariable(t))):n.isComplex()&&(n=new x(e.scope.freeVariable("arg"))),n=new z(n),this.splat&&(n=new G(n)),n.updateLocationDataIfMissing(this.locationData),this.reference=n)},t.prototype.isComplex=function(){return this.name.isComplex()},t.prototype.eachName=function(e,t){var n,r,s,o,a,c;if(null==t&&(t=this.name),n=function(t){return e("@"+t.properties[0].name.value,t)},t instanceof x)return e(t.value,t);if(t instanceof z)return n(t);for(c=t.objects,r=0,s=c.length;s>r;r++)a=c[r],a instanceof i&&null==a.context&&(a=a.variable),a instanceof i?this.eachName(e,a.value.unwrap()):a instanceof G?(o=a.name.unwrap(),e(o.value,o)):a instanceof z?a.isArray()||a.isObject()?this.eachName(e,a.base):a["this"]?n(a):e(a.base.value,a.base):a instanceof p||a.error("illegal parameter "+a.compile())},t}(r),e.Splat=G=function(e){function t(e){this.name=e.compile?e:new x(e)}return kt(t,e),t.prototype.children=["name"],t.prototype.isAssignable=Q,t.prototype.assigns=function(e){return this.name.assigns(e)},t.prototype.compileToFragments=function(e){return this.name.compileToFragments(e)},t.prototype.unwrap=function(){return this.name},t.compileSplattedArray=function(e,n,i){var r,s,o,a,c,l,h,u,p,d,f;for(h=-1;(f=n[++h])&&!(f instanceof t););if(h>=n.length)return[];if(1===n.length)return f=n[0],c=f.compileToFragments(e,F),i?c:[].concat(f.makeCode(yt("slice",e)+".call("),c,f.makeCode(")"));for(r=n.slice(h),l=u=0,d=r.length;d>u;l=++u)f=r[l],o=f.compileToFragments(e,F),r[l]=f instanceof t?[].concat(f.makeCode(yt("slice",e)+".call("),o,f.makeCode(")")):[].concat(f.makeCode("["),o,f.makeCode("]"));return 0===h?(f=n[0],a=f.joinFragmentArrays(r.slice(1),", "),r[0].concat(f.makeCode(".concat("),a,f.makeCode(")"))):(s=function(){var t,i,r,s;for(r=n.slice(0,h),s=[],t=0,i=r.length;i>t;t++)f=r[t],s.push(f.compileToFragments(e,F));return s}(),s=n[0].joinFragmentArrays(s,", "),a=n[h].joinFragmentArrays(r,", "),p=n[n.length-1],[].concat(n[0].makeCode("["),s,n[h].makeCode("].concat("),a,p.makeCode(")")))},t}(r),e.Expansion=p=function(e){function t(){return t.__super__.constructor.apply(this,arguments)}return kt(t,e),t.prototype.isComplex=D,t.prototype.compileNode=function(){return this.error("Expansion must be used inside a destructuring assignment or parameter list")},t.prototype.asReference=function(){return this},t.prototype.eachName=function(){},t}(r),e.While=J=function(e){function t(e,t){this.condition=(null!=t?t.invert:void 0)?e.invert():e,this.guard=null!=t?t.guard:void 0}return kt(t,e),t.prototype.children=["condition","guard","body"],t.prototype.isStatement=Q,t.prototype.makeReturn=function(e){return e?t.__super__.makeReturn.apply(this,arguments):(this.returns=!this.jumps({loop:!0}),this)},t.prototype.addBody=function(e){return this.body=e,this},t.prototype.jumps=function(){var e,t,n,i,r;if(e=this.body.expressions,!e.length)return!1;for(t=0,i=e.length;i>t;t++)if(r=e[t],n=r.jumps({loop:!0}))return n;return!1},t.prototype.compileNode=function(e){var t,n,i,r;return e.indent+=q,r="",n=this.body,n.isEmpty()?n=this.makeCode(""):(this.returns&&(n.makeReturn(i=e.scope.freeVariable("results")),r=""+this.tab+i+" = [];\n"),this.guard&&(n.expressions.length>1?n.expressions.unshift(new y(new O(this.guard).invert(),new x("continue"))):this.guard&&(n=s.wrap([new y(this.guard,n)]))),n=[].concat(this.makeCode("\n"),n.compileToFragments(e,L),this.makeCode("\n"+this.tab))),t=[].concat(this.makeCode(r+this.tab+"while ("),this.condition.compileToFragments(e,N),this.makeCode(") {"),n,this.makeCode("}")),this.returns&&t.push(this.makeCode("\n"+this.tab+"return "+i+";")),t},t}(r),e.Op=I=function(e){function n(e,t,n,i){if("in"===e)return new k(t,n);if("do"===e)return this.generateDo(t);if("new"===e){if(t instanceof o&&!t["do"]&&!t.isNew)return t.newInstance();(t instanceof c&&t.bound||t["do"])&&(t=new O(t))}return this.operator=r[e]||e,this.first=t,this.second=n,this.flip=!!i,this}var r,s;return kt(n,e),r={"==":"===","!=":"!==",of:"in",yieldfrom:"yield*"},s={"!==":"===","===":"!=="},n.prototype.children=["first","second"],n.prototype.isSimpleNumber=D,n.prototype.isYield=function(){var e;return"yield"===(e=this.operator)||"yield*"===e},n.prototype.isYieldReturn=function(){return this.isYield()&&this.first instanceof M},n.prototype.isUnary=function(){return!this.second},n.prototype.isComplex=function(){var e;return!(this.isUnary()&&("+"===(e=this.operator)||"-"===e)&&this.first instanceof z&&this.first.isSimpleNumber())},n.prototype.isChainable=function(){var e;return"<"===(e=this.operator)||">"===e||">="===e||"<="===e||"==="===e||"!=="===e},n.prototype.invert=function(){var e,t,i,r,o;if(this.isChainable()&&this.first.isChainable()){for(e=!0,t=this;t&&t.operator;)e&&(e=t.operator in s),t=t.first;if(!e)return new O(this).invert();for(t=this;t&&t.operator;)t.invert=!t.invert,t.operator=s[t.operator],t=t.first;return this}return(r=s[this.operator])?(this.operator=r,this.first.unwrap()instanceof n&&this.first.invert(),this):this.second?new O(this).invert():"!"===this.operator&&(i=this.first.unwrap())instanceof n&&("!"===(o=i.operator)||"in"===o||"instanceof"===o)?i:new n("!",this)},n.prototype.unfoldSoak=function(e){var t;return("++"===(t=this.operator)||"--"===t||"delete"===t)&&bt(e,this,"first")},n.prototype.generateDo=function(e){var t,n,r,s,a,l,h,u;for(l=[],n=e instanceof i&&(h=e.value.unwrap())instanceof c?h:e,u=n.params||[],r=0,s=u.length;s>r;r++)a=u[r],a.value?(l.push(a.value),delete a.value):l.push(a);return t=new o(e,l),t["do"]=!0,t},n.prototype.compileNode=function(e){var t,n,i,r,s,o;if(n=this.isChainable()&&this.first.isChainable(),n||(this.first.front=this.front),"delete"===this.operator&&e.scope.check(this.first.unwrapAll().value)&&this.error("delete operand may not be argument or var"),("--"===(r=this.operator)||"++"===r)&&(s=this.first.unwrapAll().value,Tt.call(V,s)>=0)&&this.error('cannot increment/decrement "'+this.first.unwrapAll().value+'"'),this.isYield())return this.compileYield(e);if(this.isUnary())return this.compileUnary(e);if(n)return this.compileChain(e);switch(this.operator){case"?":return this.compileExistence(e);case"**":return this.compilePower(e);case"//":return this.compileFloorDivision(e);case"%%":return this.compileModulo(e);default:return i=this.first.compileToFragments(e,E),o=this.second.compileToFragments(e,E),t=[].concat(i,this.makeCode(" "+this.operator+" "),o),E>=e.level?t:this.wrapInBraces(t)}},n.prototype.compileChain=function(e){var t,n,i,r;return i=this.first.second.cache(e),this.first.second=i[0],r=i[1],n=this.first.compileToFragments(e,E),t=n.concat(this.makeCode(" "+(this.invert?"&&":"||")+" "),r.compileToFragments(e),this.makeCode(" "+this.operator+" "),this.second.compileToFragments(e,E)),this.wrapInBraces(t)},n.prototype.compileExistence=function(e){var t,n;return this.first.isComplex()?(n=new x(e.scope.freeVariable("ref")),t=new O(new i(n,this.first))):(t=this.first,n=t),new y(new u(t),n,{type:"if"}).addElse(this.second).compileToFragments(e)},n.prototype.compileUnary=function(e){var t,i,r;return i=[],t=this.operator,i.push([this.makeCode(t)]),"!"===t&&this.first instanceof u?(this.first.negated=!this.first.negated,this.first.compileToFragments(e)):e.level>=T?new O(this).compileToFragments(e):(r="+"===t||"-"===t,("new"===t||"typeof"===t||"delete"===t||r&&this.first instanceof n&&this.first.operator===t)&&i.push([this.makeCode(" ")]),(r&&this.first instanceof n||"new"===t&&this.first.isStatement(e))&&(this.first=new O(this.first)),i.push(this.first.compileToFragments(e,E)),this.flip&&i.reverse(),this.joinFragmentArrays(i,""))},n.prototype.compileYield=function(e){var t,n;return n=[],t=this.operator,null==e.scope.parent&&this.error("yield statements must occur within a function generator."),Tt.call(Object.keys(this.first),"expression")>=0&&!(this.first instanceof W)?this.isYieldReturn()?n.push(this.first.compileToFragments(e,L)):null!=this.first.expression&&n.push(this.first.expression.compileToFragments(e,E)):(n.push([this.makeCode("("+t+" ")]),n.push(this.first.compileToFragments(e,E)),n.push([this.makeCode(")")])),this.joinFragmentArrays(n,"")},n.prototype.compilePower=function(e){var n;return n=new z(new x("Math"),[new t(new x("pow"))]),new o(n,[this.first,this.second]).compileToFragments(e)},n.prototype.compileFloorDivision=function(e){var i,r;return r=new z(new x("Math"),[new t(new x("floor"))]),i=new n("/",this.first,this.second),new o(r,[i]).compileToFragments(e)},n.prototype.compileModulo=function(e){var t;return t=new z(new x(yt("modulo",e))),new o(t,[this.first,this.second]).compileToFragments(e)},n.prototype.toString=function(e){return n.__super__.toString.call(this,e,this.constructor.name+" "+this.operator)},n}(r),e.In=k=function(e){function t(e,t){this.object=e,this.array=t}return kt(t,e),t.prototype.children=["object","array"],t.prototype.invert=S,t.prototype.compileNode=function(e){var t,n,i,r,s;if(this.array instanceof z&&this.array.isArray()&&this.array.base.objects.length){for(s=this.array.base.objects,n=0,i=s.length;i>n;n++)if(r=s[n],r instanceof G){t=!0;break}if(!t)return this.compileOrTest(e)}return this.compileLoopTest(e)},t.prototype.compileOrTest=function(e){var t,n,i,r,s,o,a,c,l,h,u,p;for(c=this.object.cache(e,E),u=c[0],a=c[1],l=this.negated?[" !== "," && "]:[" === "," || "],t=l[0],n=l[1],p=[],h=this.array.base.objects,i=s=0,o=h.length;o>s;i=++s)r=h[i],i&&p.push(this.makeCode(n)),p=p.concat(i?a:u,this.makeCode(t),r.compileToFragments(e,T));return E>e.level?p:this.wrapInBraces(p)},t.prototype.compileLoopTest=function(e){var t,n,i,r;return i=this.object.cache(e,F),r=i[0],n=i[1],t=[].concat(this.makeCode(yt("indexOf",e)+".call("),this.array.compileToFragments(e,F),this.makeCode(", "),n,this.makeCode(") "+(this.negated?"< 0":">= 0"))),st(r)===st(n)?t:(t=r.concat(this.makeCode(", "),t),F>e.level?t:this.wrapInBraces(t))},t.prototype.toString=function(e){return t.__super__.toString.call(this,e,this.constructor.name+(this.negated?"!":""))},t}(r),e.Try=Y=function(e){function t(e,t,n,i){this.attempt=e,this.errorVariable=t,this.recovery=n,this.ensure=i}return kt(t,e),t.prototype.children=["attempt","recovery","ensure"],t.prototype.isStatement=Q,t.prototype.jumps=function(e){var t;return this.attempt.jumps(e)||(null!=(t=this.recovery)?t.jumps(e):void 0)},t.prototype.makeReturn=function(e){return this.attempt&&(this.attempt=this.attempt.makeReturn(e)),this.recovery&&(this.recovery=this.recovery.makeReturn(e)),this},t.prototype.compileNode=function(e){var t,n,r,s,o;return e.indent+=q,o=this.attempt.compileToFragments(e,L),t=this.recovery?(r=e.scope.freeVariable("error"),s=new x(r),this.errorVariable?this.recovery.unshift(new i(this.errorVariable,s)):void 0,[].concat(this.makeCode(" catch ("),s.compileToFragments(e),this.makeCode(") {\n"),this.recovery.compileToFragments(e,L),this.makeCode("\n"+this.tab+"}"))):this.ensure||this.recovery?[]:[this.makeCode(" catch ("+r+") {}")],n=this.ensure?[].concat(this.makeCode(" finally {\n"),this.ensure.compileToFragments(e,L),this.makeCode("\n"+this.tab+"}")):[],[].concat(this.makeCode(this.tab+"try {\n"),o,this.makeCode("\n"+this.tab+"}"),t,n)},t}(r),e.Throw=W=function(e){function t(e){this.expression=e}return kt(t,e),t.prototype.children=["expression"],t.prototype.isStatement=Q,t.prototype.jumps=D,t.prototype.makeReturn=X,t.prototype.compileNode=function(e){return[].concat(this.makeCode(this.tab+"throw "),this.expression.compileToFragments(e),this.makeCode(";"))},t}(r),e.Existence=u=function(e){function t(e){this.expression=e}return kt(t,e),t.prototype.children=["expression"],t.prototype.invert=S,t.prototype.compileNode=function(e){var t,n,i,r;return this.expression.front=this.front,i=this.expression.compile(e,E),g.test(i)&&!e.scope.check(i)?(r=this.negated?["===","||"]:["!==","&&"],t=r[0],n=r[1],i="typeof "+i+" "+t+' "undefined" '+n+" "+i+" "+t+" null"):i=i+" "+(this.negated?"==":"!=")+" null",[this.makeCode(C>=e.level?i:"("+i+")")]},t}(r),e.Parens=O=function(e){function t(e){this.body=e}return kt(t,e),t.prototype.children=["body"],t.prototype.unwrap=function(){return this.body},t.prototype.isComplex=function(){return this.body.isComplex()},t.prototype.compileNode=function(e){var t,n,i;return n=this.body.unwrap(),n instanceof z&&n.isAtomic()?(n.front=this.front,n.compileToFragments(e)):(i=n.compileToFragments(e,N),t=E>e.level&&(n instanceof I||n instanceof o||n instanceof f&&n.returns),t?i:this.wrapInBraces(i))},t}(r),e.For=f=function(e){function t(e,t){var n;this.source=t.source,this.guard=t.guard,this.step=t.step,this.name=t.name,this.index=t.index,this.body=s.wrap([e]),this.own=!!t.own,this.object=!!t.object,this.object&&(n=[this.index,this.name],this.name=n[0],this.index=n[1]),this.index instanceof z&&this.index.error("index cannot be a pattern matching expression"),this.range=this.source instanceof z&&this.source.base instanceof j&&!this.source.properties.length,this.pattern=this.name instanceof z,this.range&&this.index&&this.index.error("indexes do not apply to range loops"),this.range&&this.pattern&&this.name.error("cannot pattern match over range loops"),this.own&&!this.object&&this.name.error("cannot use own with for-in"),this.returns=!1}return kt(t,e),t.prototype.children=["body","source","guard","step"],t.prototype.compileNode=function(e){var t,n,r,o,a,c,l,h,u,p,d,f,m,v,b,k,w,T,C,E,N,S,D,A,I,_,$,j,B,V,P,U,G,H;return t=s.wrap([this.body]),D=t.expressions,T=D[D.length-1],(null!=T?T.jumps():void 0)instanceof M&&(this.returns=!1),B=this.range?this.source.base:this.source,j=e.scope,this.pattern||(E=this.name&&this.name.compile(e,F)),v=this.index&&this.index.compile(e,F),E&&!this.pattern&&j.find(E),v&&j.find(v),this.returns&&($=j.freeVariable("results")),b=this.object&&v||j.freeVariable("i",{single:!0}),k=this.range&&E||v||b,w=k!==b?k+" = ":"",this.step&&!this.range&&(A=this.cacheToCodeFragments(this.step.cache(e,F,ot)),V=A[0],U=A[1],P=U.match(R)),this.pattern&&(E=b),H="",d="",l="",f=this.tab+q,this.range?p=B.compileToFragments(ht(e,{index:b,name:E,step:this.step,isComplex:ot})):(G=this.source.compile(e,F),!E&&!this.own||g.test(G)||(l+=""+this.tab+(S=j.freeVariable("ref"))+" = "+G+";\n",G=S),E&&!this.pattern&&(N=E+" = "+G+"["+k+"]"),this.object||(V!==U&&(l+=""+this.tab+V+";\n"),this.step&&P&&(u=0>pt(P[0]))||(C=j.freeVariable("len")),a=""+w+b+" = 0, "+C+" = "+G+".length",c=""+w+b+" = "+G+".length - 1",r=b+" < "+C,o=b+" >= 0",this.step?(P?u&&(r=o,a=c):(r=U+" > 0 ? "+r+" : "+o,a="("+U+" > 0 ? ("+a+") : "+c+")"),m=b+" += "+U):m=""+(k!==b?"++"+b:b+"++"),p=[this.makeCode(a+"; "+r+"; "+w+m)])),this.returns&&(I=""+this.tab+$+" = [];\n",_="\n"+this.tab+"return "+$+";",t.makeReturn($)),this.guard&&(t.expressions.length>1?t.expressions.unshift(new y(new O(this.guard).invert(),new x("continue"))):this.guard&&(t=s.wrap([new y(this.guard,t)]))),this.pattern&&t.expressions.unshift(new i(this.name,new x(G+"["+k+"]"))),h=[].concat(this.makeCode(l),this.pluckDirectCall(e,t)),N&&(H="\n"+f+N+";"),this.object&&(p=[this.makeCode(k+" in "+G)],this.own&&(d="\n"+f+"if (!"+yt("hasProp",e)+".call("+G+", "+k+")) continue;")),n=t.compileToFragments(ht(e,{indent:f}),L),n&&n.length>0&&(n=[].concat(this.makeCode("\n"),n,this.makeCode("\n"))),[].concat(h,this.makeCode(""+(I||"")+this.tab+"for ("),p,this.makeCode(") {"+d+H),n,this.makeCode(this.tab+"}"+(_||"")))},t.prototype.pluckDirectCall=function(e,t){var n,r,s,a,l,h,u,p,d,f,m,g,v,b,y,k;for(r=[],d=t.expressions,l=h=0,u=d.length;u>h;l=++h)s=d[l],s=s.unwrapAll(),s instanceof o&&(k=null!=(f=s.variable)?f.unwrapAll():void 0,(k instanceof c||k instanceof z&&(null!=(m=k.base)?m.unwrapAll():void 0)instanceof c&&1===k.properties.length&&("call"===(g=null!=(v=k.properties[0].name)?v.value:void 0)||"apply"===g))&&(a=(null!=(b=k.base)?b.unwrapAll():void 0)||k,p=new x(e.scope.freeVariable("fn")),n=new z(p),k.base&&(y=[n,k],k.base=y[0],n=y[1]),t.expressions[l]=new o(n,s.args),r=r.concat(this.makeCode(this.tab),new i(p,a).compileToFragments(e,L),this.makeCode(";\n"))));return r},t}(J),e.Switch=H=function(e){function t(e,t,n){this.subject=e,this.cases=t,this.otherwise=n}return kt(t,e),t.prototype.children=["subject","cases","otherwise"],t.prototype.isStatement=Q,t.prototype.jumps=function(e){var t,n,i,r,s,o,a,c;for(null==e&&(e={block:!0}),o=this.cases,i=0,s=o.length;s>i;i++)if(a=o[i],n=a[0],t=a[1],r=t.jumps(e))return r;return null!=(c=this.otherwise)?c.jumps(e):void 0},t.prototype.makeReturn=function(e){var t,n,i,r,o;for(r=this.cases,t=0,n=r.length;n>t;t++)i=r[t],i[1].makeReturn(e);return e&&(this.otherwise||(this.otherwise=new s([new x("void 0")]))),null!=(o=this.otherwise)&&o.makeReturn(e),this},t.prototype.compileNode=function(e){var t,n,i,r,s,o,a,c,l,h,u,p,d,f,m,g;for(c=e.indent+q,l=e.indent=c+q,o=[].concat(this.makeCode(this.tab+"switch ("),this.subject?this.subject.compileToFragments(e,N):this.makeCode("false"),this.makeCode(") {\n")),f=this.cases,a=h=0,p=f.length;p>h;a=++h){for(m=f[a],r=m[0],t=m[1],g=rt([r]),u=0,d=g.length;d>u;u++)i=g[u],this.subject||(i=i.invert()),o=o.concat(this.makeCode(c+"case "),i.compileToFragments(e,N),this.makeCode(":\n"));if((n=t.compileToFragments(e,L)).length>0&&(o=o.concat(n,this.makeCode("\n"))),a===this.cases.length-1&&!this.otherwise)break;s=this.lastNonComment(t.expressions),s instanceof M||s instanceof x&&s.jumps()&&"debugger"!==s.value||o.push(i.makeCode(l+"break;\n"))}return this.otherwise&&this.otherwise.expressions.length&&o.push.apply(o,[this.makeCode(c+"default:\n")].concat(Ct.call(this.otherwise.compileToFragments(e,L)),[this.makeCode("\n")])),o.push(this.makeCode(this.tab+"}")),o},t}(r),e.If=y=function(e){function t(e,t,n){this.body=t,null==n&&(n={}),this.condition="unless"===n.type?e.invert():e,this.elseBody=null,this.isChain=!1,this.soak=n.soak}return kt(t,e),t.prototype.children=["condition","body","elseBody"],t.prototype.bodyNode=function(){var e;return null!=(e=this.body)?e.unwrap():void 0},t.prototype.elseBodyNode=function(){var e;return null!=(e=this.elseBody)?e.unwrap():void 0},t.prototype.addElse=function(e){return this.isChain?this.elseBodyNode().addElse(e):(this.isChain=e instanceof t,this.elseBody=this.ensureBlock(e),this.elseBody.updateLocationDataIfMissing(e.locationData)),this},t.prototype.isStatement=function(e){var t;return(null!=e?e.level:void 0)===L||this.bodyNode().isStatement(e)||(null!=(t=this.elseBodyNode())?t.isStatement(e):void 0)},t.prototype.jumps=function(e){var t;return this.body.jumps(e)||(null!=(t=this.elseBody)?t.jumps(e):void 0)},t.prototype.compileNode=function(e){return this.isStatement(e)?this.compileStatement(e):this.compileExpression(e)},t.prototype.makeReturn=function(e){return e&&(this.elseBody||(this.elseBody=new s([new x("void 0")]))),this.body&&(this.body=new s([this.body.makeReturn(e)])),this.elseBody&&(this.elseBody=new s([this.elseBody.makeReturn(e)])),this},t.prototype.ensureBlock=function(e){return e instanceof s?e:new s([e])},t.prototype.compileStatement=function(e){var n,i,r,s,o,a,c;return r=tt(e,"chainChild"),(o=tt(e,"isExistentialEquals"))?new t(this.condition.invert(),this.elseBodyNode(),{type:"if"}).compileToFragments(e):(c=e.indent+q,s=this.condition.compileToFragments(e,N),i=this.ensureBlock(this.body).compileToFragments(ht(e,{indent:c})),a=[].concat(this.makeCode("if ("),s,this.makeCode(") {\n"),i,this.makeCode("\n"+this.tab+"}")),r||a.unshift(this.makeCode(this.tab)),this.elseBody?(n=a.concat(this.makeCode(" else ")),this.isChain?(e.chainChild=!0,n=n.concat(this.elseBody.unwrap().compileToFragments(e,L))):n=n.concat(this.makeCode("{\n"),this.elseBody.compileToFragments(ht(e,{indent:c}),L),this.makeCode("\n"+this.tab+"}")),n):a)},t.prototype.compileExpression=function(e){var t,n,i,r;return i=this.condition.compileToFragments(e,C),n=this.bodyNode().compileToFragments(e,F),t=this.elseBodyNode()?this.elseBodyNode().compileToFragments(e,F):[this.makeCode("void 0")],r=i.concat(this.makeCode(" ? "),n,this.makeCode(" : "),t),e.level>=C?this.wrapInBraces(r):r},t.prototype.unfoldSoak=function(){return this.soak&&this},t}(r),K={extend:function(e){return"function(child, parent) { for (var key in parent) { if ("+yt("hasProp",e)+".call(parent, key)) child[key] = parent[key]; } function ctor() { this.constructor = child; } ctor.prototype = parent.prototype; child.prototype = new ctor(); child.__super__ = parent.prototype; return child; }"},bind:function(){return"function(fn, me){ return function(){ return fn.apply(me, arguments); }; }"},indexOf:function(){return"[].indexOf || function(item) { for (var i = 0, l = this.length; i < l; i++) { if (i in this && this[i] === item) return i; } return -1; }"},modulo:function(){return"function(a, b) { return (+a % (b = +b) + b) % b; }"},hasProp:function(){return"{}.hasOwnProperty"},slice:function(){return"[].slice"}},L=1,N=2,F=3,C=4,E=5,T=6,q="  ",g=/^(?!\d)[$\w\x7f-\uffff]+$/,B=/^[+-]?\d+$/,m=/^[+-]?0x[\da-f]+/i,R=/^[+-]?(?:0x[\da-f]+|\d*\.?\d+(?:e[+-]?\d+)?)$/i,b=/^['"]/,v=/^\//,yt=function(e,t){var n,i;return i=t.scope.root,e in i.utilities?i.utilities[e]:(n=i.freeVariable(e),i.assign(n,K[e](t)),i.utilities[e]=n)},ut=function(e,t){return e=e.replace(/\n/g,"$&"+t),e.replace(/\s+$/,"")},pt=function(e){return null==e?0:e.match(m)?parseInt(e,16):parseFloat(e)},at=function(e){return e instanceof x&&"arguments"===e.value&&!e.asKey},ct=function(e){return e instanceof x&&"this"===e.value&&!e.asKey||e instanceof c&&e.bound||e instanceof o&&e.isSuper},ot=function(e){return e.isComplex()||("function"==typeof e.isAssignable?e.isAssignable():void 0)},bt=function(e,t,n){var i;if(i=t[n].unfoldSoak(e))return t[n]=i.body,i.body=new z(t),i}}.call(this),t.exports}(),require["./sourcemap"]=function(){var e={},t={exports:e};return function(){var e,n;e=function(){function e(e){this.line=e,this.columns=[]}return e.prototype.add=function(e,t,n){var i,r;return r=t[0],i=t[1],null==n&&(n={}),this.columns[e]&&n.noReplace?void 0:this.columns[e]={line:this.line,column:e,sourceLine:r,sourceColumn:i}},e.prototype.sourceLocation=function(e){for(var t;!((t=this.columns[e])||0>=e);)e--;return t&&[t.sourceLine,t.sourceColumn]},e}(),n=function(){function t(){this.lines=[]}var n,i,r,s;return t.prototype.add=function(t,n,i){var r,s,o,a;return null==i&&(i={}),o=n[0],s=n[1],a=(r=this.lines)[o]||(r[o]=new e(o)),a.add(s,t,i)},t.prototype.sourceLocation=function(e){var t,n,i;for(n=e[0],t=e[1];!((i=this.lines[n])||0>=n);)n--;return i&&i.sourceLocation(t)},t.prototype.generate=function(e,t){var n,i,r,s,o,a,c,l,h,u,p,d,f,m,g,v;for(null==e&&(e={}),null==t&&(t=null),v=0,s=0,a=0,o=0,d=!1,n="",f=this.lines,u=i=0,c=f.length;c>i;u=++i)if(h=f[u])for(m=h.columns,r=0,l=m.length;l>r;r++)if(p=m[r]){for(;p.line>v;)s=0,d=!1,n+=";",v++;d&&(n+=",",d=!1),n+=this.encodeVlq(p.column-s),s=p.column,n+=this.encodeVlq(0),n+=this.encodeVlq(p.sourceLine-a),a=p.sourceLine,n+=this.encodeVlq(p.sourceColumn-o),o=p.sourceColumn,d=!0}return g={version:3,file:e.generatedFile||"",sourceRoot:e.sourceRoot||"",sources:e.sourceFiles||[""],names:[],mappings:n},e.inline&&(g.sourcesContent=[t]),JSON.stringify(g,null,2)},r=5,i=1<<r,s=i-1,t.prototype.encodeVlq=function(e){var t,n,o,a;for(t="",o=0>e?1:0,a=(Math.abs(e)<<1)+o;a||!t;)n=a&s,a>>=r,a&&(n|=i),t+=this.encodeBase64(n);return t},n="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",t.prototype.encodeBase64=function(e){return n[e]||function(){throw Error("Cannot Base64 encode value: "+e)
}()},t}(),t.exports=n}.call(this),t.exports}(),require["./coffee-script"]=function(){var e={},t={exports:e};return function(){var t,n,i,r,s,o,a,c,l,h,u,p,d,f,m,g,v,b,y={}.hasOwnProperty,k=[].indexOf||function(e){for(var t=0,n=this.length;n>t;t++)if(t in this&&this[t]===e)return t;return-1};if(a=require("fs"),v=require("vm"),f=require("path"),t=require("./lexer").Lexer,d=require("./parser").parser,l=require("./helpers"),n=require("./sourcemap"),e.VERSION="1.10.0",e.FILE_EXTENSIONS=[".coffee",".litcoffee",".coffee.md"],e.helpers=l,b=function(e){return function(t,n){var i,r;null==n&&(n={});try{return e.call(this,t,n)}catch(r){if(i=r,"string"!=typeof t)throw i;throw l.updateSyntaxError(i,t,n.filename)}}},e.compile=r=b(function(e,t){var i,r,s,o,a,c,h,u,f,m,g,v,b,y,k;for(v=l.merge,o=l.extend,t=o({},t),t.sourceMap&&(g=new n),k=p.tokenize(e,t),t.referencedVars=function(){var e,t,n;for(n=[],e=0,t=k.length;t>e;e++)y=k[e],y.variable&&n.push(y[1]);return n}(),c=d.parse(k).compileToFragments(t),s=0,t.header&&(s+=1),t.shiftLine&&(s+=1),r=0,f="",u=0,m=c.length;m>u;u++)a=c[u],t.sourceMap&&(a.locationData&&!/^[;\s]*$/.test(a.code)&&g.add([a.locationData.first_line,a.locationData.first_column],[s,r],{noReplace:!0}),b=l.count(a.code,"\n"),s+=b,b?r=a.code.length-(a.code.lastIndexOf("\n")+1):r+=a.code.length),f+=a.code;return t.header&&(h="Generated by CoffeeScript "+this.VERSION,f="// "+h+"\n"+f),t.sourceMap?(i={js:f},i.sourceMap=g,i.v3SourceMap=g.generate(t,e),i):f}),e.tokens=b(function(e,t){return p.tokenize(e,t)}),e.nodes=b(function(e,t){return"string"==typeof e?d.parse(p.tokenize(e,t)):d.parse(e)}),e.run=function(e,t){var n,i,s,o;return null==t&&(t={}),s=require.main,s.filename=process.argv[1]=t.filename?a.realpathSync(t.filename):".",s.moduleCache&&(s.moduleCache={}),i=t.filename?f.dirname(a.realpathSync(t.filename)):a.realpathSync("."),s.paths=require("module")._nodeModulePaths(i),(!l.isCoffee(s.filename)||require.extensions)&&(n=r(e,t),e=null!=(o=n.js)?o:n),s._compile(e,s.filename)},e.eval=function(e,t){var n,i,s,o,a,c,l,h,u,p,d,m,g,b,k,w,T;if(null==t&&(t={}),e=e.trim()){if(o=null!=(m=v.Script.createContext)?m:v.createContext,c=null!=(g=v.isContext)?g:function(){return t.sandbox instanceof o().constructor},o){if(null!=t.sandbox){if(c(t.sandbox))w=t.sandbox;else{w=o(),b=t.sandbox;for(h in b)y.call(b,h)&&(T=b[h],w[h]=T)}w.global=w.root=w.GLOBAL=w}else w=global;if(w.__filename=t.filename||"eval",w.__dirname=f.dirname(w.__filename),w===global&&!w.module&&!w.require){for(n=require("module"),w.module=i=new n(t.modulename||"eval"),w.require=s=function(e){return n._load(e,i,!0)},i.filename=w.__filename,k=Object.getOwnPropertyNames(require),a=0,u=k.length;u>a;a++)d=k[a],"paths"!==d&&"arguments"!==d&&"caller"!==d&&(s[d]=require[d]);s.paths=i.paths=n._nodeModulePaths(process.cwd()),s.resolve=function(e){return n._resolveFilename(e,i)}}}p={};for(h in t)y.call(t,h)&&(T=t[h],p[h]=T);return p.bare=!0,l=r(e,p),w===global?v.runInThisContext(l):v.runInContext(l,w)}},e.register=function(){return require("./register")},require.extensions)for(m=this.FILE_EXTENSIONS,h=0,u=m.length;u>h;h++)s=m[h],null==(i=require.extensions)[s]&&(i[s]=function(){throw Error("Use CoffeeScript.register() or require the coffee-script/register module to require "+s+" files.")});e._compileFile=function(e,t){var n,i,s,o,c;null==t&&(t=!1),o=a.readFileSync(e,"utf8"),c=65279===o.charCodeAt(0)?o.substring(1):o;try{n=r(c,{filename:e,sourceMap:t,literate:l.isLiterate(e)})}catch(s){throw i=s,l.updateSyntaxError(i,c,e)}return n},p=new t,d.lexer={lex:function(){var e,t;return t=d.tokens[this.pos++],t?(e=t[0],this.yytext=t[1],this.yylloc=t[2],d.errorToken=t.origin||t,this.yylineno=this.yylloc.first_line):e="",e},setInput:function(e){return d.tokens=e,this.pos=0},upcomingInput:function(){return""}},d.yy=require("./nodes"),d.yy.parseError=function(e,t){var n,i,r,s,o,a;return o=t.token,s=d.errorToken,a=d.tokens,i=s[0],r=s[1],n=s[2],r=function(){switch(!1){case s!==a[a.length-1]:return"end of input";case"INDENT"!==i&&"OUTDENT"!==i:return"indentation";case"IDENTIFIER"!==i&&"NUMBER"!==i&&"STRING"!==i&&"STRING_START"!==i&&"REGEX"!==i&&"REGEX_START"!==i:return i.replace(/_START$/,"").toLowerCase();default:return l.nameWhitespaceCharacter(r)}}(),l.throwSyntaxError("unexpected "+r,n)},o=function(e,t){var n,i,r,s,o,a,c,l,h,u,p,d;return s=void 0,r="",e.isNative()?r="native":(e.isEval()?(s=e.getScriptNameOrSourceURL(),s||(r=e.getEvalOrigin()+", ")):s=e.getFileName(),s||(s="<anonymous>"),l=e.getLineNumber(),i=e.getColumnNumber(),u=t(s,l,i),r=u?s+":"+u[0]+":"+u[1]:s+":"+l+":"+i),o=e.getFunctionName(),a=e.isConstructor(),c=!(e.isToplevel()||a),c?(h=e.getMethodName(),d=e.getTypeName(),o?(p=n="",d&&o.indexOf(d)&&(p=d+"."),h&&o.indexOf("."+h)!==o.length-h.length-1&&(n=" [as "+h+"]"),""+p+o+n+" ("+r+")"):d+"."+(h||"<anonymous>")+" ("+r+")"):a?"new "+(o||"<anonymous>")+" ("+r+")":o?o+" ("+r+")":r},g={},c=function(t){var n,i;if(g[t])return g[t];if(i=null!=f?f.extname(t):void 0,!(0>k.call(e.FILE_EXTENSIONS,i)))return n=e._compileFile(t,!0),g[t]=n.sourceMap},Error.prepareStackTrace=function(t,n){var i,r,s;return s=function(e,t,n){var i,r;return r=c(e),r&&(i=r.sourceLocation([t-1,n-1])),i?[i[0]+1,i[1]+1]:null},r=function(){var t,r,a;for(a=[],t=0,r=n.length;r>t&&(i=n[t],i.getFunction()!==e.run);t++)a.push("  at "+o(i,s));return a}(),""+t+"\n"+r.join("\n")+"\n"}}.call(this),t.exports}(),require["./browser"]=function(){var exports={},module={exports:exports};return function(){var CoffeeScript,compile,runScripts,indexOf=[].indexOf||function(e){for(var t=0,n=this.length;n>t;t++)if(t in this&&this[t]===e)return t;return-1};CoffeeScript=require("./coffee-script"),CoffeeScript.require=require,compile=CoffeeScript.compile,CoffeeScript.eval=function(code,options){return null==options&&(options={}),null==options.bare&&(options.bare=!0),eval(compile(code,options))},CoffeeScript.run=function(e,t){return null==t&&(t={}),t.bare=!0,t.shiftLine=!0,Function(compile(e,t))()},"undefined"!=typeof window&&null!==window&&("undefined"!=typeof btoa&&null!==btoa&&"undefined"!=typeof JSON&&null!==JSON&&"undefined"!=typeof unescape&&null!==unescape&&"undefined"!=typeof encodeURIComponent&&null!==encodeURIComponent&&(compile=function(e,t){var n,i,r;return null==t&&(t={}),t.sourceMap=!0,t.inline=!0,i=CoffeeScript.compile(e,t),n=i.js,r=i.v3SourceMap,n+"\n//# sourceMappingURL=data:application/json;base64,"+btoa(unescape(encodeURIComponent(r)))+"\n//# sourceURL=coffeescript"}),CoffeeScript.load=function(e,t,n,i){var r;return null==n&&(n={}),null==i&&(i=!1),n.sourceFiles=[e],r=window.ActiveXObject?new window.ActiveXObject("Microsoft.XMLHTTP"):new window.XMLHttpRequest,r.open("GET",e,!0),"overrideMimeType"in r&&r.overrideMimeType("text/plain"),r.onreadystatechange=function(){var s,o;if(4===r.readyState){if(0!==(o=r.status)&&200!==o)throw Error("Could not load "+e);if(s=[r.responseText,n],i||CoffeeScript.run.apply(CoffeeScript,s),t)return t(s)}},r.send(null)},runScripts=function(){var e,t,n,i,r,s,o,a,c,l,h;for(h=window.document.getElementsByTagName("script"),t=["text/coffeescript","text/literate-coffeescript"],e=function(){var e,n,i,r;for(r=[],e=0,n=h.length;n>e;e++)c=h[e],i=c.type,indexOf.call(t,i)>=0&&r.push(c);return r}(),s=0,n=function(){var t;return t=e[s],t instanceof Array?(CoffeeScript.run.apply(CoffeeScript,t),s++,n()):void 0},i=function(i,r){var s,o;return s={literate:i.type===t[1]},o=i.src||i.getAttribute("data-src"),o?CoffeeScript.load(o,function(t){return e[r]=t,n()},s,!0):(s.sourceFiles=["embedded"],e[r]=[i.innerHTML,s])},r=o=0,a=e.length;a>o;r=++o)l=e[r],i(l,r);return n()},window.addEventListener?window.addEventListener("DOMContentLoaded",runScripts,!1):window.attachEvent("onload",runScripts))}.call(this),module.exports}(),require["./coffee-script"]}();"function"==typeof define&&define.amd?define(function(){return CoffeeScript}):root.CoffeeScript=CoffeeScript})(this);// Licensed under the Apache License, Version 2.0 (the "License"); you may not
// use this file except in compliance with the License. You may obtain a copy of
// the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
// License for the specific language governing permissions and limitations under
// the License.

function create_sandbox() {
  var sandbox = {};
  sandbox.emit = Views.emit;
  sandbox.sum = Views.sum;
  sandbox.log = log;
  sandbox.toJSON = JSON.stringify;
  sandbox.JSON = JSON;
  sandbox.provides = Mime.provides;
  sandbox.registerType = Mime.registerType;
  sandbox.start = Render.start;
  sandbox.send = Render.send;
  sandbox.getRow = Render.getRow;
  sandbox.isArray = isArray;
  return sandbox;
};

function create_filter_sandbox() {
  var sandbox = create_sandbox();
  sandbox.emit = Filter.emit;
  return sandbox;
};

function create_dreyfus_sandbox() {
  var sandbox = create_sandbox();
  sandbox.index = Dreyfus.index;
  return sandbox;
};

function create_nouveau_sandbox() {
  var sandbox = create_sandbox();
  sandbox.index = Nouveau.index;
  return sandbox;
};

function seal(obj, flag) {
  Object.freeze(obj);
};

// This is a copy from loop.js
var DDoc = (function() {
  var ddoc_dispatch = {
    "lists"     : Render.list,
    "shows"    : Render.show,
    "filters"   : Filter.filter,
    "views"     : Filter.filter_view,
    "updates"  : Render.update,
    "validate_doc_update" : Validate.validate,
    "rewrites"  : Render.rewrite
  };
  var ddocs = {};
  return {
    ddoc : function() {
      var args = [];
      for (var i=0; i < arguments.length; i++) {
        args.push(arguments[i]);
      };
      var ddocId = args.shift();
      if (ddocId == "new") {
        // get the real ddocId.
        ddocId = args.shift();
        // store the ddoc, functions are lazily compiled.
        ddocs[ddocId] = args.shift();
        print("true");
      } else {
        // Couch makes sure we know this ddoc already.
        var ddoc = ddocs[ddocId];
        if (!ddoc) throw(["fatal", "query_protocol_error", "uncached design doc: "+ddocId]);
        var funPath = args.shift();
        var cmd = funPath[0];
        // the first member of the fun path determines the type of operation
        var funArgs = args.shift();
        if (ddoc_dispatch[cmd]) {
          // get the function, call the command with it
          var point = ddoc;
          for (var i=0; i < funPath.length; i++) {
            if (i+1 == funPath.length) {
              var fun = point[funPath[i]];
              if (!fun) {
                throw(["error","not_found",
                       "missing " + funPath[0] + " function " + funPath[i] +
                       " on design doc " + ddocId]);
              }
              if (typeof fun != "function") {
                // For filter_view we want the emit() function
                // to be overridden and just toggle a flag instead of
                // accumulating rows
                var sandbox = (cmd === "views") ? create_filter_sandbox() : create_sandbox();
                fun = Couch.compileFunction(fun, ddoc, funPath.join('.'), sandbox);
                // cache the compiled fun on the ddoc
                point[funPath[i]] = fun;
              };
            } else {
              point = point[funPath[i]];
            }
          };

          // run the correct responder with the cmd body
          ddoc_dispatch[cmd].apply(null, [fun, ddoc, funArgs]);
        } else {
          // unknown command, quit and hope the restarted version is better
          throw(["fatal", "unknown_command", "unknown ddoc command '" + cmd + "'"]);
        }
      }
    }
  };
})();

// This mostly a copy from loop.js handleError
function handleError(e) {
    if (e === null) {
      // internal error, another possibility when out of memory
      // we tell the client it was a fatal error by dying
      respond(["error", "internal_error", null]);
      return false;
    }
    const type = e[0];
    if (type == "fatal") {
      e[0] = "error"; // we tell the client it was a fatal error by dying
      respond(e);
      return false;
    } else if (type == "error") {
      respond(e);
      return true;
    } else if (e.name == "InternalError") {
      // If the internal error is caught by handleViewError it will be
      // re-thrown as a ["fatal", ...] error, and we already handle that above.
      // Here we handle the case when the error is thrown outside of
      // handleViewError, for instance when serializing the rows to be sent
      // back to the user
      respond(["error", e.name, e.message]);
      return false;
    } else if (e.error && e.reason) {
      // compatibility with old error format
      respond(["error", e.error, e.reason]);
      return true;
    } else if (e.name) {
      respond(["error", e.name, e]);
      return true;
    } else {
      respond(["error","unnamed_error", e.stack]);
      return true;
    }
  };

globalThis.dispatch = function(line) {
  const cmd = JSON.parse(line);
  State.line_length = line.length;
  try {
    switch (cmd.shift()) {
    case "ddoc":
      DDoc.ddoc.apply(null, cmd);
      break;
    case "reset":
      State.reset.apply(null, cmd);
      break;
    case "add_fun":
      State.addFun.apply(null, cmd);
      break;
    case "add_lib":
      State.addLib.apply(null, cmd);
      break;
    case "map_doc":
      Views.mapDoc.apply(null, cmd);
      break;
    case "index_doc":
      Dreyfus.indexDoc.apply(null, cmd);
      break;
    case "nouveau_index_doc":
      Nouveau.indexDoc.apply(null, cmd);
      break;
    case "reduce":
      Views.reduce.apply(null, cmd);
      break;
    case "rereduce":
      Views.rereduce.apply(null, cmd);
      break;
    default:
      // unknown command, quit and hope the restarted version is better
      throw(["fatal", "unknown_command", "unknown command '" + cmdkey + "'"]);
    }
  } catch(e) {
      return handleError(e);
  };
  return true;
};
})();
