"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ObservabilityPlugin = void 0;
var _configSchema = require("@osd/config-schema");
var _server = require("../../../src/core/server");
var _shared = require("../common/constants/shared");
var _migrations = require("./adaptors/integrations/migrations");
var _opensearch_observability_plugin = require("./adaptors/opensearch_observability_plugin");
var _ppl_plugin = require("./adaptors/ppl_plugin");
var _ppl_parser = require("./parsers/ppl_parser");
var _register_settings = require("./plugin_helper/register_settings");
var _index = require("./routes/index");
var _observability_saved_object = require("./saved_objects/observability_saved_object");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class ObservabilityPlugin {
  constructor(initializerContext) {
    this.initializerContext = initializerContext;
    _defineProperty(this, "logger", void 0);
    this.logger = initializerContext.logger.get();
  }
  async setup(core, deps) {
    const {
      assistantDashboards,
      dataSource
    } = deps;
    this.logger.debug('Observability: Setup');
    const router = core.http.createRouter();
    const dataSourceEnabled = !!dataSource;
    const openSearchObservabilityClient = core.opensearch.legacy.createClient('opensearch_observability', {
      plugins: [_ppl_plugin.PPLPlugin, _opensearch_observability_plugin.OpenSearchObservabilityPlugin]
    });
    if (dataSourceEnabled) {
      dataSource.registerCustomApiSchema(_ppl_plugin.PPLPlugin);
      dataSource.registerCustomApiSchema(_opensearch_observability_plugin.OpenSearchObservabilityPlugin);
    }
    // @ts-ignore
    core.http.registerRouteHandlerContext('observability_plugin', (_context, _request) => {
      return {
        logger: this.logger,
        observabilityClient: openSearchObservabilityClient
      };
    });
    const obsPanelType = {
      name: 'observability-panel',
      hidden: false,
      namespaceType: 'single',
      mappings: {
        dynamic: false,
        properties: {
          title: {
            type: 'text'
          },
          description: {
            type: 'text'
          }
        }
      },
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: dataSourceEnabled ? '' : `/app/${_shared.observabilityPanelsID}#/${obj.id}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return `Observability Settings [${obj.id}]`;
        }
      },
      migrations: {
        '3.0.0': doc => ({
          ...doc,
          description: ''
        }),
        '3.0.1': doc => ({
          ...doc,
          description: 'Some Description Text'
        }),
        '3.0.2': doc => ({
          ...doc,
          dateCreated: parseInt(doc.dateCreated || '0', 10)
        })
      }
    };
    const integrationInstanceType = {
      name: 'integration-instance',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/installed/${obj.id}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          return obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          templateName: {
            type: 'text'
          },
          dataSource: {
            type: 'text'
          },
          creationDate: {
            type: 'date'
          },
          assets: {
            type: 'nested'
          }
        }
      }
    };
    const integrationTemplateType = {
      name: 'integration-template',
      hidden: false,
      namespaceType: 'single',
      management: {
        importableAndExportable: true,
        getInAppUrl(obj) {
          return {
            path: `/app/integrations#/available/${obj.attributes.name}`,
            uiCapabilitiesPath: 'advancedSettings.show'
          };
        },
        getTitle(obj) {
          var _obj$attributes$displ;
          return (_obj$attributes$displ = obj.attributes.displayName) !== null && _obj$attributes$displ !== void 0 ? _obj$attributes$displ : obj.attributes.name;
        }
      },
      mappings: {
        dynamic: false,
        properties: {
          name: {
            type: 'text'
          },
          version: {
            type: 'text'
          },
          displayName: {
            type: 'text'
          },
          license: {
            type: 'text'
          },
          type: {
            type: 'text'
          },
          labels: {
            type: 'text'
          },
          author: {
            type: 'text'
          },
          description: {
            type: 'text'
          },
          sourceUrl: {
            type: 'text'
          },
          statics: {
            type: 'nested'
          },
          components: {
            type: 'nested'
          },
          assets: {
            type: 'nested'
          },
          sampleData: {
            type: 'nested'
          }
        }
      },
      migrations: {
        '3.0.0': _migrations.migrateV1IntegrationToV2Integration
      }
    };
    core.savedObjects.registerType(obsPanelType);
    core.savedObjects.registerType(integrationInstanceType);
    core.savedObjects.registerType(integrationTemplateType);

    // Register server side APIs
    (0, _index.setupRoutes)({
      router,
      client: openSearchObservabilityClient,
      dataSourceEnabled
    });
    core.savedObjects.registerType((0, _observability_saved_object.getVisualizationSavedObject)(dataSourceEnabled));
    core.savedObjects.registerType((0, _observability_saved_object.getSearchSavedObject)(dataSourceEnabled));
    core.savedObjects.registerType(_observability_saved_object.notebookSavedObject);
    core.capabilities.registerProvider(() => ({
      observability: {
        show: true
      }
    }));
    assistantDashboards === null || assistantDashboards === void 0 || assistantDashboards.registerMessageParser(_ppl_parser.PPLParsers);
    (0, _register_settings.registerObservabilityUISettings)(core.uiSettings);
    core.uiSettings.register({
      'observability:defaultDashboard': {
        name: 'Observability default dashboard',
        value: '',
        description: 'The default dashboard to display in Observability overview page',
        schema: _configSchema.schema.string(),
        scope: core.workspace.isWorkspaceEnabled() ? _server.UiSettingScope.WORKSPACE : _server.UiSettingScope.GLOBAL
      }
    });
    core.uiSettings.register({
      'observability:overviewCardsDisplay': {
        name: 'Observability overview cards',
        value: true,
        description: 'Show the Observability overview page cards',
        schema: _configSchema.schema.boolean(),
        scope: core.workspace.isWorkspaceEnabled() ? _server.UiSettingScope.WORKSPACE : _server.UiSettingScope.GLOBAL
      }
    });
    return {};
  }
  start(_core) {
    this.logger.debug('Observability: Started');
    return {};
  }
  stop() {}
}
exports.ObservabilityPlugin = ObservabilityPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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