"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.request = exports.prepareRequestOptions = void 0;
var axios_1 = __importDefault(require("axios"));
var patcher_1 = require("./compat/patcher");
var digest_1 = require("./auth/digest");
var merge_1 = require("./tools/merge");
var headers_1 = require("./tools/headers");
function _request(requestOptions) {
    return (0, patcher_1.getPatcher)().patchInline("request", function (options) { return (0, axios_1.default)(options); }, requestOptions);
}
function prepareRequestOptions(requestOptions, context, userOptions) {
    var finalOptions = (0, merge_1.cloneShallow)(requestOptions);
    finalOptions.headers = (0, headers_1.mergeHeaders)(context.headers, finalOptions.headers || {}, userOptions.headers || {});
    if (typeof userOptions.data !== "undefined") {
        finalOptions.data = userOptions.data;
    }
    if (userOptions.signal) {
        finalOptions.signal = userOptions.signal;
    }
    if (context.httpAgent) {
        finalOptions.httpAgent = context.httpAgent;
    }
    if (context.httpsAgent) {
        finalOptions.httpsAgent = context.httpsAgent;
    }
    if (context.digest) {
        finalOptions._digest = context.digest;
    }
    if (typeof context.withCredentials === "boolean") {
        finalOptions.withCredentials = context.withCredentials;
    }
    if (context.maxContentLength) {
        finalOptions.maxContentLength = context.maxContentLength;
    }
    if (context.maxBodyLength) {
        finalOptions.maxBodyLength = context.maxBodyLength;
    }
    if (userOptions.hasOwnProperty("onUploadProgress")) {
        finalOptions.onUploadProgress = userOptions["onUploadProgress"];
    }
    if (userOptions.hasOwnProperty("onDownloadProgress")) {
        finalOptions.onDownloadProgress = userOptions["onDownloadProgress"];
    }
    // Take full control of all response status codes
    finalOptions.validateStatus = function () { return true; };
    return finalOptions;
}
exports.prepareRequestOptions = prepareRequestOptions;
function request(requestOptions) {
    // Client not configured for digest authentication
    if (!requestOptions._digest) {
        return _request(requestOptions);
    }
    // Remove client's digest authentication object from request options
    var _digest = requestOptions._digest;
    delete requestOptions._digest;
    // If client is already using digest authentication, include the digest authorization header
    if (_digest.hasDigestAuth) {
        requestOptions = (0, merge_1.merge)(requestOptions, {
            headers: {
                Authorization: (0, digest_1.generateDigestAuthHeader)(requestOptions, _digest)
            }
        });
    }
    // Perform the request and handle digest authentication
    return _request(requestOptions).then(function (response) {
        if (response.status == 401) {
            _digest.hasDigestAuth = (0, digest_1.parseDigestAuth)(response, _digest);
            if (_digest.hasDigestAuth) {
                requestOptions = (0, merge_1.merge)(requestOptions, {
                    headers: {
                        Authorization: (0, digest_1.generateDigestAuthHeader)(requestOptions, _digest)
                    }
                });
                return _request(requestOptions).then(function (response2) {
                    if (response2.status == 401) {
                        _digest.hasDigestAuth = false;
                    }
                    else {
                        _digest.nc++;
                    }
                    return response2;
                });
            }
        }
        else {
            _digest.nc++;
        }
        return response;
    });
}
exports.request = request;
