"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _MDSEnabledClientService = require("./MDSEnabledClientService");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class RollupService extends _MDSEnabledClientService.MDSEnabledClientService {
  constructor(...args) {
    super(...args);
    /**
     * Calls backend Put Rollup API
     */
    _defineProperty(this, "putRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const {
          seqNo,
          primaryTerm
        } = request.query;
        let method = "ism.putRollup";
        let params = {
          rollupId: id,
          if_seq_no: seqNo,
          if_primary_term: primaryTerm,
          body: JSON.stringify(request.body)
        };
        if (seqNo === undefined || primaryTerm === undefined) {
          method = "ism.createRollup";
          params = {
            rollupId: id,
            body: JSON.stringify(request.body)
          };
        }
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const putRollupResponse = await callWithRequest(method, params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: putRollupResponse
          }
        });
      } catch (err) {
        console.error("Index Management - RollupService - putRollup", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    /**
     * Calls backend Delete Rollup API
     */
    _defineProperty(this, "deleteRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          rollupId: id
        };
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const deleteRollupResponse = await callWithRequest("ism.deleteRollup", params);
        if (deleteRollupResponse.result !== "deleted") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: deleteRollupResponse.result
            }
          });
        }
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: true
          }
        });
      } catch (err) {
        console.error("Index Management - RollupService - deleteRollup:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    _defineProperty(this, "startRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          rollupId: id
        };
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const startResponse = await callWithRequest("ism.startRollup", params);
        const acknowledged = _lodash.default.get(startResponse, "acknowledged");
        if (acknowledged) {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: true
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to start rollup"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - RollupService - startRollup:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    _defineProperty(this, "stopRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          rollupId: id
        };
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const stopResponse = await callWithRequest("ism.stopRollup", params);
        const acknowledged = _lodash.default.get(stopResponse, "acknowledged");
        if (acknowledged) {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: true
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to stop rollup"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - RollupService - stopRollup:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    /**
     * Calls backend Get Rollup API
     */
    _defineProperty(this, "getRollup", async (context, request, response) => {
      try {
        const {
          id
        } = request.params;
        const params = {
          rollupId: id
        };
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const getResponse = await callWithRequest("ism.getRollup", params);
        const metadata = await callWithRequest("ism.explainRollup", params);
        const rollup = _lodash.default.get(getResponse, "rollup", null);
        const seqNo = _lodash.default.get(getResponse, "_seq_no");
        const primaryTerm = _lodash.default.get(getResponse, "_primary_term");

        //Form response
        if (rollup) {
          if (metadata) {
            return response.custom({
              statusCode: 200,
              body: {
                ok: true,
                response: {
                  _id: id,
                  _seqNo: seqNo,
                  _primaryTerm: primaryTerm,
                  rollup: rollup,
                  metadata: metadata
                }
              }
            });
          } else return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to load metadata"
            }
          });
        } else {
          return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: "Failed to load rollup"
            }
          });
        }
      } catch (err) {
        console.error("Index Management - RollupService - getRollup:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    _defineProperty(this, "getMappings", async (context, request, response) => {
      try {
        const {
          index
        } = request.body;
        const params = {
          index: index
        };
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const mappings = await callWithRequest("indices.getMapping", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: mappings
          }
        });
      } catch (err) {
        console.error("Index Management - RollupService - getMapping:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    /**
     * Performs a fuzzy search request on rollup id
     */
    _defineProperty(this, "getRollups", async (context, request, response) => {
      try {
        const {
          from,
          size,
          search,
          sortDirection,
          sortField
        } = request.query;
        const rollupSortMap = {
          _id: "rollup.rollup_id.keyword",
          "rollup.source_index": "rollup.source_index.keyword",
          "rollup.target_index": "rollup.target_index.keyword",
          "rollup.rollup.enabled": "rollup.enabled"
        };
        const params = {
          from: parseInt(from, 10),
          size: parseInt(size, 10),
          search,
          sortField: rollupSortMap[sortField] || rollupSortMap._id,
          sortDirection
        };
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const getRollupResponse = await callWithRequest("ism.getRollups", params);
        const totalRollups = getRollupResponse.total_rollups;
        const rollups = getRollupResponse.rollups.map(rollup => ({
          _seqNo: rollup._seqNo,
          _primaryTerm: rollup._primaryTerm,
          _id: rollup._id,
          rollup: rollup.rollup,
          metadata: null
        }));

        // Call getExplain if any rollup job exists
        if (totalRollups) {
          // Concat rollup job ids
          const ids = rollups.map(rollup => rollup._id).join(",");
          const explainResponse = await callWithRequest("ism.explainRollup", {
            rollupId: ids
          });
          if (!explainResponse.error) {
            rollups.map(rollup => {
              rollup.metadata = explainResponse[rollup._id];
            });
            return response.custom({
              statusCode: 200,
              body: {
                ok: true,
                response: {
                  rollups: rollups,
                  totalRollups: totalRollups,
                  metadata: explainResponse
                }
              }
            });
          } else return response.custom({
            statusCode: 200,
            body: {
              ok: false,
              error: explainResponse ? explainResponse.error : "An error occurred when calling getExplain API."
            }
          });
        }
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              rollups: rollups,
              totalRollups: totalRollups,
              metadata: {}
            }
          }
        });
      } catch (err) {
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                rollups: [],
                totalRollups: 0,
                metadata: null
              }
            }
          });
        }
        console.error("Index Management - RollupService - getRollups", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: "Error in getRollups " + err.message
          }
        });
      }
    });
  }
}
exports.default = RollupService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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