# Copyright (c) 2024 Bill Peck  <bpeck@redhat.com>
#
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import annotations

DOCUMENTATION = '''
---
module: azure_rm_arcssh
version_added: "3.1.0"
short_description: Configure SSH proxy for ARC hosts
description:
    - Configure SSH proxy for ARC hosts. Use this in combination with the
      azure_rm inventory plugin with include_arc_resource_groups to dynamically
      generate the list of hosts.

options:
    resource_group:
        description:
            - The name of the Resource Group to which the server belongs.
        required: True
        type: str
    resource_type:
        description:
            - The name of the Resource Type to which the server belongs.
        required: True
        type: str
    inventory_hostname:
        description:
            - The inventory configured hostname.
        type: str
    ansible_host:
        description:
            - The discovered hostname that is generated by the inventory config.
            - Please see the examples on how this can be set based on the resource_type.
        type: str
    private_key_file:
        description:
            - The Private key that matches the public key that you have already added to your authorized_hosts
              of local_user.
        default: ~/.ssh/id_rsa
        type: str
    local_user:
        description:
            - The local user that you use to log into your server.
        type: str
    port:
        description:
            - The port that your ssh server is listening on.
        default: '22'
        type: str
    ssh_config_file:
        description:
            - Host specific file location for ssh config.  It must match the -F argument passed to
              ansible_ssh_common_args.  This will be overwritten and should be different for each host.
            - See the Examples for how to set this variable to match both the task and inventory plugin.
        required: True
        type: str
    ssh_relay_file:
        description:
            - Host specific file location for the relay file.
        required: True
        type: str
    ssh_proxy_folder:
        description:
            - The folder which will store the client ssh proxy.
        default: ~/.clientsshproxy
        type: str

extends_documentation_fragment:
    - azure.azcollection.azure

author:
    - Bill Peck (@p3ck)
'''

EXAMPLES = '''
# Inventory plugin:
# ---
# plugin: azure.azcollection.azure_rm
# plain_host_names: yes
# include_arc_resource_groups: ['*']
#
#
# hostvar_expressions:
#   ansible_host: "resource_group + '-' + name if 'Microsoft.HybridCompute/machines' == resource_type else (public_dns_hostnames + public_ipv4_address) | first"
#   ansible_ssh_common_args: "'-F /tmp/' + resource_group + '-' + name + '/ssh_config' if 'Microsoft.HybridCompute/machines' == resource_type"
#
# keyed_groups:
# # places each host in a group named 'tag_(tag name)_(tag value)' for each tag on a VM.
#  - prefix: "type"
#    key: resource_type
#    trailing_separator: false

- name: Configure ARC SSH Proxy
  hosts: localhost
  connection: local
  tasks:
    - name: Setup Proxy
      azure.azcollection.azure_rm_arcssh:
        inventory_hostname: "{{ item }}"
        ansible_host: "{{ hostvars[item].ansible_host }}"
        local_user: admin
        resource_group: "{{ hostvars[item].resource_group }}"
        resource_type: "{{ hostvars[item].resource_type }}"
        private_key_file: "~/.ssh/id_rsa"
        ssh_config_file: "/tmp/{{ hostvars[item].resource_group }}-{{ item }}/ssh_config"
        ssh_relay_file: "/tmp/{{ hostvars[item].resource_group }}-{{ item }}/relay_info"
        ssh_proxy_folder: "~/.clientsshproxy"
      loop: "{{ groups['type_Microsoft_HybridCompute_machines'] }}"

- name: Ping ARC Hosts
  hosts: type_Microsoft_HybridCompute_machines
  tasks:
    - name: Ping
      ansible.builtin.ping:
'''
