# ----------------------------------------------------------------------------
#
#  Copyright (C) 2021-2022 Fons Adriaensen <fons@linuxaudio.org>
#    
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http:#www.gnu.org/licenses/>.
#
# ----------------------------------------------------------------------------


from zita_jacktools.jackclient import JackClient
from zita_jacktools import jackgainctl_ext


class JackGainctl(JackClient):

    """
    Multichannel dezippered audio gain control, with per
    channel and master gain and mute controls.

    The 'gain' and 'mute' settings are independent. A gain
    less than -120 dB is equivalent to 'off' (-inf dB).
    The initial master gain is 'off', not muted.
    The initial channel gain is 0 dB, not muted.

    Gain changes are dezippered as follows:
    'off' -> 'on'  : 10 ms fade in.
    'on'  -> 'off' : 10 ms fade out.
    otherwise      : gain changes by max 6 dB in each 10 ms.
    """

    def __init__(self, nchan, client_name, server_name = None):
        """
        Create a new JackGainctl instance.

        The result should be checked using get_state().
        The optional 'server_name' allows to select between running
        Jack servers.
        """
        assert (nchan > 0)
        self._jgainctl, base = jackgainctl_ext.makecaps (self, client_name, server_name, nchan)
        super().__init__(base)


    def set_gain(self, gain) :
        """
        Set master gain in dB.
        """
        return jackgainctl_ext.set_gain (self._jgainctl, -1, gain)


    def set_mute(self, mute) :
        """
        Set master mute state (boolean). 
        """
        return jackgainctl_ext.set_mute (self._jgainctl, -1, mute)


    def set_channel_gain(self, chan, gain) :
        """
        Set channel gain in dB.
        """
        assert (chan >= 0)
        return jackgainctl_ext.set_gain (self._jgainctl, chan, gain)


    def set_channel_mute(self, chan, mute) :
        """
        Set channel mute state (boolean). 
        """
        assert (chan >= 0)
        return jackgainctl_ext.set_mute (self._jgainctl, chan, mute)


