package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.40.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.40.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````%!D``#0```#4H0````0`!30`(``*
M`"@`'``;``$``'"DC@``I(X``*2.``!P````<`````0````$````!@```#0`
M```T````-````$`!``!``0``!`````0````#````F`$``)@!``"8`0``&0``
M`!D````$`````0````$````````````````````XCP``.(\```4`````$```
M`0```*R=``"LG0``K)T``)P"``"P`@``!@`````0```"````M)T``+2=``"T
MG0``&`$``!@!```&````!`````0```!T`0``=`$``'0!```D````)`````0`
M```$````!````!B/```8CP``&(\``"`````@````!`````0```!1Y71D````
M```````````````````````&````$````%+E=&2LG0``K)T``*R=``!4`@``
M5`(```0````!````!````!0````#````1TY5`*&AEIP>\<PRP#3:/ZI:]G!%
MN82-+VQI8B]L9"UL:6YU>"UA<FUH9BYS;RXS`````#L```!-````"`````@`
M```!D!`$)4`$@"B```6@AE"$@($``@LH!(PX@!)!B@``"`````!-````3P``
M`%````````````````````!2`````````%,```!4````50``````````````
M5@```%<`````````````````````````6````%D```!:`````````%L`````
M``````````````!<````70```%X```````````````````!?````8```````
M``!B`````````&,`````````9```````````````````````````````````
M`&4```````````````````!F`````````&<```````````````````!I````
M?JF<FUGOEWEK"4/6=E0\E"'_Y`X-.I<+P^4_+(D'<-/9<5@<K4OCP&L:GE#5
M9-1\F4_#FR_%F[8S8MOM:W^:?`&O-.B[XY)\LP&+;T+ZUX=#1=7L9Z_,9A6,
M)<RCX/M)S]Y'+>-'Q@'@IP(*;\7?'[[:5QO[H6Q@````````````````````
M```````H$P````````,`"@```````*`````````#`!<`"0(`````````````
M$@```'$``````````````!(```#A`@`````````````2````K`(`````````
M````$@```)X"`````````````!(```![```````````````2````B@$`````
M````````$@```!T#`````````````!(```"'```````````````2`````0``
M````````````(````)<"`````````````!(```"Y```````````````2````
M80``````````````$@```*4"`````````````!(```#G```````````````2
M````U`$`````````````$@```!```````````````"````#<````````````
M```2````M`(`````````````$@```'8``````````````!(```#W`0``````
M```````2````NP(`````````````$@```"P``````````````"````!A`@``
M```````````2````_`$`````````````$@```%,``````````````!(````G
M`0`````````````6````!`$`````````````$@```&H!`````````````!(`
M``"H```````````````2````]0``````````````$@```.H!````````````
M`!(```#<`0`````````````2````P@(`````````````$@````($````````
M`````!(```#-```````````````2````\0(`````````````$0```,T!````
M`````````!(````+`P`````````````1````#P(`````````````$@```)H`
M`````````````!(```#$`0`````````````2````O0$`````````````$@``
M`%X!`````````````!(```"K`0`````````````2````(`$`````````````
M$@```)4!`````````````!(```!2`@`````````````2````XP$`````````
M````$@```/$!`````````````!(```!@`@`````````````2````^0(`````
M````````$@```#H!`````````````!(````>`@`````````````2````%`$`
M````````````$@```"4"`````````````!(```#O`@`````````````1````
M>@(`````````````$0```$8``````````````!$```"S`0`````````````2
M````3P$`````````````$0```#8"`````````````!(```#G`@``````````
M```2````P@``````````````$@```$0"`````````````!(````N`@``````
M```````2````>P$`````````````$@```&@"`````````````!(````T`P``
M```````````2````@0(`````````````$@```,D"`````````````!(```"F
M`0`````````````2````T`(`````````````$@```(@"`````````````"(`
M``#P`P``*#```"0````2``P`5`0``%R@````````$``8`"8$````H```````
M`!``%P`S!```2*`````````0`!@`J`0``#@C``"0`@``$@`,`"@$````H```
M`````"``%P!,!```R"$``%0````2``P`M00``$2@```$````$0`7`&@$``!(
MH````````!``&``7!```B$P```0````1``X`Y@,```1+``!H`0``$@`,`*`#
M``#\.```(`,``!(`#`"_!```D"T``'0````2``P`@@,``#0C```$````$@`,
M`"P$```4&0```````!(`#`!U`@``&!8``/P"```2``P`700``%R@````````
M$``8`'T$``!<H````````!``&`"V`P``\"P``*`````2``P`000``.!)```D
M`0``$@`,`(\$``!(H````````!``%P#:`P``^"4``+0!```2``P`=`0``(Q)
M``!4````$@`,`,<$``"L)P``1`4``!(`#`""!```A#$``'@'```2``P`DP,`
M`,@E```P````$@`,`,0#```<(@``&`$``!(`#``+!```'#P``#P!```2``P`
M7`0``%R@````````$``8`)8$``!,,```.`$``!(`#```7U]G;6]N7W-T87)T
M7U\`7TE435]D97)E9VES=&5R5$U#;&]N951A8FQE`%])5$U?<F5G:7-T97)4
M34-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E<FQ?<WES7W1E<FT`4&5R;%]G
M=E]F971C:'!V`'!E<FQ?9G)E90!097)L7V=E=%]S=@!097)L7V%T9F]R:U]U
M;FQO8VL`<&5R;%]D97-T<G5C=`!097)L7VYE=UA37V9L86=S`'!E<FQ?<G5N
M`%!E<FQ?;F5W6%,`4&5R;%]S=E]S971P=FX`<&5R;%]A;&QO8P!097)L7W-V
M7W-E='!V`'!E<FQ?8V]N<W1R=6-T`&)O;W1?1'EN84QO861E<@!097)L7W-A
M9F5S>7-M86QL;V,`4$Q?8W5R<F5N=%]C;VYT97AT`%!E<FQ?8W)O86M?;F]C
M;VYT97AT`%!,7VUE;6]R>5]W<F%P`%!E<FQ?;6=?<V5T`%!E<FQ?871F;W)K
M7VQO8VL`4&5R;%]S>7-?:6YI=#,`<&5R;%]P87)S90!?7W-T86-K7V-H:U]F
M86EL`&5X:70`;'-E96LV-`!R96%D9&ER-C0`<W1R9'5P`&-L;W-E9&ER`&]P
M96XV-`!S=')R8VAR`'5N;&EN:P!S=')T;VL`<W1R;&5N`')M9&ER`')E860`
M7U]M96UC<'E?8VAK`&UK9&ER`%]?=F9P<FEN=&9?8VAK`&=E='!I9`!R96%D
M;&EN:P!R96%L;&]C`%]?<W!R:6YT9E]C:&L`7U]M96UM;W9E7V-H:P!?7V9P
M<FEN=&9?8VAK`&QS=&%T-C0`7U]L:6)C7W-T87)T7VUA:6X`<W1D97)R`&UE
M;6-H<@!?7V-X85]F:6YA;&EZ90!S=')C:'(`;65M8VUP`&UE;7-E=`!O<&5N
M9&ER`&=E='5I9`!A8V-E<W,`;65M8W!Y`'-T<F-M<`!?7V5R<FYO7VQO8V%T
M:6]N`&%B;W)T`'-T<FYC;7``7U]E;G9I<F]N`%]?<F5G:7-T97)?871F;W)K
M`%]?<W1A8VM?8VAK7V=U87)D`%]?865A8FE?=6YW:6YD7V-P<%]P<C``7U]A
M96%B:5]U;G=I;F1?8W!P7W!R,0!L:6)P97)L+G-O`&QI8F,N<V\N-@!L9"UL
M:6YU>"UA<FUH9BYS;RXS`&QI8F=C8U]S+G-O+C$`<&%R7V-U<G)E;G1?97AE
M8P!P87)?8F%S96YA;64`6%-?26YT97)N86QS7U!!4E]"3T]4`'!A<E]E;G9?
M8VQE86X`<&%R7V-U<G)E;G1?97AE8U]P<F]C`'!A<E]D:7)N86UE`'-H85]F
M:6YA;`!G971?=7-E<FYA;65?9G)O;5]G971P=W5I9`!P87)?8VQE86YU<`!?
M24]?<W1D:6Y?=7-E9`!?7V1A=&%?<W1A<G0`7U]B<W-?<W1A<G1?7P!S:&%?
M=7!D871E`'AS7VEN:70`7U]E;F1?7P!?7V)S<U]E;F1?7P!?7V)S<U]S=&%R
M=`!S:&%?:6YI=`!?96YD`'!A<E]M:W1M<&1I<@!?961A=&$`<&%R7W-E='5P
M7VQI8G!A=&@`<&%R7V9I;F1P<F]G`&UY7W!A<E]P;`!P87)?9&EE`'!A<E]I
M;FET7V5N=@!'3$E"0U\R+C0`1T-#7S,N-0!'3$E"0U\R+C,T`$=,24)#7S(N
M,S,`+W5S<B]L:6(O<&5R;#4O-2XT,"]C;W)E7W!E<FPO0T]210``````````
M`@`!``(``@`"``$``0`#``$``0`"``$``0`"``$``@`!``$``@`"``(``@`!
M``0``@`!``$``0`!``$``0`"``(``@`"``$``@`"``4``@`!``(``@`!``(`
M`@`"``(``@`"``0``@`!``(``0`"``(``@`!``(``0`"``(``0`"``(``0`&
M``,``@`"``(``@`"``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!``$``0`!`````0`!`&`#```0````(```
M`!1I:0T```4`U`0````````!``$`=`,``!`````@````529Y"P```P#>!```
M``````$``P!6`P``$`````````"TD98&```&`.8$```0````LY&6!@``!`#Q
M!```$````!1I:0T```(`U`0```````"LG0``%P```+"=```7````_)\``!<`
M```$H```%P```!2@```7````&*```!<````@H```%P```"2@```7````**``
M`!<````LH```%P```#"@```7````.*```!<````\H```%P```$2@```7````
MR)\``!4+``#,GP``%0P``-"?```5$P``U)\``!49``#8GP``$QT``-R?```5
M'@``X)\``!4?``#DGP``%2<``.B?```5*0``[)\``!4\``#PGP``%3T``/2?
M```5/P``^)\``!5,``#8G@``%@,``-R>```6!```X)X``!8%``#DG@``%@8`
M`.B>```6!P``[)X``!8(``#PG@``%@D``/2>```6#```^)X``!8-``#\G@``
M%@X```"?```6#P``!)\``!80```(GP``%A$```R?```6$@``$)\``!84```4
MGP``%A4``!B?```6%@``')\``!87```@GP``%A@``"2?```6&@``*)\``!8;
M```LGP``%AP``#"?```6(```-)\``!8A```XGP``%B(``#R?```6(P``0)\`
M`!8D``!$GP``%B4``$B?```6)@``3)\``!8H``!0GP``%BH``%2?```6*P``
M6)\``!8L``!<GP``%BT``&"?```6+@``9)\``!8O``!HGP``%C```&R?```6
M,0``<)\``!8R``!TGP``%C,``'B?```6-```?)\``!8U``"`GP``%C8``(2?
M```6-P``B)\``!8X``",GP``%CD``)"?```6.@``E)\``!8^``"8GP``%D``
M`)R?```600``H)\``!9"``"DGP``%D,``*B?```61```K)\``!9%``"PGP``
M%D8``+2?```62```N)\``!9)``"\GP``%DH``,"?```62P``Q)\``!9,```(
M0"WIB0$`ZPB`O>@$X"WE!."?Y0[@C^`(\+[EB(L```#&C^((RHSBB/N\Y0#&
MC^((RHSB@/N\Y0#&C^((RHSB>/N\Y0#&C^((RHSB</N\Y0#&C^((RHSB:/N\
MY0#&C^((RHSB8/N\Y0#&C^((RHSB6/N\Y0#&C^((RHSB4/N\Y0#&C^((RHSB
M2/N\Y0#&C^((RHSB0/N\Y0#&C^((RHSB./N\Y0#&C^((RHSB,/N\Y0#&C^((
MRHSB*/N\Y0#&C^((RHSB(/N\Y0#&C^((RHSB&/N\Y0#&C^((RHSB$/N\Y0#&
MC^((RHSB"/N\Y0#&C^((RHSB`/N\Y0#&C^((RHSB^/J\Y0#&C^((RHSB\/J\
MY0#&C^((RHSBZ/J\Y0#&C^((RHSBX/J\Y0#&C^((RHSBV/J\Y0#&C^((RHSB
MT/J\Y0#&C^((RHSBR/J\Y0#&C^((RHSBP/J\Y0#&C^((RHSBN/J\Y0#&C^((
MRHSBL/J\Y0#&C^((RHSBJ/J\Y0#&C^((RHSBH/J\Y0#&C^((RHSBF/J\Y0#&
MC^((RHSBD/J\Y0#&C^((RHSBB/J\Y0#&C^((RHSB@/J\Y0#&C^((RHSB>/J\
MY0#&C^((RHSB</J\Y0#&C^((RHSB:/J\Y0#&C^((RHSB8/J\Y0#&C^((RHSB
M6/J\Y0#&C^((RHSB4/J\Y0#&C^((RHSB2/J\Y0#&C^((RHSB0/J\Y0#&C^((
MRHSB./J\Y0#&C^((RHSB,/J\Y0#&C^((RHSB*/J\Y0#&C^((RHSB(/J\Y0#&
MC^((RHSB&/J\Y0#&C^((RHSB$/J\Y0#&C^((RHSB"/J\Y0#&C^((RHSB`/J\
MY0#&C^((RHSB^/F\Y0#&C^((RHSB\/F\Y0#&C^((RHSBZ/F\Y0#&C^((RHSB
MX/F\Y0#&C^((RHSBV/F\Y0#&C^((RHSBT/F\Y0#&C^((RHSBR/F\Y0#&C^((
MRHSBP/F\Y0#&C^((RHSBN/F\Y0#&C^((RHSBL/F\Y7!`+>D!RJ#C#,!-X,@/
MC.6(PI_E*-!-XH0RG^4,P(_@@$*?Y1P`C>4<`(WB&!"-Y1@0C>(4((WE%""-
MX@1`C^`#,)SG`#"3Y20PC>4`,*#CU?__ZU`BG^50,I_E`B"4YPP@C>4"$*#A
M`S"4YQ`PC>4#`*#A=@T`ZS0RG^4#,)3G`%#3Y0``5>-2```*<"\=[B`RG^4<
M`)WE`U"`X@0`@.(#,)_G!P%PXP,@DN>F,]+E`S"#XZ8SPN5C``"*``&@X:'_
M_^OP$9_E\"&?Y0`PH.$!$(_@&$"=Y0(@C^``$)'E!`""Y0``E.4($(/ET!&?
MY1P@G>4!$(_@#!"#Y<01G^4!`%+C``"#Y0$0C^`$$(/E70``V@00A.(,P(/B
M`D&$X`3@D>0$X*SE`0!4X?O__QH#$(+BD$&?Y0``H.,!`8/G!2"@X01`C^``
M`(WE?!&?Y0P`E.4!$(_@!O__ZP``4.,I```*:%&?Y2!@C>(%4(_@#`"5Y4K_
M_^L`0*#A5`&?Y080H.$``(_@N```ZP``4.,@```*0#&?Y0,PC^`,`)/EY?[_
MZR#__^LP(9_E\#"?Y0(@C^`#,)+G`""3Y20PG>4"(#/@`#"@XS,``!H$`*#A
M*-"-XG"`O>C^_O_K`"&?Y0``4.,"((_@#`""Y2@```H3___K[#"?Y7`O'>X#
M,)_G`S"2YSY6P^6@___J#`"4Y>'^_^O2___J"`"5Y0``4.,!```*]@@`Z]G_
M_^JX`)_E!A"@X0``C^",``#K``!0XP@`A>72__\*]?__ZIP@G^5,,)_E`B"/
MX`,PDN<`()/E)#"=Y0(@,^``,*#C"@``&GPPG^5\`)_E``"/X`,PE.<,,(WE
M`Q"@X2G__^L$$*#CI___Z@$`H.-/___K$O__ZY"(```<`0``<(@``/P````4
M`0``)`$``!B)``!4B0``5(D``"PY```4.0``Z(@``*@)``#`B```/#8``)2(
M``#\A@``2(@``+R'```H-0``6(8``"@!``#D-@```+"@XP#@H.,$$)WD#2"@
MX00@+>4$`"WE'*"?Y1@PC^(#H(K@`#"@XP0P+>4,`)_E``":YR#__^N#_O_K
M?(4``#`!```4,)_E%""?Y0,PC^`"()/G``!2XQ[_+P&)_O_J9(4````!```L
M`)_E+#"?Y0``C^`H()_E`S"/X```4^$"((_@'O\O`1@PG^4#,)+G``!3XQ[_
M+P$3_R_AO(8``+2&```PA0``!`$``#@`G^4X,)_E``"/X#0@G^4#,(_@`#!#
MX`(@C^"C'Z#A0Q&!X,$0L.$>_R\!&#"?Y0,PDN<``%/C'O\O`1/_+^%XA@``
M<(8``.R$```(`0``6#"?Y5@@G^4#,(_@`B"/X``PT^4``%/C'O\O$1!`+>D!
MRJ#C#,!-X/@/C.4T,)_E`S"2YP``4^,"```**#"?Y0,`G^?L_O_KQ___ZQPP
MG^4!(*#C`S"/X``@P^40@+WH*(8``*B$```L`0``K(4``-R%``#-___J\$<M
MZ0'*H.,,P$W@X`^,Y;PPG^6\()_E`S"/X`(PD^<`D)/E``!9XR<```H`4-#E
M`("@X0%PH.$``%7C/0!5$P%0H!,`4*`#$0``&@!`F>4``%3C#```"@E@H.$%
M(*#A"!"@X00`H.&G_O_K``!0XP(``!H%,-3G/0!3XPL```H$0+;E``!4X_/_
M_QH$`*#A\(>]Z`!0H.$!,/7E``!3XST`4Q/[__\:"%!%X.;__^H)8$;@`5"%
MX@5`A.!&8:#A!`"@X0!@A^7PA[WH"4"@X>W__^HDA```&`$``!!`+>D!RJ#C
M#,!-X(@/C.6,()_E<-!-XH@PG^4-$*#A`B"/X`!`H.$#,)+G`#"3Y6PPC>4`
M,*#C9/[_ZP``4.,``*`3!```&A`PG>4/.@/B"@I3XP$)4Q,*```*2""?Y4`P
MG^4"((_@`S"2YP`@D^5L,)WE`B`SX``PH.,'```:<-"-XA"`O>@"$*#C!`"@
MX0G^_^L0#V_AH`*@X>[__^H^_O_K0(,``!P!``#\@@``\$`MZ0'*H.,,P$W@
M<`^,Y6`AG^5\T$WB7#&?Y0!04.("((_@`S"2YP`PD^5T,(WE`#"@XP(```H`
M,-7E``!3XPH``!HT(9_E+#&?Y0(@C^`#,)+G`""3Y70PG>4"(#/@`#"@XT(`
M`!I\T(WB\("]Z+C]_^L`8%#B\?__"@8`H.$X_O_K``!0XRL```H3,-#E$T"`
MXBX`4^,"```:`3#4Y0``4^/T__\*$S#0Y2X`4^,F```*!0"@X>7]_^L`<*#A
M!`"@X>+]_^L`<(?@`G"'X@<`H.$"_O_KI#"?Y0<@H.$"$*#C`S"/X`1`C>4`
M4(WE`$"@X1[^_^L($(WB!`"@X<3]_^L!`'#C!`"@X0,```H8,)WE#SH#X@$)
M4^,2```*SOW_ZP0`H.&Q_?_K!@"@X0S^_^L``%#CT___&@8`H.';_?_K!0"@
MX?']_^N\___J`3#4Y2X`4^/5__\:`C#4Y0``4^/$__\*T?__ZJ/__^OK___J
MW?W_ZY""```<`0``9((``!PP``#P3RWI`<J@XPS`3>#0#XSE'"*?Y0S03>(8
M,I_E`4"@X0(@C^``4*#A#(*?Y0,PDN<(@(_@`#"3Y00PC>4`,*#C`##1Y3T`
M4^,!0($"!`"@X9W]_^L-$*#A`&"@X04`H.$A___K`)!0XD````J6_?_K``!6
MX3<``)K`,9_E`'"=Y0.`F.<'<:#A`##5Y0``4^,]`%,3`3"@$P`PH`,%```*
M!3"@X0$@\^4``%+C/0!2$_O__QH%,$/@`@"&X@!@F.4#`(#@I/W_ZP``4.,'
M`(;G%0``"@`PF.4%`*#A!Q"3YP$PT.0`,,'E/0!3XP``4Q,&```*`2"!X@$P
MT.0"$*#A`3#"Y```4^,]`%,3^?__&@$@H.$],*#C`##!Y0$PU.0!,.+E``!3
MX_O__QH0(9_E`#&?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XS<``!H,T(WB
M\(^]Z`&02>(!,-3D`3#IY0``4^/[__\:[O__ZL@PG^4#@)CG`'"8Y0`PE^4`
M`%/C)@``"@<PH.$$(+/E"1"@X0&0B>(``%+C^O__&@,0@>(!L:#AF#"?Y0,P
MC^``()/E``!2XQ```!H+`*#A`2"@XP`@@^5C_?_K`*!0XM7__PH'$*#A"7&@
MX0LPH.$'(*#AB?W_ZP"@B.4`D(WE`9")X@`PH.,),8KGH?__Z@L0H.$'`*#A
M@_W_ZP"@H.$``%#C``"(Y</__PH)<:#A\?__Z@.0H.$(L*#CW?__ZDS]_^L(
M@0``'`$``/B````8`0``]'\``/R```#P02WI`<J@XPS`3>```(SE(``,Y0%P
MH.'\$)_E_#"?Y0':3>(!$(_@"-!-X@!@H.$!"HWB!`"`X@2`C>(#,)'G_R\`
MXP@0H.$`,)/E`#"`Y0`PH./(`)_E``"/X$G]_^L`0%#B!@``V@$P1.(#P(W@
M!"#<Y0``4N,7```:``!3XQ0``!J<,)_E<`\=[@<@H.$&$*#A`S"?YP,`D.?3
M_/_KA""?Y0$:C>)P,)_E!!"!X@(@C^`#,)+G`""3Y0`PD>4"(#/@`#"@XQ(`
M`!H!VHWB"-"-XO"!O>@#0*#A!""@X2\0H.,(`*#A0?W_ZP``4./C__\*-`"?
MY7#/'>X$,*#A"""@X080H.$``)_G``"<Y^;\_^OA___J__S_Z[A^```<`0``
MW"P``%1_```L?@``X'X``!!`+>D!RJ#C#,!-X.@/C.5P$)_E<$\=[FP@G^40
MT$WB:`"?Y0(@C^!D,)_E`1"?YP,PC^``()+G`0"4YU00G^4,((WE`1"/X`W]
M_^M($)_E2#"?Y0#`H.-$()_E`S"/X`$0G^<"((_@`0"4YS00G^4$P(WE`1"/
MX``PC>6L_/_K$-"-XA"`O>B(?@``A'T``!`!``"(;```S"L``%1^``"X+@``
M=!<``+@K```X`)_E"-!-XC0@G^5PSQWN,!"?Y0(@C^`L,)_E``"?YP,PC^`!
M()+G(!"?Y0``G.<!$(_@!""-Y0C0C>+E_/_J['T``.A\```0`0``[&L``#0K
M``#P0"WI`<J@XPS`3>```(SE`<I,X@``C.4P``SEX""?Y0+:3>+<,)_E'-!-
MX@(@C^`"&HWB%!"!X@Q`C>(!:HWB`S"2YQ!@AN(`,)/E`#"!Y0`PH..X_/_K
MK""?Y0(0H..H,)_E`B"/X`4`C>@#,(_@!`"@X0$@`>.[_/_K!`"@X0$JH.,&
M$*#AL?S_ZP!`4.(`4*"S"@``N@%PA.('`*#AC?S_ZP!04.(%```*!S"@X080
MH.$$(*#A5_S_ZP`PH.,$,,7G2""?Y0(:C>(T,)_E%!"!X@(@C^`#,)+G`""3
MY0`PD>4"(#/@`#"@XP,``!H%`*#A`MJ-XAS0C>+P@+WH>/S_ZWQ\```<`0``
M["H``-@J``#4>P``N/__ZO!/+>D!RJ#C#,!-X```C.6H``SE1"*?Y4+=3>)`
M,I_E!-!-X@(@C^``8*#A-`*?Y4'-C>(0@(WB/,",X@,PDN<!0*#A``"/X`@0
MH.$`,)/E`#",Y0`PH..\_?_K+Q"@XP!0H.$&`*#A`/S_ZP``4.,2```*\`&?
MY080H.$``(_@=_[_ZP8`H.'@(9_E01V-XLPQG^4\$('B`B"/X`,PDN<`()/E
M`#"1Y0(@,^``,*#C:0``&D+=C>($T(WB\(^]Z`0`H.$T_/_KI!&?Y0$0C^!#
M_/_K`$!0XN7__PJ4D9_E@*"-X@`PU.4$<$KB"9"/X```4^,[```*``!5XP0`
M``H$$*#A!0"@X6C\_^L``%#C*P``"@0`H.$$_/_K`3!`X@,PA.``L*#A`P!4
MX0`PA#``$*`S`0"$,@,``#H(``#J`P!0X0`0P^4"```*`2!SY2\`4N/Y__\*
M!`"@X?/[_^L`L*#A!@"@X?#[_^L+`(#@`0"`X@$*4..]__\J^`"?Y0DPH.$!
M*J#C`A"@XP``C^``0(WE00"-Z0<`H.$K_/_K"!"@X0<`H.'1^__K``!0XP,`
M`!I@,!KE#SH#X@()4^,5```*M!"?Y0``H.,!$(_@`_S_ZP!`4.*E__\*`##4
MY0``4^/#__\:``!5XQ8```H`,-7E+@!3XP,```J`0)_E`;"@XP1`C^#5___J
M`3#5Y0``4^/J__\*]___Z@$0H.,'`*#AK_O_ZP``4./D__\:4`"?Y0<0H.$`
M`(_@"/[_ZP<`H.'4^__KCO__ZCA`G^4!L*#C!$"/X,'__^K:^__K:'L``!P!
M``#X*0``S"D``/1Z``",*0``<"D``-@H``",*```-"@``!`H``#D)P``$$`M
MZ0'*H.,,P$W@^`^,Y2\0H.,`0*#A?_O_ZP``4.,!0(`2!`"@X1!`O>BV^__J
M\$`MZ0'*H.,,P$W@``",Y2``#.5\(9_E`=I-XG@QG^4,T$WB`B"/X`$:C>($
M$('B`%!0X@,PDN<`,)/E`#"!Y0`PH.,Q```*`##5Y0``4^,N```*A?O_ZP$@
M@.(`0*#A`0I2XSH``(H$<(WB!1"@X0<`H.&#^__K`3!$X@,PA^`#`%?A"0``
M*@0`A^`%((WB`!"@XP(``.H"`%#A`!#`Y0(```H!,'#E+P!3X_G__PHO$*#C
M!P"@X4O[_^L``%#C)0``"@<`4.$`,*"#`##`A0<`H.&`^__KP""?Y0$:C>*T
M,)_E!!"!X@(@C^`#,)+G`""3Y0`PD>4"(#/@`#"@XR,``!H!VHWB#-"-XO"`
MO>B,()_E`1J-XGPPG^4$$('B`B"/X`,PDN<`()/E`#"1Y0(@,^``,*#C%0``
M&F0`G^4``(_@`=J-X@S0C>+P0+WH8?O_Z@``H./>___J2""?Y0$:C>(P,)_E
M!!"!X@(@C^`#,)+G`""3Y0`PD>4"(#/@`#"@XP(``!H@`)_E``"/X.O__^I<
M^__KJ'@``!P!``#D=P``K'<``#@F``!@=P``["4``'!`+>D!RJ#C#,!-X.@/
MC.7$))_E"-!-XL`TG^4-0*#A`B"/X+ADG^6X5)_E!F"/X`,PDN<%4(_@`#"3
MY00PC>4`,*#C!!"@X08`H.&B_/_K``!0XPX```J,-)_E``"=Y0,PE><`$)/E
M`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I4
M9)_E!F"/X`00H.$&`*#AC/S_ZP``4.,.```*-#2?Y0``G>4#,)7G`!"3Y0`Q
M@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J`&2?
MY09@C^`$$*#A!@"@X7;\_^L``%#C#@``"MPSG^4``)WE`S"5YP`0D^4`,8'@
M!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZJQCG^4&
M8(_@!!"@X08`H.%@_/_K``!0XPX```J$,Y_E``"=Y0,PE><`$)/E`#&!X`0@
MD^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I88Y_E!F"/
MX`00H.$&`*#A2OS_ZP``4.,.```*+#.?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E
M`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J!&.?Y09@C^`$
M$*#A!@"@X33\_^L``%#C#@``"M0RG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A
M@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZK!BG^4&8(_@!!"@
MX08`H.$>_/_K``!0XPX```I\,I_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G
M``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I<8I_E!F"/X`00H.$&
M`*#A"/S_ZP``4.,.```*)#*?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``
M4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J"&*?Y09@C^`$$*#A!@"@
MX?+[_^L``%#C#@``"LPQG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C
M\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZK1AG^4&8(_@!!"@X08`H.'<
M^__K``!0XPX```IT,9_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'_
M_PH$,(/B!""3Y00@@^0``%+C^___&NO__^I@89_E!F"/X`00H.$&`*#AQOO_
MZP``4.,.```*'#&?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*
M!#"#X@0@D^4$((/D``!2X_O__QKK___J#`&?Y000H.$``(_@L?O_ZP`04.("
M```*^`"?Y0``C^!Q_/_K\`"?Y000H.$``(_@J/O_ZP`04.("```*W`"?Y0``
MC^!H_/_KU`"?Y000H.$``(_@G_O_ZP`04.(2```*P`"?Y0``C^!?_/_KN""?
MY60PG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,0```:F!"?Y9@`G^4!$(_@
M``"/X`C0C>)P0+WH3_S_ZH0`G^4$$*#A``"/X(;[_^L`$%#BZ?__"G``G^4`
M`(_@1OS_Z^7__^H>^O_K^'8``!P!``#()0``Y'8``!@!``!H)0``&"4``,PD
M``!\)```-"0``'@C``"0(P``1",``/@B```D(@``4"(``"0B``!`(@``0"(`
M`#PB``!@(0``H'(``!@B``!\(0``Z"$``&PA```$X"WE`<J@XPS`3>#P#XSE
M>""?Y0S03>)T,)_E#1"@X0(@C^!L`)_E``"/X`,PDN<`,)/E!#"-Y0`PH.-5
M^__K``!0XP4```H``-#E``!0XS``4!,!`*`#``"@$P$`(.(T()_E*#"?Y0(@
MC^`#,)+G`""3Y00PG>4"(#/@`#"@XP$``!H,T(WB!/"=Y.#Y_^NT<0``'`$`
M`,P@``!L<0``#P`MZ0'*H.,$X"WE#,!-X/`/C.5($)_E#-!-XD0@G^44,(WB
M`1"/X#P`G^4`,(WE.,"?Y0``C^`"()'G`A"@XP`@DN4$((WE`""@XQ`@G>4,
MP)#G``"<Y;#Y_^O_`*#C__G_ZQ!Q```<`0```'$``"`!``#P1RWI`,"@XP'H
MH.,,P$W@#N!,X`'*3.(``(SE#@!<X?O__QHX``[E`=A-XAC03>+,P9_E`A"@
MXP'HC>(,P(_@%.".X@`0C>4`(*#CM!&?Y0`PH.,`@*#A`1"<YP`0D>4`$([E
M`!"@XY[Y_^L!`%#B`'#!XL<OH.$","#@`C!3X',P_^8","/@`C!3X`(@PN`#
M8%#@`G#'X```5^,X``"Z")"@XP&00.,,4(WB`!"@XP8@H.$',*#A"`"@X0`0
MC>6(^?_K`0!QXP$`<`,]```*"2"@X040H.$(`*#A2_G_ZP$`<.,I```*!P!0
MXQ\``)H(`$#B`$"5X!P``"H(H9_E"J"/X`(``.H!0$3B!0!4X18``#H((*#C
M"A"@X00`H.$3^?_K``!0X_;__QH%0$3@!@"4X,0?I^#0()_E`<B-XL`PG^44
MP(SB`B"/X`,PDN<`()/E`#"<Y0(@,^``,*#C)@``&@'8C>(8T(WB\(>]Z`%H
M5N(`<,?B`0!WXP$(=@/)__\:``#@XP`0H.'I___J?""?Y0$8C>)H,)_E%!"!
MX@(@C^`#,)+G`""3Y0`PD>4"(#/@`#"@XQ```!I4`)_E``"/X'#__^M,()_E
M`1B-XC`PG^44$('B`B"/X`,PDN<`()/E`#"1Y0(@,^``,*#C`@``&B0`G^4`
M`(_@8O__ZS[Y_^N$<```'`$``/QP``!X;P``(&\``+`>``#H;@``:!X``!!`
M+>D!RJ#C#,!-X/@/C.7O^/_K$OG_ZP``4.,``)`5$("]Z/!`+>D!RJ#C#,!-
MX-`/C.4`(9_E'-!-XOPPG^40$(WB`B"/X/1PG^4`4*#A!W"/X`,PDN<'`*#A
M`#"3Y10PC>4`,*#C?/K_ZP!`4.("```*`##4Y0``4^,.```:P""?Y;0PG^4"
M((_@`S"2YP`@D^44,)WE`B`SX``PH.,D```:H`"?Y040H.$``(_@'-"-XO!`
MO>@M^__J!0"@X=[X_^L`8*#A!`"@X=OX_^L`8(;@`F"&X@8`H.'[^/_K:#"?
MY6@0G^4&(*#A`S"/X`A`C>4!$(_@`%"-Y000C>4"$*#C`$"@X13Y_^M$()_E
M*#"?Y0(@C^`#,)+G`""3Y10PG>4"(#/@`#"@XP00H`$'`*`!W?__"N?X_^M8
M;@``'`$```@>```4;@``K!T``(`<``!T'```B&T``/!/+>D!RJ#C#,!-X.`.
MC.7\T$WBU":?Y=06G^60<(WB`B"/X,S&G^4@((WE`1"/X,0FG^4,P(_@'`"-
MY0PPC.(88(SBJ$"-X@3@H.$PP(SB`B"1YYQ0C>(H@(WB`""2Y?0@C>4`(*#C
M!P"3Z`<`A^@/`+;H#P"NZ`,`EN@#`([H!P"<Z`<`A>AP!I_E"!"@X0``C^`:
M^O_K`&!0X@(```H`,-;E``!3XRD!`!IQ^/_KE/C_ZP``4..(```*`&"0Y0``
M5N.%```*!@"@X83X_^N`(*#A`;""X@L`H.&D^/_K`##6Y0!0H.$``%/C)`"-
M%0N0H!$D`0`*!*:?Y0J@C^`+(*#A!0"@X0`PC>4"$*#C"C"@X0)0A>*Y^/_K
M"P!9X0LPH#$),*`A"R!#X`*P2^(#L$O@`B""X@,`4N$"L(L@`["+,`$P]N4`
M`%/CZ___&K`%G^4``(_@`@``Z@0`M.4``%#C#0``"@@0H.'D^?_K`%!0XOC_
M_PH`,-7E``!3X_7__PH5^O_K``!0X_+__PH%`*#A;?C_ZP!`4.(-```:!$"7
MY```5.,*```*`##4Y0``4^/3```*!`"@X0?Z_^L``%#C]?__"@0`H.%?^/_K
M`$!0XO'__PH$`*#A0/C_ZR1@G>4`4*#A!@"@X3SX_^L`4(7@`5N%X@10A>(%
M`*#A6_C_ZP`UG^4`<*#A!2"@X0,PC^`(,(WE\#2?Y0(0H.,,8(WE`S"/X`0P
MC>7@-)_E`$"-Y0,PC^!Q^/_K!QV@XP<`H.'>]__K`0!PXP,``!J)^/_K`!"0
MY1$`4>.O```:"!"@X0<`H.%0^/_K`$!0XC$```J<%)_E(`"=Y1PPG>64))_E
M`0"0YP(0H.,`,)/E`B"/X`!PC>4``)#E.OC_ZP!@H.-T))_E3#2?Y0(@C^`#
M,)+G`""3Y?0PG>4"(#/@`#"@X_P``!H&`*#A_-"-XO"/O>@=^/_K`&!0XHL`
M`!H$`)7D``!0XPD```H($*#A@_G_ZP`P4.+X__\*`##3Y0``4^/R__\:!`"5
MY```4./U__\:"&2?Y0T`H.,-D*#C$_C_ZP9@C^``4*#A";"@X5,PH.,D`(WE
M;O__ZC@PG>4/.@/B`0E3X\G__QI`8)WER/?_ZP``5N'%__\:.#"=Y5,PZ.<'
M#5/CP?__&@4`H.'_]__K`&"@X:@#G^4($*#A``"/X%[Y_^L`$%#B@```"APP
MG>4``)/EA?O_ZP"@4.)[```*"@"@X<WW_^L`D*#A`P!0XPP``)IL$Y_E!`!`
MX@``BN`!$(_@)?C_ZP``4.,%```:!`!9XZ<```H)D(K@!3!9Y2\`4^.C```*
M/`.?Y0@0H.$``(_@0?G_ZP``4.,#```*`##0Y3``4^,``%,3!```&@H`H.$`
M$*#CO_?_ZP"`4.)=```:!!.?Y00#G^4!$(_@`*.?Y0``C^#\DI_E]/G_Z_B"
MG^7A]__K]#*?Y?0BG^4"$*#C`S"/X`0PC>7H,I_E`B"/X`@`C>4&`*#A#""-
MY0,PC^`%(*#A`'"-Y0J@C^`)D(_@"("/X-SW_^L.``#J^/?_ZP`PD.41`%/C
M#P``&LKW_^L!0(3B`,"@X0HPH.$%(*#A`A"@XP8`H.$0D(WE#$"-Y8`1C>C,
M]__K!QV@XP8`H.$Y]__K`0!PX^O__PH'`*#A9??_ZV`"G^4&$*#A``"/X,;Y
M_^L&`*#A</[_ZV;__^H#0*#A,?__ZG/W_^N`(*#A`;""XNW^_^KP(9_E(`"=
MY1PPG>4"`)#G`#"3Y1PBG^4$$(WE`A"@XP!PC>4"((_@``"0Y8[W_^M2___J
M6_[_Z_PAG^68,9_E`B"/X`,PDN<`()/E]#"=Y0(@,^``,*#C3P``&@8`H.'\
MT(WB\$^]Z'#W_^HD`(WE[O[_ZAPPG>4`H)/E@/__ZK;]_^L`D*#A`*!1XIW_
M_[H*(%#B`+"@XP`PRN((`*#A`+"-Y6?W_^O`$(WB!B"@XP@`H.$<$(WE+/?_
MZQP0G>4&`%#C-0``&L`@G>5#/*#C03-$XP,`4N&*__\:M"#1X4@U!.,#`%+A
MAO__&C(@6>((`*#A`##*XLA`C>(`L(WE4/?_ZP@`H.$H(*#C!!"@X1;W_^LH
M`%#C(```&APQG^4%(*#A&!&?Y08`H.$#,(_@!#"-Y0PQG^4!$(_@"$"-Y0P0
MC>4#,(_@`A"@XP!PC>7PL,WE8O?_ZYG__^H<,)WE!)"3Y0``6>-7__\*"0"@
MX1/W_^L#`%#C4___FL@0G^4$`$#B``")X`$0C^!L]__K``!0XPF@H`%+___J
M+_?_ZZ@@G^4P,)_E`B"/X`,PDN<`()/E]#"=Y0(@,^``,*#C]?__&H@`G^4`
M`(_@1OW_ZR!M```4;0``2&X``!P!``!@&P``&!P``%@;```8&P``]!D``.@:
M```@`0``W!H``)1J``#H&0``0!H```0:``!L&```K!@``"@8``"$&0``*!H`
M`)`7``#`%P``2!H``(`9``#\%@``[!<``.!G``#,%0``5!@``(`7```H%P``
M>&8``/`6``!P0"WI`<J@XPS`3>#H#XSEH!*?Y0C03>*<(I_E<$\=[@$0C^"4
M,I_EE`*?Y0U@H.$"()'G``"/X`T0H.$`()+E!""-Y0`@H.,#,)_G`U"4YV0P
MU>5E,,7E2_C_ZP``4.,"```*`##0Y3$`4^-O```*COO_ZTPRG^4#,)_G`U"4
MYT02G^4%`*#A`S"@XP$@H.,!$(_@BO;_ZP!04.(,```**#*?Y0P@E>4#,)_G
M`!"2Y1PBG^4#`)3G`B"/X(;V_^L,,)7E`!"3Y0@PD>4!!1/C80``&OP1G^4#
M,*#C`2"@XP$0G^<!`)3G[!&?Y0$0C^!S]O_K`%!0X@L```K<`9_E!A"@X0``
MC^`?^/_K##"5Y0`@4.(`$)/E2```"L`QG^4#,)_G`P"4YVOV_^NT,9_E`2"@
MX[`1G^4!$(_@`S"?YP-0E.<%`*#A3O;_ZYPAG^4`$*#A!0"@X0(@C^"049_E
M7?;_ZXPQG^6,`9_E!5"/X``@H.,&$*#A``"/X`,PG^<#,)3G92##Y?_W_^L`
M0*#A``!0XP@`A>42```*!QV@XP0`H.$H]O_K`2"`X@`PH.$1`%#C`0!2$RT`
M`(H\(9_E]#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XR,``!H(T(WB<("]
MZ`0`E>6E_?_K`$"@X0``4.,(`(7EYO__&OP@G^6P,)_E`B"/X`,PDN<`()/E
M!#"=Y0(@,^``,*#C$@``&MP`G^4``(_@A_;_ZP$PT.4``%/CD/__"HO__^K$
M,)_E`0"@X0,PC^`$,)/E`!"3Y23Y_^NS___JK#"?Y0,PG^<#`)3G7?;_ZYC_
M_^ID]O_KF!"?Y3P@G^4!$(_@`B"1YP`0DN4$()WE`1`RX``@H./U__\:>!"?
MY00@H.%T`)_E`1"/X```C^`$$)'E`!"1Y67V_^NH90``'`$``(QF``"8%```
M7&8``!`6```L9@``[!4``/1E``#`%0``@!,``+1E``"890``>!4``'`5``#<
M90``6&4````3```$9```P&,``*@4````90``=&0``$QC``"D9```3!0``#!`
M+>D!RJ#C#,!-X.@/C.4,(9_E`$"@X0@QG^4,T$WB`B"/X``!G^4-$*#A``"/
MX`,PDN<`,)/E!#"-Y0`PH..)]__K`#!0X@H```H`,-/E,`!3XP``4Q,!,*`#
M`#"@$P$P(^(``%3C`#"@`P$P`Q(``%/C"@``&K`@G^6D,)_E`B"/X`,PDN<`
M()/E!#"=Y0(@,^``,*#C(```&@S0C>(P@+WH`##4Y0``4^/Q__\*!`"@X4CZ
M_^LO$*#C`%"@X<#U_^L``%#C`5"`$@4`H.'X]?_K6!"?Y00@H.,!$(_@)/;_
MZP``4./B__\:1""?Y3`PG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,#```:
M!`"@X0S0C>(P0+WHM??_ZO#U_^N(8@``'`$``)P1```H8@``M!$``+1A``#P
M3RWI`<J@XPS`3>"`#HSE$$R?Y5??3>(,[)_E$#"-X@1`C^`#$*#A'""`XES`
M@.(.X)3G`.">Y53AC>4`X*#CC0IB].``\/,"`%SAW0I!]/K__QH&2]WM"#O=
M[1$>C>(*6]WM````ZK)18O((*]/M-""#X@`;T^T(,(/B`0!3X8\'8O2Q$4+S
ML0%`\[0!0/.S06/RM3%E\J`88/(P`.'SL`%A\@X+P^WN__\:($"0Z9DY!^."
M.D7C#$"0Y1#`G>4`8)#E!!`NX!!PD.4#P(S@!1`!X&61H.$'P(S@!!`AX!0@
MG>7FS8S@`<",X`D0+N`#((+@!A`!X&:!H.$$((+@&*"=Y0X0(>`"$('@""`I
MX`.@BN`,(`+@#J"*X`D@(N`*((+@'*"=Y>P=@>!LP:#A`Z"*X`F@BN`,D"C@
M`9`)X.$M@N`(D"G@81&@X0J0B>`@H)WEXIV)X`.@BN`(H(K@`8`LX`*`".!B
M(:#A#(`HX`*P(>`*@(C@)*"=Y0FP"^#IC8C@`Z"*X`R@BN!IP:#A`9`KX`RP
M(N`*D(G@**"=Y0BP"^#HG8G@`Z"*X`&@BN!H$:#A`H`KX`&P+.`*@(C@+*"=
MY0FP"^#IC8C@`Z"*X`*@BN!I(:#A#)`KX`*P(>`*D(G@,*"=Y0BP"^#HG8G@
M`Z"*X`R@BN!HP:#A`8`KX`RP(N`*@(C@-*"=Y0FP"^#IC8C@`Z"*X`&@BN!I
M$:#A`I`KX`&P+.`*D(G@.*"=Y0BP"^#HG8G@`Z"*X`*@BN!H(:#A#(`KX`*P
M(>`*@(C@/*"=Y0FP"^#IC8C@`Z"*X`R@BN!IP:#A`9`KX`RP(N`*D(G@0*"=
MY0BP"^#HG8G@`Z"*X`&@BN!H$:#A`H`KX`&P+.`*@(C@1*"=Y0FP"^#IC8C@
M`Z"*X`*@BN!I(:#A#)`KX`*P(>`*D(G@2*"=Y0BP"^#HG8G@`Z"*X`R@BN!H
MP:#A`8`KX`RP(N`*@(C@3*"=Y0FP"^#IC8C@`Z"*X`&@BN!I$:#A`I`KX`&P
M+.`*D(G@4*"=Y0BP"^#HG8G@`Z"*X`*@BN!H(:#A#(`KX`J`B.!4H)WEZ8V(
MX`.@BN`,P(K@`J`AX`F@"N!ID:#A`:`JX`R@BN!8P)WEZ*V*X`/`C.`!P(S@
M"1`BX`@0`>!H@:#A`A`AX`P0@>!<P)WEZAV!X`,PC.`(P"G@"L`,X`(P@^`)
MP"S@8""=Y0/`C."A.P[CV3Y&XVJAH.$#((+@X<V,X`F0@N!D()WE`R""X`@@
M@N`*@"C@`8`HX`B`B>!AD:#A:!"=Y>R-B.`#$('@"A"!X`F@*N`,H"K@;,&@
MX0H@@N#HK8+@;""=Y0,@@N`)((+@#)`IX`B0*>!H@:#A"1"!X.J=@>!P$)WE
M`Q"!X`P0@>`(P"S@"L`LX&JAH.$,((+@Z<V"X'0@G>4#((+@""""X`J`*.`)
M@"C@:9&@X0@0@>#LC8'@>!"=Y0,0@>`*$('@":`JX`R@*N!LP:#A"B""X.BM
M@N!\()WE`R""X`D@@N`,D"G@")`IX&B!H.$)$('@ZIV!X(`0G>4#$('@#!"!
MX`C`+.`*P"S@:J&@X0P@@N#IS8+@A""=Y0,@@N`(((+@"H`HX`F`*.!ID:#A
M"!"!X.R-@>"($)WE`Q"!X`H0@>`)H"K@#*`JX&S!H.$*((+@Z*V"X(P@G>4#
M((+@:K&@X0D@@N`,D"G@")`IX&B!H.$)$('@ZIV!X)`0G>4#$('@#!"!X`C`
M+.`*P"S@#"""X.FM@N"4()WE`R""X`@@@N`+@"C@"<`HX&F1H.$,$('@F,"=
MY>J-@>`)$"O@`\",X`H0(>`+P(S@G+"=Y6JAH.$!((+@`["+X.@=@N`)L(O@
M"I`IX`@@*>!H@:#A`L",X*`@G>7AG8S@`R""X`H@@N`(H"K@`<`JX*2@G>5A
M$:#A#,"+X`.@BN#IS8S@"*"*X`&`*.`)@"C@:9&@X0@@@N"PL)WE[(V"X*@@
MG>4#((+@`2""X`D0(>`,$"'@;,&@X0&@BN`,$"G@Z*V*X`@0(>`!((+@:(&@
MX>H=@N"L()WE`R""X`@P+.`*,"/@"2""X`,@@N#</`OC&S](XVJ1H.$#L(O@
M":"(X0RPB^"TP)WEX2V"X`&@"N`#P(S@81&@X0C`C.`)@`C@"*"*X0&`B>$+
MH(K@`H`(X`&P">#BK8K@"X"(X;RPG>4,@(C@N,"=Y6(AH.$#L(O@`\",X`&P
MB^`)D(S@`L"!X0K`#.`"$`'@`<",X>J-B.`)P(S@P)"=Y6JAH.'HS8S@"A""
MX0.0B>`"D(G@"!`!X`H@`N!H@:#A`A"!X0@@"N`+L('@"!"*X0P0`>#LO8O@
M`A"!X6S!H.$)$('@Q)"=Y0P@B.'K'8'@`Y")X`L@`N`*D(G@R*"=Y6NQH.$#
MH(K@"*"*X`R`".`(((+A88&@X0F0@N`+((SAX9V)X`$@`N#,$)WE`Q"!X`P0
M@>`+P`S@#,""X0@@B^$*P(S@"2`"X`B@"^#IS8S@"B""X="@G>5ID:#A`1""
MX`.@BN`)((CA"Z"*X-2PG>4,(`+@[!V!X`.PB^!LP:#A"+"+X`F`".`(((+A
M88&@X0J@@N`,((GAX:V*X`$@`N#8$)WE`Q"!X`D0@>`,D`G@"9""X0@@C.$+
MD(G@"B`"X`BP#.#JG8G@"R""X>"PG>4!$(+@W""=Y6JAH.$#L(O@`R""X`BP
MB^`,P(+@"B"(X0D@`N`*@`C@Z1V!X`B`@N%ID:#A#("(X`D@BN%AP:#AX8V(
MX`$@`N#D$)WE`Q"!X`H0@>`)H`K@"J""X0P@B>$+H(K@""`"X`RP">#HK8K@
M"R""X>RPG>4!$(+@Z""=Y6B!H.$#L(O@`R""X`RPB^`)D(+@""",X0H@`N`(
MP`S@ZAV!X`P@@N%JH:#A"9""X`H@B.%AP:#AX9V)X`$@`N#P$)WE`Q"!X`@0
M@>`*@`C@"(""X0P@BN$+@(C@"2`"X`RP"N#IC8C@"R""X?BPG>4!((+@]!"=
MY6F1H.$#L(O@`Q"!X`RPB^`*H('@"1",X0@0`>`)P`S@#,"!X>@M@N`*P(S@
M_*"=Y6B!H.'BS8S@"!")X0,PBN`),(/@`A`!X`B0">!B(:#A`J"(X0D0@>$+
M$('@#*`*X`*P".#L'8'@;)&@X0N@BN$`P9WE`Z"*X-8Q#.-B.DSC`\",X.&M
MBN`(@(S@!,&=Y6&QH.$#P(S@`L",X`D@(N`!("+@"!&=Y0*`B.`+("G@ZHV(
MX`H@(N`#$('@:J&@X0D0@>`"P(S@"I`KX.C-C.`(D"G@#"&=Y0D0@>!H@:#A
M[)V!X!`1G>4#((+@`Q"!X`L@@N`*$('@"*`JX`R@*N!LP:#A"B""X.FM@N`4
M(9WE`R""X`@@@N`,@"C@"8`HX&F1H.$($('@ZHV!X!@1G>4#$('@#!"!X`G`
M+.`*P"S@:J&@X0P@@N#HS8+@'"&=Y0,@@N`)((+@"I`IX`B0*>!H@:#A"1"!
MX.R=@>`@$9WE`Q"!X`H0@>`(H"K@#*`JX&S!H.$*((+@Z:V"X"0AG>4#((+@
M""""X`R`*.`)@"C@:9&@X0@0@>#JC8'@*!&=Y0,0@>`,$('@"<`LX`K`+.!J
MH:#A#"""X.C-@N`L(9WE`R""X`D@@N`*D"G@")`IX&B!H.$)$('@[)V!X#`1
MG>4#$('@"A"!X`B@*N`,H"K@;,&@X0H@@N#IK8+@-"&=Y0,@@N`(((+@#(`H
MX`F`*.!ID:#A"!"!X.J-@>`X$9WE`Q"!X`P0@>`)P"S@"L`LX&JAH.$,((+@
MZ,V"X#PAG>4#((+@"2""X`J0*>`(D"G@:(&@X0D0@>!`D9WE[!V!X`.0B>`*
MD(G@"*`JX`R@*N!LP:#A"B""X$2AG>7A+8+@`Z"*X`B@BN`,@"C@`8`HX&$1
MH.$(D(G@2(&=Y>*=B>`#@(C@#("(X`'`+.`"P"S@8B&@X0S`BN`"<(?@Z<V,
MX!!P@.5LX8[@"."-Y4SAG>4#,([@!C"#X`$P@^`"$"'@"1`AX&F1H.$!$(C@
M"2`BX.P=@>`,("+@`C"#X`E`A.#A/8/@#$"-Y0`PC>4!4(7@/""?Y010C>7?
M"FWT`B"/X"@PG^6/"D#T`S"2YP`@D^54,9WE`B`SX``PH.,!```:5]^-XO"/
MO>C@\O_K3&$``!P!``!T50``$$`MZ0'*H.,,P$W@^`^,Y6``H./4\O_K!PO?
M[0@KW^T).]_M`""@XP0+P.T8((#E7""`Y=\J0/00@+WH\.'2PP`````!(T5G
MB:O-[_[<NIAV5#(0\$<MZ0'*H.,,P$W@X`^,Y10PD.4"4*#A@B&@X0!@H.$"
M,)/@&""0Y10P@.4!<*#A7`"0Y0$@@B*E+H+@&""&Y0``4.,<H(8")```&C\`
M5>,<``#:0(!%XB53H.$!D$7B!T"@X053A^`$,*#A"B"@X4`0A.(`X)/E$#"#
MX@S`$^40((+B"``3Y01`$^4!`%/A!$`"Y1#@`N4,P`+E"``"Y?/__QH#0*#A
M!@"@X;'\_^L%`%3AZ___&D!PA^()4TC@/X#(XP=PB.`%(*#A!Q"@X0H`H.%Q
M\O_K7%"&Y?"'O>A`0&#B'*"&X@4`5.$``(K@!4"@H00@H.%H\O_K7#"6Y0,P
MA.!<,(;E0`!3X_"'O1@&`*#A!%!%X`1PA^"5_/_KR?__ZG!`+>D!RJ#C`HLM
M[0S`3>#H#XSE`4"@X100D>4`4*#A'&"$XC$_O^88`)3ET1'EYS`OO^88*T/L
M`2"!XG\PX.,X`%+C`@"&X`$PQN=!``#:0"!BX@`0H.,:\O_K!`"@X7O\_^LL
M,(3B4`#`\@\*1O0/"D/T/#"$X@\*0_1,,(3B#P=#]%0PA.($`*#A#X<#]&_\
M_^L#,-3E`##%Y00`H.&R,-3A`3#%Y0`PE.4C-*#A`C#%Y0`PE.4#,,7E!S#4
MY00PQ>6V,-3A!3#%Y00PE.4C-*#A!C#%Y00PE.4',,7E"S#4Y0@PQ>6Z,-3A
M"3#%Y0@PE.4C-*#A"C#%Y0@PE.4+,,7E#S#4Y0PPQ>6^,-3A#3#%Y0PPE.4C
M-*#A#C#%Y0PPE.4/,,7E$S#4Y1`PQ>6R,=3A$3#%Y1`PE.4C-*#A$C#%Y1`P
ME.43,,7E`HN]['!`O>CJ\?_J."!BX@`0H./8\?_KQO__Z@@PG^4#,(_@`#"3
MY3#R_^J,4P``"$`MZ0B`O>@!``(`/@```$$````,````00```!$```!!````
M)P```$$````V````00````,```!!````!@```$$````5````00````\```!!
M````#@```$$````F````00```!,```!!````*0```$$````"````00```!D`
M``!!````"@```$$````+````00``````````````)7,O)7,````O<')O8R]S
M96QF+V5X90``1'EN84QO861E<CHZ8F]O=%]$>6YA3&]A9&5R`$EN=&5R;F%L
M<SHZ4$%2.CI"3T]4`````"]P<F]C+R5I+R5S`&5X90`N````4$%27U1%35``
M````4$%27U!23T=.04U%`````#H````E<R5S)7,``"\```!015),-4Q)0@``
M``!015),3$E"`%!%4DPU3U!4`````%!%4DQ)3P``4$%27TE.251)04Q)6D5$
M`%!!4E]34$%73D5$`%!!4E]#3$5!3@```%!!4E]$14)51P```%!!4E]#04-(
M10```%!!4E]'3$]"04Q?1$5"54<`````4$%27T=,3T)!3%]435!$25(```!0
M05)?5$U01$E2``!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@``
M```Q````;'-E96L@9F%I;&5D`````')E860@9F%I;&5D`$Q$7TQ)0E)!4EE?
M4$%42`!365-414T``"4P,G@`````)7,E<R5S)7,`````<&%R+0`````E<SH@
M8W)E871I;VX@;V8@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE
M<G)N;STE:2D*```E<SH@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS
M869E("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO
M;BD*``!0051(`````'!A<FP`````+G!A<@````!S:&]R="!R96%D```E<R5S
M8V%C:&4M)7,E<P``)7,E<W1E;7`M)74E<P```"5S)7-T96UP+25U+25U)7,`
M````&````'!E<FP`````,````%!!4CHZ4&%C:V5R.CI615)324].`````#$N
M,#8S````56YA8FQE('1O(&-R96%T92!C86-H92!D:7)E8W1O<GD`````)7,Z
M(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T;W)Y("5S(&9A
M:6QE9"`H97)R;F\])6DI"@`````M90``+2T``%5315(`````55-%4DY!344`
M````5$5-4$1)4@!435``+W1M<`````!E=F%L("=E>&5C("]U<W(O8FEN+W!E
M<FP@("U3("0P("1[,2LB)$`B?2<*:68@,#L@(R!N;W0@<G5N;FEN9R!U;F1E
M<B!S;VUE('-H96QL"G!A8VMA9V4@7U]P87)?<&P["FUY("@D4$%27TU!1TE#
M+"`D1DE,15]O9F9S971?<VEZ92P@)&-A8VAE7VYA;65?<VEZ92D["D)%1TE.
M('L*)%!!4E]-04=)0R`](")<;E!!4BYP;5QN(CL*)$9)3$5?;V9F<V5T7W-I
M>F4@/2`T.R`@(",@<&%C:R@B3B(I"B1C86-H95]N86UE7W-I>F4@/2`T,#L*
M)%!+6DE07TU!1TE#(#T@(E!+7#`P,UPP,#0B.PI]"G-U8B!F:6YD7W!A<E]M
M86=I8PI["FUY("@D9F@I(#T@0%\["FUY("1C:'5N:U]S:7IE(#T@-C0@*B`Q
M,#(T.PIM>2`D8G5F.PIM>2`D<VEZ92`]("US("1F:#L*;7D@)'!O<R`]("@D
M<VEZ92TQ*2`M("@D<VEZ92TQ*2`E("1C:'5N:U]S:7IE.R`@("`@(",@3D]4
M13H@)'!O<R!I<R!A(&UU;'1I<&QE(&]F("1C:'5N:U]S:7IE"G=H:6QE("@D
M<&]S(#X](#`I('L*<V5E:R`D9F@L("1P;W,L(#`["G)E860@)&9H+"`D8G5F
M+"`D8VAU;FM?<VEZ92`K(&QE;F=T:"@D4$%27TU!1TE#*3L*:68@*"AM>2`D
M:2`](')I;F1E>"@D8G5F+"`D4$%27TU!1TE#*2D@/CT@,"D@>PIR971U<FX@
M)'!O<R`K("1I.PI]"B1P;W,@+3T@)&-H=6YK7W-I>F4["GT*<F5T=7)N("TQ
M.PI]"FUY("@D<&%R7W1E;7`L("1P<F]G;F%M92P@0'1M<&9I;&4L("5-;V1U
M;&5#86-H92D["D5.1"![(&EF("@D14Y6>U!!4E]#3$5!3GTI('L*<F5Q=6ER
M92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIR97%U:7)E
M($9I;&4Z.E-P96,["FUY("1T;W!D:7(@/2!&:6QE.CI"87-E;F%M93HZ9&ER
M;F%M92@D<&%R7W1E;7`I.PIO=71S*'%Q6U)E;6]V:6YG(&9I;&5S(&EN("(D
M<&%R7W1E;7`B72D["D9I;&4Z.D9I;F0Z.F9I;F1D97!T:"AS=6(@>R`H("UD
M("D@/R!R;61I<B`Z('5N;&EN:R!]+"`D<&%R7W1E;7`I.PIR;61I<B`D<&%R
M7W1E;7`["FEF("@M9"`D<&%R7W1E;7`@)B8@)%Y/(&YE("=-4U=I;C,R)RD@
M>PIM>2`D=&UP(#T@;F5W($9I;&4Z.E1E;7`H"E1%35!,051%(#T^("=T;7!8
M6%A86"<L"D1)4B`]/B!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D=&]P9&ER
M*2P*4U5&1DE8(#T^("<N8VUD)RP*54Y,24Y+(#T^(#`L"BD["FUY("1F:6QE
M;F%M92`]("1T;7`M/F9I;&5N86UE.PIP<FEN="`D=&UP(#P\(BXN+B(["G@]
M,3L@=VAI;&4@6R!<)'@@+6QT(#$P(%T[(&1O"G)M("UR9B`G)'!A<E]T96UP
M)PII9B!;(%PA("UD("<D<&%R7W1E;7`G(%T[('1H96X*8G)E86L*9FD*<VQE
M97`@,0IX/6!E>'!R(%PD>"`K(#%@"F1O;F4*<FT@)R1F:6QE;F%M92<*+BXN
M"F-L;W-E("1T;7`["F-H;6]D(#`W,#`L("1F:6QE;F%M93L*;7D@)&-M9"`]
M("(D9FEL96YA;64@/B]D978O;G5L;"`R/B8Q("8B.PIS>7-T96TH)&-M9"D[
M"F]U=',H<7%;4W!A=VYE9"!B86-K9W)O=6YD('!R;V-E<W,@=&\@<&5R9F]R
M;2!C;&5A;G5P.B`D9FEL96YA;65=*3L*?0I]('T*0D5'24X@>PI);G1E<FYA
M;',Z.E!!4CHZ0D]/5"@I(&EF(&1E9FEN960@)DEN=&5R;F%L<SHZ4$%2.CI"
M3T]4.PIE=F%L('L*7W!A<E]I;FET7V5N=B@I.PIM>2`D<75I970@/2`A)$5.
M5GM005)?1$5"54=].PIM>2`E<WES(#T@*`IP871H7W-E<"`@("`]/B`H)%Y/
M(#U^("]>35-7:6XO(#\@)SLG(#H@)SHG*2P*7V5X92`@("`@("`@/3X@*"1>
M3R`]?B`O7B@_.DU35VEN?$]3,GQC>6=W:6XI+R`_("<N97AE)R`Z("<G*2P*
M7V1E;&EM("`@("`@/3X@*"1>3R`]?B`O7DU35VEN?$]3,B\@/R`G7%PG(#H@
M)R\G*2P**3L*7W-E=%]P<F]G;F%M92@I.PIO=71S*'%Q6UPD<')O9VYA;64@
M/2`B)'!R;V=N86UE(ETI.PI?<V5T7W!A<E]T96UP*"D["F]U=',H<7%;7"1%
M3E9[4$%27U1%35!](#T@(B1%3E9[4$%27U1%35!](ETI.PIM>2`H)'-T87)T
M7W!O<RP@)'-T87)T7V]F7T9)3$5?<V5C=&EO;BD["DU!1TE#.B!["FQO8V%L
M("1324=[7U]705).7U]](#T@<W5B('M].PIU;FQE<W,@*&]P96X@7T9(+"`G
M/#IR87<G+"`D<')O9VYA;64I('L*;W5T<RAQ<5M#86XG="!R96%D(&9R;VT@
M9FEL92`B)'!R;V=N86UE(ETI.R`@(R!D;VXG="!U<V4@)"$@:&5R92!A<R!I
M="!R97%U:7)E<R!%<G)N;RYP;0IL87-T($U!1TE#.PI]"FUY("1M86=I8U]P
M;W,@/2!F:6YD7W!A<E]M86=I8R@J7T9(*3L*:68@*"1M86=I8U]P;W,@/"`P
M*2!["F]U=',H<7%;0V%N)W0@9FEN9"!M86=I8R!S=')I;F<@(B1005)?34%'
M24,B(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0SL*?0IO=71S
M*")&;W5N9"!005(@;6%G:6,@870@<&]S:71I;VX@)&UA9VEC7W!O<R(I.PIS
M965K(%]&2"P@)&UA9VEC7W!O<R`M("1&24Q%7V]F9G-E=%]S:7IE("T@;&5N
M9W1H*")<,$-!0TA%(BDL(#`["G)E860@7T9(+"`D8G5F+"!L96YG=&@H(EPP
M0T%#2$4B*3L*:68@*"1B=68@;F4@(EPP0T%#2$4B("8F("1B=68@;F4@(EPP
M0TQ%04XB*2!["F]U=',H(DYO(&-A8VAE(&UA<FME<B!F;W5N9"(I.PIL87-T
M($U!1TE#.PI]"F]U=',H<7%;0V%C:&4@;6%R:V5R("(D8G5F(B!F;W5N9%TI
M.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M("1&24Q%7V]F9G-E=%]S:7IE+"`P
M.PIR96%D(%]&2"P@)&)U9BP@)$9)3$5?;V9F<V5T7W-I>F4["FUY("1O9F9S
M970@/2!U;G!A8VLH(DXB+"`D8G5F*3L*;W5T<R@B3V9F<V5T(&9R;VT@<W1A
M<G0@;V8@1DE,17,@:7,@)&]F9G-E="(I.PIS965K(%]&2"P@)&UA9VEC7W!O
M<R`M("1&24Q%7V]F9G-E=%]S:7IE("T@)&]F9G-E="P@,#L*)'-T87)T7V]F
M7T9)3$5?<V5C=&EO;B`]('1E;&P@7T9(.PIM>2`E<F5Q=6ER95]L:7-T.PIR
M96%D(%]&2"P@)&)U9BP@-#L@("`@("`@("`@("`@("`@("`@("`@("`@("`C
M(')E860@=&AE(&9I<G-T(")&24Q%(@IW:&EL92`H)&)U9B!E<2`B1DE,12(I
M('L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH
M(DXB+"`D8G5F*3L*;7D@)&9U;&QN86UE(#T@)&)U9CL*;W5T<RAQ<5M5;G!A
M8VMI;F<@1DE,12`B)&9U;&QN86UE(BXN+ETI.PIM>2`D8W)C(#T@*"`D9G5L
M;&YA;64@/7X@<WQ>*%MA+69<9%U[.'TI+WQ\("D@/R`D,2`Z('5N9&5F.PIM
M>2`H)&)A<V5N86UE+"`D97AT*2`]("@D8G5F(#U^(&U\*#\Z+BHO*3\H+BHI
M*%PN+BHI?"D["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U9BP@
M=6YP86-K*").(BP@)&)U9BD["FEF("AD969I;F5D*"1E>'0I(&%N9"`D97AT
M("%^("]<+B@_.G!M?'!L?&EX?&%L*20O:2D@>PIM>2`D9FEL96YA;64@/2!?
M<V%V95]A<R@B)&-R8R1E>'0B+"`D8G5F+"`P-S4U*3L*)%!!4CHZ2&5A=GDZ
M.D9U;&Q#86-H97LD9G5L;&YA;65](#T@)&9I;&5N86UE.PHD4$%2.CI(96%V
M>3HZ1G5L;$-A8VAE>R1F:6QE;F%M97T@/2`D9G5L;&YA;64["GT*96QS92![
M"B1-;V1U;&5#86-H97LD9G5L;&YA;65](#T@>PIB=68@/3X@)&)U9BP*8W)C
M(#T^("1C<F,L"FYA;64@/3X@)&9U;&QN86UE+`I].PHD<F5Q=6ER95]L:7-T
M>R1F=6QL;F%M97TK*SL*?0IR96%D(%]&2"P@)&)U9BP@-#L*?0IL;V-A;"!`
M24Y#(#T@*'-U8B!["FUY("@D<V5L9BP@)&UO9'5L92D@/2!`7SL*<F5T=7)N
M(&EF(')E9B`D;6]D=6QE(&]R("$D;6]D=6QE.PIM>2`D:6YF;R`]("1-;V1U
M;&5#86-H97LD;6]D=6QE?3L*<F5T=7)N('5N;&5S<R`D:6YF;SL*9&5L971E
M("1R97%U:7)E7VQI<W1[)&UO9'5L97T["B1)3D-[)&UO9'5L97T@/2`B+VQO
M861E<B\D:6YF;R\D;6]D=6QE(CL*:68@*"1%3E9[4$%27T-,14%.?2!A;F0@
M9&5F:6YE9"@F24\Z.D9I;&4Z.FYE=RDI('L*;7D@)&9H(#T@24\Z.D9I;&4M
M/FYE=U]T;7!F:6QE(&]R(&1I92`B0V%N)W0@8W)E871E('1E;7`@9FEL93H@
M)"$B.PHD9F@M/F)I;FUO9&4H*3L*)&9H+3YP<FEN="@D:6YF;RT^>V)U9GTI
M.PHD9F@M/G-E96LH,"P@,"D["G)E='5R;B`D9F@["GT*96QS92!["FUY("1F
M:6QE;F%M92`](%]S879E7V%S*"(D:6YF;RT^>V-R8WTN<&TB+"`D:6YF;RT^
M>V)U9GTI.PHD:6YF;RT^>V9I;&5](#T@)&9I;&5N86UE.PIO<&5N(&UY("1F
M:"P@)SPZ<F%W)RP@)&9I;&5N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D
M9FEL96YA;64B.B`D(5T["G)E='5R;B`D9F@["GT*9&EE(")";V]T<W1R87!P
M:6YG(&9A:6QE9#H@8V%N)W0@9FEN9"!M;V1U;&4@)&UO9'5L92$B.PI]+"!`
M24Y#*3L*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!005(Z.DAE879Y.PIR
M97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*
M4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PIR97%U:7)E($E/.CI&
M:6QE.PIW:&EL92`H;7D@)&9I;&5N86UE(#T@*'-O<G0@:V5Y<R`E<F5Q=6ER
M95]L:7-T*5LP72D@>PIU;FQE<W,@*"1)3D-[)&9I;&5N86UE?2!O<B`D9FEL
M96YA;64@/7X@+T)31%!!3B\I('L*:68@*"1F:6QE;F%M92`]?B`O7"YP;6,_
M)"]I*2!["G)E<75I<F4@)&9I;&5N86UE.PI]"F5L<V4@>PID;R`D9FEL96YA
M;64@=6YL97-S("1F:6QE;F%M92`]?B`O<VET96-U<W1O;6EZ95PN<&PD+SL*
M?0I]"F1E;&5T92`D<F5Q=6ER95]L:7-T>R1F:6QE;F%M97T["GT*:68@*"1B
M=68@;F4@)%!+6DE07TU!1TE#*2!["F]U=',H<7%;3F\@>FEP(&9O=6YD(&%F
M=&5R($9)3$4@<V5C=&EO;B!I;B!F:6QE("(D<')O9VYA;64B72D["FQA<W0@
M34%'24,@.PI]"B1S=&%R=%]P;W,@/2`H=&5L;"!?1D@I("T@-#L@("`@("`@
M("`@("`@("`@(R!S=&%R="!O9B!Z:7`*?0IM>2!`<&%R7V%R9W,["FUY("@D
M;W5T+"`D8G5N9&QE+"`D;&]G9F@L("1C86-H95]N86UE*3L*9&5L971E("1%
M3E9[4$%27T%04%]2155317T[(",@<V%N:71I>F4@*%)%55-%(&UA>2!B92!A
M('-E8W5R:71Y('!R;V)L96TI"FEF("@A)'-T87)T7W!O<R!O<B`H)$%21U9;
M,%T@97$@)RTM<&%R+6]P=&EO;G,G("8F('-H:69T*2D@>PIM>2`E9&ES=%]C
M;60@/2!Q=R@*<"`@(&)L:6)?=&]?<&%R"FD@("!I;G-T86QL7W!A<@IU("`@
M=6YI;G-T86QL7W!A<@IS("`@<VEG;E]P87(*=B`@('9E<FEF>5]P87(**3L*
M:68@*$!!4D=6(&%N9"`D05)'5ELP72!E<2`G+2UR975S92<I('L*<VAI9G0@
M0$%21U8["B1%3E9[4$%27T%04%]2155317T@/2!S:&EF="!`05)'5CL*?0IE
M;'-E('L@(R!N;W)M86P@<&%R;"!B96AA=FEO=7(*;7D@0&%D9%]T;U]I;F,[
M"G=H:6QE("A`05)'5BD@>PHD05)'5ELP72`]?B`O7BTH6T%)34]"3&)Q<&EU
M<U1V72DH+BHI+R!O<B!L87-T.PII9B`H)#$@97$@)TDG*2!["G!U<V@@0&%D
M9%]T;U]I;F,L("0R.PI]"F5L<VEF("@D,2!E<2`G32<I('L*979A;"`B=7-E
M("0R(CL*?0IE;'-I9B`H)#$@97$@)T$G*2!["G5N<VAI9G0@0'!A<E]A<F=S
M+"`D,CL*?0IE;'-I9B`H)#$@97$@)T\G*2!["B1O=70@/2`D,CL*?0IE;'-I
M9B`H)#$@97$@)V(G*2!["B1B=6YD;&4@/2`G<VET92<["GT*96QS:68@*"0Q
M(&5Q("=")RD@>PHD8G5N9&QE(#T@)V%L;"<["GT*96QS:68@*"0Q(&5Q("=Q
M)RD@>PHD<75I970@/2`Q.PI]"F5L<VEF("@D,2!E<2`G3"<I('L*;W!E;B`D
M;&]G9F@L("(^/B(L("0R(&]R(&1I92!Q<5M#86XG="!O<&5N(&QO9R!F:6QE
M("(D,B(Z("0A73L*?0IE;'-I9B`H)#$@97$@)U0G*2!["B1C86-H95]N86UE
M(#T@)#(["GT*<VAI9G0H0$%21U8I.PII9B`H;7D@)&-M9"`]("1D:7-T7V-M
M9'LD,7TI('L*9&5L971E("1%3E9[)U!!4E]414U0)WT["FEN:71?:6YC*"D[
M"G)E<75I<F4@4$%2.CI$:7-T.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@I('5N
M;&5S<R!`05)'5CL*)GLB4$%2.CI$:7-T.CHD8VUD(GTH)%\I(&9O<B!`05)'
M5CL*97AI=#L*?0I]"G5N<VAI9G0@0$E.0RP@0&%D9%]T;U]I;F,["GT*?0II
M9B`H)&]U="D@>PI["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@07)C:&EV
M93HZ6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!.PI]"FUY("1P87(@/2!S:&EF
M="A`05)'5BD["FUY("1Z:7`["FEF("AD969I;F5D("1P87(I('L*;W!E;B!M
M>2`D9F@L("<\.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T(&9I;F0@<&%R
M(&9I;&4@(B1P87(B.B`D(5T["F)L97-S*"1F:"P@)TE/.CI&:6QE)RD["B1Z
M:7`@/2!!<F-H:79E.CI::7`M/FYE=SL**"`D>FEP+3YR96%D1G)O;49I;&5(
M86YD;&4H)&9H+"`D<&%R*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0IO
M<B!D:64@<7%;17)R;W(@<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!A<B)=.PI]
M"FUY("5M971A7W!A<B`](&1O('L*:68@*"1Z:7`@86YD(&UY("1M971A(#T@
M)'II<"T^8V]N=&5N=',H)TU%5$$N>6UL)RDI('L*)&UE=&$@/7X@<R\N*EYP
M87(Z)"\O;7,["B1M971A(#U^(',O7EQ3+BHO+VUS.PHD;65T82`]?B`O7B`@
M*%M>.ETK*3H@*"XK*20O;6<["GT*?3L*:68@*&1E9FEN960@)'!A<BD@>PIO
M<&5N(&UY("1P:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@<F5A
M9"!P87(@9FEL92`B)'!A<B(Z("0A73L*;7D@)&)U9CL*<F5A9"`D<&@L("1B
M=68L(#0["F1I92!Q<5LB)'!A<B(@:7,@;F]T(&$@<&%R(&9I;&5=('5N;&5S
M<R`D8G5F(&5Q("102UI)4%]-04=)0SL*8VQO<V4@)'!H.PI]"D-R96%T95!A
M=&@H)&]U="D@.PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W*`HD;W5T+`I)3SHZ
M1FEL93HZ3U]#4D5!5"@I('P@24\Z.D9I;&4Z.D]?4D174B@I('P@24\Z.D9I
M;&4Z.D]?5%)53D,H*2P*,#<W-RP**2!O<B!D:64@<7%;0V%N)W0@8W)E871E
M(&9I;&4@(B1O=70B.B`D(5T["B1F:"T^8FEN;6]D92@I.PIS965K(%]&2"P@
M,"P@,#L*;7D@)&QO861E<CL*:68@*&1E9FEN960@)'-T87)T7V]F7T9)3$5?
M<V5C=&EO;BD@>PIR96%D(%]&2"P@)&QO861E<BP@)'-T87)T7V]F7T9)3$5?
M<V5C=&EO;CL*?2!E;'-E('L*;&]C86P@)"\@/2!U;F1E9CL*)&QO861E<B`]
M(#Q?1D@^.PI]"FEF("@A)$5.5GM005)?5D520D%424U](&%N9"`D;&]A9&5R
M(#U^("]>*#\Z(R%\7$!R96TI+RD@>PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0
M;V13=')I<#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&QO861E
M<BP@)#`I.PI]"B1F:"T^<')I;G0H)&QO861E<BD*;W(@9&EE('%Q6T5R<F]R
M('=R:71I;F<@;&]A9&5R('1O("(D;W5T(CH@)"%=.PII9B`H)&)U;F1L92D@
M>PIR97%U:7)E(%!!4CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L
M;V%D97(H*3L*:6YI=%]I;F,H*3L*<F5Q=6ER95]M;V1U;&5S*"D["FUY($!I
M;F,@/2!G<F5P('L@(2]"4T1004XO('T*9W)E<"!["B@D8G5N9&QE(&YE("=S
M:71E)RD@;W(**"1?(&YE("1#;VYF:6<Z.D-O;F9I9WMA<F-H;&EB97AP?2!A
M;F0*)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>W!R:79L:6)E>'!]*3L*?2!`24Y#
M.PIS?"\K?"]\9RP@<WPO)'Q\(&9O<F5A8V@@0&EN8SL*;7D@)69I;&5S.PHD
M9FEL97-[)%]]*RL@9F]R($!$>6YA3&]A9&5R.CID;%]S:&%R961?;V)J96-T
M<RP@=F%L=65S("5)3D,["FUY("1L:6)?97AT(#T@)$-O;F9I9SHZ0V]N9FEG
M>VQI8E]E>'1].R`@("`@("`@(",@6%A8(&QI8E]E>'0@=G,@9&QE>'0@/PIM
M>2`E=W)I='1E;CL*9F]R96%C:"!M>2`D:V5Y("AS;W)T(&ME>7,@)69I;&5S
M*2!["FUY("@D9FEL92P@)&YA;64I.PII9B`H9&5F:6YE9"AM>2`D9F,@/2`D
M4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1K97E]*2D@>PHH)&9I;&4L("1N86UE
M*2`]("@D:V5Y+"`D9F,I.PI]"F5L<V4@>PIF;W)E86-H(&UY("1D:7(@*$!I
M;F,I('L*:68@*"1K97D@/7X@;7Q>7%$D9&ER7$4O*"XJ*21\:2D@>PHH)&9I
M;&4L("1N86UE*2`]("@D:V5Y+"`D,2D["FQA<W0["GT*:68@*"1K97D@/7X@
M;7Q>+VQO861E<B];7B]=*R\H+BHI)'PI('L*:68@*&UY("1R968@/2`D36]D
M=6QE0V%C:&5[)#%]*2!["B@D9FEL92P@)&YA;64I(#T@*"1R968L("0Q*3L*
M;&%S=#L*?0II9B`H+68@(B1D:7(O)#$B*2!["B@D9FEL92P@)&YA;64I(#T@
M*"(D9&ER+R0Q(BP@)#$I.PIL87-T.PI]"GT*?0I]"FYE>'0@=6YL97-S(&1E
M9FEN960@)&YA;64["FYE>'0@:68@)'=R:71T96Y[)&YA;65]*RL["FYE>'0@
M:68@(7)E9B@D9FEL92D@86YD("1F:6QE(#U^("]<+EQ1)&QI8E]E>'1<120O
M:3L*;W5T<RAS<')I;G1F*'%Q6U!A8VMI;F<@1DE,12`B)7,B+BXN72P@<F5F
M("1F:6QE(#\@)&9I;&4M/GMN86UE?2`Z("1F:6QE*2D["FUY("1C;VYT96YT
M.PII9B`H<F5F*"1F:6QE*2D@>PHD8V]N=&5N="`]("1F:6QE+3Y[8G5F?3L*
M?0IE;'-E('L*;&]C86P@)"\@/2!U;F1E9CL*;W!E;B!M>2`D=&@L("<\.G)A
M=R<L("1F:6QE(&]R(&1I92!Q<5M#86XG="!R96%D("(D9FEL92(Z("0A73L*
M)&-O;G1E;G0@/2`\)'1H/CL*8VQO<V4@)'1H.PI005(Z.D9I;'1E<CHZ4&]D
M4W1R:7`M/F%P<&QY*%PD8V]N=&5N="P@(CQE;6)E9&1E9#XO)&YA;64B*0II
M9B`A)$5.5GM005)?5D520D%424U](&%N9"`D;F%M92`]?B`O7"XH/SIP;7QI
M>'QA;"DD+VD["E!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0M/FYE=RT^87!P
M;'DH7"1C;VYT96YT+"`D9FEL92P@)&YA;64I.PI]"B1F:"T^<')I;G0H(D9)
M3$4B+`IP86-K*"=.)RP@;&5N9W1H*"1N86UE*2`K(#DI+`IS<')I;G1F*"(E
M,#AX+R5S(BP@07)C:&EV93HZ6FEP.CIC;VUP=71E0U)#,S(H)&-O;G1E;G0I
M+"`D;F%M92DL"G!A8VLH)TXG+"!L96YG=&@H)&-O;G1E;G0I*2P*)&-O;G1E
M;G0I"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG(&5M8F5D9&5D($9)3$4@=&\@
M(B1O=70B.B`D(5T["F]U=',H<7%;5W)I='1E;B!A<R`B)&YA;64B72D["GT*
M?0II9B`H)'II<"D@>PHD>FEP+3YW<FET951O1FEL94AA;F1L92@D9F@I(#T]
M($%R8VAI=F4Z.EII<#HZ05I?3TL*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@
M>FEP('!A<G0@;V8@(B1O=70B73L*?0II9B`H)&UE=&%?<&%R>V-L96%N?2D@
M>PHD9F@M/G!R:6YT*")<,$-,14%.(BD["GT*96QS92!["FEF("@A9&5F:6YE
M9"`D8V%C:&5?;F%M92D@>PIM>2`D8W1X(#T@1&EG97-T.CI32$$M/FYE=R@Q
M*3L*<V5E:R`D9F@L(#`L(#`["B1C='@M/F%D9&9I;&4H)&9H*3L*<V5E:R`D
M9F@L(#`L(#(["B1C86-H95]N86UE(#T@)&-T>"T^:&5X9&EG97-T.PI]"B1C
M86-H95]N86UE(#T@<W5B<W1R("1C86-H95]N86UE+"`P+"`D8V%C:&5?;F%M
M95]S:7IE.PHD8V%C:&5?;F%M92`N/2`B7#`B('@@*"1C86-H95]N86UE7W-I
M>F4@+2!L96YG=&@@)&-A8VAE7VYA;64I.PHD9F@M/G!R:6YT*"1C86-H95]N
M86UE+`HB7#!#04-(12(I.PI]"FUY("1O9F9S970@/2`D9F@M/G1E;&P@+2!L
M96YG=&@H)&QO861E<BD["B1F:"T^<')I;G0H<&%C:R@G3B<L("1O9F9S970I
M+`HD4$%27TU!1TE#*3L*)&9H+3YC;&]S92!O<B!D:64@<7%;17)R;W(@=W)I
M=&EN9R!T<F%I;&5R(&]F("(D;W5T(CH@)"%=.PIC:&UO9"`P-S4U+"`D;W5T
M.PIE>&ET.PI]"GL*;&%S="!U;FQE<W,@9&5F:6YE9"`D<W1A<G1?<&]S.PI?
M9FEX7W!R;V=N86UE*"D["G)E<75I<F4@4$%2.PI005(Z.DAE879Y.CI?:6YI
M=%]D>6YA;&]A9&5R*"D["GL*<F5Q=6ER92!&:6QE.CI&:6YD.PIR97%U:7)E
M($%R8VAI=F4Z.EII<#L*?0IM>2`D9F@@/2!)3SHZ1FEL92T^;F5W.R`@("`@
M("`@("`@("`@("`@("`@("`@("`@("`@(R!!<F-H:79E.CI::7`@;W!E<F%T
M97,@;VX@86X@24\Z.DAA;F1L90HD9F@M/F9D;W!E;BAF:6QE;F\H7T9(*2P@
M)W(G*2!O<B!D:64@<7%;9F1O<&5N*"D@9F%I;&5D.B`D(5T["D%R8VAI=F4Z
M.EII<#HZ<V5T0VAU;FM3:7IE*"US(%]&2"D["FUY("1Z:7`@/2!!<F-H:79E
M.CI::7`M/FYE=SL**"1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P
M<F]G;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;
M17)R;W(@<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!R;V=N86UE(ET["D%R8VAI
M=F4Z.EII<#HZ<V5T0VAU;FM3:7IE*#8T("H@,3`R-"D["G!U<V@@0%!!4CHZ
M3&EB0V%C:&4L("1Z:7`["B1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97T@/2`D
M>FEP.PIO=71S*")%>'1R86-T:6YG('II<"XN+B(I.PII9B`H9&5F:6YE9"`D
M14Y6>U!!4E]414U0?2D@>R`C('-H;W5L9"!B92!S970@870@=&AI<R!P;VEN
M="$*9F]R96%C:"!M>2`D;65M8F5R("@@)'II<"T^;65M8F5R<R`I('L*;F5X
M="!I9B`D;65M8F5R+3YI<T1I<F5C=&]R>3L*;7D@)&UE;6)E<E]N86UE(#T@
M)&UE;6)E<BT^9FEL94YA;64["FYE>'0@=6YL97-S("1M96UB97)?;F%M92`]
M?B!M>PI>"B\_<VAL:6(O"B@_.B1#;VYF:6<Z.D-O;F9I9WMA<F-H;F%M97TO
M*3\**%M>+UTK*0HD"GUX.PIM>2`D97AT<F%C=%]N86UE(#T@)#$["FUY("1D
M97-T7VYA;64@/2!&:6QE.CI3<&5C+3YC871F:6QE*"1%3E9[4$%27U1%35!]
M+"`D97AT<F%C=%]N86UE*3L*:68@*"UF("1D97-T7VYA;64@)B8@+7,@7R`]
M/2`D;65M8F5R+3YU;F-O;7!R97-S9613:7IE*"DI('L*;W5T<RAQ<5M3:VEP
M<&EN9R`B)&UE;6)E<E]N86UE(B!S:6YC92!I="!A;')E861Y(&5X:7-T<R!A
M="`B)&1E<W1?;F%M92)=*3L*?2!E;'-E('L*;W5T<RAQ<5M%>'1R86-T:6YG
M("(D;65M8F5R7VYA;64B('1O("(D9&5S=%]N86UE(BXN+ETI.PHH)&UE;6)E
M<BT^97AT<F%C=%1O1FEL94YA;65D*"1D97-T7VYA;64I(#T]($%R8VAI=F4Z
M.EII<#HZ05I?3TLH*2D*;W(@9&EE('%Q6T5R<F]R(&5X=')A8W1I;F<@>FEP
M(&UE;6)E<B!T;R`B)&1E<W1?;F%M92)=.PIC:&UO9"@P-34U+"`D9&5S=%]N
M86UE*2!I9B`D7D\@97$@(FAP=7@B.PI]"GT*?0I]"G5N;&5S<R`H)%!!4CHZ
M3&EB0V%C:&5[)'!R;V=N86UE?2D@>PID:64@/#P@(BXB('5N;&5S<R!`05)'
M5CL*57-A9V4Z("0P(%L@+4%L:6(N<&%R(%T@6R`M261I<B!=(%L@+4UM;V1U
M;&4@72!;('-R8RYP87(@72!;('!R;V=R86TN<&P@70HD,"!;("U"?"UB(%T@
M6RU/;W5T9FEL95T@<W)C+G!A<@HN"B1%3E9[4$%27U!23T=.04U%?2`]("1P
M<F]G;F%M92`]("0P(#T@<VAI9G0H0$%21U8I.PI]"G-U8B!#<F5A=&50871H
M('L*;7D@*"1N86UE*2`]($!?.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIM
M>2`H)&)A<V5N86UE+"`D<&%T:"P@)&5X="D@/2!&:6QE.CI"87-E;F%M93HZ
M9FEL97!A<G-E*"1N86UE+"`H)UPN+BHG*2D["G)E<75I<F4@1FEL93HZ4&%T
M:#L*1FEL93HZ4&%T:#HZ;6MP871H*"1P871H*2!U;FQE<W,H+64@)'!A=&@I
M.R`C(&UK<&%T:"!D:65S('=I=&@@97)R;W(*?0IS=6(@<F5Q=6ER95]M;V1U
M;&5S('L*<F5Q=6ER92!L:6(["G)E<75I<F4@1'EN84QO861E<CL*<F5Q=6ER
M92!I;G1E9V5R.PIR97%U:7)E('-T<FEC=#L*<F5Q=6ER92!W87)N:6YG<SL*
M<F5Q=6ER92!V87)S.PIR97%U:7)E($-A<G`["G)E<75I<F4@0V%R<#HZ2&5A
M=GD["G)E<75I<F4@17)R;F\["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PIR
M97%U:7)E($5X<&]R=&5R.PIR97%U:7)E($9C;G1L.PIR97%U:7)E($9I;&4Z
M.E1E;7`["G)E<75I<F4@1FEL93HZ4W!E8SL*<F5Q=6ER92!84TQO861E<CL*
M<F5Q=6ER92!#;VYF:6<["G)E<75I<F4@24\Z.DAA;F1L93L*<F5Q=6ER92!)
M3SHZ1FEL93L*<F5Q=6ER92!#;VUP<F5S<SHZ6FQI8CL*<F5Q=6ER92!!<F-H
M:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["G)E<75I<F4@4$%2.PIR
M97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@4$%2.CI$:7-T.PIR97%U:7)E
M(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ
M4&%T8VA#;VYT96YT.PIR97%U:7)E(&%T=')I8G5T97,["F5V86P@>R!R97%U
M:7)E($-W9"!].PIE=F%L('L@<F5Q=6ER92!7:6XS,B!].PIE=F%L('L@<F5Q
M=6ER92!38V%L87(Z.E5T:6P@?3L*979A;"![(')E<75I<F4@07)C:&EV93HZ
M56YZ:7`Z.D)U<G-T('T["F5V86P@>R!R97%U:7)E(%1I93HZ2&%S:#HZ3F%M
M961#87!T=7)E('T["F5V86P@>R!R97%U:7)E(%!E<FQ)3SL@<F5Q=6ER92!0
M97)L24\Z.G-C86QA<B!].PIE=F%L('L@<F5Q=6ER92!U=&8X('T["GT*<W5B
M(%]S971?<&%R7W1E;7`@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2!A
M;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO*2!["B1P87)?=&5M<"`]("0Q
M.PIR971U<FX["GT*9F]R96%C:"!M>2`D<&%T:"`H"BAM87`@)$5.5GLD7WTL
M('%W*"!005)?5$U01$E2(%1-4$1)4B!414U01$E2(%1%35`@5$U0("DI+`IQ
M=R@@0SI<7%1%35`@+W1M<"`N("D**2!["FYE>'0@=6YL97-S(&1E9FEN960@
M)'!A=&@@86YD("UD("1P871H(&%N9"`M=R`D<&%T:#L*;7D@)'5S97)N86UE
M.PIM>2`D<'=U:60["F5V86P@>R@D<'=U:60I(#T@9V5T<'=U:60H)#XI(&EF
M(&1E9FEN960@)#X[?3L*:68@*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.86UE
M*2`I('L*)'5S97)N86UE(#T@)E=I;C,R.CI,;V=I;DYA;64["GT*96QS:68@
M*&1E9FEN960@)'!W=6ED*2!["B1U<V5R;F%M92`]("1P=W5I9#L*?0IE;'-E
M('L*)'5S97)N86UE(#T@)$5.5GM54T523D%-17T@?'P@)$5.5GM54T52?2!\
M?"`G4UE35$5-)SL*?0IM>2`D<W1M<&1I<B`]("(D<&%T:"1S>7-[7V1E;&EM
M?7!A<BTB+G5N<&%C:R@B2"HB+"`D=7-E<FYA;64I.PIM:V1I<B`D<W1M<&1I
M<BP@,#<U-3L*;7D@)&-A8VAE7V1I<CL*:68@*"1%3E9[4$%27T-,14%.?2D@
M>PHD8V%C:&5?9&ER(#T@(G1E;7`M)"0B.PI]"F5L<V4@>PIO<&5N(&UY("1F
M:"P@(CPZ<F%W(BP@)'!R;V=N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D
M<')O9VYA;64B.B`D(5T["FEF("@H;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R
M7VUA9VEC*"1F:"DI(#X](#`I('L*<V5E:R`D9F@L("1M86=I8U]P;W,*+2`D
M1DE,15]O9F9S971?<VEZ90HM(&QE;F=T:"@B7#!#04-(12(I+"`P.PIM>2`D
M8G5F.PIR96%D("1F:"P@)&)U9BP@;&5N9W1H*")<,$-!0TA%(BD["FEF("@D
M8G5F(&5Q(")<,$-!0TA%(BD@>PIS965K("1F:"P@)&UA9VEC7W!O<PHM("1&
M24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*")<,$-!0TA%(BD*+2`D8V%C:&5?
M;F%M95]S:7IE+"`P.PIR96%D("1F:"P@)&)U9BP@)&-A8VAE7VYA;65?<VEZ
M93L*)&)U9B`]?B!S+UPP+R]G.PHD8V%C:&5?9&ER(#T@(F-A8VAE+21B=68B
M.PI]"GT*8VQO<V4@)&9H.PI]"FEF("@A)&-A8VAE7V1I<BD@>PHD8V%C:&5?
M9&ER(#T@(G1E;7`M)"0B.PHD14Y6>U!!4E]#3$5!3GT@/2`Q.PI]"B1S=&UP
M9&ER("X]("(D<WES>U]D96QI;7TD8V%C:&5?9&ER(CL*;6MD:7(@)'-T;7!D
M:7(L(#`W-34["B1%3E9[4$%27U1%35!](#T@)'-T;7!D:7(["FQA<W0["GT*
M)'!A<E]T96UP(#T@)#$@:68@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%2
M7U1%35!](#U^("\H+BLI+SL*?0IS=6(@7W-A=F5?87,@>PIM>2`H)&YA;64L
M("1C;VYT96YT<RP@)&UO9&4I(#T@0%\["FUY("1F=6QL;F%M92`]("(D<&%R
M7W1E;7`O)&YA;64B.PIU;FQE<W,@*"UE("1F=6QL;F%M92D@>PIM>2`D=&5M
M<&YA;64@/2`B)&9U;&QN86UE+B0D(CL*;7D@)&9H.PHH;W!E;B`D9F@L("<^
M.G)A=R<L("1T96UP;F%M92D*)B8@*'!R:6YT("1F:"`D8V]N=&5N=',I"B8F
M("AC;&]S92`D9F@I"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG("(D=&5M<&YA
M;64B.B`D(5T["F-H;6]D("1M;V1E+"`D=&5M<&YA;64@:68@9&5F:6YE9"`D
M;6]D93L*<F5N86UE*"1T96UP;F%M92P@)&9U;&QN86UE*2!O<B!U;FQI;FLH
M)'1E;7!N86UE*3L*?0IR971U<FX@)&9U;&QN86UE.PI]"G-U8B!?<V5T7W!R
M;V=N86UE('L*:68@*&1E9FEN960@)$5.5GM005)?4%)/1TY!345](&%N9"`D
M14Y6>U!!4E]04D]'3D%-17T@/7X@+R@N*RDO*2!["B1P<F]G;F%M92`]("0Q
M.PI]"B1P<F]G;F%M92!\?#T@)#`["FEF("@D14Y6>U!!4E]414U0?2!A;F0@
M:6YD97@H)'!R;V=N86UE+"`D14Y6>U!!4E]414U0?2D@/CT@,"D@>PHD<')O
M9VYA;64@/2!S=6)S='(H)'!R;V=N86UE+"!R:6YD97@H)'!R;V=N86UE+"`D
M<WES>U]D96QI;7TI("L@,2D["GT*:68@*"$D14Y6>U!!4E]04D]'3D%-17T@
M;W(@:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#X](#`I('L*:68@
M*&]P96X@;7D@)&9H+"`G/"<L("1P<F]G;F%M92D@>PIR971U<FX@:68@+7,@
M)&9H.PI]"FEF("@M<R`B)'!R;V=N86UE)'-Y<WM?97AE?2(I('L*)'!R;V=N
M86UE("X]("1S>7-[7V5X97T["G)E='5R;CL*?0I]"F9O<F5A8V@@;7D@)&1I
M<B`H<W!L:70@+UQ1)'-Y<WMP871H7W-E<'U<12\L("1%3E9[4$%42'TI('L*
M;F5X="!I9B!E>&ES=',@)$5.5GM005)?5$5-4'T@86YD("1D:7(@97$@)$5.
M5GM005)?5$5-4'T["B1D:7(@/7X@<R]<421S>7-[7V1E;&EM?5Q%)"\O.PHH
M*"1P<F]G;F%M92`]("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?
M97AE?2(I+"!L87-T*0II9B`M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M
M921S>7-[7V5X97TB.PHH*"1P<F]G;F%M92`]("(D9&ER)'-Y<WM?9&5L:6U]
M)'!R;V=N86UE(BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R
M;V=N86UE(CL*?0I]"G-U8B!?9FEX7W!R;V=N86UE('L*)#`@/2`D<')O9VYA
M;64@?'P]("1%3E9[4$%27U!23T=.04U%?3L*:68@*&EN9&5X*"1P<F]G;F%M
M92P@)'-Y<WM?9&5L:6U]*2`\(#`I('L*)'!R;V=N86UE(#T@(BXD<WES>U]D
M96QI;7TD<')O9VYA;64B.PI]"FUY("1P=V0@/2`H9&5F:6YE9"`F0W=D.CIG
M971C=V0I(#\@0W=D.CIG971C=V0H*0HZ("@H9&5F:6YE9"`F5VEN,S(Z.D=E
M=$-W9"D@/R!7:6XS,CHZ1V5T0W=D*"D@.B!@<'=D8"D["F-H;VUP*"1P=V0I
M.PHD<')O9VYA;64@/7X@<R]>*#\]7"Y<+C]<421S>7-[7V1E;&EM?5Q%*2\D
M<'=D)'-Y<WM?9&5L:6U]+SL*)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N
M86UE.PI]"G-U8B!?<&%R7VEN:71?96YV('L*:68@*"`D14Y6>U!!4E])3DE4
M24%,25I%1'TK*R`]/2`Q("D@>PIR971U<FX["GT@96QS92!["B1%3E9[4$%2
M7TE.251)04Q)6D5$?2`](#(["GT*9F]R("AQ=R@@4U!!5TY%1"!414U0($-,
M14%.($1%0E5'($-!0TA%(%!23T=.04U%("D@*2!["F1E;&5T92`D14Y6>R)0
M05)?)%\B?3L*?0IF;W(@*'%W+R!435!$25(@5$5-4"!#3$5!3B!$14)51R`O
M*2!["B1%3E9[(E!!4E\D7R)](#T@)$5.5GLB4$%27T=,3T)!3%\D7R)](&EF
M(&5X:7-T<R`D14Y6>R)005)?1TQ/0D%,7R1?(GT["GT*:68@*"1%3E9[4$%2
M7U1%35!]*2!["F1E;&5T92`D14Y6>U!!4E]#3$5!3GT["GT*?0IS=6(@;W5T
M<R!["G)E='5R;B!I9B`D<75I970["FEF("@D;&]G9F@I('L*<')I;G0@)&QO
M9V9H(")`7UQN(CL*?0IE;'-E('L*<')I;G0@(D!?7&XB.PI]"GT*<W5B(&EN
M:71?:6YC('L*<F5Q=6ER92!#;VYF:6<["G!U<V@@0$E.0RP@9W)E<"!D969I
M;F5D+"!M87`@)$-O;F9I9SHZ0V]N9FEG>R1??2P@<7<H"F%R8VAL:6)E>'`@
M<')I=FQI8F5X<"!S:71E87)C:&5X<"!S:71E;&EB97AP"G9E;F1O<F%R8VAE
M>'`@=F5N9&]R;&EB97AP"BD["GT*<&%C:V%G92!M86EN.PIR97%U:7)E(%!!
M4CL*=6YS:&EF="!`24Y#+"!<)E!!4CHZ9FEN9%]P87(["E!!4BT^:6UP;W)T
M*$!P87)?87)G<RD["F1I92!Q<5MP87(N<&PZ($-A;B=T(&]P96X@<&5R;"!S
M8W)I<'0@(B1P<F]G;F%M92(Z($YO('-U8V@@9FEL92!O<B!D:7)E8W1O<GE<
M;ET*=6YL97-S("UE("1P<F]G;F%M93L*9&\@)'!R;V=N86UE.PI#3U)%.CIE
M>&ET*"0Q*2!I9B`H)$`@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@
M)$`@:68@)$`["GT["B0Z.E]?15)23U(@/2`D0"!I9B`D0#L*?0I#3U)%.CIE
M>&ET*"0Q*2!I9B`H)#HZ7U]%4E)/4B`]?B]>7U1+7T58251?7"@H7&0K*5PI
M+RD["F1I92`D.CI?7T524D]2(&EF("0Z.E]?15)23U(["C$["E]?14Y$7U\*
M`'!E<FQX<VDN8P````!#04-(10``/@```$$````,````00```!$```!!````
M)P```$$````V````00````,```!!````!@```$$````5````00````\```!!
M````#@```$$````F````00```!,```!!````*0```$$````"````00```!D`
M``!!````"@```$$````+````00``````````````@;(!@;"PK`<`````A`(!
M@;`/L0``````A;(!@;"PKG\`````=(?_?["J"8!HBO]_`0```&"-_W^PJQZ`
MX([_?P$````DD?]_N/__?U"2_W^PJ`.`])+_?P$```"TGO]_K/__?R"?_W^P
M__]_/*'_?["PJ(!8H?]_`0```(BB_W^PKSZ`^*G_?["J`8`0K?]_`0``````
M```$````$`````$```!'3E4```````,````"````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````"$
M&@``$!H```$```!+`P```0```%8#```!````8`,```$```!T`P``'0```/P$
M```,````*!,```T```"`3```&0```*R=```;````!````!H```"PG0``'```
M``0```#U_O]OM`$```4```#X"0``!@```$@#```*````'P4```L````0````
M%0`````````#````S)X```(```#@`0``%````!$````7````2!$``!$```!P
M$```$@```-@````3````"````!X````(````^___;P$```C^__]O\`\``/__
M_V\#````\/__;Q@/``#Z__]O#@``````````````````````````````````
M``````````````````````"TG0`````````````T$P``-!,``#03```T$P``
M-!,``#03```T$P``-!,``#03```T$P``-!,``#03```T$P``-!,``#03```T
M$P``-!,``#03```T$P``-!,``#03```T$P``-!,``#03```T$P``-!,``#03
M```T$P``-!,``#03```T$P``-!,``#03```T$P``-!,``#03```T$P``-!,`
M`#03```T$P``-!,``#03```T$P``-!,``#03```T$P``-!,``#03```T$P``
M-!,``#03```T$P``-!,``#03```T$P``-!,``#03```T$P``-!,``#03````
M````````````````````````````````````````````````````````````
M````````&!8````````$H```"E!!4BYP;0H`````9%```'A-````````.$X`
M`#Q.``!84```@$T``&!0````````1%```$Q0````````;%```$=#0SH@*$=.
M52D@,30N,BXQ(#(P,C4P,C`W`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(W+C!?
M<F,Q+C=?8S@W-#-E82D@,30N,BXQ(#(P,C4P,C`W`$$T````865A8FD``2H`
M```%-RU!``8*!T$(`0D""@,,`1($$P$4`14!%P,8`1D!&@(<`2(!`"YS:'-T
M<G1A8@`N;F]T92YG;G4N8G5I;&0M:60`+FEN=&5R<``N9VYU+FAA<V@`+F1Y
M;G-Y;0`N9'EN<W1R`"YG;G4N=F5R<VEO;@`N9VYU+G9E<G-I;VY?<@`N<F5L
M+F1Y;@`N<F5L+G!L=``N:6YI=``N=&5X=``N9FEN:0`N<F]D871A`"Y!4DTN
M97AT86(`+D%232YE>&ED>``N96A?9G)A;64`+FYO=&4N04))+71A9P`N:6YI
M=%]A<G)A>0`N9FEN:5]A<G)A>0`N9'EN86UI8P`N9V]T`"YD871A`"YB<W,`
M+F-O;6UE;G0`+D%232YA='1R:6)U=&5S````````````````````````````
M```````````````````````````````+````!P````(```!T`0``=`$``"0`
M``````````````0`````````'@````$````"````F`$``)@!```9````````
M```````!`````````"8```#V__]O`@```+0!``"T`0``E`$```0`````````
M!`````0````P````"P````(```!(`P``2`,``+`&```%`````P````0````0
M````.`````,````"````^`D``/@)```?!0`````````````!`````````$``
M``#___]O`@```!@/```8#P``U@````0``````````@````(```!-````_O__
M;P(```#P#P``\`\``(`````%`````P````0`````````7`````D````"````
M<!```'`0``#8````!``````````$````"````&4````)````0@```$@1``!(
M$0``X`$```0````6````!`````@```!N`````0````8````H$P``*!,```P`
M``````````````0`````````:0````$````&````-!,``#03``#D`@``````
M```````$````!````'0````!````!@```!@6```8%@``:#8`````````````
M"`````````!Z`````0````8```"`3```@$P```@```````````````0`````
M````@`````$````"````B$P``(A,``#X00`````````````$`````````(@`
M```!`````@```(".``"`C@``)```````````````!`````````"3`````0``
M<((```"DC@``I(X``'`````,``````````0`````````G@````$````"````
M%(\``!2/```$```````````````$`````````*@````'`````@```!B/```8
MCP``(```````````````!`````````"V````#@````,```"LG0``K)T```0`
M``````````````0````$````P@````\````#````L)T``+"=```$````````
M```````$````!````,X````&`````P```+2=``"TG0``&`$```4`````````
M!`````@```#7`````0````,```#,G@``S)X``#0!``````````````0````$
M````W`````$````#`````*````"@``!(```````````````$`````````.(`
M```(`````P```$B@``!(H```%```````````````!`````````#G`````0``
M`#``````````2*```%0```````````````$````!````\`````,``'``````
M`````)R@```U```````````````!``````````$````#``````````````#1
7H`````$``````````````0``````````
