"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var searchImpl = require("./search-implementation");
var hlimpl = require("../parser/highLevelImpl");
var sourceFinder = require("./sourceFinder");
/**
 * Finds declaration of the entity located in the particular compilation unit at offset.
 *
 * @param unit - compilation unit.
 * @param offset - character offset in unit counting from 0.
 * @param nodePart - optionally specifies the relevant instruction part of the node.
 * @returns {ll.ICompilationUnit|hl.IParseResult} - either the declaration node or compilation unit
 */
function findDeclaration(unit, offset, nodePart) {
    return searchImpl.findDeclaration(unit, offset, nodePart);
}
exports.findDeclaration = findDeclaration;
/**
 * Finds usages of the entity located in the particular compilation unit at offset.
 * @param unit - compilation unit.
 * @param offset - character offset in unit counting from 0.
 * @returns {FindUsagesResult}
 */
function findUsages(unit, offset) {
    return searchImpl.findUsages(unit, offset);
}
exports.findUsages = findUsages;
/**
 * Finds global declarations (high-level nodes defined at the top-level of the unit) of the unit,
 * current node belongs to. If current unit is included by the external unit, also adds up declarations of the
 * external unit.
 * @param n
 * @returns {hl.IHighLevelNode[]}
 */
function globalDeclarations(n) {
    return searchImpl.globalDeclarations(n);
}
exports.globalDeclarations = globalDeclarations;
/**
 * Finds references for the node starting from a given root. In a general case its better to use findUsages method.
 * @param root - root of the unit to start from.
 * @param node - node to find references for,
 * @param result - array to put results to.
 */
function refFinder(root, node, result) {
    searchImpl.refFinder(root, node, result);
}
exports.refFinder = refFinder;
/**
 * Finds declaration of the entity set by a node.
 * @param node - node, which declaration to find
 * @param nodePart - optionally specifies the relevant instruction part of the node.
 * @returns {ll.ICompilationUnit|hl.IParseResult} - either the declaration node or compilation unit
 */
function findDeclarationByNode(node, nodePart) {
    return searchImpl.findDeclarationByNode(node, nodePart);
}
exports.findDeclarationByNode = findDeclarationByNode;
/**
 * Location kind inside the instruction
 */
var LocationKind;
(function (LocationKind) {
    /**
     * In value
     */
    LocationKind[LocationKind["VALUE_COMPLETION"] = 0] = "VALUE_COMPLETION";
    /**
     * In key
     */
    LocationKind[LocationKind["KEY_COMPLETION"] = 1] = "KEY_COMPLETION";
    /**
     * In path
     */
    LocationKind[LocationKind["PATH_COMPLETION"] = 2] = "PATH_COMPLETION";
    /**
     * In directive
     */
    LocationKind[LocationKind["DIRECTIVE_COMPLETION"] = 3] = "DIRECTIVE_COMPLETION";
    /**
     * In RAML version
     */
    LocationKind[LocationKind["VERSION_COMPLETION"] = 4] = "VERSION_COMPLETION";
    /**
     * In annotation
     */
    LocationKind[LocationKind["ANNOTATION_COMPLETION"] = 5] = "ANNOTATION_COMPLETION";
    /**
     * In key of the sequence
     */
    LocationKind[LocationKind["SEQUENCE_KEY_COPLETION"] = 6] = "SEQUENCE_KEY_COPLETION";
    /**
     * In comment
     */
    LocationKind[LocationKind["INCOMMENT"] = 7] = "INCOMMENT";
})(LocationKind = exports.LocationKind || (exports.LocationKind = {}));
/**
 * Determines location kind inside the instruction.
 * @param text - RAML code
 * @param offset - offeset counting from 0.
 * @returns {any}
 */
function determineCompletionKind(text, offset) {
    return searchImpl.determineCompletionKind(text, offset);
}
exports.determineCompletionKind = determineCompletionKind;
/**
 * Determines names, which can appear in a certain place of code defined by a node and a proprty.
 * @param property - definition property
 * @param node - node.
 * @returns {string[]} - list of names
 */
function enumValues(property, node) {
    return searchImpl.enumValues(property, node);
}
exports.enumValues = enumValues;
/**
 * Returns a string representation of node name in a context. In case of deep unit dependencies returns contcatenated path.
 * @param node - node, which name to determine
 * @param context - sets up a node, which unit is used as a context to calculate potential unit paths from
 * @returns {string}
 */
function qName(node, context) {
    return hlimpl.qName(node, context);
}
exports.qName = qName;
/**
 * Return all sub types of a given type visible from the particular node.
 * @param type - type, which subtypes to find
 * @param context - context node to determine subtypes visibility.
 * @returns {ITypeDefinition[]} - list of subtypes
 */
function subTypesWithLocals(type, context) {
    return searchImpl.subTypesWithLocals(type, context);
}
exports.subTypesWithLocals = subTypesWithLocals;
/**
 * Finds nodes, which declare a type in the context of a specific node.
 * @param type - type, which declarations to find.
 * @param context - context node.
 * @returns {hl.IHighLevelNode[]} - list of results
 */
function nodesDeclaringType(type, context) {
    return searchImpl.nodesDeclaringType(type, context);
}
exports.nodesDeclaringType = nodesDeclaringType;
/**
 * Checks if a particular attribute declares an example.
 * @param attribute
 * @returns {boolean}
 */
function isExampleNodeContent(attribute) {
    return searchImpl.isExampleNodeContent(attribute);
}
exports.isExampleNodeContent = isExampleNodeContent;
/**
 * Finds the type, which declares the content of an example defined in the attribute.
 * @param attribute
 * @returns {hl.INodeDefinition}
 */
function findExampleContentType(attribute) {
    return searchImpl.findExampleContentType(attribute);
}
exports.findExampleContentType = findExampleContentType;
/**
 * Parses the value of example attribute
 * @param attribute - attribute, which value to parse
 * @param definition - type, which defines example contents.
 * @returns {hl.IHighLevelNode}
 */
function parseDocumentationContent(attribute, definition) {
    return searchImpl.parseDocumentationContent(attribute, definition);
}
exports.parseDocumentationContent = parseDocumentationContent;
/**
 * Parses example node contents.
 * @param exampleNode - example node, which value to parse
 * @param definition - type, which defines example contents.
 * @returns {hl.IHighLevelNode}
 */
function parseStructuredExample(exampleNode, definition) {
    return searchImpl.parseStructuredExample(exampleNode, definition);
}
exports.parseStructuredExample = parseStructuredExample;
/**
 * Checks whether the node defines an example.
 * @param node
 * @returns {boolean}
 */
function isExampleNode(node) {
    return searchImpl.isExampleNode(node);
}
exports.isExampleNode = isExampleNode;
/**
 * Finds out, which references a particular node property targets.
 * @param property - node property.
 * @param node - node
 * @returns {hl.IHighLevelNode[]} - list of references
 */
function referenceTargets(property, node) {
    return searchImpl.referenceTargets(property, node);
}
exports.referenceTargets = referenceTargets;
/**
 * Finds out the source of the nominal type.
 * @param nominalType - type, which source to find.
 * @returns {IHighLevelSourceProvider} - source provider. both provider and its getSource() can be null
 */
function getNominalTypeSource(nominalType) {
    return sourceFinder.getNominalTypeSource(nominalType);
}
exports.getNominalTypeSource = getNominalTypeSource;
/**
 * Finds out all subtypes of the type being a range of the property, in the context of the specific node.
 * @param property - property, which range type's subtypes to find.
 * @param node - context node.
 * @returns {hl.ITypeDefinition[]} - list of subtypes.
 */
function findAllSubTypes(property, node) {
    return searchImpl.findAllSubTypes(property, node);
}
exports.findAllSubTypes = findAllSubTypes;
/**
 * Finds the root of the unit of the particular node.
 * @param node
 * @returns {hl.IHighLevelNode}
 */
function declRoot(node) {
    return searchImpl.declRoot(node);
}
exports.declRoot = declRoot;
/**
 * Finds a node located in certain place of the code.
 * @param parent - searches inside this node
 * @param offset - start position
 * @param end - end position
 * @param goToOtherUnits - whether to check other units
 * @param returnAttrs - whether to return attributes, or only high level nodes
 * @returns {hl.IParseResult} - search result
 */
function deepFindNode(parent, offset, end, goToOtherUnits, returnAttrs) {
    if (goToOtherUnits === void 0) { goToOtherUnits = true; }
    if (returnAttrs === void 0) { returnAttrs = true; }
    return searchImpl.deepFindNode(parent, offset, end, goToOtherUnits, returnAttrs);
}
exports.deepFindNode = deepFindNode;
/**
 * Finds all children of the node, recursively.
 * @param node
 * @returns {hl.IParseResult[]}
 */
function allChildren(node) {
    return searchImpl.allChildren(node);
}
exports.allChildren = allChildren;
//# sourceMappingURL=search-interface.js.map