"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.ZoHAdapter = void 0;
const node_net_1 = require("node:net");
const node_path_1 = require("node:path");
const zigbee_on_host_1 = require("zigbee-on-host");
const logger_1 = require("zigbee-on-host/dist/utils/logger");
const logger_2 = require("../../../utils/logger");
const queue_1 = require("../../../utils/queue");
const wait_1 = require("../../../utils/wait");
const waitress_1 = require("../../../utils/waitress");
const ZSpec = __importStar(require("../../../zspec"));
const Zcl = __importStar(require("../../../zspec/zcl"));
const Zdo = __importStar(require("../../../zspec/zdo"));
const adapter_1 = require("../../adapter");
const serialPort_1 = require("../../serialPort");
const socketPortUtils_1 = require("../../socketPortUtils");
const utils_1 = require("./utils");
const NS = 'zh:zoh';
const DEFAULT_REQUEST_TIMEOUT = 15000;
class ZoHAdapter extends adapter_1.Adapter {
    serialPort;
    socketPort;
    /** True when adapter is currently closing */
    closing;
    interpanLock;
    driver;
    queue;
    zclWaitress;
    zdoWaitress;
    constructor(networkOptions, serialPortOptions, backupPath, adapterOptions) {
        super(networkOptions, serialPortOptions, backupPath, adapterOptions);
        this.hasZdoMessageOverhead = true;
        this.manufacturerID = Zcl.ManufacturerCode.CONNECTIVITY_STANDARDS_ALLIANCE;
        this.closing = false;
        const channel = networkOptions.channelList[0];
        this.driver = new zigbee_on_host_1.OTRCPDriver({
            txChannel: channel,
            ccaBackoffAttempts: 1,
            ccaRetries: 4,
            enableCSMACA: true,
            headerUpdated: true,
            reTx: false,
            securityProcessed: true,
            txDelay: 0,
            txDelayBaseTime: 0,
            rxChannelAfterTxDone: channel,
        }, 
        // NOTE: this information is overwritten on `start` if a save exists
        {
            // TODO: make this configurable
            eui64: Buffer.from([0x5a, 0x6f, 0x48, 0x6f, 0x6e, 0x5a, 0x32, 0x4d]).readBigUInt64LE(0),
            panId: this.networkOptions.panID,
            extendedPANId: Buffer.from(this.networkOptions.extendedPanID).readBigUInt64LE(0),
            channel,
            nwkUpdateId: 0,
            txPower: this.adapterOptions.transmitPower ?? /* v8 ignore next */ 5,
            // ZigBeeAlliance09
            tcKey: Buffer.from([0x5a, 0x69, 0x67, 0x42, 0x65, 0x65, 0x41, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x30, 0x39]),
            tcKeyFrameCounter: 0,
            networkKey: Buffer.from(this.networkOptions.networkKey),
            networkKeyFrameCounter: 0,
            networkKeySequenceNumber: 0,
        }, (0, node_path_1.dirname)(this.backupPath));
        this.queue = new queue_1.Queue(this.adapterOptions.concurrent || /* v8 ignore next */ 8); // ORed to avoid 0 (not checked in settings/queue constructor)
        this.zclWaitress = new waitress_1.Waitress(this.zclWaitressValidator, this.waitressTimeoutFormatter);
        this.zdoWaitress = new waitress_1.Waitress(this.zdoWaitressValidator, this.waitressTimeoutFormatter);
        this.interpanLock = false;
    }
    /**
     * Init the serial or socket port and hook parser/writer.
     */
    /* v8 ignore start */
    async initPort() {
        await this.closePort(); // will do nothing if nothing's open
        if ((0, socketPortUtils_1.isTcpPath)(this.serialPortOptions.path)) {
            const pathUrl = new URL(this.serialPortOptions.path);
            const hostname = pathUrl.hostname;
            const port = Number.parseInt(pathUrl.port, 10);
            logger_2.logger.debug(`Opening TCP socket with ${hostname}:${port}`, NS);
            this.socketPort = new node_net_1.Socket();
            this.socketPort.setNoDelay(true);
            this.socketPort.setKeepAlive(true, 15000);
            this.driver.writer.pipe(this.socketPort);
            this.socketPort.pipe(this.driver.parser);
            this.driver.parser.on('data', this.driver.onFrame.bind(this.driver));
            return await new Promise((resolve, reject) => {
                const openError = async (err) => {
                    await this.stop();
                    reject(err);
                };
                this.socketPort.on('connect', () => {
                    logger_2.logger.debug('Socket connected', NS);
                });
                this.socketPort.on('ready', () => {
                    logger_2.logger.info('Socket ready', NS);
                    this.socketPort.removeListener('error', openError);
                    this.socketPort.once('close', this.onPortClose.bind(this));
                    this.socketPort.on('error', this.onPortError.bind(this));
                    resolve();
                });
                this.socketPort.once('error', openError);
                this.socketPort.connect(port, hostname);
            });
        }
        const serialOpts = {
            path: this.serialPortOptions.path,
            baudRate: typeof this.serialPortOptions.baudRate === 'number' ? this.serialPortOptions.baudRate : 115200,
            rtscts: typeof this.serialPortOptions.rtscts === 'boolean' ? this.serialPortOptions.rtscts : false,
            autoOpen: false,
            parity: 'none',
            stopBits: 1,
            xon: false,
            xoff: false,
        };
        // enable software flow control if RTS/CTS not enabled in config
        if (!serialOpts.rtscts) {
            logger_2.logger.info('RTS/CTS config is off, enabling software flow control.', NS);
            serialOpts.xon = true;
            serialOpts.xoff = true;
        }
        logger_2.logger.debug(() => `Opening serial port with [path=${serialOpts.path} baudRate=${serialOpts.baudRate} rtscts=${serialOpts.rtscts}]`, NS);
        this.serialPort = new serialPort_1.SerialPort(serialOpts);
        this.driver.writer.pipe(this.serialPort);
        this.serialPort.pipe(this.driver.parser);
        this.driver.parser.on('data', this.driver.onFrame.bind(this.driver));
        try {
            await this.serialPort.asyncOpen();
            await this.serialPort.asyncFlush();
            logger_2.logger.info('Serial port opened', NS);
            this.serialPort.once('close', this.onPortClose.bind(this));
            this.serialPort.on('error', this.onPortError.bind(this));
        }
        catch (error) {
            await this.stop();
            throw error;
        }
    }
    /* v8 ignore stop */
    /**
     * Handle port closing
     * @param err A boolean for Socket, an Error for serialport
     */
    /* v8 ignore start */
    async onPortClose(error) {
        if (error) {
            logger_2.logger.error('Port closed unexpectedly.', NS);
        }
        else {
            logger_2.logger.info('Port closed.', NS);
        }
    }
    /* v8 ignore stop */
    /**
     * Handle port error
     * @param error
     */
    /* v8 ignore start */
    async onPortError(error) {
        logger_2.logger.error(`Port ${error}`, NS);
        this.emit('disconnected');
    }
    /* v8 ignore stop */
    /* v8 ignore start */
    async closePort() {
        if (this.serialPort?.isOpen) {
            try {
                await this.serialPort.asyncFlushAndClose();
            }
            catch (err) {
                logger_2.logger.error(`Failed to close serial port ${err}.`, NS);
            }
            this.serialPort.removeAllListeners();
            this.serialPort = undefined;
        }
        else if (this.socketPort != null && !this.socketPort.closed) {
            this.socketPort.destroy();
            this.socketPort.removeAllListeners();
            this.socketPort = undefined;
        }
    }
    /* v8 ignore stop */
    async start() {
        (0, logger_1.setLogger)(logger_2.logger); // pass the logger to ZoH
        await this.initPort();
        let result = 'resumed';
        const currentNetParams = await this.driver.readNetworkState();
        if (currentNetParams) {
            // Note: channel change is handled by Controller
            if (
            // TODO: add eui64 whenever added as configurable
            this.networkOptions.panID !== currentNetParams.panId ||
                Buffer.from(this.networkOptions.extendedPanID).readBigUInt64LE(0) != currentNetParams.extendedPANId ||
                !Buffer.from(this.networkOptions.networkKey).equals(currentNetParams.networkKey)) {
                await this.driver.resetNetwork();
                result = 'reset';
            }
        }
        else {
            // no save detected, brand new network
            result = 'reset';
        }
        await this.driver.start();
        await this.driver.formNetwork();
        this.driver.on('frame', this.onFrame.bind(this));
        this.driver.on('gpFrame', this.onGPFrame.bind(this));
        this.driver.on('deviceJoined', this.onDeviceJoined.bind(this));
        this.driver.on('deviceRejoined', this.onDeviceRejoined.bind(this));
        this.driver.on('deviceLeft', this.onDeviceLeft.bind(this));
        this.driver.on('deviceAuthorized', this.onDeviceAuthorized.bind(this));
        return result;
    }
    async stop() {
        this.closing = true;
        this.driver.removeAllListeners();
        this.queue.clear();
        this.zclWaitress.clear();
        this.zdoWaitress.clear();
        await this.driver.stop();
    }
    async getCoordinatorIEEE() {
        return `0x${(0, utils_1.bigUInt64ToHexBE)(this.driver.netParams.eui64)}`;
    }
    async getCoordinatorVersion() {
        return {
            type: 'ZigBee on Host',
            meta: {
                major: this.driver.protocolVersionMajor,
                minor: this.driver.protocolVersionMinor,
                version: this.driver.ncpVersion,
                apiVersion: this.driver.rcpAPIVersion,
                revision: `https://github.com/Nerivec/zigbee-on-host (using: ${this.driver.ncpVersion})`,
            },
        };
    }
    /* v8 ignore start */
    async reset(type) {
        throw new Error(`Reset ${type} not support`);
    }
    /* v8 ignore stop */
    /* v8 ignore start */
    async supportsBackup() {
        return false;
    }
    /* v8 ignore stop */
    /* v8 ignore start */
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async backup(ieeeAddressesInDatabase) {
        throw new Error('ZigBee on Host handles backup internally');
    }
    /* v8 ignore stop */
    async getNetworkParameters() {
        return {
            panID: this.driver.netParams.panId,
            extendedPanID: `0x${(0, utils_1.bigUInt64ToHexBE)(this.driver.netParams.extendedPANId)}`,
            channel: this.driver.netParams.channel,
            nwkUpdateID: this.driver.netParams.nwkUpdateId,
        };
    }
    /* v8 ignore start */
    async addInstallCode(ieeeAddress, key) {
        throw new Error(`not supported ${ieeeAddress}, ${key.toString('hex')}`);
    }
    /* v8 ignore stop */
    /* v8 ignore start */
    waitFor(networkAddress, endpoint, frameType, direction, transactionSequenceNumber, clusterID, commandIdentifier, timeout) {
        const waiter = this.zclWaitress.waitFor({
            sender: networkAddress,
            endpoint,
            clusterId: clusterID,
            commandId: commandIdentifier,
            transactionSequenceNumber,
        }, timeout);
        const cancel = () => this.zclWaitress.remove(waiter.ID);
        return { cancel, promise: waiter.start().promise };
    }
    async sendZdo(ieeeAddress, networkAddress, clusterId, payload, disableResponse) {
        if (networkAddress === ZSpec.COORDINATOR_ADDRESS) {
            // mock ZDO response using driver layer data for coordinator
            // seqNum doesn't matter since waitress bypassed, so don't bother doing any logic for it
            const response = this.driver.getCoordinatorZDOResponse(clusterId, payload);
            if (!response) {
                throw new Error(`Coordinator does not support ZDO cluster ${clusterId}`);
            }
            const respClusterId = clusterId | 0x8000;
            const result = Zdo.Buffalo.readResponse(this.hasZdoMessageOverhead, respClusterId, response);
            this.emit('zdoResponse', respClusterId, result);
            return result;
        }
        return await this.queue.execute(async () => {
            this.checkInterpanLock();
            logger_2.logger.debug(() => `~~~> [ZDO to=${ieeeAddress}:${networkAddress} clusterId=${clusterId} disableResponse=${disableResponse}]`, NS);
            const [, zdoSeqNum] = await this.driver.sendZDO(payload, networkAddress, // nwkDest16
            undefined, // nwkDest64 XXX: avoid passing EUI64 whenever not absolutely necessary
            clusterId);
            if (!disableResponse) {
                const responseClusterId = Zdo.Utils.getResponseClusterId(clusterId);
                if (responseClusterId) {
                    const resp = await this.zdoWaitress
                        .waitFor({
                        sender: responseClusterId === Zdo.ClusterId.NETWORK_ADDRESS_RESPONSE ? ieeeAddress : networkAddress,
                        clusterId: responseClusterId,
                        transactionSequenceNumber: zdoSeqNum,
                    }, DEFAULT_REQUEST_TIMEOUT)
                        .start().promise;
                    return resp.response;
                }
            }
        }, networkAddress);
    }
    async permitJoin(seconds, networkAddress) {
        if (networkAddress === undefined) {
            // send ZDO BCAST
            this.driver.allowJoins(seconds, true);
            this.driver.gpEnterCommissioningMode(seconds);
            const clusterId = Zdo.ClusterId.PERMIT_JOINING_REQUEST;
            // `authentication`: TC significance always 1 (zb specs)
            const zdoPayload = Zdo.Buffalo.buildRequest(this.hasZdoMessageOverhead, clusterId, seconds, 1, []);
            await this.sendZdo(ZSpec.BLANK_EUI64, ZSpec.BroadcastAddress.DEFAULT, clusterId, zdoPayload, true);
        }
        else if (networkAddress === ZSpec.COORDINATOR_ADDRESS) {
            this.driver.allowJoins(seconds, true);
            this.driver.gpEnterCommissioningMode(seconds);
        }
        else {
            // send ZDO to networkAddress
            this.driver.allowJoins(seconds, false);
            const clusterId = Zdo.ClusterId.PERMIT_JOINING_REQUEST;
            // `authentication`: TC significance always 1 (zb specs)
            const zdoPayload = Zdo.Buffalo.buildRequest(this.hasZdoMessageOverhead, clusterId, seconds, 1, []);
            const result = await this.sendZdo(ZSpec.BLANK_EUI64, networkAddress, clusterId, zdoPayload, false);
            /* v8 ignore start */
            if (!Zdo.Buffalo.checkStatus(result)) {
                throw new Zdo.StatusError(result[0]);
            }
            /* v8 ignore stop */
        }
    }
    // #endregion
    // #region ZCL
    async sendZclFrameToEndpoint(ieeeAddr, networkAddress, endpoint, zclFrame, timeout, disableResponse, disableRecovery, sourceEndpoint) {
        /* v8 ignore start */
        if (networkAddress === ZSpec.COORDINATOR_ADDRESS) {
            // TODO: handle e.g. GP permit join
            logger_2.logger.debug(() => `~x~> [ZCL clusterId=${zclFrame.cluster.ID} destEp=${endpoint} sourceEp=${sourceEndpoint}] Not sending to coordinator`, NS);
            return;
        }
        /* v8 ignore stop */
        let commandResponseId;
        if (zclFrame.command.response !== undefined && disableResponse === false) {
            commandResponseId = zclFrame.command.response;
        }
        else if (!zclFrame.header.frameControl.disableDefaultResponse) {
            commandResponseId = Zcl.Foundation.defaultRsp.ID;
        }
        return await this.queue.execute(async () => {
            this.checkInterpanLock();
            logger_2.logger.debug(() => `~~~> [ZCL to=${ieeeAddr}:${networkAddress} clusterId=${zclFrame.cluster.ID} destEp=${endpoint} sourceEp=${sourceEndpoint}]`, NS);
            for (let i = 0; i < 2; i++) {
                try {
                    await this.driver.sendUnicast(zclFrame.toBuffer(), sourceEndpoint === ZSpec.GP_ENDPOINT && endpoint === ZSpec.GP_ENDPOINT ? ZSpec.GP_PROFILE_ID : ZSpec.HA_PROFILE_ID, zclFrame.cluster.ID, networkAddress, // nwkDest16
                    undefined, // nwkDest64 XXX: avoid passing EUI64 whenever not absolutely necessary
                    endpoint, // destEp
                    sourceEndpoint ?? 1);
                    if (commandResponseId !== undefined) {
                        const resp = await this.zclWaitress
                            .waitFor({
                            sender: networkAddress,
                            clusterId: zclFrame.cluster.ID,
                            endpoint,
                            commandId: commandResponseId,
                            transactionSequenceNumber: zclFrame.header.transactionSequenceNumber,
                        }, timeout)
                            .start().promise;
                        return resp;
                    }
                    return;
                }
                catch (error) {
                    if (disableRecovery || i == 1) {
                        throw error;
                    } // else retry
                }
                /* v8 ignore start */
            } // coverage detection failure
            /* v8 ignore stop */
        });
    }
    async sendZclFrameToGroup(groupID, zclFrame, sourceEndpoint) {
        return await this.queue.execute(async () => {
            this.checkInterpanLock();
            logger_2.logger.debug(() => `~~~> [ZCL GROUP to=${groupID} clusterId=${zclFrame.cluster.ID} sourceEp=${sourceEndpoint}]`, NS);
            await this.driver.sendGroupcast(zclFrame.toBuffer(), ZSpec.HA_PROFILE_ID, zclFrame.cluster.ID, groupID, sourceEndpoint ?? 1);
            // settle
            await (0, wait_1.wait)(500);
        });
    }
    async sendZclFrameToAll(endpoint, zclFrame, sourceEndpoint, destination) {
        return await this.queue.execute(async () => {
            this.checkInterpanLock();
            logger_2.logger.debug(() => `~~~> [ZCL BROADCAST to=${destination} destEp=${endpoint} sourceEp=${sourceEndpoint}]`, NS);
            await this.driver.sendBroadcast(zclFrame.toBuffer(), sourceEndpoint === ZSpec.GP_ENDPOINT && endpoint === ZSpec.GP_ENDPOINT ? ZSpec.GP_PROFILE_ID : ZSpec.HA_PROFILE_ID, zclFrame.cluster.ID, destination, endpoint, sourceEndpoint);
            // settle
            await (0, wait_1.wait)(500);
        });
    }
    // #endregion
    // #region InterPAN
    /* v8 ignore start */
    async setChannelInterPAN(channel) {
        throw new Error(`not supported ${channel}`);
    }
    /* v8 ignore stop */
    /* v8 ignore start */
    async sendZclFrameInterPANToIeeeAddr(zclFrame, ieeeAddress) {
        throw new Error(`not supported ${JSON.stringify(zclFrame)}, ${ieeeAddress}`);
    }
    /* v8 ignore stop */
    /* v8 ignore start */
    async sendZclFrameInterPANBroadcast(zclFrame, timeout) {
        throw new Error(`not supported ${JSON.stringify(zclFrame)}, ${timeout}`);
    }
    /* v8 ignore stop */
    /* v8 ignore start */
    async restoreChannelInterPAN() {
        throw new Error(`not supported`);
    }
    /* v8 ignore stop */
    // #endregion
    // #region Implementation-Specific
    /* v8 ignore start */
    checkInterpanLock() {
        if (this.interpanLock) {
            throw new Error(`[INTERPAN MODE] Cannot execute non-InterPAN commands.`);
        }
    }
    /* v8 ignore stop */
    /**
     * @param sender16 If undefined, sender64 is expected defined
     * @param sender64 If undefined, sender16 is expected defined
     * @param apsHeader
     * @param apsPayload
     */
    onFrame(sender16, sender64, apsHeader, apsPayload, rssi) {
        if (apsHeader.profileId === Zdo.ZDO_PROFILE_ID) {
            logger_2.logger.debug(() => `<~~~ APS ZDO[sender=${sender16}:${sender64} clusterId=${apsHeader.clusterId}]`, NS);
            try {
                const result = Zdo.Buffalo.readResponse(this.hasZdoMessageOverhead, apsHeader.clusterId, apsPayload);
                if (apsHeader.clusterId === Zdo.ClusterId.NETWORK_ADDRESS_RESPONSE) {
                    // special case to properly resolve a NETWORK_ADDRESS_RESPONSE following a NETWORK_ADDRESS_REQUEST (based on EUI64 from ZDO payload)
                    // NOTE: if response has invalid status (no EUI64 available), response waiter will eventually time out
                    if (Zdo.Buffalo.checkStatus(result)) {
                        this.zdoWaitress.resolve({ sender: result[1].eui64, clusterId: apsHeader.clusterId, response: result, seqNum: apsPayload[0] });
                    }
                }
                else {
                    this.zdoWaitress.resolve({ sender: sender16, clusterId: apsHeader.clusterId, response: result, seqNum: apsPayload[0] });
                }
                this.emit('zdoResponse', apsHeader.clusterId, result);
                /* v8 ignore start */
            }
            catch (error) {
                logger_2.logger.error(`${error.message}`, NS);
            }
            /* v8 ignore stop */
        }
        else {
            logger_2.logger.debug(() => `<~~~ APS[sender=${sender16}:${sender64} profileId=${apsHeader.profileId} clusterId=${apsHeader.clusterId}]`, NS);
            const payload = {
                clusterID: apsHeader.clusterId,
                header: Zcl.Header.fromBuffer(apsPayload),
                address: sender64 !== undefined ? `0x${(0, utils_1.bigUInt64ToHexBE)(sender64)}` : sender16,
                data: apsPayload,
                endpoint: apsHeader.sourceEndpoint,
                linkquality: rssi, // TODO: convert RSSI to LQA
                groupID: apsHeader.group,
                wasBroadcast: apsHeader.frameControl.deliveryMode === 2 /* BCAST */,
                destinationEndpoint: apsHeader.destEndpoint,
            };
            this.zclWaitress.resolve(payload);
            this.emit('zclPayload', payload);
        }
    }
    onGPFrame(cmdId, payload, macHeader, nwkHeader, rssi) {
        // transform into a ZCL frame
        const data = Buffer.alloc((nwkHeader.frameControlExt?.appId === 0x02 /* ZGP */ ? /* v8 ignore next */ 20 : 15) + payload.byteLength);
        let offset = 0;
        data.writeUInt8(0b00000001, offset); // frameControl: FrameType.SPECIFIC + Direction.CLIENT_TO_SERVER + disableDefaultResponse=false
        offset += 1;
        data.writeUInt8(macHeader.sequenceNumber ?? /* v8 ignore next */ 0, offset);
        offset += 1;
        data.writeUInt8(cmdId === 0xe0 ? 0x04 /* commissioning notification */ : 0x00 /* notification */, offset);
        offset += 1;
        if (nwkHeader.frameControlExt) {
            /* v8 ignore start */
            if (cmdId === 0xe0) {
                data.writeUInt16LE((nwkHeader.frameControlExt.appId & 0x7) |
                    (((nwkHeader.frameControlExt.rxAfterTx ? 1 : 0) & 0x1) << 3) |
                    ((nwkHeader.frameControlExt.securityLevel & 0x3) << 4), offset);
                /* v8 ignore stop */
            }
            else {
                data.writeUInt16LE((nwkHeader.frameControlExt.appId & 0x7) |
                    ((nwkHeader.frameControlExt.securityLevel & 0x3) << 6) |
                    /* v8 ignore next */ (((nwkHeader.frameControlExt.rxAfterTx ? 1 : 0) & 0x3) << 11), offset);
            }
        }
        else {
            data.writeUInt16LE(0, offset); // options, only srcID present
        }
        offset += 2;
        /* v8 ignore start */
        if (nwkHeader.frameControlExt?.appId === 0x02 /* ZGP */) {
            data.writeBigUInt64LE(macHeader.source64, offset);
            offset += 8;
            data.writeUInt8(nwkHeader.endpoint, offset);
            offset += 1;
            /* v8 ignore stop */
        }
        else {
            data.writeUInt32LE(nwkHeader.sourceId, offset);
            offset += 4;
        }
        data.writeUInt32LE(nwkHeader.securityFrameCounter ?? 0, offset);
        offset += 4;
        data.writeUInt8(cmdId, offset);
        offset += 1;
        data.writeUInt8(payload.byteLength, offset);
        offset += 1;
        data.set(payload, offset);
        const zclPayload = {
            clusterID: 0x21 /* Green Power */,
            header: Zcl.Header.fromBuffer(data),
            address: macHeader.source64 !== undefined ? /* v8 ignore next */ `0x${(0, utils_1.bigUInt64ToHexBE)(macHeader.source64)}` : nwkHeader.sourceId & 0xffff,
            data,
            endpoint: ZSpec.GP_ENDPOINT,
            linkquality: rssi, // TODO: convert RSSI to LQA
            groupID: ZSpec.GP_GROUP_ID,
            wasBroadcast: macHeader.destination64 === undefined && macHeader.destination16 >= 0xfff8,
            destinationEndpoint: ZSpec.GP_ENDPOINT,
        };
        this.zclWaitress.resolve(zclPayload);
        this.emit('zclPayload', zclPayload);
    }
    onDeviceJoined(source16, source64, capabilities) {
        // XXX: don't delay if no cap? (joined through router)
        if (capabilities && capabilities.rxOnWhenIdle) {
            this.emit('deviceJoined', { networkAddress: source16, ieeeAddr: `0x${(0, utils_1.bigUInt64ToHexBE)(source64)}` });
        }
        else {
            // XXX: end devices can be finicky about finishing the key authorization, Z2M interview can create a bottleneck, so delay it
            setTimeout(() => {
                this.emit('deviceJoined', { networkAddress: source16, ieeeAddr: `0x${(0, utils_1.bigUInt64ToHexBE)(source64)}` });
            }, 5000);
        }
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    onDeviceRejoined(source16, source64, capabilities) {
        this.emit('deviceJoined', { networkAddress: source16, ieeeAddr: `0x${(0, utils_1.bigUInt64ToHexBE)(source64)}` });
    }
    onDeviceLeft(source16, source64) {
        this.emit('deviceLeave', { networkAddress: source16, ieeeAddr: `0x${(0, utils_1.bigUInt64ToHexBE)(source64)}` });
    }
    /* v8 ignore start */
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    onDeviceAuthorized(source16, source64) { }
    /* v8 ignore stop */
    waitressTimeoutFormatter(matcher, timeout) {
        return `Timeout after ${timeout}ms [sender=${matcher.sender} clusterId=${matcher.clusterId} cmdId=${matcher.commandId}]`;
    }
    zclWaitressValidator(payload, matcher) {
        return (
        // no sender in Touchlink
        (matcher.sender === undefined || payload.address === matcher.sender) &&
            payload.clusterID === matcher.clusterId &&
            payload.endpoint === matcher.endpoint &&
            payload.header.commandIdentifier === matcher.commandId &&
            (matcher.transactionSequenceNumber === undefined || payload.header.transactionSequenceNumber === matcher.transactionSequenceNumber));
    }
    zdoWaitressValidator(payload, matcher) {
        return matcher.sender === payload.sender && matcher.clusterId === payload.clusterId && matcher.transactionSequenceNumber === payload.seqNum;
    }
}
exports.ZoHAdapter = ZoHAdapter;
//# sourceMappingURL=zohAdapter.js.map