package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.40.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.40.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`MP`!`````!\```````!``````````,@!`0``
M`````````$``.``*`$``&@`9``8````$````0`````````!``````````$``
M````````,`(````````P`@````````@``````````P````0```"4`@``````
M`)0"````````E`(````````;`````````!L``````````0`````````!````
M!0```````````````````````````````````-B@````````V*``````````
M``$```````$````&````</L```````!P^P$``````'#[`0``````&`4`````
M``!`!0```````````0```````@````8```"`^P```````(#[`0``````@/L!
M```````@`@```````"`"````````"``````````$````!````'`"````````
M<`(```````!P`@```````"0`````````)``````````$``````````0````$
M````N*````````"XH````````+B@````````(``````````@``````````0`
M````````4.5T9`0`````F0````````"9`````````)D````````,`0``````
M``P!````````!`````````!1Y71D!@``````````````````````````````
M```````````````````````````0`````````%+E=&0$````</L```````!P
M^P$``````'#[`0``````D`0```````"0!`````````$`````````!````!0`
M```#````1TY5`&QO0NR\]3Y]?589&2E@32X?N934+VQI8B]L9"UL:6YU>"UA
M87)C:#8T+G-O+C$``#L```!)````!`````@````A$!``!<`$A"@"$`6`AD"$
M@HD`!@D@!(B(@!!!N@`""`````!)````2P```$P```````````````````!.
M`````````$\```!0````40``````````````4@```%,`````````````````
M````````5````%4```!6`````````%<```````````````````!8````60``
M`%H```````````````````!;````7`````````!>`````````%\`````````
M8````````````````````````````````````&$```````````````````!B
M`````````&,```````````````````!E````?JF<FUGOEWEK"4/6=E0\E"'_
MY`X-.I<+P^4_+(D'<-/9<5@<K4OCP&L:GE#59-1\F4_#FR_%F[8S8MOM:W^:
M?`&O-.B[XY)\LP&+;T+ZUX=#1=7L9Z_,9A6,)<RCX/M)S]Y'+>-'Q@'@IP(*
M;\7?'[[:5QO[H6Q@```````````````````````````````````````````#
M``H`X!L````````````````````````#`!8````"`````````````````'$`
M```2`````````````````````````,T!```2````````````````````````
M`'L````2`````````````````````````'L!```2````````````````````
M`````)P!```2`````````````````````````#4"```2````````````````
M`````````-0!```2`````````````````````````(<````2````````````
M``````````````$````@`````````````````````````!8"```2````````
M``````````````````\"```2`````````````````````````+(#```2````
M`````````````````````+D````2`````````````````````````/4"```1
M`````````````````````````&$````2`````````````````````````*P"
M```2`````````````````````````.<````2````````````````````````
M`!\"```2`````````````````````````)<!```2````````````````````
M`````,4!```2`````````````````````````*0!```2````````````````
M`````````'D"```B`````````````````````````!`````@````````````
M`````````````-P````2`````````````````````````"P````@````````
M`````````````````+4!```2`````````````````````````-$"```2````
M`````````````````````%("```2`````````````````````````*4"```2
M`````````````````````````&L"```1`````````````````````````,L"
M```2`````````````````````````%,````2````````````````````````
M`"<!```6``````````````````````````0!```2````````````````````
M`````%L!```2`````````````````````````.(!```2````````````````
M`````````*@````2`````````````````````````/4````2````````````
M`````````````)X"```2`````````````````````````+H"```2````````
M`````````````````.,"```2`````````````````````````.@!```2````
M`````````````````````$,"```2`````````````````````````,T````2
M`````````````````````````(8!```2```````````````````````````"
M```2`````````````````````````)H````2````````````````````````
M`"`!```2`````````````````````````$\!```2````````````````````
M`````%$"```2`````````````````````````+,"```2````````````````
M`````````)8"```2`````````````````````````.T!```2````````````
M`````````````*X!```2`````````````````````````#H!```2````````
M`````````````````(@"```2`````````````````````````!0!```2````
M`````````````````````/H!```2`````````````````````````'("```2
M`````````````````````````$8````1`````````````````````````'8`
M```2`````````````````````````,(````2````````````````````````
M`&P!```2`````````````````````````%D"```2````````````````````
M`````-D"```1`````````````````````````+X!```2````````````````
M`````````-L!```2`````````````````````````"<"```2````````````
M`````````````-L"```1`````````````````````````(\"```2````````
M`````````````````*`#```2``P`,#8````````P``````````0$```0`!<`
ML``"`````````````````-8#```0`!8````"`````````````````.,#```0
M`!<`B``"`````````````````%@$```2``P`1"D```````#T`@```````-@#
M```@`!8````"`````````````````/P#```2``P`X"<````````T````````
M`&4$```1`!8`@``"```````(`````````!@$```0`!<`B``"````````````
M`````,<#```1``X`8%<````````$`````````)8#```2``P`8%4```````#<
M`0```````%`#```2``P`2#\```````"@`@```````&\$```2``P`X#,`````
M``"H`````````#(#```2``P`0"D````````$`````````-P#```2``P``!\`
M```````T`````````&8"```2``P`@!P```````!T`@````````T$```0`!<`
ML``"`````````````````"T$```0`!<`L``"`````````````````&8#```2
M``P`8#,```````"``````````/$#```2``P`*%0````````P`0```````#\$
M```0`!8`B``"`````````````````(H#```2``P`@"P```````"\`0``````
M`"0$```2``P`X%,```````!(`````````'<$```2``P`0"X````````8!0``
M`````#($```2``P`J#<```````"@!P```````$,#```2``P`0"P````````\
M`````````'0#```2``P`("@````````<`0```````+L#```2``P`Z$$`````
M```8`0````````P$```0`!<`L``"`````````````````$8$```2``P`8#8`
M``````!(`0````````!?7V=M;VY?<W1A<G1?7P!?251-7V1E<F5G:7-T97)4
M34-L;VYE5&%B;&4`7TE435]R96=I<W1E<E1-0VQO;F5486)L90!03%]D;U]U
M;F1U;7``4&5R;%]S>7-?=&5R;0!097)L7V=V7V9E=&-H<'8`<&5R;%]F<F5E
M`%!E<FQ?9V5T7W-V`%!E<FQ?871F;W)K7W5N;&]C:P!P97)L7V1E<W1R=6-T
M`%!E<FQ?;F5W6%-?9FQA9W,`<&5R;%]R=6X`4&5R;%]N97=84P!097)L7W-V
M7W-E='!V;@!P97)L7V%L;&]C`%!E<FQ?<W9?<V5T<'8`<&5R;%]C;VYS=')U
M8W0`8F]O=%]$>6YA3&]A9&5R`%!E<FQ?<V%F97-Y<VUA;&QO8P!03%]C=7)R
M96YT7V-O;G1E>'0`4&5R;%]C<F]A:U]N;V-O;G1E>'0`4&5R;%]M9U]S970`
M4&5R;%]A=&9O<FM?;&]C:P!097)L7W-Y<U]I;FET,P!P97)L7W!A<G-E`%]?
M<W1A8VM?8VAK7V9A:6P`97AI=`!L<V5E:S8T`')E861D:7(V-`!S=')D=7``
M8VQO<V5D:7(`;W!E;C8T`'-T<G)C:'(`=6YL:6YK`'-T<G1O:P!S=')L96X`
M<FUD:7(`<F5A9`!?7VUE;6-P>5]C:&L`;6MD:7(`7U]V9G!R:6YT9E]C:&L`
M9V5T<&ED`')E861L:6YK`')E86QL;V,`7U]S<')I;G1F7V-H:P!?7VUE;6UO
M=F5?8VAK`%]?9G!R:6YT9E]C:&L`;'-T870V-`!?7VQI8F-?<W1A<G1?;6%I
M;@!S=&1E<G(`;65M8VAR`%]?8WAA7V9I;F%L:7IE`'-T<F-H<@!M96US970`
M;W!E;F1I<@!G971U:60`86-C97-S`&UE;6-P>0!S=')C;7``7U]E<G)N;U]L
M;V-A=&EO;@!A8F]R=`!S=')N8VUP`%]?96YV:7)O;@!?7W)E9VES=&5R7V%T
M9F]R:P!?7W-T86-K7V-H:U]G=6%R9`!L:6)P97)L+G-O`&QI8F,N<V\N-@!L
M9"UL:6YU>"UA87)C:#8T+G-O+C$`<&%R7V-U<G)E;G1?97AE8P!P87)?8F%S
M96YA;64`6%-?26YT97)N86QS7U!!4E]"3T]4`'!A<E]E;G9?8VQE86X`<&%R
M7V-U<G)E;G1?97AE8U]P<F]C`'!A<E]D:7)N86UE`'-H85]F:6YA;`!G971?
M=7-E<FYA;65?9G)O;5]G971P=W5I9`!P87)?8VQE86YU<`!?24]?<W1D:6Y?
M=7-E9`!?7V1A=&%?<W1A<G0`7U]B<W-?<W1A<G1?7P!S:&%?=7!D871E`'AS
M7VEN:70`7U]E;F1?7P!?7V)S<U]E;F1?7P!?7V)S<U]S=&%R=`!S:&%?:6YI
M=`!?96YD`'!A<E]M:W1M<&1I<@!?961A=&$`<&%R7W-E='5P7VQI8G!A=&@`
M<&%R7V9I;F1P<F]G`&UY7W!A<E]P;`!P87)?9&EE`'!A<E]I;FET7V5N=@!'
M3$E"0U\R+C$W`$=,24)#7S(N,S0`1TQ)0D-?,BXS,P`O=7-R+VQI8B]P97)L
M-2\U+C0P+V-O<F5?<&5R;"]#3U)%``````````$``@`!``$``@`"``(``0`!
M``(``@`"``$``P`!``(``0`"``(``@`"``(``0`!``$``@`"``0``@`"``(`
M`0`!``$``0`"``$``0`"``(``@`"``(``0`"``(``0`"``$`!``"``(``@`"
M``$``@`!``(``@`!``(``0`!``4``@`"``(``@`"``(``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!
M``$````!``$`'`,``!`````@````EY&6!@```P"$!`````````$``P`2`P``
M$`````````"TD98&```%`(\$```0````LY&6!@``!`":!```$````)>1E@8`
M``(`A`0```````!P^P$```````,$````````("````````!X^P$```````,$
M````````S!\```````!8_P$```````,$````````0%<```````#P_P$`````
M``,$````````@!P````````(``(```````,$````````"``"```````@``(`
M``````,$````````0%L````````H``(```````,$````````T%<````````X
M``(```````,$````````P%@```````!```(```````,$````````(%L`````
M``!(``(```````,$````````*%L```````!0``(```````,$````````,%L`
M``````!8``(```````,$````````.%L```````!H``(```````,$````````
M"%L```````!P``(```````,$````````$%L```````"```(```````,$````
M````2%L```````#@_0$```````$$```#``````````````#H_0$```````$$
M```$``````````````#P_0$```````$$```%``````````````#X_0$`````
M``$$```&````````````````_@$```````$$```'```````````````(_@$`
M``````$$```(```````````````0_@$```````$$```)```````````````8
M_@$```````$$```*```````````````@_@$```````$$```+````````````
M```H_@$```````$$```,```````````````P_@$```````$$```-````````
M```````X_@$```````$$```.``````````````!`_@$```````$$```/````
M``````````!(_@$```````$$```0``````````````!0_@$```````$$```1
M``````````````!8_@$```````$$```2``````````````!@_@$```````$$
M```3``````````````!H_@$```````$$```4``````````````!P_@$`````
M``$$```5``````````````!X_@$```````$$```6``````````````"`_@$`
M``````$$```7``````````````"(_@$```````$$```8``````````````"0
M_@$```````$$```9``````````````"8_@$```````$$```:````````````
M``"@_@$```````$$```;``````````````"H_@$```````$$```<````````
M``````"P_@$```````$$```=``````````````"X_@$```````$$```>````
M``````````#`_@$```````$$```?``````````````#(_@$```````$$```@
M``````````````#0_@$```````$$```B``````````````#8_@$```````8$
M```C``````````````#@_@$```````$$```D``````````````#H_@$`````
M``$$```E``````````````#P_@$```````$$```F``````````````#X_@$`
M``````$$```G````````````````_P$```````$$```H```````````````(
M_P$```````$$```I```````````````0_P$```````$$```J````````````
M```8_P$```````$$```L```````````````@_P$```````$$```M````````
M```````H_P$```````$$```N```````````````P_P$```````$$```O````
M```````````X_P$```````$$```P``````````````!`_P$```````$$```Q
M``````````````!(_P$```````$$```R``````````````!0_P$```````$$
M```S``````````````!@_P$```````$$```T``````````````!H_P$`````
M``$$```U``````````````!P_P$```````$$```V``````````````!X_P$`
M``````$$```W``````````````"`_P$```````$$```X``````````````"(
M_P$```````$$```Y``````````````"0_P$```````$$```Z````````````
M``"8_P$```````$$```[``````````````"@_P$```````$$```\````````
M``````"H_P$```````$$```]``````````````"P_P$```````$$```^````
M``````````"X_P$```````$$```_``````````````#`_P$```````$$``!`
M``````````````#(_P$```````$$``!!``````````````#0_P$```````$$
M``!$``````````````#8_P$```````$$``!%``````````````#@_P$`````
M``$$``!&``````````````#H_P$```````$$``!'``````````````#X_P$`
M``````$$``!(``````````````"X_0$```````($```+``````````````#`
M_0$```````($```A``````````````#(_0$```````($```K````````````
M``#0_0$```````($``!"```````````````?(`/5_7N_J?T#`)'2``"4_7O!
MJ,`#7]8``````````/![OZGP``#0$=I&^1#"-I$@`A_6'R`#U1\@`]4?(`/5
M\```T!'>1OD0XC:1(`(?UO```-`1XD;Y$`(WD2`"'];P``#0$>9&^1`B-Y$@
M`A_6\```T!'J1OD00C>1(`(?U@``````````````````````````````````
M````````_X,!T>,``-!C)$?Y_7L#J?W#`)'BAP"IXB,`D>%#`)'@'P"YX',`
MD?-3!*GU6P6I9`!`^>07`/D$`(#2XP``T&/D1_E@`#_6X@``T$(,1_G@``#0
M`'1'^>$#`JKC``#08ZQ'^6``/];@``#0`-A'^0``0#G`"@`VX```T`!L1_E!
MT#O5\P``\")H8/CT'T"Y06A2.8`2`!&4#@`1(00`,D%H$CGA``#0(<Q'^0!\
M?9,@`#_6XP,`JN8+0/GA``#P8D("D>`?0+DE0$#Y)```D,<`0/F$X"N19Q``
MJ2$``)`A`"R10P0`^64$`:D?!`!Q#0L`5"$`@-(?(`/5'R`#U20,`!'%>&'X
M(00`D678)/@?``%K;/__5``,`!%U0@*1?]@@^.4``-"E_$;Y`0``L*`.0/GB
M`Q0J(:`=D00`@-*@`#_60`8`-'5"`I'VDP"1X0``T"&@1_F@#D#Y(``_UO0#
M`"KA`Q:J(```D`!`(9&4``"48`4`M'-"`I'A``#0(?!&^6`.0/D@`#_6X```
MT`!H1_D``#_6X```T``D1_GB%T#Y`0!`^4(``>L!`(#2P04`5/U[0ZG@`Q0J
M\U-$J?5;1:G_@P&1P`-?UN```-``3$?Y\P``\```/]9B0@*10`P`^:`#`+3A
M``#0(8!'^2``/];@``#0`&Q'^4'0.]4B:&#X7U@D.:#__Q>@#D#YX0``T"$@
M1_D@`#_6R___%Z`*0/E@``"TT`@`E-/__Q?A`Q:J(```D`!@'Y%A``"4H`H`
M^:#Y_[3("`"4R___%X``@%*R__\7X0``T"$X1_D@`(!2(``_UN```-``F$?Y
M```_UA\@`]4?(`/5'R`#U1\@`]4=`(#2'@"`TN4#`*KA`T#YXB,`D>8#`)'@
M``#0`/A'^0,`@-($`(#22?__ET#__Y?@``#0`!!'^4```+0X__\7P`-?UA\@
M`]4?(`/5'R`#U1\@`]4?(`/5'R`#U>```/``(`*1X0``\"$@`I$_``#KP```
M5.$``-`A2$?Y80``M/`#`:H``A_6P`-?UN```/``(`*1X0``\"$@`I$A``#+
M(OQ_TT$,@8LA_$&3P0``M.(``-!"4$?Y8@``M/`#`JH``A_6P`-?UOU[OJG]
M`P"1\PL`^?,``/!@(D(Y0`$`-^$``-`A1$?Y@0``M.```/``!$#Y(``_UMG_
M_Y<@`(!28"(".?,+0/G]>\*HP`-?UA\@`]4?(`/5'R`#U=S__Q<?(`/5'R`#
MU1\@`]4?(`/5'R`#U1\@`]4?(`/5_7N[J>(``+!"]$?Y_0,`D?-3`:GY(P#Y
M60!`^;D&`+3U6P*I]@,!JA4`@%+W8P.I]P,`J@``0#D?]`!Q!!A`>B$#`%0S
M`T#Y$P(`M/@``+`86T?YM7Y`D_0#&:H?(`/5X@,5JN$#%ZK@`Q.J``,_UH``
M`#5@:G4X'_0`<2`"`%23CD#X\_[_M?5;0JGW8T.IX`,3JODC0/GS4T&I_7O%
MJ,`#7];U`Q>JH!Y`.!_T`'$$&$!ZH?__5+4"%TOB__\7E`(9R[4&`)%S`A6+
ME/Y#D]0"`+G@`Q.J^2-`^?5;0JGW8T.I\U-!J?U[Q:C``U_6$P"`TN?__Q<?
M(`/5'R`#U?_#`M'B``"P0B1'^>$C`)']>PFI_4,"D?-3`/GS`P"J0P!`^>-'
M`/D#`(#2X@``L$*P1_E``#_60`(`->$;0+D"`)12(0P4$C\00'$D$$)ZP`$`
M5.$``+`A)$?YXT=`^2(`0/EC``+K`@"`TN$!`%3S4T#Y_7M)J?_#`I'``U_6
M``"`4O3__Q?B``"P0F!'^>`#$ZI!`(!20``_UA\``''@%Y\:[/__%^```+``
MF$?Y```_UA\@`]4?(`/5'R`#U?]#`]'A``"P(21'^?U[":G]0P*1(@!`^>)'
M`/D"`(#2P```M/-3"JGS`P"J`0!`.8$!`#7S4TJIX```L``D1_GB1T#Y`0!`
M^4(``>L!`(#2@0L`5/U[2:G_0P.1P`-?UN$``+`AN$?Y(``_UO0#`*H@_O^T
M]V,,J?<``+#W0D?Y]5L+J>`#%*K@`C_6``<`M`%,0#D53`"1/[@`<6$``%2A
M!D`Y`?__-`!,0#D?N`!Q8`<`5/@``+`8[T?YX`,3J@`#/];V`P"JX`,5J@`#
M/];A``"P(:1'^=8"``O6"@`1UGY`D^`#%JH@`#_6Y@``L,;P1_GE`Q6JX@,6
MJO4#`*I!`(!2Y`,3J@,``/!CH!V1P``_UN(``+!"7$?YX2,`D>`#%:I``#_6
M'P0`,:```%3@&T"Y``P4$A\00''``P!4X0``L"'T1OG@`Q6J(``_UN$``+`A
MW$?YX`,5JB``/];@`Q2JX`(_UD#Y_[7A``"P(51'^>`#%*H@`#_6X0``L"%X
M1_G@`Q.J(``_UO-32JGU6TNI]V-,J:C__Q>@!D`Y'[@`<8'X_U2@"D`YP/;_
M-,'__Q?@`Q6JDO__E^7__Q?@``"P`)A'^?-3"JGU6PNI]V,,J0``/]8?(`/5
M'R`#U?_#`=']>P&I_4,`D?-3`JGT`P&J\P,`JO5;`ZGW8P2I^6L%J?MS!JD@
M`$`YX0``L"$D1_D?]`!Q]P``L/?N1_DB`$#YX@<`^0(`@-*4%I2:X`,4JN`"
M/];V`P"JX1,`D>`#$ZK__O^7X`@`M/4#`*K@`C_6WP(`ZZD'`%3X!X"Y^0``
ML#GW1_G[``"P>Z='^1CS?=-@`D`Y'_0`<0080'H@#0!4Y`,3JH,<0#A_]`!Q
M9!A`>J'__U2$`!/+A'Q`DS4#0/G`"@"1```$BV`#/]:@:CCXX`(`M"$#0/G@
M`Q.J(VAX^`$40#AA```Y/_0`<2080'H``0!4'R`#U1\@`]4"%$`X8AP`.%_T
M`'%$&$!Z@?__5*$'@%(``(#2810`.()J8#AB:"`X``0`D:+__S7@``"P`"1'
M^>('0/D!`$#Y0@`!ZP$`@-)A"`!4_7M!J?-30JGU6T.I]V-$J?EK1:G[<T:I
M_\,!D<`#7]8!`(#2@FIA.*)J(3@A!`"1HO__->S__Q?Y``"P.?='^14`@%(\
M`T#Y@`-`^:`%`+3@`QRJ`8Q`^.(#%2JU!@`1H?__M4(,`!%7?'V3X0``T""0
M0+F@`@`U^P``L'NG1_DB`(!2X`,7JB*0`+E@`S_6^@,`JH#Z_[3D``"PA`1'
M^;A^?9/C`Q>JX0,<JN(#&*J``#_6.@,`^:`&`!'U!P"Y7]L@^*'__Q?B``"P
M0C1'^>$#%ZK@`QRJ0``_UB`#`/GZ`P"J`/C_M/L``+![IT?YN'Y]D_'__Q<$
M`(#2GO__%Q<"@-+:__\7X```L`"81_D``#_6_7N]J0T"@M+]`P"1\U,!J?5;
M`JG_8RW+_P,"^?,#`*KC``"P8R1'^?4C`)'T`P&JXO^!TF``0/G@!PCY``"`
MTN$#%:KC``"P8Q1'^0```/``P!V18``_UA\``''-``!4!`0`4>(#`"J@RF0X
M(`,`->0"`#7@``"P`&Q'^4/0.]7B`Q2JX0,3JF!H8/CC``"P8S!'^6``/];@
M``"P`"1'^>('2/D!`$#Y0@`!ZP$`@-)A`P!4#0*"TO]C+8OS4T&I]5M"J?U[
MPZC``U_6X@,$*N,``+!CU$?Y5GQ`D^(#%JK@`Q6JX06`4F``/]9`_/^TX```
ML`!L1_E$T#O5XP,6JN(#%:KA`Q.J@&A@^.0``+"$E$?Y@``_UN#__Q?@``"P
M`)A'^0``/];]>[ZI(P``\`$``/#]`P"18^`BD?-3`:E4T#O5\P``L'-N1_GB
M``"P0G!'^2$`'I'D``"PA.!'^8!J<_B``#_6@&IS^`0``/#S4T&IA,`KD?U[
MPJCC`P2J\```L!!^1_D"``"P`0``\$(@/9$A@!Z1!0"`4@`"'];@``"P`&Q'
M^430.]7B``"P0G!'^2,``/#P``"P$.)'^6/@(I&`:&#X`0``\"$`'I$``A_6
M'R`#U1\@`]4?(`/5_7N]J0T$A-+]`P"1\U,!J?5;`JG_8RW+_P,"^>```+``
M)$?Y\R,`D10`@-(!`$#YX0\0^0$`@-+@``"P`!A'^0``/];D`P`JY@``L,;P
M1_DB`(+200"`4N`#$ZH%``#P`P``\*7@'I%C`!^1P``_UN,``+!C%$?Y``*"
MTO9C((L"`(+2X`,3JN$#%JI@`#_60`+X-^$``+`AI$?Y%00`$?,#`*JU?D"3
MX`,5JB``/];T`P"J(`$`M.0``+"$O$?Y<WY`D^,#%:KA`Q:JX@,3JH``/]:?
M:C,XX```L``D1_GB#U#Y`0!`^4(``>L!`(#2`0$`5`T$A-+_8RV+X`,4JO-3
M0:GU6T*I_7O#J,`#7];@``"P`)A'^0``/]8?(`/5N/__%_U[NJD-%H+2_0,`
MD?-3`:GU6P*I_V,MR_\#`OGT`P"JX@``L$(D1_GU`P&JX9,`D4``0/G@5PCY
M``"`T@```/``8!^1KOW_E_,#`*KB``"P0LA'^>`#%*KA!8!20``_UF`"`+3A
M`Q2J````\`"@'Y&+_O^7X`,4JN$``+`A)$?YXU=(^2(`0/EC``+K`@"`T@$2
M`%0-%H+2_V,MB_-30:GU6T*I_7O&J,`#7];@``"P`,!'^?=S"/GW``"P]PI'
M^?AW"/G[@PCYX0,`JOR'"/G@`Q6JX0\`^18``/`@`#_6P>(?D>`"/];\`P"J
M@`X`M.```+``7$?Y^*,"D?L``+![\T?YX`<`^?4``+"U[D?Y^7L(^>```+``
MM$?YX`L`^?I_"/F``T`YP`<`--,``+3B"T#YX0,<JN`#$ZI``#_6@`4`-.`#
M'*J@`C_6`@0`T?D#%:J"`P*+^@,`JI\#`NNC``!4"@``%%_T'SB?`P+K@```
M5$``0#D?O`!Q8/__5.`#'*J@`C_6^@,`JN`#%*H@`S_6```:BP`$`)$?_#_Q
M*`,`5`(`@M)!`(!2Y`,<JN8#%*K@`QBJ!0``\`,``/"E`""18R`@D6`#/];B
M!T#YX:,`D>`#&*I``#_6H```->`[0+D`#!02'R!`<4`#`%3!XA^1``"`TN`"
M/];\`P"J(/G_M?=S2/GX=TCY^7M(^?I_2/G[@TCY_(=(^9+__Q<S!`"T8`)`
M.1^X`''@``!4!```\#H`@-+Y``"P.>]'^9Q`'Y'2__\78`9`.2#]_S3X__\7
MX@``L$)@1_G@`QBJ(0"`4D``/]9`_/\UX0,8J@```/``H!^1"?[_E^$/0/G@
M`QBJ(``_UO=S2/GX=TCY^7M(^?I_2/G[@TCY_(=(^7;__Q<$``#P.@"`TOD`
M`+`Y[T?YG$`?D;7__Q?W<TCY^'=(^?N#2/G\ATCY9O__%^```+``F$?Y]W,(
M^?AW"/GY>PCY^G\(^?N#"/G\APCY```_UA\@`]4?(`/5_7N^J>$%@%+B``"P
M0CQ'^?T#`)'S"P#Y\P,`JD``/]8?``#Q8`:`FO,+0/G]>\*H\```L!#"1_D`
M`A_6'R`#U?U[O:D-`H+2_0,`D?]C+<O_`P+YX0``L"$D1_DB`$#YX@<(^0(`
M@-)`!P"T]1L(^?4#`*H!`$`YH08`-.$``+`A[$?Y\Q,(^?07"/D@`#_6`@0`
MD?,#`*I?!$#QR`<`5.,``+!C+$?Y]",`D>$#%:K@`Q2J8``_UF`&`-&``@"+
M'P`4ZZ@``%0'```4'_0?.!\`%.N```!4`0!`.3^\`'%@__]4X@``L$(\1_G@
M`Q2JX06`4D``/]8@!0"T'P`4ZV@$`%3A``"P(<!'^>`#%*H@`#_6X0``L"$D
M1_GC!TCY(@!`^6,``NL"`(#2804`5/,32/D-`H+2]!=(^?4;2/G_8RV+_7O#
MJ,`#7];U&TCYX```L``D1_GB!TCY`0!`^4(``>L!`(#2X0,`5/```+`0PD?Y
M````\`!`'Y$-`H+2_V,MB_U[PZ@``A_6'P``.=W__Q<``(#2W___%^```+``
M)$?YX@=(^0$`0/E"``'K`0"`TB$!`%3S$TCY````\/072/D`0!^1]1M(^?``
M`+`0PD?YZO__%^```+``F$?Y```_UN```+``F$?Y\Q,(^?07"/GU&PCY```_
MUA\@`]7_@P'1X```L``D1_G]>P&I_4,`D?-3`JD4``#P\Q,`D?5;`ZF40B"1
M]V,$J?DK`/GU``"PM?9'^0$`0/GA!P#Y`0"`TN$#$ZK@`Q2J;?S_E^`!`+2B
M`D#YXP>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#
M%*I?_/^78/[_M?4``+"U]D?Y%```\)2"()'A`Q.JX`,4JE?\_Y?@`0"TH@)`
M^>,'@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`Q2J
M2?S_EV#^_[7U``"PM?9'^10``/"4HB"1X0,3JN`#%*I!_/^7``(`M*("0/GC
M!X"Y0`P#BP$$0/E!>"/XX?[_M!\@`]4!"$#Y`00`^0`@`)&A__^UX0,3JN`#
M%*HR_/^70/[_M?4``+"U]D?Y%```\)3B()'A`Q.JX`,4JBK\_Y<@`@"TH@)`
M^>,'@+E`#`.+`01`^4%X(_CA_O^T'R`#U1\@`]4!"$#Y`00`^0`@`)&A__^U
MX0,3JN`#%*H:_/^7(/[_M?8``+#6]D?Y%```\)4"(9'A`Q.JX`,5JA+\_Y<@
M`@"TP@)`^>,'@+E`#`.+`01`^4%X(_CA_O^T'R`#U1\@`]4!"$#Y`00`^0`@
M`)&A__^UX0,3JN`#%:H"_/^7(/[_M?8``)#6]D?Y%0``T+5"(9'A`Q.JX`,5
MJOK[_Y<@`@"TP@)`^>,'@+E`#`.+`01`^4%X(_CA_O^T'R`#U1\@`]4!"$#Y
M`00`^0`@`)&A__^UX0,3JN`#%:KJ^_^7(/[_M?<``)#W]D?Y%0``T+9B'Y'A
M`Q.JX`,6JN+[_Y<@`@"TX@)`^>,'@+E`#`.+`01`^4%X(_CA_O^T'R`#U1\@
M`]4!"$#Y`00`^0`@`)&A__^UX0,3JN`#%JK2^_^7(/[_M?@``)`8]T?Y%@``
MT->"(9'A`Q.JX`,7JLK[_Y<@`@"T`@-`^>,'@+E`#`.+`01`^4%X(_CA_O^T
M'R`#U1\@`]4!"$#Y`00`^0`@`)&A__^UX0,3JN`#%ZJZ^_^7(/[_M?D``)`Y
M]T?Y%P``T/C"(9'A`Q.JX`,8JK+[_Y<@`@"T(@-`^>,'@+E`#`.+`01`^4%X
M(_CA_O^T'R`#U1\@`]4!"$#Y`00`^0`@`)&A__^UX0,3JN`#&*JB^_^7(/[_
MM?D``)`Y]T?Y&```T!@#(I'A`Q.JX`,8JIK[_Y<@`@"T(@-`^>,'@+E`#`.+
M`01`^4%X(_CA_O^T'R`#U1\@`]4!"$#Y`00`^0`@`)&A__^UX0,3JN`#&*J*
M^_^7(/[_M?D``)`Y]T?Y&```T!BC'Y'A`Q.JX`,8JH+[_Y<@`@"T(@-`^>,'
M@+E`#`.+`01`^4%X(_CA_O^T'R`#U1\@`]4!"$#Y`00`^0`@`)&A__^UX0,3
MJN`#&*IR^_^7(/[_M>$#$ZH```#0`$`BD6W[_Y?A`P"J8```M.#"(9%1_/^7
MX0,3J@```-``H"*19?O_E^$#`*J```"T````T```(Y%(_/^7X0,3J@```-``
M0".17/O_E^$#`*J``@"TH&(?D4#\_Y?@``"0`"1'^>('0/D!`$#Y0@`!ZP$`
M@-*!`@!4_7M!J8`"(9'U6T.I`0``T/-30JDAX".1]V-$J?DK0/G_@P&1+_S_
M%^$#$ZH```#0`(`CD4/[_Y?A`P"J0/W_M,""(9$G_/^7Y___%^```)``F$?Y
M```_UA\@`]4?(`/5_X,`T0```-#B``"00B1'^>$3`)']>P&I_4,`D0"`(9%#
M`$#YXP<`^0,`@-(M^_^7X`$`M```0#D?P`!Q!!A`>N`'GQKA``"0(21'^>,'
M0/DB`$#Y8P`"ZP(`@-+!``!4_7M!J?^#`)'``U_6``"`4O7__Q?@``"0`)A'
M^0``/];_`P71[`:`$ND``)`I)4?Y[0,$D?U[!ZG]PP&1ZP^`$NH``)!*94?Y
MX8L0J>(#`*KCDQ&I00"`4N,#`)'EFQ*IYY\`^>`'!*WB#P6MY!<&K>8?!ZTD
M`4#YY#<`^00`@-+D`P61Y),"J>T?`/GL+P@I0`%`^?Z#PCS_@\,\Y```D(2<
M1_G^`X`]_P>`/?Z#A#S_@X4\@``_UN`?@%+A``"0(3A'^2``/];]>[RI_0,`
MD?-3`:GU6P*I]V,#J?]#0-'_`P+Y_T,`T?\#`OGD0T"1XP``D&,D1_D!`(#2
M]```D)0"1_EE`$#YA00`^04`@-+S`P`J0@"`4H`"/]88!`#1X`,8ZP$_0)(`
M/$"2($2`VA@#`.N$!@!4]0``D+6.1_GV`P"1]P``L.$#&*K@`Q,J`@"`4H`"
M/]8?!`"QX`8`5`(!@-+A`Q:JX`,3*B(`H/*@`C_6'P0`,:`$`%0`?$"3'QP`
M\:D#`%0`(`#1P`(`JZ,``%09```4``0`T1\`%NO#`@!4`0!`^>(*0/E?``'K
M0?__5```%LL``!B+X0``D"$D1_GC0T"19`1`^2(`0/F$``+K`@"`TF$$`%3_
M0P"1_T-`D?-30:GU6T*I]V-#J?U[Q*C``U_6&$-`T1]#0+%!^O]4``"`DNW_
M_Q?@``"0`"1'^>)#0)%#!$#Y`0!`^6,``>L!`(#2X0$`5````-``0"21>___
ME^```)``)$?YXD-`D4,$0/D!`$#Y8P`!ZP$`@-*!``!4````T```))%P__^7
MX```D`"81_D``#_6X```D`"$1_G]>[^I_0,`D0``/];A``"0(1Q'^2``/]9`
M``"T``!`^?U[P:C``U_6_T,!T>$``)`A)$?Y_7L!J?U#`)'S4P*I]`,`JO5;
M`ZD5``#0MH(DD2(`0/GB!P#Y`@"`TN`#%JKA$P"1:?K_EX```+3S`P"J``!`
M.>`!`#7@``"0`"1'^>('0/D!`$#Y0@`!ZP$`@-(A!@!4_7M!J>$#%*J@@B21
M\U-"J?5;0ZG_0P&1/_O_%_<C`/G@`Q2J]P``D/?N1_G@`C_6]0,`JN`#$ZK@
M`C_6X0``D"&D1_FU`@"+M0H`D>`#%:H@`#_6YP``D.?P1_GF`Q.JX@,5JD$`
M@%+D`Q2J\P,`J@4``-`#``#0I>`?D6,@()'@`#_6X```D``D1_GB!T#Y`0!`
M^4(``>L!`(#2H0$`5/U[0:GA`Q.JX`,6JO-30JGU6T.I]R-`^?]#`9$6^_\7
MX```D`"81_GW(P#Y```_UN```)``F$?Y```_UO\#!]'B``"P0D``D>/#`9%$
MH`"1_7L6J?V#!9'A``"0(21'^>`/`/E()$&I\U,7J?,#`9%&G$6I]5L8J?1C
M`9'W8QFI^/,`D5Z`PCP@`$#YX*\`^0``@-*=_$"MZ"<$J0```-!%$$#YYI\%
MJ>$#&*I$-$#Y?G0`K0!@'Y%E"@#YA`H`^7\(@#T$^O^7@```M`$`0#GU`P"J
M`2H`->```)``A$?Y^6L:J?MS&ZG@%P#Y```_UN$``)`A'$?Y(``_UL`3`+0;
M`$#YFQ,`M/H``)!:[T?YX`,;JD`#/]87^'_3]P8`D?4``)"UID?YX`,7JOD#
M%ZJ@`C_6X!,`^60#0#G\`P"JQ"@`-/8``)#6\D?Y%```T)3B))'B`Q>JX`,<
MJD$`@%+C`Q2JP`(_UIP+`)$_`Q?KX`H`T2$CEYID'T`X-P`7R_<*`)'_`@'K
M]R*!FO<"`<OW`@"+!/[_-0```-#WXP&1```CD0,``!3@AD#X0`(`M.$#&*K(
M^?^7]`,`JF#__[0!`$`Y(?__-`/Z_Y?@_O\TX0``D"'`1_G@`Q2J(``_UO<#
M`*H``@"U'R`#U1\@`]4?(`/5]```D)3"1_EW`D#Y-P$`M.`"0#D@"P`TX`,7
MJO'Y_Y<`"@`U=P9`^7,B`)$W__^UX`,7JD`#/];Y$T#Y\P,`JAP``-#@`QFJ
M0`,_UG0"``N4$A`1E'Y`D^`#%*J@`C_6YP,9JN0#%ZKY``"0.=-'^>(#%*J%
M`R"1!@``T`,``-#&`"618R`ED?<#`*I!`(!2P`(_UN`#%ZH!.(!2(`,_UA\$
M`#'A``!4X```D`"(1_D``#_6!0!`N;]$`'&!&0!4X@``D$*P1_GA@P*1X`,7
MJD``/];S`P`JH```->"S0+D`#!02'Q!`<4`)`%3A#T#Y`@``T.```)``9$?Y
MY`,7JB,`0/E"8":1``!`^4$`@%+E``"0I9!'^:``/]85`(#2X```D``D1_GB
MKT#Y`0!`^4(``>L!`(#2@20`5/EK6JG@`Q6J^W-;J?U[5JGS4U>I]5M8J?=C
M6:G_`P>1P`-?UN`#%ZIS(@"1@`(_UO<#`*K`]/^TK___%Q<`@-*M__\7]0``
MD+7"1_F``D#Y(`$`M.$#&*I3^?^78```M`$`0#E!`@`U@`9`^90B`)$@__^U
M]0``D+6F1_F@`8#2N0&`TO<#&:KZ``"06N]'^:`"/]8&``#0_`,`JMO`))%D
M"H!2X!,`^5O__Q>@`C_6E"(`D?L#`*J@_/^T^@``D%KO1_E``S_6%_A_T_<&
M`)%(__\7X!=`^?N[0+D``#_6?P,`:V'V_U3@LT"Y`"``$A\`!W'A]?]4X`,4
MJJ`"/];A`QBJ]0,`J@```-``P">1)?G_E^$#`*K@$`"TX`]`^0``0/EA^_^7
M^P,`JD`0`+3@`QNJ0`,_UN(#`*H?#`!Q30(`5.,``)!CM$?Y`'Q`D^(/`JD`
M$`#1`0``T&`#`(LAX">18``_U@`!`#7B$T#Y7Q``<2`5`%1"%`!18,MB.!^\
M`'&@%`!4X0,8JA@``-``@R&1!/G_EZ```+0``$`Y'\``<0080'H!`0!4X@``
MD$+H1_G@`QNJ`0"`4D``/];[`P`JX`L`-?L``)![&T?Y`(,AD0$``-`AX".1
MVOG_EY@#()$:``#08`,_UN8#`"KE`QBJY`,7JN(#%*K@`Q6J1\,KD0,``-!!
M`(!28Z`HD<`"/]8```#0`.`HD?P``)"<BT?YX`\`^1(``!2``S_6``!`N1]$
M`'%A`@!48`,_UG,&`!'C#T#Y0<,KD>$#`/GF`P`JYP,3*N4#&*KD`Q>JX@,4
MJN`#%:I!`(!2P`(_UN`#%:H!.(!2(`,_UA\$`#%@_?]4X0``D"'<1_G@`Q>J
M(``_UN$#%:H```#0`&`?D:GY_Y?@`Q6J1_[_ET[__Q?A#T#YY`,7JN```)``
M9$?Y`@``T",`0/E"8"61``!`^4$`@%+F``"0QI!'^<``/]9`__\7./[_E^``
M`)``)$?YXJ]`^0$`0/E"``'K`0"`TL$,`%3]>U:IX`,5JO-35ZGU6UBI]V-9
MJ?\#!Y'P``"0$,)'^0`"'];V``"0UO)'^<[^_Q?@#T#Y&P!`^7W__Q>K_?^7
MY`,`J@#T_[?C``"08P!'^>,3`JD!*`#1^@``D%J/1_D"`(!2X`,;*F``/];A
M@P21X`,;*L(`@-+A#P#Y0`,_UA\8`/'D%T#YX8]!J:$&`%3B(T&Y`&"(4B!H
MJ')?``!K(?'_5"$(0'D`J8A2/P``:Z'P_U2!R`#1`@"`4N`#&RKSHP218``_
MUN`#&RKA`Q.J`@6`TD`#/]8?H`#Q(00`5.8#$ZJ%`R"1X@,4JN0#%ZK@`Q6J
M!P``T`,``-#GP"N18V`HD4$`@%+_0P4YP`(_UIS__Q?@#T#Y`@1`^4+K_[3@
M`P*JX@\`^4`#/]8?#`#QJ>K_5.(/0/D`$`#1`0``T"$`*)%```"+XA=`^4``
M/]8?``!QX@]`^7L3@II*__\7X```D``D1_GBKT#Y`0!`^4(``>L!`(#2@0``
M5````-``("B1+_W_E^```)``F$?Y```_UN```)``F$?Y^6L:J?MS&ZD``#_6
M_T,!T0```-#B``"00B1'^0``(9']>P&I_4,`D?-3`JE4T#O5\P``D'-N1_GU
M6P.I]A,`D>$#%JKW8P2IE6IS^$,`0/GC!P#Y`P"`TJ+"0CFBQ@(Y*?C_EX``
M`+0!`$`Y/\0`<:`-`%2D^_^7E6IS^.`#%:KW``"0]RI'^0$``-!C`(!2(4`I
MD2(`@%+@`C_6]0,`JJ`)`+2A"D#Y`@``T(!J<_A"8"F1(0!`^?@``)`8,T?Y
M``,_UJ`*0/D!`$#Y(`Q`N<`+L#>`:G/X`0``L&,`@%(A@"F1(@"`4N`"/];W
M``"0]0,`JF`!`+3A`Q:J````L`"@'Y$`^/^7H0I`^>(#`*HA`$#YX`@`M(!J
M<_@``S_6PP``\&/X1ODB`(!2E6IS^`$``+`AH"F1X`,5JF``/]8"``"PX0,`
MJD(`*I'@`Q6J``,_UH)J<_CT0@*1X0,6J@```+``8!^17\0".>;W_Y>`"@#Y
M\P,`JD`#`+3"``#P0M!'^0$X@%+@`Q.J0``_U@$$`!$_!`!Q!(A1>F$&`%3`
M``#P`"1'^>('0/D!`$#Y0@`!ZP$`@-(A!0!4_7M!J?-30JGU6T.I]V-$J?]#
M`9'``U_6V```\!@S1_F^__\7@`9`^:+]_Y>`"@#Y\P,`JF#\_[7```#P`"1'
M^>('0/D!`$#Y0@`!ZP$`@-*!`@!4P0``\"'$1_D```"P`"`JD2``/]8`!$`Y
MH/+_-)+__Q?B0@*1X`,!JD$$0/DA`$#Y,?G_E[?__Q>`:G/XP@``\$*H1_E`
M`#_6G___%\```/``F$?Y```_UL$``/`A)$?YXP=`^2(`0/EC``+K`@"`TN'^
M_U3W0@*1XP,`*L0``/"$Q$?Y````L.$&0/GB`Q.J`,`JD2$`0/F``#_6_\,`
MT<$``/`A)$?Y_7L!J?U#`)'S4P*I\P,`J@```+`B`$#YX@<`^0(`@-(`@"&1
MX1,`D8GW_Y<@`0"T``!`.1_``'$$&$!ZX`>?&G\"`/'A!Y\:/P``:H$!`%3`
M``#P`"1'^>('0/D!`$#Y0@`!ZP$`@-+!!`!4_7M!J?-30JG_PP"1P`-?UF`"
M0#F`_O\TX`,3JH'Z_Y?"``#P0CQ'^?0#`*KA!8!20``_UA\``/'!``#P(<!'
M^8`&@)H@`#_6PP``\&-81_D!``"P@@"`TB$`)9%@`#_6(/S_-<```/``)$?Y
MX@=`^0$`0/E"``'K`0"`TL$``%3]>T&IX`,3JO-30JG_PP"1Q/?_%\```/``
MF$?Y```_UOU[OZD!X`"1!.`!D<,``/!C)$?Y_0,`D14$`$__0PK1EN0`;V4`
M0/GE1P'Y!0"`TO?E`&_B`P"16N0`;^,#`)$[?$!M/03!/'L[GPZ_!V!OO@=8
M;[P'2&]["R`N_5=(3_]76$_>'SI.>#N53KT?-D[_'S=.WA^\3K\?OT[Y'[Y.
M>(R?3#\`!.O!_?]4^1_`/4*@"9'XY0!OX:,!D?_[0*W]\T&M^^M"K2%``)$@
M@-@\(0#?//4?/&[?'[Y.(!P@;KX?O4Z='[Q.?!^[3A4<-6Y;'[I..A^Y3K8&
M86^U5D%/U1ZU3KD>.$XY@(`\7P`!Z\']_U0&)$"I)3./T@<@0:E%4*ORX3=`
MJ21TG=(*$$#YPOQ;TP,!!\I"%`:J8P`)BBO]0M-C``C*(0`%BR$`"HMB0"*+
M0@`!"VMY":IN?4"2K0$%BT'\6]/,`0?*(10"JHP!!HHA``V+C`$'RNU#0:G#
M_$+3C`$(BV-X!JHA``P+:P$#RFL!`HI/_$+3+/Q;TVL!#LJM`06+C!4!JHP!
M#8MK`0>+BP$+"^)Y`JIO?$"28P`"RF,``8IM_5O3#`(%BV,`#\JM%0NJ8P`.
MBZT!#(LL_$+3[A-`^:T!`PN!>0&J)-NM\JS]6]-#``'*S@$%BXP5#:I"?$"2
MC`$.BV,`"XKNPT*I8P`"RF,`#XMO_4+3@P$#"^MY"ZHO?$"2(0`+RFS\6],A
M``V*S@$%BXP5`ZHA``_*C`$.BR$``HNB_4+3@0$!"T)X#:IL`0+*;GU`DHP!
M`XHM_%O3"P(%BXP!#LJM%0&JC`$/BZT!"XMO_$+3\!]`^:L!#`OC>0.J3WQ`
MDD(``\IL_5O30@`!B@T"!8N,%0NJ0@`/RHP!#8M"``Z++?Q"TX(!`@OP(T#Y
MH7D!JFP``<IN?$"2C`$+BDW\6],#`@6+C`$.RJT5`JJ,`0^+K0$#BV_]0M/P
M)T#YHP$,"^MY"ZHO?$"2;/Q;TR$`"\H-`@6+(0`"BHP5`ZHA``_*C`$-BTW\
M0M.B>0*J(0`.BX$!`0ML`0+*;GU`DHP!`XJ,`0[*+?Q;TXP!#XNM%0&J\#]%
MJ0L"!8MP_$+3`WH#JJT!"XM0?$"20@`#RJL!#`M"``&*0@`0RNT!!8ML_5O3
M0@`.B^Y'1JF,%0NJC`$-BRW\0M."`0(+H7D!JFP``<IO?$"2C`$+BDW\6]/#
M`06+C`$/RJT5`JJ,`1"+K0$#BV/]0M.N`0P+:W@+JB,`"\HP?$"28P`"BL']
M6],L`@6+(10.JF,`$,HA``R+8P`/BTS\0M,A``,+@GD"JN-'1ZEL`0+*;WU`
MDHP!#HHM_%O3C`$/RJT5`:J,`1"+4'Q`DF,`!8NM`0.+P_U"TZT!#`MC>`ZJ
M0@`#RBP"!8NK_5O30@`!BFL5#:I"`!#*:P$,BT(`#XMO`0(++/Q"T^)'2*F!
M>0&J:P`!RFY\0))K`0V*[/U;TVL!#LJ,%0^J:P$0BS!\0))"``6+C`$"BZ+]
M0M.,`0L+0G@-JB$``LHK`@6+@_U;TR$`#XIC%`RJ(0`0RB$`#HMC``N+;0`!
M"^']0M/N2T#Y(7@/JDL``<JC_5O30GQ`DFL!#(IC%`VJ:P$"RLX!!8MR`1"+
MPT$CBXO]0M-Q`!(+;'D,JNY/0/DK``S*+_Y;TR%\0)+O%1&J:P$-BFL!`<K%
M`06+?@$"BZ)`+XM/`!X+I?U"T^Y30/FM>`VJ\/U;TV,`$HL0%@^JD@$-RE("
M`TK#`02+4@(!BV%`,(LP`!(+(_Y"T^M70/EE>!&J#OY;TZ,!!<K.%1"J0@`>
MBVL!!(MB``)*3$`LBV)!+HM1``P+X_U"T^M;0/EC>`^J+OY;TZ\``\K.%1&J
M(0`2B^$!`4IK`02++4`MBV%!+HLK``T+#_Y"T^Y?0/GP>1"J;_U;TT(`#(OO
M%0NJ;``0RH(!`DK.`02+14`EB\)!+XM>``4++/Y"T^YC0/F,>1&JS_];TR$`
M#8OO%1ZJ#0(,RJ$!`4K-`02+(T`CBZ%!+XLQ``,+;?U"T^YG0/FK>0NJ+_Y;
MTT(`!8OO%1&JA0$+RJ(``DK.`02+34`PB\)!+XM2``T+Q?]"T^YK0/FE>!ZJ
M3_Y;TR$``XOO%1*J8P$%RF$``4K.`02++$`LB\%!+XLP``P+(_Y"T^YO0/EC
M>!&J#_Y;TT(`#8OO%1"JK0`#RJ(!`DK.`02+2T`KB\)!+XM1``L+3?Y"T^YS
M0/FM>1*J+_Y;TR$`#(OO%1&J;``-RH$!`4K.`02+)4`EB\%!+XLR``4+#/Y"
MT^YW0/F,>1"J3_Y;TT(`"XOO%1*JJP$,RF(!`DK.`02+0T`CB\)!+XM0``,+
M*_Y"T^Y[0/EK>1&J#_Y;TR$`!8OO%1"JA0$+RJ$``4K.`02++4`MB\%!+XLQ
M``T+1?Y"T^Y_0/FE>!*J+_Y;TT(``XOO%1&J8P$%RF(``DK.`02+3$`LB\)!
M+XM.``P+`_Y"T^^#0/EC>!"JT/U;TR$`#8L0%@ZJK0`#RJ$!`4KM`02+*T`K
MBZ%!,(LP``L++_Y"T^V'0/GQ>1&J#_Y;TT(`#(OO%1"J;``1RH(!`DJL`02+
M14`EBX)!+XM,``4+S_U"T^V+0/GN>0ZJC_U;TR$`"XOO%0RJ*P(.RF$!`4JK
M`02+(T`CBV%!+XLK``,+#_Y"T^V/0/GP>1"JT@$0RF_]6]-"``6+Y14+JD("
M`DJM`02+3T`QBX+]0M-,>`RJI4$EB[$`#PLA``.+`P(,RJ4`#XOMDT#Y8P`!
M2F']0M,B_EO3*W@+JD(4$:J/`0O*K0$$B^\!!4JB02*+_D$PBW)`+HOMOU*I
M3@`2"R/^0M/!_5O38W@1JB$4#JJE`02+30`2BZ%`(8MB`0/*,``>"T(`#4KM
M`02+0D`LBP7^6],A`!Z+I100JJ4`#8NL``(+POU"T^V?0/E">`ZJ;@`"RH7]
M6]/!`0%*K0$$BR%`*XND%`RJ"_Y"TX0`#8MP>1"JCP`!"^6C0/E+`!"J`0("
M"FL!#(IK`0&JY/U;TX&;E]*$%`^J8>.Q\J4``8MC02.+A``%BX7]0M.-``,+
MY*=`^:QX#*H+`@RJA0$0"FL!#XJC_5O3:P$%JH0``8MC%`VJ8D$BBV,`!(OE
M_4+3Y*M`^:]X#ZIN``(+@@$/JN,!#`I"``V*R_U;TT(``ZIK%0ZJ@P`!BT5`
M,(MK`0.+L?U"TV(!!0OCKT#Y+7H-JNL!#:JD`0\*:P$.BD7\6]-C``&+:P$$
MJJ44`JID02R+I0`#B\O]0M/CLT#Y;GD.JJP`!`ND`0ZJQ0$-"H0``HJ+_5O3
MA``%JFL5#*IE``&+4/Q"TX-`+XMK`06+!'H"JN6W0/EP`0,+RP$$JH(`#@IK
M`0R*`_Y;TZ4``8MK`0*J8Q00JF)!+8MC``6+B_U"T^6[0/EL>0RJ;P`""X(`
M#*J#`00*0@`0BNO]6]-"``.J:Q4/JJ,``8L-_D+314`NBVL!`XNB>1"JX[]`
M^6T!!0N+`0*J3@`,"FL!#XJE_5O3:P$.JF,``8NE%`VJ9$$DBZ4``XOK_4+3
MX\-`^6]Y#ZJN``0+1``/JN4!`@J$``V*R_U;TX0`!:IK%0ZJ90`!BVL!!8N#
M0"R+L/U"TVP!`PODQT#Y#7H-JNL!#:JE`0\*:P$.BH/]6]-K`06JA``!BV,4
M#*K%_4+3KG@.JF)!(HMC``2+Q0$-"F,``@NB`0ZJ0@`,BI#]0M-"``6J:_Q;
MTT5`+XMK%0.JY#]9J0QZ#*J$``&+[P$!BVL!!(N$`0X*8@$%"\L!#*IK`0.*
M1?Q;TVL!!*JE%`*J9$$MBZ4`#XMK_$+3[=-`^:\`!`ME>0.J4/Q"TX0!!:JC
M``P*A``"BNO]6].M`0&+A``#JFL5#ZJ#0"Z+:P$-BP1Z`JIN`0,+JP`$JH(`
M!0IK`0^*:P$"JL/]6]-B02R+8Q0.JNVS6JGK_4+3K0$!BXP!`8MC``V+;0`"
M"V-Y#ZJ"``.J<``$"D(`#HJK_5O3S_U"TT(`$*IK%0VJ14`EBVL!#(OB>0ZJ
M;P$%"VL``JI.``,*:P$-BFL!#JKE_5O39$$DBZ44#ZKLKUNIC`$!BVL!`8NE
M``R+K/U"TZX`!`N->0VJ10`-JK`!`@JE``^*Q/U;TZ4`$*KL_4+3A!0.JH]Y
M#ZJC0".+A``+B^7G0/F,``,+I`$/JNL!#0J$``Z*@_U;TX0`"ZJE``&+@D`B
MBV,4#*K$_4+38P`%BX5X#JIK``(+XP$%JJX`#PIC``R*8OU;TV,`#JI"%`NJ
M<D`MBX/]0M/D-UVI;'@,JHX!!0J$``&+H0$!BX)`(HND``RJ4``2"X0`"XJ$
M``ZJ`_Y;TY%`+XMC%!"J(T`CBV']0M-D`!$+*W@+JH\!"\I!`!*+COQ;TV,`
M$8O"%02J[@$!2L5!)8O!.IC2[3M>J4%,N?(/_D+3\'D0JG$!$,HC`@-*K0$!
MB\X!`8NB02*+;$`LBTT`!0M"``6+K_U;T^\5#:K#02^+C_Q"TWX`#`OD>02J
M[OM`^<__6],%`@3*[Q4>JJ(``DK%`0&+2T`KBZ)`+XM2``L+I?U"T^[_0/FM
M>`VJ3_Y;TX4`#<KO%1*J8P`,B\X!`8NC``-*94`PB\-!+XMQ``4+S/]"T^X#
M0?F,>1ZJ+_Y;TT(`"XOO%1&JJP$,RF(!`DK.`0&+1$`DB\)!+XM0``0+2_Y"
MT^X'0?EK>1*J#_Y;TV,`!8OO%1"JA0$+RJ,``TK.`0&+;4`MB\-!+XMR``T+
M)?Y"T^X+0?FE>!&J3_Y;TT(`!(OO%1*J9`$%RH(``DK.`0&+3$`LB\)!+XM1
M``P+!/Y"T^X/0?F$>!"J+_Y;TV,`#8OO%1&JK0`$RJ,!`TK.`0&+:T`KB\-!
M+XMP``L+3?Y"T^X30?FM>1*J#_Y;TT(`#(OO%1"JC``-RH(!`DK.`0&+14`E
MB\)!+XM2``4++/Y"T^X70?F,>1&J3_Y;TV,`"XOO%1*JJP$,RF,!`TK.`0&+
M9$`DB\-!+XMQ``0+"_Y"T^X;0?EK>1"J+_Y;TT(`!8OO%1&JA0$+RJ(``DK.
M`0&+34`MB\)!+XM0``T+1?Y"T^X?0?FE>!*J#_Y;TV,`!(OO%1"J9`$%RH,`
M`TK.`0&+;$`LB\-!+XMR``P+)/Y"T^XC0?F$>!&J3_Y;TT(`#8OO%1*JK0`$
MRJ(!`DK-`0&+2T`KBZ)!+XM1``L+#?Y"T^XG0?FP>1"J+_Y;TV,`#(OO%1&J
MC``0RH,!`TK,`0&+94`EBX-!+XM^``4+3?Y"T^PK0?FM>1*JSO];TT(`"XO.
M%1ZJ"P(-RF(!`DJ+`0&+1$`DBV)!+HM,``0+*_Y"T^\O0?EQ>1&JCOU;TV,`
M!8O.%0RJI0$1RJ4``TKC`0&+LD`PBVY`+HO%`1(+R_]"T^,S0?EK>1ZJK_Q;
MTS`""\I"``2+Y!4%J@("`DIC``&+7D`MBV)`)(M#`!X+A/U"T^\W0?F,>`RJ
M9/Q;TW`!#,K.`1*+A!0#JA`"#DKN`0&+$4(QB\1!)(OO.T'YC@`1"ZW\0M.$
M`!&+K7D%JD4`'HN0`0W*POU;TQ`"!4I"%`ZJY0$!BPM"*XM"``6+9?Q"T^\_
M0?E"``L+HW@#JLQ`+(NE`0/*2_Q;TV85`JJD``1*X0$!B\7]0M.$``&+Q@`,
MBZ%X#JJE`0H+R@``T$HE1_E"``D+(0`'"V,`"`N$``8+!`@`J0$,`:D%$`#Y
MXD=!^4$!0/E"``'K`0"`TH$``%3_0PJ1_7O!J,`#7];```#0`)A'^0``/]8?
M(`/5'R`#U<$``-`AI$?Y_7N_J0`0@-+]`P"1(``_UA\8`/DB``"0(0``D!]X
M`+E=.,(](@``D#XTPCU?/,(]'G0`K1\(@#W]>\&HP`-?UOU[NZE"?$"3_0,`
MD?-3`:GT`P"J]5L"J?=C`ZGW`P*J^`,!JOEK!*D%C$*I`'A`N:0,`@N_``3K
M8Y2#FF)T0HN$B@*IP`0`-=D``-`Y+T?YE>(`D?_^`''-`@!4]GX&4_,#&*K:
M!@!1UF8:4Q8#%HL?(`/5'R`#U7UR0*W@`Q2J?WI!K7,"`9&]<@"MOWH!K9+[
M_Y=_`A;K`?__5!@#`9$`"(!2]P(!4?<:&DM88Z";X0,8JN`#%:KB?D"3(`,_
MUI=Z`+GS4T&I]5M"J?=C0ZGY:T2I_7O%J,`#7]83"(!2<P(`2W\"%VN5X@"1
MV0``T#DO1_ESTI<:H,(@BW9^0)/B`Q:J(`,_UH!Z0+E@`@`+@'H`N1\``7%A
M_?]4]P(32Q@#%HO@`Q2J;/O_E\K__Q<?(`/5'R`#U?U[O*GC#X`2_0,`D?-3
M`:GT`P"J\P,!JO5;`JDVX`"1[QL`_36`0JFB(@-3#P!GGD$$`!$@?$"3P\HB
M.,`"`(L_X`!QS0L`5,,``-!C_$?Y`@B`4D(``4L!`(!20GQ`DV``/];@`Q.J
M3OO_EQ\$`$]_@H,\W_X`K7\V`/GZ!7!__P5H?_L%>'^^`F>>X`,3JMP':'_=
M!W!_7P<#;MX'>']_!P5N_P4';I\'"6Z_!PMNWP<-;K\>#TY_.@#].?O_EV$"
M0/G@`Q.JT```T!#>1_DA_%C3@0(`.6$"0/DA_%#3@08`.6$"0/DA_$C3@0H`
M.6$"0/F!#@`Y809`^2'\6-.!$@`Y809`^2'\4-.!%@`Y809`^2'\2-.!&@`Y
M809`^8$>`#EA"D#Y(?Q8TX$B`#EA"D#Y(?Q0TX$F`#EA"D#Y(?Q(TX$J`#EA
M"D#Y@2X`.6$.0/DA_%C3@3(`.6$.0/DA_%#3@38`.6$.0/DA_$C3@3H`.6$.
M0/F!/@`Y81)`^2'\6-.!0@`Y81)`^2'\4-.!1@`Y81)`^2'\2-.!2@`Y81)`
M^8%.`#GO&T#]\U-!J?5;0JG]>\2H``(?UL,``-!C_$?Y`@>`4D(``4L!`(!2
M0GQ`DV``/]:I__\7'R`#U<,``/!C!$#Y/O'_%Q\@`]7]>[^I_0,`D?U[P:C`
M`U_6`0`"```````E<R\E<P```"]P<F]C+W-E;&8O97AE``!$>6YA3&]A9&5R
M.CIB;V]T7T1Y;F%,;V%D97(``````$EN=&5R;F%L<SHZ4$%2.CI"3T]4````
M`&5X90``````+W!R;V,O)6DO)7,``````"X`````````4$%27U1%35``````
M`````%!!4E]04D]'3D%-10`````Z`````````"\`````````)7,E<R5S``!0
M15),-4Q)0@``````````4$523$Q)0@!015),-4]05```````````4$523$E/
M``!005)?24Y)5$E!3$E:140`4$%27U-005=.140``````%!!4E]#3$5!3@``
M``````!005)?1$5"54<`````````4$%27T-!0TA%`````````%!!4E]'3$]"
M04Q?1$5"54<``````````%!!4E]'3$]"04Q?5$U01$E2`````````%!!4E]4
M35!$25(```````!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@``
M````````,0````````!L<V5E:R!F86EL960`````<F5A9"!F86EL960`````
M`$Q$7TQ)0E)!4EE?4$%42`!365-414T``"4P,G@`````<&%R+0`````E<R5S
M)7,E<P``````````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@<W5B9&ER96-T
M;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@```````"5S.B!P<FEV871E('-U
M8F1I<F5C=&]R>2`E<R!I<R!U;G-A9F4@*'!L96%S92!R96UO=F4@:70@86YD
M(')E=')Y('EO=7(@;W!E<F%T:6]N*0H```````!0051(`````'!A<FP`````
M+G!A<@````!S:&]R="!R96%D````````)7,E<V-A8VAE+25S)7,``"5S)7-T
M96UP+25U)7,````E<R5S=&5M<"TE=2TE=25S```````````8`````````'!E
M<FP`````,`````````!005(Z.E!A8VME<CHZ5D524TE/3@`````Q+C`V,P``
M`%5N86)L92!T;R!C<F5A=&4@8V%C:&4@9&ER96-T;W)Y```````````E<SH@
M8W)E871I;VX@;V8@<')I=F%T92!C86-H92!S=6)D:7)E8W1O<GD@)7,@9F%I
M;&5D("AE<G)N;STE:2D*```````````M90```````"TM````````55-%4@``
M``!54T523D%-10``````````5$U01$E2``!414U01$E2`%1%35``````5$U0
M```````O=&UP`````&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@
M)'LQ*R(D0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE
M;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F
M9G-E=%]S:7IE+"`D8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!
M1TE#(#T@(EQN4$%2+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@
M(R!P86-K*").(BD*)&-A8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'
M24,@/2`B4$M<,#`S7#`P-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@
M*"1F:"D@/2!`7SL*;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B
M=68["FUY("1S:7IE(#T@+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@
M*"1S:7IE+3$I("4@)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES
M(&$@;75L=&EP;&4@;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@
M>PIS965K("1F:"P@)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S
M:7IE("L@;&5N9W1H*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X
M*"1B=68L("1005)?34%'24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D[
M"GT*)'!O<R`M/2`D8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P
M87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*
M14Y$('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E
M;7`["G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E
M8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?
M=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=
M*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER
M(#H@=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@
M*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@
M/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2
M(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@
M/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M
M<"T^9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;
M(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@
M+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X
M<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M
M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M
M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3
M<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z
M("1F:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"
M3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@
M>PI?<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)5
M1WT["FUY("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I
M;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z
M35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@
M("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?
M<V5T7W!R;V=N86UE*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA
M;64B72D["E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-
M4'T@/2`B)$5.5GM005)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A
M<G1?;V9?1DE,15]S96-T:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!
M4DY?7WT@/2!S=6(@>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P
M<F]G;F%M92D@>PIO=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O
M9VYA;64B72D[("`C(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S
M($5R<FYO+G!M"FQA<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?
M<&%R7VUA9VEC*"I?1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ
M<5M#86XG="!F:6YD(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL
M92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!
M4B!M86=I8R!A="!P;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D
M;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#
M2$4B*2P@,#L*<F5A9"!?1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII
M9B`H)&)U9B!N92`B7#!#04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*
M;W5T<R@B3F\@8V%C:&4@;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*
M;W5T<RAQ<5M#86-H92!M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(
M+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(
M+"`D8G5F+"`D1DE,15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C
M:R@B3B(L("1B=68I.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%
M<R!I<R`D;V9F<V5T(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?
M;V9F<V5T7W-I>F4@+2`D;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T
M:6]N(#T@=&5L;"!?1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D
M8G5F+"`T.R`@("`@("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@
M9FER<W0@(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&
M2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I
M.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%
M("(D9G5L;&YA;64B+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S
M?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA
M;64L("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*
M<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB
M+"`D8G5F*3L*:68@*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z
M<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D
M8W)C)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE
M>R1F=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C
M:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A
M8VAE>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*
M;F%M92`]/B`D9G5L;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE
M?2LK.PI]"G)E860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B
M('L*;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M
M;V1U;&4@;W(@(21M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M
M;V1U;&5].PIR971U<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?
M;&ES='LD;6]D=6QE?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O
M+R1M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)
M3SHZ1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I
M;&4@;W(@9&EE(")#86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^
M8FEN;6]D92@I.PHD9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E
M:R@P+"`P*3L*<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@
M7W-A=F5?87,H(B1I;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I
M;F9O+3Y[9FEL97T@/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G
M+"`D9FEL96YA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z
M("0A73L*<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D
M.B!C86XG="!F:6YD(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U
M:7)E(%A33&]A9&5R.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R
M<#HZ2&5A=GD["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y
M.CI?:6YI=%]D>6YA;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE
M("AM>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=
M*2!["G5N;&5S<R`H)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O
M0E-$4$%.+RD@>PII9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q
M=6ER92`D9FEL96YA;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@
M)&9I;&5N86UE(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E
M("1R97%U:7)E7VQI<W1[)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:
M25!?34%'24,I('L*;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S
M96-T:6]N(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*
M)'-T87)T7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C
M('-T87)T(&]F('II<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD
M;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!0
M7U)%55-%?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@
M<')O8FQE;2D*:68@*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP
M87(M;W!T:6]N<R<@)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP
M("`@8FQI8E]T;U]P87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?
M<&%R"G,@("!S:6=N7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@
M86YD("1!4D=66S!=(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.
M5GM005)?05!07U)%55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO
M<FUA;"!P87)L(&)E:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!
M4D=6*2!["B1!4D=66S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO
M(&]R(&QA<W0["FEF("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@
M)#(["GT*96QS:68@*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L
M<VEF("@D,2!E<2`G02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L
M<VEF("@D,2!E<2`G3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G
M8B<I('L*)&)U;F1L92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2![
M"B1B=6YD;&4@/2`G86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE
M="`](#$["GT*96QS:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^
M(BP@)#(@;W(@9&EE('%Q6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=
M.PI]"F5L<VEF("@D,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS
M:&EF="A`05)'5BD["FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID
M96QE=&4@)$5.5GLG4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!0
M05(Z.D1I<W0["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6
M.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]
M"GT*=6YS:&EF="!`24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2![
M"GL*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E
M<75I<F4@1&EG97-T.CI32$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*
M;7D@)'II<#L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ
M<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A
M<B(Z("0A73L*8FQE<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI
M=F4Z.EII<"T^;F5W.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L
M("1P87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%
M<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?
M<&%R(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT
M96YT<R@G345402YY;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*
M)&UE=&$@/7X@<R]>7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H
M+BLI)"]M9SL*?0I].PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H
M+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE
M("(D<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE
M('%Q6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@
M)%!+6DE07TU!1TE#.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`[
M"FUY("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-2
M14%4*"D@?"!)3SHZ1FEL93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.
M0R@I+`HP-S<W+`HI(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U
M="(Z("0A73L*)&9H+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D
M;&]A9&5R.PII9B`H9&5F:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2![
M"G)E860@7T9(+"`D;&]A9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI]
M(&5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*
M:68@*"$D14Y6>U!!4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC
M(7Q<0')E;2DO*2!["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI0
M05(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*
M)&9H+3YP<FEN="@D;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L
M;V%D97(@=&\@(B1O=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@
M4$%2.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII
M;FET7VEN8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@
M>R`A+T)31%!!3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH
M)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D
M0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG
M+"!S?"\D?'P@9F]R96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK
M*R!F;W(@0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@
M)4E.0SL*;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[
M("`@("`@("`@(R!86%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N
M.PIF;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F
M:6QE+"`D;F%M92D["FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y
M.CI&=6QL0V%C:&5[)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL
M("1F8RD["GT*96QS92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H
M)&ME>2`]?B!M?%Y<421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I
M(#T@*"1K97DL("0Q*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R
M+UM>+UTK+R@N*BDD?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD
M,7TI('L**"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF
M("@M9B`B)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B
M+"`D,2D["FQA<W0["GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M
M93L*;F5X="!I9B`D=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F
M:6QE*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P
M<FEN=&8H<7%;4&%C:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D
M9FEL92T^>VYA;65](#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H
M)&9I;&4I*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL
M;V-A;"`D+R`]('5N9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@
M;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`]
M(#PD=&@^.PIC;&]S92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P
M;'DH7"1C;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!
M4E]615)"051)37T@86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*
M4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E
M;G0L("1F:6QE+"`D;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH
M)TXG+"!L96YG=&@H)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!
M<F-H:79E.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*
M<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE
M('%Q6T5R<F]R('=R:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A
M73L*;W5T<RAQ<5M7<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP
M*2!["B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ
M6FEP.CI!6E]/2PIO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O
M9B`B)&]U=")=.PI]"FEF("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I
M;G0H(EPP0TQ%04XB*3L*?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N
M86UE*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F
M:"P@,"P@,#L*)&-T>"T^861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*
M)&-A8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@
M/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C
M86-H95]N86UE("X](")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T
M:"`D8V%C:&5?;F%M92D["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!
M0TA%(BD["GT*;7D@)&]F9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A
M9&5R*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'
M24,I.PHD9F@M/F-L;W-E(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL
M97(@;V8@(B1O=70B.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*
M>PIL87-T('5N;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA
M;64H*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D
M97(H*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ
M6FEP.PI]"FUY("1F:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@
M("`@("`@("`@("`@("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)
M3SHZ2&%N9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I
M92!Q<5MF9&]P96XH*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#
M:'5N:U-I>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W
M.PHH)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]
M/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D
M:6YG('II<"!A<F-H:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS
M971#:'5N:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@
M)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H
M(D5X=')A8W1I;F<@>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%
M35!]*2![(",@<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H
M(&UY("1M96UB97(@*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB
M97(M/FES1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF
M:6QE3F%M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S
M:&QI8B\**#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI
M"B0*?7@["FUY("1E>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]
M($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T
M7VYA;64I.PII9B`H+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M
M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M
M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N
M86UE(ETI.PI](&5L<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?
M;F%M92(@=&\@(B1D97-T7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T
M5&]&:6QE3F%M960H)&1E<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/
M2R@I*0IO<B!D:64@<7%;17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O
M("(D9&5S=%]N86UE(ET["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>
M3R!E<2`B:'!U>"(["GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD
M<')O9VYA;65]*2!["F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@
M)#`@6R`M06QI8BYP87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C
M+G!A<B!=(%L@<')O9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE
M72!S<F,N<&%R"BX*)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@
M)#`@/2!S:&EF="A`05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA
M;64I(#T@0%\["G)E<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA
M;64L("1P871H+"`D97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H
M)&YA;64L("@G7"XN*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0
M871H.CIM:W!A=&@H)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H
M(&1I97,@=VET:"!E<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U
M:7)E(&QI8CL*<F5Q=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97([
M"G)E<75I<F4@<W1R:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A
M<G,["G)E<75I<F4@0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER
M92!%<G)N;SL*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP
M;W)T97(["G)E<75I<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q
M=6ER92!&:6QE.CI3<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O
M;F9I9SL*<F5Q=6ER92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR
M97%U:7)E($-O;7!R97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*
M<F5Q=6ER92!$:6=E<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2
M.CI(96%V>3L*<F5Q=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT
M97(Z.E!O9%-T<FEP.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E
M;G0["G)E<75I<F4@871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T[
M"F5V86P@>R!R97%U:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA
M<CHZ571I;"!].PIE=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R
M<W0@?3L*979A;"![(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@
M?3L*979A;"![(')E<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A
M;&%R('T["F5V86P@>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?
M=&5M<"!["FEF("AD969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!
M4E]414U0?2`]?B`O*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*
M?0IF;W)E86-H(&UY("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]4
M35!$25(@5$U01$E2(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-
M4"`O=&UP("X@*0HI('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@
M+60@)'!A=&@@86YD("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I
M9#L*979A;"![*"1P=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D
M/CM].PII9B`H(&1E9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E
M<FYA;64@/2`F5VEN,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D
M<'=U:60I('L*)'5S97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA
M;64@/2`D14Y6>U5315).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG
M.PI]"FUY("1S=&UP9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP
M86-K*")(*B(L("1U<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM
M>2`D8V%C:&5?9&ER.PII9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D
M:7(@/2`B=&5M<"TD)"(["GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B
M+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z
M("0A73L*:68@*"AM>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H
M*2D@/CT@,"D@>PIS965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E
M=%]S:7IE"BT@;&5N9W1H*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@
M)&9H+"`D8G5F+"!L96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP
M0T%#2$4B*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T
M7W-I>F4*+2!L96YG=&@H(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L
M(#`["G)E860@)&9H+"`D8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^
M(',O7#`O+V<["B1C86-H95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S
M92`D9F@["GT*:68@*"$D8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M
M<"TD)"(["B1%3E9[4$%27T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S
M>7-[7V1E;&EM?21C86-H95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*
M)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@
M/2`D,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@
M+R@N*RDO.PI]"G-U8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S
M+"`D;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M
M92(["G5N;&5S<R`H+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D
M9G5L;&YA;64N)"0B.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E
M;7!N86UE*0HF)B`H<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F
M:"D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*
M8VAM;V0@)&UO9&4L("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA
M;64H)'1E;7!N86UE+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I
M.PI]"G)E='5R;B`D9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII
M9B`H9&5F:6YE9"`D14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!2
M3T=.04U%?2`]?B`O*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N
M86UE('Q\/2`D,#L*:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O
M9VYA;64L("1%3E9[4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U
M8G-T<B@D<')O9VYA;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM
M?2D@*R`Q*3L*?0II9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D
M<')O9VYA;64L("1S>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D
M9F@L("<\)RP@)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@
M*"US("(D<')O9VYA;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y
M<WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O
M7%$D<WES>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X
M:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0
M?3L*)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE
M(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA
M<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE
M?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B
M*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]
M"GT*<W5B(%]F:7A?<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.
M5GM005)?4%)/1TY!345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D
M96QI;7TI(#P@,"D@>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G
M;F%M92(["GT*;7D@)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#
M=V0Z.F=E=&-W9"@I"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I
M;C,R.CI'971#=V0H*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M
M92`]?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D
M96QI;7TO.PHD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B
M(%]P87)?:6YI=%]E;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK
M(#T](#$@*2!["G)E='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:
M141](#T@,CL*?0IF;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@
M0T%#2$4@4%)/1TY!344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]
M"F9O<B`H<7<O(%1-4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB
M4$%27R1?(GT@/2`D14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%
M3E9[(E!!4E]'3$]"04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*
M9&5L971E("1%3E9[4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N
M(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB
M.PI]"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR
M97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D
M0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP
M('-I=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L
M:6)E>'`**3L*?0IP86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T
M($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S
M*3L*9&EE('%Q6W!A<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R
M;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@
M+64@)'!R;V=N86UE.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF
M("@D0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*
M?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF
M("@D.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z
M.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH`<&5R;'AS:2YC
M``````````!#04-(10```2-%9P````")J\WO`````/[<NI@`````=E0R$```
M``#PX=+#`````````````````1L#.PP!```@````@(/__[`&````AO__)`$`
M`&"&__\X`0``D(;__TP!``#,AO__8`$``""'__^$`0``0(?__Y@!``!`B/__
M``(```")__\H`@``X(K__X@"``!`C?__S`(``&B.__\``P``X([__R0#```@
MC___.`,``$"0__]L`P``1)#__X`#``!`D___``0``("3__\D!```0)7__Y`$
M``!@FO__S`0``.":___P!```B)O__PP%```PG?__4`4``&"=__]P!0``J)[_
M_\`%``!(IO__1`8``.BH__]\!@```*K__^`&``#@NO__#`<``"B[__\L!P``
M8+S__V@'``!`OO__H`<````````0``````````%Z4@`$>!X!&PP?`!`````8
M````U(3__S0`````00<>$````"P````@A?__,``````````0````0````#R%
M__\\`````````"````!4````9(7__T@`````00X@G02>`T*3`D[>W=,.````
M`!````!X````E(7__P0`````````9````(P```"@A?__^`````!!#E"="IX)
M19,(E`>9`D.6!94&0Y@#EP17UM5!V-=$WMW9T]0.`$$.4),(E`>5!I8%EP28
M`YD"G0J>"4_6U4'8UT+>W=G3U`X`00Y0DPB4!YD"G0J>"0`D````]````#B&
M__^T`````$$.L`%$G02>`T*3`E@*T]W>#@!!"P``7````!P!``#0AO__V`$`
M``!!#M`!0YT(G@=&E`63!D34TTG=W@X`00[0`9,&E`6=")X'1I@!EP)#E@.5
M!`)#"M330=;50=C700M)T]35UM?80Y0%DP9!E@.5!$&8`9<"0````'P!``!0
MB/__8`(```!!#G!!G0R>"T*3"I0)1I4(E@>7!I@%F02:`YL"G`$"4`K;W-G:
MU]C5UM/4W=X.`$$+```P````P`$``&R*__\H`0```$$.,)T&G@5%#L`@DP24
M`Y4"E@%I"@XP0][=U=;3U`X`00L`(````/0!``!@B___>`````!!#B"=!)X#
M19,"E`%/WMW3U`X`$````!@"``"TB___-``````````P````+`(``."+__\<
M`0```$$.,)T&G@5%#M!`DP24`Y4"E@%Y"@XP1-[=U=;3U`X`00L`$````&`"
M``#,C/__!`````````!\````=`(``+R,___T`@```$$.8)T,G@M%#I`BDPJ4
M"94(E@=A"@Y@0][=U=;3U`X`00M#EP9#F`5!FP)"G`%1F01$F@-Y"M=!V$'9
M0=I!VT'<00M;"M=!V$'90=I!VT'<00M&V=I!UT'80=M!W$27!D&8!4&9!$&:
M`T&;`D&<`2````#T`@``.(___SP`````00X@G02>`T63`D;>W=,.`````&@`
M```8`P``5(___[P!````00XPG0:>!4,.P"!(E0)&DP1!E`-JTT+40=5!#C!!
MWMT.`$$.P""5`IT&G@5!U4T.,$'>W0X`00[`(),$E`.5`IT&G@5,"M-"U$+5
M0PM#T]350Y,$090#094"`#@```"$`P``J)#__Q@%````00Y@0YT*G@E"DPB4
M!T:5!I8%EP28`YD"`RT!"MG7V-76T]3=W@X`00L``"````#``P``C)7__X``
M````00X@19T"G@%4"MW>#@!!"P```!@```#D`P``Z)7__Z@`````00[``D6=
M&IX9``!```````0``'26__^H`0```$$.0)T(G@=%#L"`!),&E`65!)8#EP*8
M`4(.T(`$?@H.P(`$00Y`1-[=U]C5UM/4#@!!"QP```!$!```V)?__S``````
M0PX0G0*>`4C>W0X`````3````&0$``#HE___2`$```!!#E!#G0B>!T*3!I0%
M0I4$E@-9"M76T]3=W@X`00M!EP)FUT'5UM/4W=X.`$$.4),&E`65!)8#G0B>
M!T.7`@"`````M`0``."8__^@!P```$$.P`-%G0R>"T:3"I0)194(E@>7!I@%
M69H#F01!G`&;`@*3"MK90MS;1=?8U=;3U-W>#@!!"P*PV=K;W$[7V-76T]3=
MW@X`0P[``Y,*E`F5")8'EP:8!9D$F@.;`IP!G0R>"P)<V=K;W$.:`YD$09P!
MFP(T````.`4``/R?__^@`@```$$.4$6=")X'0I,&E`5$E026`T.7`I@!`F(*
MU]C5UM/4W=X.`$$+`#````!P!0``9*+__Q@!````00XP0YT$G@-"DP*4`5L*
MT]3=W@X`00M@"M/4W=X.`$$+```L````I`4``,A\__]T`@```$$.8$.=!IX%
M2),$E`.5`I8!`F8*U=;3U-W>#@!!"P`H````U`4``!BC___8$````$$.$)T"
MG@%'#J`%`RD$"@X00=[=#@!!"P```!P`````!@``S+/__T@`````0PX0G0*>
M`4[>W0X`````.````"`&``#TL___,`$```!!#E"="IX)0Y,(E`=#E0:6!9<$
MF`-#F0*:`6P*WMW9VM?8U=;3U`X`00L`-````%P&``#PM/__W`$```!!#D"=
M")X'0Y,&E`5#E026`T(%3P("90K>W=76T]0&3PX`00L````0````E`8``)BV
M__\,```````````````$````$`````$```!'3E4```````,````'````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````("````````#,'P````````$`````````!P,`
M```````!`````````!(#`````````0`````````<`P```````!T`````````
MI00````````,`````````.`;````````#0````````!,5P```````!D`````
M````</L!```````;``````````@`````````&@````````!X^P$``````!P`
M````````"`````````#U_O]O`````+`"````````!0````````#P#0``````
M``8`````````2`0````````*`````````,@$````````"P`````````8````
M`````!4````````````````````#`````````*#]`0```````@````````!@
M`````````!0`````````!P`````````7`````````(`;````````!P``````
M``#H$P````````@`````````F`<````````)`````````!@`````````'@``
M```````(`````````/O__V\``````0``"`````#^__]O`````(@3````````
M____;P`````"`````````/#__V\`````N!(```````#Y__]O``````\`````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````'``````````<`````````!P`````````'```
M`````(#[`0``````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````$!7````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M@!P`````````````````````````````"``"```````*4$%2+G!M"@``````
M````0%L```````#05P``````````````````P%@````````@6P```````"A;
M````````,%L````````X6P``````````````````"%L````````06P``````
M````````````2%L```````!'0T,Z("A'3E4I(#$T+C(N,2`R,#(U,#(P-P!'
M0T,Z("AC<F]S<W1O;VPM3D<@,2XR-RXP7W)C,2XW7V,X-S0S96$I(#$T+C(N
M,2`R,#(U,#(P-P``+G-H<W1R=&%B`"YN;W1E+F=N=2YB=6EL9"UI9``N:6YT
M97)P`"YG;G4N:&%S:``N9'EN<WEM`"YD>6YS='(`+F=N=2YV97)S:6]N`"YG
M;G4N=F5R<VEO;E]R`"YR96QA+F1Y;@`N<F5L82YP;'0`+FEN:70`+G1E>'0`
M+F9I;FD`+G)O9&%T80`N96A?9G)A;65?:&1R`"YE:%]F<F%M90`N;F]T92Y!
M0DDM=&%G`"YI;FET7V%R<F%Y`"YF:6YI7V%R<F%Y`"YD>6YA;6EC`"YG;W0`
M+F1A=&$`+F)S<P`N8V]M;65N=```````````````````````````````````
M```````````````````````````````````````````````````````+````
M!P````(`````````<`(```````!P`@```````"0````````````````````$
M````````````````````'@````$````"`````````)0"````````E`(`````
M```;`````````````````````0```````````````````"8```#V__]O`@``
M``````"P`@```````+`"````````E`$````````$``````````@`````````
M```````````P````"P````(`````````2`0```````!(!````````*@)````
M````!0````,````(`````````!@`````````.`````,````"`````````/`-
M````````\`T```````#(!````````````````````0``````````````````
M`$````#___]O`@````````"X$@```````+@2````````S@`````````$````
M``````(``````````@````````!-````_O__;P(`````````B!,```````"(
M$P```````&``````````!0````(````(````````````````````7`````0`
M```"`````````.@3````````Z!,```````"8!P````````0`````````"```
M```````8`````````&8````$````0@````````"`&P```````(`;````````
M8``````````$````%0````@`````````&`````````!P`````0````8`````
M````X!L```````#@&P```````!@````````````````````$````````````
M````````:P````$````&```````````<`````````!P```````!@````````
M````````````$````````````````````'8````!````!@````````"`'```
M`````(`<````````S#H``````````````````$````````````````````!\
M`````0````8`````````3%<```````!,5P```````!0`````````````````
M```$````````````````````@@````$````"`````````&!7````````8%<`
M``````"@00``````````````````$````````````````````(H````!````
M`@``````````F0````````"9````````#`$```````````````````0`````
M``````````````"8`````0````(`````````$)H````````0F@```````*@&
M```````````````````(````````````````````H@````<````"````````
M`+B@````````N*`````````@````````````````````!```````````````
M`````+`````.`````P````````!P^P$``````'#[````````"```````````
M``````````@`````````"`````````"\````#P````,`````````>/L!````
M``!X^P````````@````````````````````(``````````@`````````R```
M``8````#`````````(#[`0``````@/L````````@`@````````4`````````
M"``````````0`````````-$````!`````P````````"@_0$``````*#]````
M````8`(```````````````````@`````````"`````````#6`````0````,`
M```````````"``````````$``````(@````````````````````(````````
M````````````W`````@````#`````````(@``@``````B``!```````H````
M````````````````"````````````````````.$````!````,```````````
M`````````(@``0``````5`````````````````````$``````````0``````
M```!`````P````````````````````````#<``$``````.H`````````````
5```````!````````````````````
