"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UsePolling = void 0;
exports.usePolling = usePolling;
var _react = require("react");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class UsePolling {
  constructor(fetchFunction, interval = 5000, onPollingSuccess, onPollingError, configurations) {
    this.fetchFunction = fetchFunction;
    this.interval = interval;
    this.onPollingSuccess = onPollingSuccess;
    this.onPollingError = onPollingError;
    this.configurations = configurations;
    _defineProperty(this, "data", null);
    _defineProperty(this, "error", null);
    _defineProperty(this, "loading", true);
    _defineProperty(this, "shouldPoll", false);
    _defineProperty(this, "intervalRef", void 0);
  }
  async fetchData(params) {
    this.loading = true;
    try {
      const result = await this.fetchFunction(params);
      this.data = result;
      this.loading = false;
      if (this.onPollingSuccess && this.onPollingSuccess(result, this.configurations)) {
        this.stopPolling();
      }
    } catch (err) {
      this.error = err;
      this.loading = false;
      if (this.onPollingError && this.onPollingError(this.error)) {
        this.stopPolling();
      }
    }
  }
  startPolling(params) {
    this.shouldPoll = true;
    if (!this.intervalRef) {
      this.intervalRef = setInterval(() => {
        if (this.shouldPoll) {
          this.fetchData(params);
        }
      }, this.interval);
    }
  }
  stopPolling() {
    this.shouldPoll = false;
    if (this.intervalRef) {
      clearInterval(this.intervalRef);
      this.intervalRef = undefined;
    }
  }
}
exports.UsePolling = UsePolling;
function usePolling(fetchFunction, interval = 5000, onPollingSuccess, onPollingError, configurations) {
  const [data, setData] = (0, _react.useState)(null);
  const [error, setError] = (0, _react.useState)(null);
  const [loading, setLoading] = (0, _react.useState)(true);
  const intervalRef = (0, _react.useRef)(undefined);
  const unmounted = (0, _react.useRef)(false);
  const shouldPoll = (0, _react.useRef)(false);
  const startPolling = params => {
    shouldPoll.current = true;
    const intervalId = setInterval(() => {
      if (shouldPoll.current) {
        fetchData(params);
      }
    }, interval);
    intervalRef.current = intervalId;
    if (unmounted.current) {
      clearInterval(intervalId);
    }
  };
  const stopPolling = () => {
    shouldPoll.current = false;
    clearInterval(intervalRef.current);
  };
  const fetchData = async params => {
    try {
      const result = await fetchFunction(params);
      setData(result);
      // Check the success condition and stop polling if it's met
      if (onPollingSuccess && onPollingSuccess(result, configurations)) {
        stopPolling();
      }
    } catch (err) {
      setError(err);

      // Check the error condition and stop polling if it's met
      if (onPollingError && onPollingError(err)) {
        stopPolling();
      }
    } finally {
      setLoading(false);
    }
  };
  (0, _react.useEffect)(() => {
    return () => {
      unmounted.current = true;
    };
  }, []);
  return {
    data,
    loading,
    error,
    startPolling,
    stopPolling
  };
}