from typing import Dict, Iterator, List, Union


def string_equals_operation(
    expected_value: Union[List[str], str], actual_value: str
) -> bool:
    if isinstance(expected_value, str):
        return actual_value == expected_value

    if isinstance(expected_value, list):
        return actual_value in expected_value

    return False


CONDITION_OPERATIONS = {"StringEquals": string_equals_operation}


class TrustCondition:
    """Single condition object"""

    def __init__(
        self,
        condition: str,
        expected_value: List[Union[List[str], str]],
        expected_value_source: List[str],
    ) -> None:
        if condition not in CONDITION_OPERATIONS:
            raise NotImplementedError(f"Unsupported condition: {condition}")

        self._condition = condition

        self._expected_value_source = expected_value_source
        self._expected_value = expected_value

    @property
    def context_keys(self) -> List[str]:
        return self._expected_value_source

    def verify_condition(self, actual_values: List[str]) -> bool:
        verify_action = CONDITION_OPERATIONS[self._condition]

        for index, actual_value in enumerate(actual_values):
            if not verify_action(self._expected_value[index], actual_value):
                return False

        return True


ConditionData = Dict[str, Dict[str, Union[str, List[str]]]]


class TrustRelationShipConditions:
    """Trust relationship conditions segment model"""

    def __init__(self, conditions: ConditionData) -> None:
        self._conditions: List[TrustCondition] = []
        for conditions_operation, condition_values in conditions.items():
            expected_value_source: List[str] = list(condition_values.keys())
            expected_value_data: List[Union[str, List[str]]] = list(
                condition_values.values()
            )

            self._conditions.append(
                TrustCondition(
                    conditions_operation, expected_value_data, expected_value_source
                )
            )

    def __iter__(self) -> Iterator[TrustCondition]:
        return self._conditions.__iter__()
