"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _constants = require("../../utils/constants");
var _helpers = require("./utils/helpers");
var _MDSEnabledClientService = require("./MDSEnabledClientService");
var _constants2 = require("./utils/constants");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class MonitorService extends _MDSEnabledClientService.MDSEnabledClientService {
  constructor(...args) {
    super(...args);
    _defineProperty(this, "createMonitor", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const createResponse = await client('alerting.createMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "createWorkflow", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const createResponse = await client('alerting.createWorkflow', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const response = await client('alerting.deleteMonitor', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const response = await client('alerting.deleteWorkflow', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          headers: _constants2.DEFAULT_HEADERS
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getMonitor', params);
        let monitor = _lodash.default.get(getResponse, 'monitor', null);
        const version = _lodash.default.get(getResponse, '_version', null);
        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);
        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);
        const associated_workflows = _lodash.default.get(getResponse, 'associated_workflows', null);
        if (monitor) {
          const client = this.getClientBasedOnDataSource(context, req);
          const aggsParams = {
            index: _constants.INDEX.ALL_ALERTS,
            body: {
              size: 0,
              query: {
                bool: {
                  must: {
                    term: {
                      monitor_id: id
                    }
                  }
                }
              },
              aggs: {
                active_count: {
                  terms: {
                    field: 'state'
                  }
                },
                '24_hour_count': {
                  date_range: {
                    field: 'start_time',
                    ranges: [{
                      from: 'now-24h/h'
                    }]
                  }
                }
              }
            }
          };
          const searchResponse = await client('alerting.getMonitors', aggsParams);
          const dayCount = _lodash.default.get(searchResponse, 'aggregations.24_hour_count.buckets.0.doc_count', 0);
          const activeBuckets = _lodash.default.get(searchResponse, 'aggregations.active_count.buckets', []);
          const activeCount = activeBuckets.reduce((acc, curr) => curr.key === 'ACTIVE' ? curr.doc_count : acc, 0);
          if (associated_workflows) {
            monitor = {
              ...monitor,
              associated_workflows,
              associatedCompositeMonitorCnt: associated_workflows.length
            };
          }
          monitor = {
            ...monitor,
            item_type: monitor.workflow_type || monitor.monitor_type,
            id,
            version
          };
          return res.ok({
            body: {
              ok: true,
              resp: monitor,
              activeCount,
              dayCount,
              version,
              ifSeqNo,
              ifPrimaryTerm
            }
          });
        } else {
          return res.ok({
            body: {
              ok: false
            }
          });
        }
      } catch (err) {
        console.error('Alerting - MonitorService - getMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getWorkflow', params);
        let workflow = _lodash.default.get(getResponse, 'workflow', null);
        const version = _lodash.default.get(getResponse, '_version', null);
        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);
        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);
        workflow.monitor_type = workflow.workflow_type;
        workflow = {
          ...workflow,
          item_type: workflow.workflow_type,
          id,
          version
        };
        return res.ok({
          body: {
            ok: true,
            resp: workflow,
            activeCount: 0,
            dayCount: 0,
            version,
            ifSeqNo,
            ifPrimaryTerm
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - getWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "updateMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body,
          refresh: 'wait_for'
        };
        const {
          type
        } = req.body;

        // TODO DRAFT: Are we sure we need to include ifSeqNo and ifPrimaryTerm from the UI side when updating monitors?
        const {
          ifSeqNo,
          ifPrimaryTerm
        } = req.query;
        if (ifSeqNo && ifPrimaryTerm) {
          params.if_seq_no = ifSeqNo;
          params.if_primary_term = ifPrimaryTerm;
        }
        const client = this.getClientBasedOnDataSource(context, req);
        const updateResponse = await client(`alerting.${type === 'workflow' ? 'updateWorkflow' : 'updateMonitor'}`, params);
        const {
          _version,
          _id
        } = updateResponse;
        return res.ok({
          body: {
            ok: true,
            version: _version,
            id: _id
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - updateMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getMonitors", async (context, req, res) => {
      try {
        const {
          from,
          size,
          search,
          sortDirection,
          sortField,
          state,
          monitorIds
        } = req.query;
        let must = {
          match_all: {}
        };
        if (search.trim()) {
          // This is an expensive wildcard query to match monitor names such as: "This is a long monitor name"
          // search query => "long monit"
          // This is acceptable because we will never allow more than 1,000 monitors
          must = {
            query_string: {
              default_field: 'monitor.name',
              default_operator: 'AND',
              query: `*${search.trim().split(' ').join('* *')}*`
            }
          };
        }
        const should = [];
        const mustList = [must];
        if (monitorIds !== undefined) {
          mustList.push({
            terms: {
              _id: Array.isArray(monitorIds) ? monitorIds : [monitorIds]
            }
          });
        } else if (monitorIds === 'empty') {
          mustList.push({
            terms: {
              _id: []
            }
          });
        }
        if (state !== 'all') {
          const enabled = state === 'enabled';
          should.push({
            term: {
              'monitor.enabled': enabled
            }
          });
          should.push({
            term: {
              'workflow.enabled': enabled
            }
          });
        }
        const monitorSorts = {
          name: 'monitor.name.keyword'
        };
        const monitorSortPageData = {
          size: 1000
        };
        if (monitorSorts[sortField]) {
          monitorSortPageData.sort = [{
            [monitorSorts[sortField]]: sortDirection
          }];
          monitorSortPageData.size = _lodash.default.defaultTo(size, 1000);
          monitorSortPageData.from = _lodash.default.defaultTo(from, 0);
        }
        const params = {
          body: {
            seq_no_primary_term: true,
            version: true,
            ...monitorSortPageData,
            query: {
              bool: {
                should,
                minimum_should_match: state !== 'all' ? 1 : 0,
                must: mustList
              }
            },
            aggregations: {
              associated_composite_monitors: {
                nested: {
                  path: 'workflow.inputs.composite_input.sequence.delegates'
                },
                aggs: {
                  monitor_ids: {
                    terms: {
                      field: 'workflow.inputs.composite_input.sequence.delegates.monitor_id'
                    }
                  }
                }
              }
            }
          }
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getMonitors', params);
        const totalMonitors = _lodash.default.get(getResponse, 'hits.total.value', 0);
        const monitorKeyValueTuples = _lodash.default.get(getResponse, 'hits.hits', []).map(result => {
          const {
            _id: id,
            _version: version,
            _seq_no: ifSeqNo,
            _primary_term: ifPrimaryTerm,
            _source
          } = result;
          const monitor = _source.monitor ? _source.monitor : _source;
          monitor['item_type'] = monitor.workflow_type || monitor.monitor_type;
          const {
            name,
            enabled,
            item_type
          } = monitor;
          return [id, {
            id,
            version,
            ifSeqNo,
            ifPrimaryTerm,
            name,
            enabled,
            item_type,
            monitor
          }];
        }, {});
        const monitorMap = new Map(monitorKeyValueTuples);
        const associatedCompositeMonitorCountMap = {};
        _lodash.default.get(getResponse, 'aggregations.associated_composite_monitors.monitor_ids.buckets', []).forEach(({
          key,
          doc_count
        }) => {
          associatedCompositeMonitorCountMap[key] = doc_count;
        });
        const monitorIdsOutput = [...monitorMap.keys()];
        const aggsOrderData = {};
        const aggsSorts = {
          active: 'active',
          acknowledged: 'acknowledged',
          errors: 'errors',
          ignored: 'ignored',
          lastNotificationTime: 'last_notification_time'
        };
        if (aggsSorts[sortField]) {
          aggsOrderData.order = {
            [aggsSorts[sortField]]: sortDirection
          };
        }
        const aggsParams = {
          index: _constants.INDEX.ALL_ALERTS,
          body: {
            size: 0,
            query: {
              terms: {
                monitor_id: monitorIdsOutput
              }
            },
            aggregations: {
              uniq_monitor_ids: {
                terms: {
                  field: 'monitor_id',
                  ...aggsOrderData,
                  size: from + size
                },
                aggregations: {
                  active: {
                    filter: {
                      term: {
                        state: 'ACTIVE'
                      }
                    }
                  },
                  acknowledged: {
                    filter: {
                      term: {
                        state: 'ACKNOWLEDGED'
                      }
                    }
                  },
                  errors: {
                    filter: {
                      term: {
                        state: 'ERROR'
                      }
                    }
                  },
                  ignored: {
                    filter: {
                      bool: {
                        filter: {
                          term: {
                            state: 'COMPLETED'
                          }
                        },
                        must_not: {
                          exists: {
                            field: 'acknowledged_time'
                          }
                        }
                      }
                    }
                  },
                  last_notification_time: {
                    max: {
                      field: 'last_notification_time'
                    }
                  },
                  latest_alert: {
                    top_hits: {
                      size: 1,
                      sort: [{
                        start_time: {
                          order: 'desc'
                        }
                      }],
                      _source: {
                        includes: ['last_notification_time', 'trigger_name']
                      }
                    }
                  }
                }
              }
            }
          }
        };
        const esAggsResponse = await client('alerting.getMonitors', aggsParams);
        const buckets = _lodash.default.get(esAggsResponse, 'aggregations.uniq_monitor_ids.buckets', []).map(bucket => {
          const {
            key: id,
            last_notification_time: {
              value: lastNotificationTime
            },
            ignored: {
              doc_count: ignored
            },
            acknowledged: {
              doc_count: acknowledged
            },
            active: {
              doc_count: active
            },
            errors: {
              doc_count: errors
            },
            latest_alert: {
              hits: {
                hits: [{
                  _source: {
                    trigger_name: latestAlert
                  }
                }]
              }
            }
          } = bucket;
          const monitor = monitorMap.get(id);
          monitorMap.delete(id);
          return {
            ...monitor,
            id,
            lastNotificationTime,
            ignored,
            latestAlert,
            acknowledged,
            active,
            errors,
            currentTime: Date.now(),
            associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[id] || 0
          };
        });
        const unusedMonitors = [...monitorMap.values()].map(monitor => ({
          ...monitor,
          lastNotificationTime: null,
          ignored: 0,
          active: 0,
          acknowledged: 0,
          errors: 0,
          latestAlert: '--',
          currentTime: Date.now(),
          associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[monitor.id] || 0
        }));
        let results = _lodash.default.orderBy(buckets.concat(unusedMonitors), [sortField], [sortDirection]);
        // If we sorted on monitor name then we already applied from/size to the first query to limit what we're aggregating over
        // Therefore we do not need to apply from/size to this result set
        // If we sorted on aggregations, then this is our in memory pagination
        if (!monitorSorts[sortField]) {
          results = results.slice(from, from + size);
        }
        return res.ok({
          body: {
            ok: true,
            monitors: results,
            totalMonitors
          }
        });
      } catch (err) {
        if ((0, _helpers.isIndexNotFoundError)(err)) {
          // Config index is not created unitl a monitor is created.
          return res.ok({
            body: {
              ok: false,
              resp: {
                totalMonitors: 0,
                monitors: [],
                message: "No monitors created"
              }
            }
          });
        } else {
          // If the index is created, some error in retrieving the monitors.
          console.error('Alerting - MonitorService - getMonitors', err);
          return res.ok({
            body: {
              ok: false,
              resp: err.message
            }
          });
        }
      }
    });
    _defineProperty(this, "acknowledgeAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const acknowledgeResponse = await client('alerting.acknowledgeAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "acknowledgeChainedAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id,
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, this.dataSourceEnabled, req, this.esDriver);
        const acknowledgeResponse = await client('alerting.acknowledgeChainedAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeChainedAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "executeMonitor", async (context, req, res) => {
      try {
        const {
          dryrun = 'true'
        } = req.query;
        const params = {
          body: req.body,
          dryrun
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const executeResponse = await client('alerting.executeMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: executeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - executeMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    //TODO: This is temporarily a pass through call which needs to be deprecated
    _defineProperty(this, "searchMonitors", async (context, req, res) => {
      try {
        const {
          query,
          index,
          size
        } = req.body;
        const params = {
          index,
          size,
          body: query
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const results = await client('alerting.getMonitors', params);
        return res.ok({
          body: {
            ok: true,
            resp: results
          }
        });
      } catch (err) {
        if ((0, _helpers.isIndexNotFoundError)(err)) {
          // Config index is not created unitl a monitor is created.
          return res.ok({
            body: {
              ok: false,
              resp: {
                totalMonitors: 0,
                monitors: [],
                message: "No monitors created"
              }
            }
          });
        } else {
          // If the index is created, some error in retrieving the monitors.
          console.error('Alerting - MonitorService - searchMonitor:', err);
          return res.ok({
            body: {
              ok: false,
              resp: err.message
            }
          });
        }
      }
    });
  }
}
exports.default = MonitorService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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